/*
 * static char *rcsid_spells_c =
 *   "$Id: spells.c,v 1.8 1993/04/25 16:33:14 frankj Exp $";
 */

/*
    CrossFire, A Multiplayer game for X-windows

    Copyright (C) 1992 Frank Tore Johansen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be reached via e-mail to frankj@ifi.uio.no.
*/

#include <global.h>
#ifndef __CEXTRACT__
#include <sproto.h>
#endif
#include <spells.h>
#include <object.h>

#define MAX_PET_MONSTERS 5
char mage_pet_monsters [MAX_PET_MONSTERS][16] = 
                {"bat","spider","stalker","beholder","dark_elf"};
int mage_num_called [MAX_PET_MONSTERS] = {2,1,1,2,3};
char priest_pet_monsters [MAX_PET_MONSTERS][16] = 
                {"bee","killer_bee","devil","angel","panther"};
int priest_num_called [MAX_PET_MONSTERS] = {3,2,2,2,5};
char altern_pet_monsters [MAX_PET_MONSTERS][16] =
                {"bird","pixie","skeleton","skull","vampire"};
int altern_num_called [MAX_PET_MONSTERS] = {1,1,2,1,1};

spell *find_spell(int spelltype) {
  if(spelltype<0||spelltype>NROFREALSPELLS)
    return NULL;
  return &spells[spelltype];
}

int check_spell_known(object *op,int sp) {
  int i;
  for(i=0;i<op->contr->nrofknownspells;i++)
    if(op->contr->known_spells[i]==sp)
      return 1;
  return 0;
}

/*
 * cast_spell():
 * Fires spell "type" in direction "dir".
 * If "ability" is true, the spell is the innate ability of a monster.
 * (ie, don't check for blocks_magic(), and don't add AT_MAGIC to attacktype.
 */

int cast_spell(object *op,int dir,int type,int ability,SpellTypeFrom item) {
  spell *s=find_spell(type);
  int success=0,bonus;
  if(s==NULL) {
    LOG(llevError,"Error, unknown spell: %d\n",type);
    return 0;
  }
  if(!IS_WIZ(op)&&op->type==PLAYER&&op->contr->shoottype==2&&
     op->stats.sp<s->sp && !(IS_SUMMON_SPELL(type)&&op->contr->golem!=NULL))
  {
    draw_info(op,"You don't have enough spellpoints.");
    op->contr->count_left=0;
    return 0;
  }
  if(!ability&&blocks_magic(op->map,op->x,op->y)) {
    if(op->type!=PLAYER)
      return 0;
    switch(op->contr->shoottype) {
    case 2:
      draw_info(op,"Something blocks your spellcasting.");
      break;
    case 3:
      draw_info(op,"Something blocks the magic of your wand.");
      break;
    case 4:
      draw_info(op,"Something blocks the magic of your scroll.");
      break;
    }
    return 0;
  }
  if(item == spellNormal && op->type==PLAYER&&s->cleric&&
     RANDOM()%100<s->level*2+cleric_chance[op->stats.Wis]) {
    draw_info(op,"You fumble the spell.");
    if(s->sp==0) /* Shouldn't happen... */
      return 0;
    return RANDOM()%(s->sp)+1;
  }

/*
 * This is a simplification of the time it takes to cast a spell.
 * In the future, the time will have to be spent before the
 * spell takes effect, and the caster can possibly be disturbed.
 * (maybe that should depend upon the spell cast?)
 */
  op->speed_left -= s->time * FABS(op->speed);

  switch(type) {
  case SP_BULLET:
    success=fire_arch(op,dir,find_archetype("bullet"),1);
    break;
  case SP_LARGE_BULLET:
    success=fire_arch(op,dir,find_archetype("lbullet"),1);
    break;
  case SP_S_FIREBALL:
    success=fire_arch(op,dir,find_archetype("firebullet_s"),!ability);
    break;
  case SP_M_FIREBALL:
    success=fire_arch(op,dir,find_archetype("firebullet_m"),!ability);
    break;
  case SP_L_FIREBALL:
    success=fire_arch(op,dir,find_archetype("firebullet_l"),!ability);
    break;
  case SP_HELLFIRE:
    success=fire_arch(op,dir,find_archetype("hellfire"),!ability);
    break;
  case SP_BURNING_HANDS:
    success=cast_cone(op,dir,5,SP_BURNING_HANDS,!ability);
    break;
  case SP_FIREBREATH:
    success=cast_cone(op,dir,14,SP_BURNING_HANDS,!ability);
    break;
  case SP_ICESTORM:
    success=cast_cone(op,dir,6,SP_ICESTORM,!ability);
    break;
  case SP_LARGE_ICESTORM:
    success=cast_cone(op,dir,14,SP_LARGE_ICESTORM,!ability);
    break;
  case SP_S_LIGHTNING:
    success=fire_bolt(op,dir,"lightning_s",!ability);
    break;
  case SP_L_LIGHTNING:
    success=fire_bolt(op,dir,"lightning_l",!ability);
    break;
  case SP_M_MISSILE:
    success=fire_arch(op,dir,find_archetype("magic_missile"),!ability);
    break;
  case SP_BOMB:
    success=create_bomb(op,dir,"bomb");
    break;
  case SP_GOLEM:
    success=summon_monster(op,dir,"golem");
    break;
  case SP_FIRE_ELEM:
    success=summon_monster(op,dir,"fire_elemental");
    break;
  case SP_WATER_ELEM:
    success=summon_monster(op,dir,"water_elemental");
    break;
  case SP_EARTH_ELEM:
    success=summon_monster(op,dir,"earth_elemental");
    break;
  case SP_AIR_ELEM:
    success=summon_monster(op,dir,"air_elemental");
    break;
  case SP_PET:
    success = summon_pet(op,dir, item);
    break;
  case SP_D_DOOR:
    success=dimension_door(op,dir);
    break;
  case SP_FIRE_WALL:
  case SP_FROST_WALL:
  case SP_EARTH_WALL:
    success=magic_wall(op,dir,type);
    break;
  case SP_PARALYZE:
    success=cast_cone(op,dir,5,SP_PARALYZE,!ability);
    break;
  case SP_SLOW:
    success=cast_cone(op,dir,5,SP_SLOW,!ability);
    break;
  case SP_MAGIC_MAPPING:
    if(op->type==PLAYER) {
      draw_map(op);
      success=1;
    }
    break;
  case SP_TURN_UNDEAD:
    success=cast_cone(op,dir,3+turn_bonus[op->stats.Wis],SP_TURN_UNDEAD,0);
    break;
  case SP_FEAR:
    if(op->type==PLAYER)
      bonus=fear_bonus[op->stats.Cha];
    else
      bonus=op->head==NULL?op->level/3+1:op->head->level/3+1;
    success=cast_cone(op,dir,4+bonus,SP_FEAR,!ability);
    break;
  case SP_POISON_CLOUD:
    success=fire_arch(op,dir,find_archetype("spellball"),!ability);
    break;
  case SP_WOW:
    success=cast_wow(op,dir,ability, item);
    break;
  case SP_DESTRUCTION:
    success=cast_destruction(op,5+op->stats.Int,AT_MAGIC);
    break;
  case SP_PERCEIVE:
    success=perceive_self(op);
    break;
  case SP_WOR:
    success=cast_wor(op);
    break;
  case SP_INVIS:
  case SP_INVIS_UNDEAD:
  case SP_IMPROVED_INVIS:
    success=cast_invisible(op,type);
    break;
  case SP_PROBE:
    success=probe(op,dir);
    break;
  case SP_HOLY_WORD:
    success=cast_cone(op,dir,2+turn_bonus[op->stats.Wis],SP_HOLY_WORD,0);
    break;
  case SP_CREATE_FOOD:
    success=cast_create_obj(op,dir,"food");
    break;
  case SP_EARTH_DUST:
    success=cast_earth2dust(op);
    break;
  case SP_STRENGTH:
  case SP_DEXTERITY:
  case SP_CONSTITUTION:
  case SP_CHARISMA:
  case SP_ARMOUR:
  case SP_PROT_COLD:
  case SP_PROT_FIRE:
  case SP_PROT_ELEC:
  case SP_PROT_POISON:
  case SP_PROT_SLOW:
  case SP_PROT_DRAIN:
  case SP_PROT_PARALYZE:
  case SP_PROT_ATTACK:
  case SP_PROT_MAGIC:
  case SP_CONFUSION:
  case SP_LEVITATE:
    success=cast_change_attr(op,dir,type);
    break;
  case SP_HEAL:
  case SP_MINOR_HEAL:
  case SP_MED_HEAL:
  case SP_MAJOR_HEAL:
  case SP_CURE_POISON:
    success=cast_heal(op,dir,type);
    break;
  case SP_REGENERATE_SPELLPOINTS:
    success = cast_regenerate_spellpoints(op);
    break;
  case SP_SMALL_SPEEDBALL:
  case SP_LARGE_SPEEDBALL:
    success=cast_speedball(op,dir,type);
    break;
  case SP_POLYMORPH:
    success = cast_polymorph(op,dir);
    break;
  case SP_CHARGING:
    success = recharge(op);
    break;
  case SP_CANCELLATION:
    success=fire_cancellation(op,dir,find_archetype("cancellation"),!ability);
    break;
  case SP_MASS_CONFUSION:
    success = cast_cone(op, dir, 5, SP_MASS_CONFUSION, 0);
    break;
  case SP_ALCHEMY:
    success = alchemy(op);
    break;
  }
  return success?s->sp:0;
}

int recharge(object *op) {
  object *wand;
  char buf[MAX_BUF];
  for(wand = op->inv; wand != NULL; wand = wand->below)
    if(wand->type == WAND && IS_APPLIED(wand))
      break;
  if(wand == NULL)
    return 0;
  if(!(RANDOM()%4)) {
    sprintf(buf,"The %s vibrates violently, then explodes!",query_name(wand));
    draw_info(op,buf);
    remove_ob(wand);
    free_object(wand);
    return 1;
  }
  sprintf(buf,"The %s glows with power.",query_name(wand));
  draw_info(op,buf);
  wand->stats.food += RANDOM()%spells[wand->stats.sp].charges + 1;
  if(wand->arch&&ANIMATED(&wand->arch->clone))
  {
    SET_ANIMATE(wand);
    wand->speed = wand->arch->clone.speed;
  }
  return 1;
}

#define MAX_ALT 80

void polymorph_living(object *op) {
  archetype *altern[MAX_ALT], *at;
  int nr = 0, x = op->x, y = op->y, choice, friendly;
  mapstruct *map = op->map;
  object *tmp, *next, *owner;

  if(op->head != NULL || op->more != NULL)
    return;
  for(at = first_archetype ; at != NULL; at = at->next)
    if(IS_MONSTER((&at->clone)) == IS_MONSTER(op) &&
       IS_GENERATOR((&at->clone)) == IS_GENERATOR(op) &&
       at->more == NULL && EDITABLE((&at->clone)))
    {
      altern[nr] = at;
      if(++nr == MAX_ALT)
        break;
    }
  if(!nr)
    return;
  choice = RANDOM()%nr;
  for(tmp = op->inv; tmp != NULL; tmp = next) {
    next = tmp->below;
    if(IS_APPLIED(tmp))
      apply(op,tmp);
    if(tmp->type == ABILITY) {
      remove_ob(tmp);
      free_object(tmp);
    }
  }
  remove_ob(op);
  owner = get_owner(op);
  friendly = IS_FRIENDLY(op);
  if (friendly)
    remove_friendly_object(op);
  copy_object(&(altern[choice]->clone),op);
  if (owner != NULL)
    set_owner(op,owner);
  if (friendly) {
    SET_FRIENDLY(op);
    op->move_type = PETMOVE;
    add_friendly_object(op);
  }
  op->x = x; op->y = y;
  insert_ob_in_map(op,map);
  update_object(op);
  if(op->arch->randomitems != NULL)
    create_treasure(op->arch->randomitems,op,GT_INVISIBLE,map->difficulty);
  for(tmp = op->inv, nr = 0; tmp != NULL && ++nr < 20; tmp = next) {
    next = tmp->below;
    (void) monster_check_apply(op,tmp);
  }
}

void polymorph_item(object *op) {
  archetype *altern[MAX_ALT], *at;
  int nr = 0, magic = op->magic, min_value, max_value, nrof = op->nrof;
  int x = op->x, y = op->y, choice, charges = op->stats.food;
  mapstruct *map = op->map;

  min_value = RANDOM()%5 ? op->arch->clone.value / 6 : 0;
  max_value = op->arch->clone.value * 4;
  if(max_value > 20000)
    max_value = 20000 + (max_value - 20000) / 2;

  for(at = first_archetype ; at != NULL; at = at->next)
    if(at->clone.type == op->type &&
       at->clone.value < max_value && at->clone.value > min_value)
    {
      altern[nr] = at;
      if(++nr == MAX_ALT)
        break;
    }
  if(!nr)
    return;
  choice = RANDOM()%nr;
  remove_ob(op);
  copy_object(&(altern[choice]->clone),op);
  op->x = x; op->y = y;
  insert_ob_in_map(op,map);
  fix_generated_item(op,0,0);
  if(magic)
    set_abs_magic(op,magic);
  if(charges)
    op->stats.food = charges;
  if(nrof && op->nrof)
    op->nrof = nrof;
  update_object(op);
}

void polymorph(object *op, object *who) {
  char buf[MAX_BUF];
  if(op->type == PLAYER)
    return; /* WILL add this later 8) */
  if(IS_MONSTER(op) || IS_GENERATOR(op)) {
    polymorph_living(op);
    return;
  }
  if(IS_ALIVE(op))
    return;
  if(FABS(op->speed) > 0.001 && !ANIMATED(op))
    return; /* Don't want to morph flying arrows, etc... */
  if(op->type == 0 || op->type==PLAYER || op->arch == NULL || NO_PICK(op) 
     || NO_PASS(op) || op->type == TREASURE)
    return;
  if(!(RANDOM()%15)) {
    sprintf(buf,"%s%s glows red, melts and evaporates!",
            op->nrof?"":"The ",query_name(op));
    draw_info(who,buf);
    remove_ob(op);
    free_object(op);
    return;
  }
  polymorph_item(op);
}

int cast_polymorph(object *op, int dir) {
  object *tmp, *next;
  int range;
  archetype *poly;

  if(dir == 0)
    return 0;
  poly = find_archetype("polymorph");
  for(range = 1;;range++) {
    int x=op->x+freearr_x[dir]*range,y=op->y+freearr_y[dir]*range;
    object *image;
    if(wall(op->map,x,y) || blocks_magic(op->map,x,y))
      break;
    for(tmp = get_map_ob(op->map,x,y); tmp != NULL && tmp->above != NULL;
        tmp = tmp->above);
    for(; tmp != NULL; tmp = next) {
      next = tmp->below;
      polymorph(tmp, op);
    }
    image = arch_to_object(poly);
    image->x = x; image->y = y;
    image->stats.food += range;
    image->speed_left = 0.1;
    insert_ob_in_map(image,op->map);
  }
  return 1;
}

int cast_speedball(object *op, int dir, int type) {
  object *spb;
  if(blocked(op->map,op->x+freearr_x[dir],op->y+freearr_y[dir]))
    return 0;
  spb=clone_arch(SPEEDBALL);
  spb->x=op->x+freearr_x[dir],spb->y=op->y+freearr_y[dir];
  spb->speed_left= -0.1;
  if(type==SP_LARGE_SPEEDBALL)
    spb->stats.dam=30;
  insert_ob_in_map(spb,op->map);
  return 1;
}

int probe(object *op, int dir) {
  int r;
  object *tmp;

  if(!dir) {
    examine(op,op);
    return 1;
  }
  for(r=1;;r++) {
    int x=op->x+r*freearr_x[dir],y=op->y+r*freearr_y[dir];
    if(out_of_map(op->map,x,y))
      break;
    if(blocks_magic(op->map,x,y)) {
      draw_info(op,"Something blocks your magic.");
      return 0;
    }
    for(tmp=get_map_ob(op->map,x,y);tmp!=NULL;tmp=tmp->above)
      if(IS_ALIVE(tmp)&&(tmp->type==PLAYER||IS_MONSTER(tmp))) {
        draw_info(op,"You detect something.");
        if(tmp->head!=NULL)
          tmp=tmp->head;
        examine_monster(op,tmp);
        return 1;
      }
  }
  draw_info(op,"You detect nothing.");
  return 1;
}

int cast_invisible(object *op, int spell_type) {
  object *tmp;

  if(op->invisible>1000) {
    draw_info(op,"You are already as invisible as you can get.");
    return 0;
  }
  switch(spell_type) {
  case SP_INVIS:
    UNSET_UNDEAD(op);
    op->invisible+=300;
    if(op->type==PLAYER)
      op->contr->tmp_invis=1;
    break;
  case SP_INVIS_UNDEAD:
    SET_UNDEAD(op);
    op->invisible+=300;
    if(op->type==PLAYER)
      op->contr->tmp_invis=1;
    break;
  case SP_IMPROVED_INVIS:
    op->invisible+=200;
    break;
  }
  draw_info(op,"You can't see your hands!");
  update_object(op);
  for (tmp = objects; tmp != NULL; tmp = tmp->next)
    if (tmp->enemy == op)
      tmp->enemy = NULL;
  return 1;
}

int
cast_heal(object *op,int dir,int spell_type) {
  object *tmp;
  archetype *at;
  object *poison;
  int heal=0;

  if(op->type!=PLAYER)
    tmp=op;
  else
    for(tmp=get_map_ob(op->map,op->x+freearr_x[dir],op->y+freearr_y[dir]);
        tmp!=NULL;
        tmp=tmp->above)
      if(tmp->type==PLAYER)
        break;
  if(tmp==NULL)               /* didn't find a player there */
    tmp=op;
  switch(spell_type) {
  case SP_MINOR_HEAL:
    heal=(RANDOM()%8)+1;
    draw_info(tmp, "Your wounds start to close.");
    break;
  case SP_MED_HEAL:
    heal=(RANDOM()%6)+(RANDOM()%6)+(RANDOM()%6)+6;
    draw_info(tmp, "Your wounds start to fade.");
    break;
  case SP_MAJOR_HEAL:
    draw_info(tmp, "Your skin looks as good as new!");
    heal=(RANDOM()%9)+(RANDOM()%9)+(RANDOM()%9)+6;
    break;
  case SP_HEAL:
    heal=op->stats.maxhp;
    draw_info(tmp, "You feel great!");
    break;
  case SP_CURE_POISON: 
    at = find_archetype("poisoning");
    poison=present_arch_in_ob(at,tmp);

    if (poison) {
      draw_info(tmp, "Your body feels cleansed");
      poison->stats.food -= 25;
      if (poison->stats.food < 1)       /* make it 1 so poison_more can */
        poison->stats.food = 1;         /* clean up.                    */ 
    }
    break;                            
  }
  if (tmp->stats.hp==tmp->stats.maxhp)
    return 0;
  tmp->stats.hp+=heal;
  if(tmp->stats.hp>tmp->stats.maxhp)
    tmp->stats.hp=tmp->stats.maxhp;
  if(tmp->type==PLAYER)
    draw_stats(tmp);
  if(op!=tmp && op->type==PLAYER)
    draw_stats(op);
  op->speed_left= -FABS(op->speed)*3; /* Freeze them for a short while */
  return 1;
}

int cast_regenerate_spellpoints(object *op) {
  op->stats.sp = op->stats.maxsp;
  if (op->type == PLAYER)
    draw_stats(op);
  draw_info(op, "Magical energies surge through your body!");
  return 1;
}

int
cast_change_attr(object *op,int dir,int spell_type) {
  object *tmp;
  object *force;
  int i;

  if(op->type!=PLAYER)
    tmp=op;
  else
    for(tmp=get_map_ob(op->map,op->x+freearr_x[dir],op->y+freearr_y[dir]);
        tmp!=NULL;
        tmp=tmp->above)
      if(tmp->type==PLAYER)
        break;
  if(tmp==NULL)               /* didn't find a player there */
    return 0;

  if((force=present_in_ob(FORCE,tmp))!=NULL)
    remove_force(force);
  force=get_archetype("force");
  switch(spell_type) {
  case SP_STRENGTH:
    if(tmp->type!=PLAYER)
      break;
    for(i=20,force->stats.Str=1;i>tmp->contr->orig_stats.Str;i-=2)
      force->stats.Str++;
    break;
  case SP_DEXTERITY:
    if(tmp->type!=PLAYER)
      break;
    for(i=20,force->stats.Dex=1;i>tmp->contr->orig_stats.Dex;i-=2)
      force->stats.Dex++;
    break;
  case SP_CONSTITUTION:
    if(tmp->type!=PLAYER)
      break;
    for(i=20,force->stats.Con=1;i>tmp->contr->orig_stats.Con;i-=2)
      force->stats.Con++;
    break;
  case SP_CHARISMA:
    if(tmp->type!=PLAYER)
      break;
    for(i=20,force->stats.Cha=1;i>tmp->contr->orig_stats.Cha;i-=2)
      force->stats.Cha++;
    break;
  case SP_ARMOUR:
    force->stats.ac=2+(op->level+3)/4;
    if(force->stats.ac>5)
      force->stats.ac=5;
    force->armour = 5+op->level*2;
    if (force->armour > 25)
      force->armour = 25;
    draw_info(tmp,"A force shimmers around you.");
    break;
  case SP_CONFUSION:
    force->attacktype |= (AT_CONFUSION|AT_PHYSICAL);
    force->protected |= AT_CONFUSION;
    break;
  case SP_PROT_COLD:
    force->protected|=AT_COLD;
    break;
  case SP_PROT_FIRE:
    force->protected|=AT_FIRE;
    break;
  case SP_PROT_ELEC:
    force->protected|=AT_ELECTRICITY;
    break;
  case SP_PROT_POISON:
    force->protected|=AT_POISON;
    break;
  case SP_PROT_SLOW:
    force->immune|=AT_SLOW;
    force->protected|=AT_PARALYZE;
    break;
  case SP_PROT_PARALYZE:
    force->immune|=AT_PARALYZE;
    force->protected|=AT_SLOW;
    break;
  case SP_PROT_DRAIN:
    force->protected|=AT_DRAIN;
    break;
  case SP_PROT_ATTACK:
    force->protected|=AT_PHYSICAL;
    break;
  case SP_PROT_MAGIC:
    force->protected|=AT_MAGIC;
    break;
  case SP_PROT_CONFUSE:
    force->protected|=AT_CONFUSION;
    break;
  case SP_PROT_CANCEL:
    force->protected|=AT_CANCELLATION;
    break;
  case SP_PROT_DEPLETE:
    force->protected|=AT_DEPLETE;
    break;
  case SP_LEVITATE:
    SET_FLY(force);
    break;
  }
  force->speed_left= -1-op->level*0.1;
  SET_APPLIED(force);
  insert_ob_in_ob(force,tmp);
  change_abil(tmp,force); /* Mostly to display any messages */
  fix_player(tmp);        /* This takes care of some stuff that change_abil() */
			  /* unfortunately is incapable off. */
  if(tmp->type==PLAYER)
    draw_stats(tmp);
  return 1;
}

int
cast_create_obj(object *op,int dir,char *name) {
  object *tmp;

  if(blocked(op->map,op->x+freearr_x[dir],op->y+freearr_y[dir])) {
    draw_info(op,"Something is in the way.");
    return 0;
  }
  tmp=get_archetype(name);
  tmp->x=op->x+freearr_x[dir],tmp->y=op->y+freearr_y[dir];
  insert_ob_in_map(tmp,op->map);
  return 1;
}

int 
cast_earth2dust( object *op ) {
  object *tmp, *next;
  int strength,i,j;

  if(op->type!=PLAYER)
    return 0;
  strength=op->level/3+1;
  strength=(strength>15)?15:strength;
  for(i= -strength;i<strength;i++)
    for(j= -strength;j<strength;j++) {
      if(out_of_map(op->map,op->x+i,op->y+j))
        continue;
      for(tmp=get_map_ob(op->map,op->x+i,op->y+j);tmp!=NULL;tmp=next) {
        next=tmp->above;
        if(tmp&&TEAR_DOWN(tmp))
          hit_player(tmp,9999,op,AT_PHYSICAL);
      }
    }
  return 1;
}

int cast_wor(object *op) {
  object *dummy;
  if(op->type!=PLAYER)
    return 0;
  if(blocks_magic(op->map,op->x,op->y)) {
    draw_info(op,"Something blocks your spell.");
    return 0;
  }
  dummy=get_object();
  dummy->invisible=1;
  dummy->speed=0.01;
  dummy->speed_left= -1;
  dummy->type=WORD_OF_RECALL;
  EXIT_PATH(dummy)=add_string(first_map_path);
  insert_ob_in_ob(dummy,op);
  draw_info(op,"You feel a force starting to build up inside you.");
  return 1;
}

int cast_wow(object *op, int dir, int ability, SpellTypeFrom item) {
  int sp;
  if(!(RANDOM()%4))
    return cast_cone(op,0,10,SP_WOW,0);
  do
    sp=RANDOM()%NROFREALSPELLS;
  while (!spells[sp].books);
  return cast_spell(op,dir,sp,ability,item);
}

int perceive_self(object *op) {
  char *cp=describe_item(op);
  if(*cp=='\0')
    draw_info(op,"You feel very mundane");
  else {
    draw_info(op,"You have:");
    draw_info(op,cp);
  }
  return 1;
}

int cast_destruction(object *op, int dam, int attacktype) {
  int i,j;
  object *tmp;
  if(op->type!=PLAYER)
    return 0;
  for(i= -5;i<5;i++)
    for(j= -5;j<5;j++) {
      if(out_of_map(op->map,op->x+i,op->y+j))
        continue;
      tmp=get_map_ob(op->map,op->x+i,op->y+j);
      while(tmp!=NULL&&(!IS_ALIVE(tmp)||tmp->type==PLAYER))
        tmp=tmp->above;
      if(tmp==NULL)
        continue;
      hit_player(tmp,dam,op,attacktype);
    }
  return 1;
}

int magic_wall(object *op,int dir,int spell_type) {
  object *tmp;
  if(!dir) {
    draw_info(op,"In what direction?");
    return 0;
  }
  if(blocked(op->map,op->x+freearr_x[dir],op->y+freearr_y[dir])) {
    draw_info(op,"Something is in the way.");
    return 0;
  }
  switch(spell_type) {
  case SP_EARTH_WALL:
    tmp=get_archetype("earthwall");
    tmp->immune=0,tmp->protected=0;
    tmp->stats.hp = op->level*8; /* More solid, since they can be torn down */
    tmp->stats.maxhp = tmp->stats.hp;
    /*
     *  Now, make agressive monsters tear down the walls:
     */
    SET_FRIENDLY(tmp);
    add_friendly_object(tmp);
    break;
  case SP_FIRE_WALL:
    tmp=get_archetype("firebreath");
    tmp->attacktype |= AT_MAGIC;
    tmp->stats.hp=200 + op->level*10;
    tmp->stats.dam=1;
    tmp->stats.food=1;                        /* so it doesn't propagate */
#if 0
    SET_SLOW_MOVE(tmp);
    SET_SLOW_PENALTY(tmp,2);
#endif
    SET_WALK_ON(tmp);
    SET_FLY_ON(tmp);
    set_owner(tmp,op);
    break;
  case SP_FROST_WALL:
    tmp=get_archetype("icestorm");
    tmp->attacktype |= AT_MAGIC;
    tmp->stats.hp=160 + op->level*10;
    tmp->stats.food=1;                        /* so it doesn't propagate */
    tmp->stats.dam=1;
#if 0
    SET_SLOW_MOVE(tmp);
    SET_SLOW_PENALTY(tmp,3);
#endif
    SET_WALK_ON(tmp);
    SET_FLY_ON(tmp);
    set_owner(tmp,op);
    break;
  default:
    LOG(llevError,"Unimplemented magic_wall spell: %d\n",spell_type);
    return 0;
  }
  tmp->x=op->x+freearr_x[dir],tmp->y=op->y+freearr_y[dir];
  insert_ob_in_map(tmp,op->map);
  if(BLOCKSVIEW(tmp))
    update_all_los(op->map);
  if(op->type==PLAYER)
    draw(op);
  else
    SET_SCARED(op); /* We don't want them to walk through the wall! */
  return 1;
}

int dimension_door(object *op,int dir) {
  int dist;
  if(!dir) {
    draw_info(op,"In what direction?");
    return 0;
  }
  if(op->type!=PLAYER)
    return 0;
  if(op->contr->count) {
    for(dist=0;dist<op->contr->count&&
               !blocks_magic(op->map,op->x+freearr_x[dir]*(dist+1),
                             op->y+freearr_y[dir]*(dist+1));
        dist++);
    if(dist<op->contr->count) {
      draw_info(op,"Something blocks your magic.\n");
      op->contr->count=0;
      return 0;
    }
    op->contr->count=0;
    if(blocked(op->map,op->x+freearr_x[dir]*dist,
                       op->y+freearr_y[dir]*dist))
    { /* Now the fun starts... 8) */
      int x=RANDOM()%op->map->mapx,y=RANDOM()%op->map->mapy;
      if(blocked(op->map,x,y)) { /* Lucky after all... */
        draw_info(op,"You cast your spell, but nothing happens.\n");
        return 1; /* Maybe the penalty should be more severe... */
      }
      remove_ob(op);
      op->x=x,op->y=y;
      insert_ob_in_map(op,op->map);
      draw(op);
      return 1;
    }
  } else {
    for(dist=0;!blocks_view (op->map,op->x+freearr_x[dir]*(dist+1),
                                     op->y+freearr_y[dir]*(dist+1))&&
               !blocks_magic(op->map,op->x+freearr_x[dir]*(dist+1),
                                     op->y+freearr_y[dir]*(dist+1));
        dist++);
    for(;dist>0&&blocked(op->map,op->x+freearr_x[dir]*dist,
                                 op->y+freearr_y[dir]*dist);dist--);
    if(!dist) {
      draw_info(op,"Your spell fail.\n");
      return 0;
    }
  }
  remove_ob(op);
  op->x+=freearr_x[dir]*dist,op->y+=freearr_y[dir]*dist;
  insert_ob_in_map(op,op->map);
  draw(op);
  op->speed_left= -FABS(op->speed)*5; /* Freeze them for a short while */
  return 1;
}

int summon_monster(object *op,int dir,char *name) {
  object *tmp;
  if(op->type==PLAYER)
    if(op->contr->golem!=NULL&&!IS_FREED(op->contr->golem)) {
      control_golem(op->contr->golem,dir);
      return 0;
    }
  if(!dir)
    dir=find_free_spot(NULL,op->map,op->x,op->y,1,9);
  if(blocked(op->map,op->x+freearr_x[dir],op->y+freearr_y[dir])) {
    draw_info(op,"There is something in the way.");
    if(op->type==PLAYER)
      op->contr->count_left=0;
    return 0;
  }
  tmp=get_archetype(name);
  if(op->type==PLAYER) {
    UNSET_MONSTER(tmp);
    tmp->stats.exp=0;
    add_friendly_object(tmp);
    tmp->type=GOLEM;
/* Don't see any point in setting this when monsters summon monsters: */
    set_owner(tmp,op);
    op->contr->golem=tmp;
  } else {
    if(IS_FRIENDLY(op)) {
      object *owner = get_owner(op);
      if(owner != NULL) /* For now, we transfer ownership */
        set_owner(tmp,owner);
      tmp->move_type = PETMOVE;
      add_friendly_object(tmp);
      SET_FRIENDLY(tmp);
    }
    SET_MONSTER(tmp);
  }
  tmp->speed_left= -1;
  tmp->x=op->x+freearr_x[dir],tmp->y=op->y+freearr_y[dir];
  tmp->direction=dir;
  insert_ob_in_map(tmp,op->map);
  return 1;
}

int summon_pet(object *op, int dir, SpellTypeFrom item) {
  int level, number, i;
  char *monster;
  archetype *at;

  level = ((op->head?op->head->level:op->level) / 4);
  if (level >= MAX_PET_MONSTERS)
    level = MAX_PET_MONSTERS - 1;
  switch(RANDOM()%3) {
  case 0:
    number = priest_num_called[level];
    monster = priest_pet_monsters[level];
    break;
  case 1:
    number = mage_num_called[level];
    monster = mage_pet_monsters[level];
    break;
  default:
    number = altern_num_called[level];
    monster = altern_pet_monsters[level];
    break;
  }
  at = find_archetype(monster);
  if(at == NULL) {
    LOG(llevError,"Unknown archetype in summon pet: %s\n",monster);
    return 0;
  }
  if (!dir)
    dir = find_free_spot(at, op->map, op->x, op->y, 1, SIZEOFFREE);
  if(arch_blocked(at,op->map, op->x + freearr_x[dir], op->y+freearr_y[dir]))
  {
    draw_info(op, "There is something in the way.");
    if(op->type == PLAYER)
      op->contr->count_left = 0;
    return 0;
  }
  if (item != spellNormal)
    op->stats.sp -= 5 + 10*level + op->level;
  for (i = 1; i < number + 1; i++) {
    archetype *atmp;
    object *prev = NULL, *head = NULL; /* We want to summon dragons *grin* */

    for(atmp = at; atmp!=NULL; atmp = atmp->more) {
      object *tmp;
      tmp = arch_to_object(atmp);
      if (atmp == at) {
        set_owner(tmp, op);
        SET_MONSTER(tmp);
        if (op->type == PLAYER) {
          tmp->stats.exp = 0;
          add_friendly_object(tmp);
          SET_FRIENDLY(tmp);
          tmp->move_type = PETMOVE;
        } else
          if(IS_FRIENDLY(op)) {
            add_friendly_object(tmp);
            SET_FRIENDLY(tmp);
            tmp->move_type = PETMOVE;
          } else
        tmp->speed_left = -1;
        tmp->enemy = op->enemy;
        tmp->type = 0;
      }
      if(head == NULL)
        head = tmp;
      tmp->x = op->x + freearr_x[dir] + tmp->arch->clone.x;
      tmp->y = op->y + freearr_y[dir] + tmp->arch->clone.y;
      tmp->map = op->map;
      if(head != tmp)
        tmp->head = head, prev->more = tmp;
      prev = tmp;
    }
    head->direction = dir;
    insert_ob_in_map(head, op->map);
    if(at->randomitems != NULL) {
      object *tmp;
      create_treasure(at->randomitems,head,GT_INVENTORY,6);
      for(tmp = head->inv; tmp != NULL; tmp = tmp->below)
        if(!tmp->nrof)
          SET_NO_DROP(tmp);
    }
    dir = absdir(dir + 1);
    if (arch_blocked(at,op->map, op->x + freearr_x[dir],
                     op->y + freearr_y[dir]))
    {
      if (i < number) {
        draw_info(op, "There is something in the way,");
        draw_info(op, "no more pets for this casting.");
        if (item != spellNormal) {
          op->stats.sp += (5 + 12 * level + op->level) / (number - i);
          if (op->stats.sp < 0)
            op->stats.sp = 0;
        }
        return 1;
      }
    }
  }
  if (item != spellNormal && op->stats.sp < 0)
    op->stats.sp = 0;
  return 1;
}

int create_bomb(object *op,int dir,char *name) {
  object *tmp;
  int dx=op->x+freearr_x[dir],dy=op->y+freearr_y[dir];
  if(wall(op->map,dx,dy)) {
    draw_info(op,"There is something in the way.");
    return 0;
  }
  tmp=get_archetype(name);
  set_owner(tmp,op);
  tmp->x=dx,tmp->y=dy;
  insert_ob_in_map(tmp,op->map);
  return 1;
}


int small_fire(object *op,int dir,char *name) {
  object *tmp;
  int dx=op->x+freearr_x[dir],dy=op->y+freearr_y[dir];
  if(wall(op->map,dx,dy)) {
    draw_info(op,"There is something in the way.");
    return 0;
  }
  tmp=get_archetype(name);
  set_owner(tmp,op);
  tmp->x=dx,tmp->y=dy;
  insert_ob_in_map(tmp,op->map);
  return 1;
}

int ok_to_put_more(mapstruct *m,int x,int y,object *op,int immune_stop) {
  object *tmp,*head;
  for(tmp=get_map_ob(m,x,y);tmp!=NULL;tmp=tmp->above) {
    head=tmp->head==NULL?tmp:tmp->head;
    if((IS_ALIVE(op) && head->immune & immune_stop) ||
       (head->stats.maxhp == op->stats.maxhp && head->type == op->type))
      return 0;
  }
  return 1;
}

int burning_hands(object *op,int dir,int strength) {
  return cast_cone(op,dir,strength,SP_BURNING_HANDS,0);
}

int icestorm(object *op,int dir,int strength) {
  return cast_cone(op,dir,strength,SP_ICESTORM,0);
}

int fire_bolt(object *op,int dir,char *name,int magic) {
  object *tmp=get_archetype(name);
  if(tmp==NULL)
    return 0;
  if(magic)
    tmp->attacktype|=AT_MAGIC;
  tmp->x=op->x,tmp->y=op->y;
  tmp->direction=dir;
  if(IS_TURNABLE(tmp))
    tmp->face.number=tmp->arch->faces[dir];
  set_owner(tmp,op);
#if 0
  if(op->type==PLAYER)
    tmp->stats.wc=5+(op->contr->shootstrength-5)/5,
    tmp->stats.exp=(op->contr->shootstrength-5)/3+12,
    tmp->stats.hp=8+(op->contr->shootstrength-5)/8;
#endif
  tmp->x+=DIRX(tmp),tmp->y+=DIRY(tmp);
  if(wall(op->map,tmp->x,tmp->y)) {
    if(!REFLECTING(tmp)) {
      free_object(tmp);
      return 0;
    }
    tmp->x=op->x,tmp->y=op->y;
    tmp->direction=absdir(tmp->direction+4);
  }
  insert_ob_in_map(tmp,op->map);
  move_bolt(tmp);
  return 1;
}

int fire_arch(object *op,int dir,archetype *at, int magic) {
  object *tmp;
  if(at==NULL)
    return 0;
  tmp=arch_to_object(at);
  if(tmp==NULL)
    return 0;
  tmp->x=op->x,tmp->y=op->y;
  tmp->direction=dir;
  if(magic)
    tmp->attacktype|=AT_MAGIC;
  if(IS_TURNABLE(tmp))
    tmp->face.number=tmp->arch->faces[dir];
#if 0
  if(op->type==PLAYER)
    tmp->stats.hp=(op->contr->shootstrength-10)/10+10;
#endif
  set_owner(tmp,op);
  if(op->type==PLAYER)
    D_LOCK(op);
  insert_ob_in_map(tmp,op->map);
  if(tmp->type==MMISSILE) /* I'll get rid of this kludge soon */
    move_missile(tmp);
  else
    move_fired_arch(tmp);
  if(op->type==PLAYER)
    D_UNLOCK(op);
  return 1;
}

int
cast_cone(object *op, int dir, int strength, int spell_type, int magic)
{
  object *tmp;
  int i,success=0,range_min= -1,range_max=1;
  if(!dir)
    range_min= -3,range_max=4,strength/=2;

  for(i=range_min;i<=range_max;i++) {
    int x=op->x+freearr_x[absdir(dir+i)],
        y=op->y+freearr_y[absdir(dir+i)];
    if(wall(op->map,x,y))
      continue;
    success=1;
    switch(spell_type) {
    case SP_BURNING_HANDS:
    case SP_FIREBREATH:
      tmp=get_archetype("firebreath");
      if(magic)
        tmp->attacktype|=AT_MAGIC;
      break;
    case SP_ICESTORM:
    case SP_LARGE_ICESTORM:
      tmp=get_archetype("icestorm");
      if(magic)
        tmp->attacktype|=AT_MAGIC;
      break;
    case SP_PARALYZE:
      tmp=get_archetype("paralyze");
      if(magic)
        tmp->attacktype|=AT_MAGIC;
      break;
    case SP_TURN_UNDEAD:
      tmp=get_archetype("turn_undead");
      break;
    case SP_FEAR:
      tmp=get_archetype("fear");
      break;
    case SP_HOLY_WORD:
      tmp=get_archetype("holy_word");
      break;
    case SP_WOW:
      tmp=get_archetype("flowers");
      break;
    case SP_MASS_CONFUSION:
      tmp=get_archetype("confuse");
      break;
    case SP_SLOW:
      tmp=get_archetype("slow");
      break;
    default:
      LOG(llevError,"Unimplemented spell: %d\n",spell_type);
      return 0;
    }
    set_owner(tmp,op);
    tmp->x=x,tmp->y=y;
    if(dir)
      tmp->stats.sp=dir;
    else
      tmp->stats.sp=i;
    tmp->stats.hp=strength;
    tmp->stats.maxhp=tmp->count;
    SET_FLY(tmp);
    insert_ob_in_map(tmp,op->map);
  }
  return success;
}

void move_cone(object *op) {
  int i;

  if (LIFESAVE(op)) {/* lava saves it's life, but not yours  :) */
      hit_map(op,0,op->attacktype);
      return;
  }
  if(get_owner(op)==NULL) {
    remove_ob(op);
    free_object(op);
    return;
  }
  hit_map(op,0,op->attacktype);
  if((op->stats.hp-=2)<0) {
    if(op->stats.exp) {
      op->speed=0;
      op->stats.exp=0;
      op->stats.sp=0; /* so they will join */
    } else {
      remove_ob(op);
      free_object(op);
    }
    return;
  }
  if(op->stats.food)
    return;
  op->stats.food=1;
  for(i= -1;i<2;i++) {
    int x=op->x+freearr_x[absdir(op->stats.sp+i)],
        y=op->y+freearr_y[absdir(op->stats.sp+i)];
    if(!wall(op->map,x,y)&&ok_to_put_more(op->map,x,y,op,op->attacktype) &&
       !blocks_view(op->map,x,y)) {
      object *tmp=arch_to_object(op->arch);
      set_owner(tmp,op->owner);
      tmp->x=x, tmp->y=y;
      tmp->stats.sp=op->stats.sp,tmp->stats.hp=op->stats.hp+1;
      tmp->stats.maxhp=op->stats.maxhp;
      tmp->attacktype=op->attacktype;
      insert_ob_in_map(tmp,op->map);
    }
  }
}

void fire_a_ball(object *op,int dir,int strength) {
  object *tmp=clone_arch(FBULLET);

  if(!dir)
    LOG(llevError,"Tried to fire a ball without direction.\n");
  set_owner(tmp,op);
  tmp->direction=dir;
  tmp->x=op->x,tmp->y=op->y;
  tmp->speed=1;
  tmp->stats.hp=strength;
  tmp->face.number=tmp->arch->faces[dir];
  SET_FLY(tmp);
  insert_ob_in_map(tmp,op->map);
  move_fired_arch(tmp);
}

void explosion(object *op) {
  object *tmp;
  mapstruct *m=op->map; /* In case we free b */
  int i;

  if(--(op->stats.hp)<0) {
    remove_ob(op);
    free_object(op);
    return;
  }
  if(op->above!=NULL&&op->above->type!=PLAYER) {
    remove_ob(op);
    insert_ob_in_map(op,op->map);
  }
  hit_map(op,0,op->attacktype);
  if(op->stats.hp>2&&!op->value) {
    op->value=1;
    for(i=1;i<9;i++) {
      int dx,dy;
      if(wall(op->map,dx=op->x+freearr_x[i],dy=op->y+freearr_y[i]))
        continue;
      if(blocks_view(op->map, dx, dy))
        continue;
      if(ok_to_put_more(op->map,dx,dy,op,op->attacktype)) {
        tmp=get_object();
        copy_object(op,tmp); /* This is probably overkill on slow computers.. */
        tmp->state=0;
        tmp->speed_left= -0.21;
        tmp->stats.hp--;
        tmp->value=0;
        tmp->x=dx,tmp->y=dy;
        insert_ob_in_map(tmp,m);
      }
    }
  }
}

int reflwall(mapstruct *m,int x,int y) {
  object *op;
  if(out_of_map(m,x,y)) return 0;
  for(op=get_map_ob(m,x,y);op!=NULL;op=op->above)
    if(REFL_SPELL(op))
      return 1;
  return 0;
}

void move_bolt(object *op) {
  object *tmp;
  int w,r;
  if(--(op->stats.hp)<0) {
    remove_ob(op);
    free_object(op);
    return;
  }
  hit_map(op,0,op->attacktype);
  if(!op->value&&--(op->stats.exp)>0) {
    op->value=1;
    if(!op->direction)
      return;
    tmp=get_map_ob(op->map,op->x,op->y);
    while(tmp!=NULL&&(!IS_ALIVE(tmp)||!(tmp->immune&op->attacktype)))
      tmp=tmp->above;
    if(tmp!=NULL) {
      remove_ob(op);
      free_object(op);
      return;
    }
    if(blocks_view(op->map,op->x+DIRX(op),op->y+DIRY(op)))
      return;
    w=wall(op->map,op->x+DIRX(op),op->y+DIRY(op));
    r=reflwall(op->map,op->x+DIRX(op),op->y+DIRY(op));
    if(w&&!REFLECTING(op))
      return;
    if(w||r) {
      if(!REFLECTING(op))
        return;
      op->value=0;
      if(op->direction&1)
        op->direction=absdir(op->direction+4);
      else {
        int left= wall(op->map,op->x+freearr_x[absdir(op->direction-1)],
                              op->y+freearr_y[absdir(op->direction-1)]),
            right=wall(op->map,op->x+freearr_x[absdir(op->direction+1)],
                              op->y+freearr_y[absdir(op->direction+1)]);
        if(left==right)
          op->direction=absdir(op->direction+4);
        else if(left)
          op->direction=absdir(op->direction+2);
        else if(right)
          op->direction=absdir(op->direction-2);
      }
      update_turn_face(op); /* A bolt *must* be IS_TURNABLE */
      return;
    }
    else {
      tmp=get_object();
      copy_object(op,tmp);
      tmp->speed_left= -0.1;
      tmp->value=0;
      tmp->stats.hp++;
      tmp->x+=DIRX(tmp),tmp->y+=DIRY(tmp);
      insert_ob_in_map(tmp,op->map);
    }
  }
}

void move_golem(object *op) {
  if(IS_MONSTER(op))
    return; /* Has already been moved */
  if(get_owner(op)==NULL) {
    LOG(llevDebug,"Golem without owner destructed.\n");
    remove_ob(op);
    free_object(op);
    return;
  }
  if(--op->stats.hp<0) {
    draw_info(op->owner,"Your golem dissolved.");
    remove_friendly_object(op);
    op->owner->contr->golem=NULL;
    remove_ob(op);
    free_object(op);
    return;
  }
  if(!move_ob(op,op->direction)&&
     !out_of_map(op->map,op->x+freearr_x[op->direction],
                 op->y+freearr_y[op->direction])) {
    update_object(op);
    hit_map(op,op->direction,AT_PHYSICAL);
  }
}

void control_golem(object *op,int dir) {
  op->direction=dir;
}

void animate_bomb(object *op) {
  int i;
  object *env;
  if(op->state!=op->arch->animations-1)
    return;
  for(i=1;i<9;i++)
    fire_arch(op,i,find_archetype("splint"),0);
  for(env=op;env->env!=NULL;env=env->env);
  remove_ob(op);
  op->x=env->x,op->y=env->y,op->map=env->map;
  if(!explode_object(op)) /* Boom 8) */
    LOG(llevError,"Error: bomb refused to go off.\n");
  return;
}

void move_missile(object *op) {
  int i;

  remove_ob(op);
  op->x+=DIRX(op),op->y+=DIRY(op);
  if(!op->direction||wall(op->map,op->x,op->y)||
     blocks_view(op->map,op->x,op->y)) {
    free_object(op);
    return;
  }
  if(blocked(op->map,op->x,op->y)) {
    hit_map(op,0,AT_MAGIC);
    free_object(op);
    return;
  }
  i=find_dir(op->map,op->x,op->y,get_owner(op));
  if(i&&i!=op->direction){
    op->direction=absdir(op->direction+((op->direction-i+8)%8<4?-1:1));
    op->face.number=op->arch->faces[op->direction];
  }
  insert_ob_in_map(op,op->map);
}

int explode_object(object *op) {
  object *tmp;
  if(op->other_arch==NULL)
    return 0;
  tmp=arch_to_object(op->other_arch);
  if(op->attacktype&AT_MAGIC)
    tmp->attacktype|=AT_MAGIC;
  if(op->owner)
    set_owner(tmp,op->owner);
  if(op->stats.hp)
    tmp->stats.hp=op->stats.hp;
  tmp->stats.maxhp=op->count; /* Unique ID */
  tmp->x=op->x,tmp->y=op->y;
  if (wall(op->map,tmp->x,tmp->y))
    tmp->x-=DIRX(op),tmp->y-=DIRY(op);
  if (wall(op->map, tmp->x, tmp->y))
    free_object(tmp);
  else
    insert_ob_in_map(tmp,op->map);
  free_object(op);
  return 1;
}

void check_fired_arch(object *op) {
  if(blocked(op->map,op->x,op->y)) {
    object *tmp;
    remove_ob(op);
    if(explode_object(op))
      return;
    for(tmp=get_map_ob(op->map,op->x,op->y);tmp!=NULL;tmp=tmp->above)
      if(IS_ALIVE(tmp))
        break;
    if(tmp!=NULL)
      op->stats.dam-=hit_player(tmp,op->stats.dam,op,op->attacktype);
    if(blocked(op->map,op->x,op->y)) {
      free_object(op);
      return;
    }
    insert_ob_in_map(op,op->map);
  }
}

void move_fired_arch(object *op) {
  remove_ob(op);
  op->x+=DIRX(op),op->y+=DIRY(op);
  if(!op->direction||wall(op->map,op->x,op->y)) {
    if(explode_object(op))
      return;
    free_object(op);
    return;
  }
  if(reflwall(op->map,op->x,op->y)) {
    op->direction=absdir(op->direction+4);
    insert_ob_in_map(op,op->map);
    update_turn_face(op);
    return;
  }
  if(blocked(op->map,op->x,op->y)) {
    object *tmp;
    if(explode_object(op))
      return;
    for(tmp=get_map_ob(op->map,op->x,op->y);tmp!=NULL;tmp=tmp->above)
      if(IS_ALIVE(tmp))
        break;
    if(tmp!=NULL)
      op->stats.dam-=hit_player(tmp,op->stats.dam,op,op->attacktype);
    if(blocked(op->map,op->x,op->y)) {
      free_object(op);
      return;
    }
  }
  insert_ob_in_map(op,op->map);
}

int fire_cancellation(object *op,int dir,archetype *at, int magic) {
  object *tmp;
  if(at==NULL)
    return 0;
  tmp=arch_to_object(at);
  if(tmp==NULL)
    return 0;
  tmp->x=op->x,tmp->y=op->y;
  tmp->direction=dir;
  if(magic)
    tmp->attacktype|=AT_MAGIC;
  set_owner(tmp,op);
  if(op->type==PLAYER)
    D_LOCK(op);
  insert_ob_in_map(tmp,op->map);
  move_cancellation(tmp);
  if(op->type==PLAYER)
    D_UNLOCK(op);
  return 1;
}

void move_cancellation(object *op) {
  remove_ob(op);
  op->x+=DIRX(op),op->y+=DIRY(op);
  if(!op->direction||wall(op->map,op->x,op->y)) {
    free_object(op);
    return;
  }
  if(reflwall(op->map,op->x,op->y)) {
    op->direction=absdir(op->direction+4);
    insert_ob_in_map(op,op->map);
    return;
  }
  hit_map(op, 0, op->attacktype);
  insert_ob_in_map(op,op->map);
}

void cancellation(object *op)
{
  object *tmp;

  if(IS_ALIVE(op)||IS_CONTAINER(op)) { /* Recur through the inventory */
    for(tmp=op->inv;tmp!=NULL;tmp=tmp->below)
      cancellation(tmp);
    if(op->type==PLAYER) {
      draw_all_inventory(op);
      draw_all_look(op);
      fix_player(op);	/* Remember that wc/dam has probably changed */
    }
  }
  else				/* Nullify this object. */
    if(FABS(op->magic)>=(RANDOM()%6))
      op->magic=0;
}



/* Alchemy code by Mark Wedel (master@cats.ucsc.edu)
 *
 * This code adds a new spell, called alchemy.  Alchemy will turn
 * objects to gold nuggets, the value of the gold nuggets being
 * about 90% of that of the item itself.  It uses the value of the
 * object before charisma adjustments, because the nuggets themselves
 * will be will be adjusted by charisma when sold.
 *
 * Large nuggets are worth 25 gp each (base).  You will always get
 * the maximum number of large nuggets you could get.
 * Small nuggets are worth 1 gp each (base).  You will get from 0
 * to the max amount of small nuggets as you could get.
 *
 * For example, if an item is worth 110 gold, you will get
 * 4 large nuggets, and from 0-10 small nuggets.
 *
 * There is also a chance (1:30) that you will get nothing at all
 * for the object.  There is also a maximum weight that will be
 * alchemied.
 */
 
/* I didn't feel like passing these as arguements to the
 * two functions that need them.  Real values are put in them
 * when the spell is cast, and these are freed when the spell
 * is finished.
 */
static object *small, *large;

static void alchemy_object(object *obj, int *small_nuggets,
	 int *large_nuggets, int *weight)
{
	int	value=query_cost(obj, NULL, F_TRUE);

	/* Give half price when we alchemy money (This should hopefully
	 * make it so that it isn't worth it to alchemy money, sell
	 * the nuggets, alchemy the gold from that, etc.
	 * Otherwise, give 9 silver on the gold for other objects,
	 * so that it would still be more affordable to haul
	 * the stuff back to town.
	 */
    if (obj->type==GOLDCOIN || obj->type==SILVERCOIN)
	value /=2;
    else
	value *= 0.9;

    if ((obj->value>0) && RANDOM()%30) {
	int tmp = (value % large->value) / small->value;

	*large_nuggets += value/ large->value;
	if (tmp)
	    *small_nuggets += RANDOM() % (tmp + 1);
    }

    /* Turn 25 small nuggets into 1 large nugget.  If the value
     * of large nuggets is not evenly divisable my the small nugget
     * value, take off an extra small_nugget (Assuming small_nuggets!=0)
     */
    if (*small_nuggets * small->value >= large->value) {
	(*large_nuggets)++;
	*small_nuggets -= large->value / small->value;
	if (*small_nuggets && large->value % small->value)
		(*small_nuggets)--;
    }
    weight += obj->weight;
    remove_ob(obj);
    free_object(obj);
}

static void update_map(object *op, int small_nuggets, int large_nuggets,
	int x, int y)
{
	object *tmp;

	if (small_nuggets) {
		tmp = get_object();
		copy_object(small, tmp);
		tmp-> nrof = small_nuggets;
		tmp->x = x;
		tmp->y = y;
		insert_ob_in_map(tmp, op->map);
	}
	if (large_nuggets) {
		tmp = get_object();
		copy_object(large, tmp);
		tmp-> nrof = large_nuggets;
		tmp->x = x;
		tmp->y = y;
		insert_ob_in_map(tmp, op->map);
	}
}

int alchemy(object *op)
{
	int x,y,weight=0,weight_max,large_nuggets,small_nuggets;
	object *next,*tmp;

  if(op->type!=PLAYER)
    return 0;
  /* Put a maximum weight of items that can be alchemied.  Limits the power
   * some, and also prevents people from alcheming every table/chair/clock
   * in sight
   */
  weight_max = 100000 + 50000*op->level;
  small=get_archetype("smallnugget"),
  large=get_archetype("largenugget");
  for(y= op->y-1;y<=op->y+1;y++) {
    for(x= op->x-1;x<=op->x+1;x++) {
      if(out_of_map(op->map,x,y) || wall(op->map,x,y) ||
	 blocks_view(op->map,x,y))
        continue;

      small_nuggets=0;
      large_nuggets=0;

	for(tmp=get_map_ob(op->map,x,y);tmp!=NULL;tmp=next) {
          next=tmp->above;
	  if (tmp->weight>0 && !NO_PICK(tmp) && !IS_ALIVE(tmp)) {
	    if (tmp->inv) {
		object *next1,*tmp1;
		for (tmp1 = tmp->inv; tmp1!=NULL; tmp1=next1) {
		    next1 = tmp1->below;
		    if (tmp1->weight>0 && !NO_PICK(tmp1) && !IS_ALIVE(tmp1))
		        alchemy_object(tmp1, &small_nuggets, &large_nuggets,
			   &weight);
		}
	    }
	    alchemy_object(tmp, &small_nuggets, &large_nuggets, &weight);
	    
	    if (weight>weight_max) {
		update_map(op, small_nuggets, large_nuggets, x, y);
		free_object(large);
		free_object(small);
		return 1;
	    }
	  }
	}
/* Insert all the nuggets at one time.  This probably saves time, but
 * it also prevents us from alcheming nuggets that were just created
 * with this spell.
 */
	update_map(op, small_nuggets, large_nuggets, x, y);
    }
  }
  free_object(large);
  free_object(small);
  return 1;
}
