/*
 * engine.c
 *
 * The rules engine for 3Dc.  At the moment, this is
 * just the definition of the move-stack functions,
 * but later will include the computer's heuristics
 * for movement
 */
/*

    3Dc, a game of 3-Dimensional Chess
    Copyright (C) 1995  Paul Hicks

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    E-Mail: P.Hicks@net-cs.ucd.ie
*/
#include <malloc.h>
#include <stdlib.h>

#include "machine.h"
#include "3Dc.h"
#include "3DcErr.h"
int n3DcErr = 0;

#ifndef ABS
#define ABS(a) ((a) < 0 ? -(a) : (a))
#endif /* ABS */
#ifndef MAX
#define MAX(a,b) ((a) < (b) ? (b) : (a))
#define MIN(a,b) ((a) > (b) ? (b) : (a))
#endif /* MAX */

void stackPush(stack *s, const Move *newMove)
{
  struct stack_el *newEl;

  newEl = (struct stack_el *)malloc(sizeof(struct stack_el));
  newEl->mvt = (Move *)malloc(sizeof(Move));
  memcpy(newEl->mvt, newMove, sizeof(Move));
  newEl->below = s->top;
  s->top = newEl;
  s->nSize++;
}

Move *stackPop(stack *s)
{
  Move *oldMove;
  struct stack_el *oldEl;

  if (!(s->top))
    return NULL;

  oldMove = s->top->mvt;
  oldEl = s->top;
  s->top = s->top->below;
  s->nSize--;
  free(oldEl);

  return oldMove;
}

Move *stackPeek(stack *s, int numMoves)
{
  struct stack_el *oldEl;

  if (numMoves >= s->nSize)
    {
      Err3Dc("Oops! Tried to peek too far.", False);
      return NULL;
    }

  for (oldEl = s->top; numMoves; --numMoves)
    oldEl = oldEl->below;

  return oldEl->mvt;
}

/* Returns a pointer to any one piece threatening the mentioned square
 * which is an enemy of the mentioned piece.  Will return NULL is the
 * square is not threatened */
Piece *pieceThreatened(Piece *piece,
                       const int xFile, const int yRank, const int zLevel)
{
  Colour bwFriend, bwEnemy;
  int xLooper, yLooper, zLooper;

  bwFriend = piece->bwSide;
  bwEnemy = ((bwFriend == white) ? black : white);
  /*
   * First check the simple pieces: knights and cannons
   */
  if (zLevel == 1)
    { /* Middle level - must check knights */
      for (xLooper = 0; xLooper < 2; xLooper++)
        {
          if (Muster[bwEnemy][idx(knight, xLooper)]->bVisible == True &&
              pieceMayMove(Muster[bwEnemy][idx(knight, xLooper)],
                           xFile, yRank, zLevel))
            return Muster[bwEnemy][idx(knight, xLooper)];
        }
    }

      for (xLooper = 0; xLooper < 4; xLooper++)
        {
          if (Muster[bwEnemy][idx(cannon, xLooper)]->bVisible == True &&
              pieceMayMove(Muster[bwEnemy][idx(cannon, xLooper)],
                           xFile, yRank, zLevel))
            return Muster[bwEnemy][idx(cannon, xLooper)];
        }

  /*
   * The remainder of the algorithm works thus:
   * All remaining pieces can only move in straight lines,
   * horizonatally, vertically, diagonally, or up/down in some
   * way.
   * So we trace along all directions from the given point (that is,
   * anywhere from 4 to 26 directions) until we reach a piece.  If
   * the colour of the piece is not bwFriendand bMayMove for it returns
   * true, then the square is threatened.  If the square is threatened,
   * return True.  If no direction causes a True to be returned, then
   * the square isn't threatened, so return False.
   */
  /*
   * It would be more efficient to have nested conditionals,
   * but then everything gets more complicated, and this condition
   * is big enough already.
   */
  if (xFile > 0)
    { /* Can move left */
      for (xLooper = xFile - 1;
           xLooper >= 0;
           --xLooper)
        {
          if (Board[zLevel][yRank][xLooper])
            {
              if (Board[zLevel][yRank][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLevel][yRank][xLooper], xFile, yRank, zLevel))
                return Board[zLevel][yRank][xLooper];
            }
        }
    }

  if (xFile > 0 && yRank > 0)
    { /* Can move back and left */
      for (xLooper = xFile - 1, yLooper = yRank - 1;
           xLooper >= 0 && yLooper >= 0;
           --xLooper, --yLooper)
        {
          if (Board[zLevel][yLooper][xLooper])
            {
              if (Board[zLevel][yLooper][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLevel][yLooper][xLooper], xFile, yRank, zLevel))
                return Board[zLevel][yLooper][xLooper];
            }
        }
    }

  if (yRank > 0)
    { /* Can move back */
      for (yLooper = yRank - 1;
           yLooper >= 0;
           --yLooper)
        {
          if (Board[zLevel][yLooper][xFile])
            {
              if (Board[zLevel][yLooper][xFile]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLevel][yLooper][xFile], xFile, yRank, zLevel))
                return Board[zLevel][yLooper][xFile];
            }
        }
    }

  if (xFile < 7 && yRank > 0)
    { /* Can move back and right */
      for (xLooper = xFile + 1, yLooper = yRank - 1;
           xLooper <= 7 && yLooper >= 0;
           ++xLooper, --yLooper)
        {
          if (Board[zLevel][yLooper][xLooper])
            {
              if (Board[zLevel][yLooper][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLevel][yLooper][xLooper], xFile, yRank, zLevel))
                return Board[zLevel][yLooper][xLooper];
            }
        }
    }

  if (xFile < 7)
    { /* Can move right */
      for (xLooper = xFile + 1;
           xLooper <= 7;
           ++xLooper)
        {
          if (Board[zLevel][yRank][xLooper])
            {
              if (Board[zLevel][yRank][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLevel][yRank][xLooper], xFile, yRank, zLevel))
                return Board[zLevel][yRank][xLooper];
            }
        }
    }

  if (xFile < 7 && yRank < 7)
    { /* Can move forward and right */
      for (xLooper = xFile + 1, yLooper = yRank + 1;
           xLooper <= 7 && yLooper <= 7;
           ++xLooper, ++yLooper)
        {
          if (Board[zLevel][yLooper][xLooper])
            {
              if (Board[zLevel][yLooper][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLevel][yLooper][xLooper], xFile, yRank, zLevel))
                return Board[zLevel][yLooper][xLooper];
            }
        }
    }

  if (yRank < 7)
    { /* Can move forward */
      for (yLooper = yRank + 1;
      yLooper <= 7;
      ++yLooper)
        {
          if (Board[zLevel][yLooper][xFile])
            {
              if (Board[zLevel][yLooper][xFile]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLevel][yLooper][xFile], xFile, yRank, zLevel))
                return Board[zLevel][yLooper][xFile];
            }
        }
    }

  if (xFile > 0 && yRank < 7)
    { /* Can move forward and left */
      for (xLooper = xFile - 1, yLooper = yRank + 1;
           xLooper >= 0 && yLooper <= 7;
           --xLooper, ++yLooper)
        {
          if (Board[zLevel][yLooper][xLooper])
            {
              if (Board[zLevel][yLooper][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLevel][yLooper][xLooper], xFile, yRank, zLevel))
                return Board[zLevel][yLooper][xLooper];
            }
        }
    }

  if (zLevel > 0)
    { /* Can move down */
      for (zLooper = zLevel - 1;
           zLooper >= 0;
           --zLooper)
        {
          if (Board[zLooper][yRank][xFile])
            {
              if (Board[zLooper][yRank][xFile]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yRank][xFile], xFile, yRank, zLevel))
                return Board[zLooper][yRank][xFile];
            }
        }
    }

  if (xFile > 0 && zLevel > 0)
    { /* Can move down and left */
      for (xLooper = xFile - 1, zLooper = zLevel - 1;
           xLooper >= 0 && zLooper >=0;
           --xLooper, --zLooper)
        {
          if (Board[zLooper][yRank][xLooper])
            {
              if (Board[zLooper][yRank][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yRank][xLooper], xFile, yRank, zLevel))
                return Board[zLooper][yRank][xLooper];
            }
        }
    }

  if (xFile > 0 && yRank > 0 && zLevel > 0)
    { /* Can move down, back and left */
      for (xLooper = xFile - 1, yLooper = yRank - 1, zLooper = zLevel - 1;
           xLooper >= 0 && yLooper >= 0 && zLooper >= 0;
           --xLooper, --yLooper, --zLooper)
        {
          if (Board[zLooper][yLooper][xLooper])
            {
              if (Board[zLooper][yLooper][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yLooper][xLooper], xFile, yRank, zLevel))
                return Board[zLooper][yLooper][xLooper];
            }
        }
    }

  if (yRank > 0 && zLevel > 0)
    { /* Can move down and back */
      for (yLooper = yRank - 1, zLooper = zLevel - 1;
           yLooper >= 0 && zLooper >= 0;
           --yLooper, --zLooper)
        {
          if (Board[zLooper][yLooper][xFile])
            {
              if (Board[zLooper][yLooper][xFile]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yLooper][xFile], xFile, yRank, zLevel))
                return Board[zLooper][yLooper][xFile];
            }
        }
    }

  if (xFile < 7 && yRank > 0 && zLevel > 0)
    { /* Can move down, back and right */
      for (xLooper = xFile + 1, yLooper = yRank - 1, zLooper = zLevel - 1;
           xLooper <= 7 && yLooper >= 0 && zLooper >= 0;
           ++xLooper, --yLooper, --zLooper)
        {
          if (Board[zLooper][yLooper][xLooper])
            {
              if (Board[zLooper][yLooper][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yLooper][xLooper], xFile, yRank, zLevel))
                return Board[zLooper][yLooper][xLooper];
            }
        }
    }

  if (xFile < 7 && zLevel > 0)
    { /* Can move down and right */
      for (xLooper = xFile + 1, zLooper = zLevel - 1;
           xLooper <= 7 && zLooper >= 0;
           ++xLooper,  --zLooper)
        {
          if (Board[zLooper][yRank][xLooper])
            {
              if (Board[zLooper][yRank][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yRank][xLooper], xFile, yRank, zLevel))
                return Board[zLooper][yRank][xLooper];
            }
        }
    }

  if (xFile < 7 && yRank < 7 && zLevel > 0)
    { /* Can move down, forward and right */
      for (xLooper = xFile + 1, yLooper = yRank + 1, zLooper = zLevel - 1;
           xLooper <= 7 && yLooper <= 7 && zLooper >= 0;
           ++xLooper, ++yLooper, --zLooper)
        {
          if (Board[zLooper][yLooper][xLooper])
            {
              if (Board[zLooper][yLooper][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yLooper][xLooper], xFile, yRank, zLevel))
                return Board[zLooper][yLooper][xLooper];
            }
        }
    }

  if (yRank < 7 && zLevel > 0)
    { /* Can move down and forward */
      for (yLooper = yRank + 1, zLooper = zLevel - 1;
           yLooper <= 7, zLooper >= 0;
           ++yLooper, --zLooper)
        {
          if (Board[zLooper][yLooper][xFile])
            {
              if (Board[zLooper][yLooper][xFile]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yLooper][xFile], xFile, yRank, zLevel))
                return Board[zLooper][yLooper][xFile];
            }
        }
    }

  if (xFile > 0 && yRank < 7 && zLevel > 0)
    { /* Can move down, forward and left */
      for (xLooper = xFile - 1, yLooper = yRank + 1, zLooper = zLevel - 1;
           xLooper >= 0 && yLooper <= 7 && zLooper >= 0;
           --xLooper, ++yLooper, --zLooper)
        {
          if (Board[zLooper][yLooper][xLooper])
            {
              if (Board[zLooper][yLooper][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yLooper][xLooper], xFile, yRank, zLevel))
                return Board[zLooper][yLooper][xLooper];
            }
        }
    }

  if (zLevel < 2)
    { /* Can move down */
      for (zLooper = zLevel + 1;
           zLooper <= 2;
           ++zLooper)
        {
          if (Board[zLooper][yRank][xFile])
            {
              if (Board[zLooper][yRank][xFile]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yRank][xFile], xFile, yRank, zLevel))
                return Board[zLooper][yRank][xFile];
            }
        }
    }

  if (xFile > 0 && zLevel < 2)
    { /* Can move down and left */
      for (xLooper = xFile - 1, zLooper = zLevel + 1;
           xLooper >= 0 && zLooper >=0;
           --xLooper, ++zLooper)
        {
          if (Board[zLooper][yRank][xLooper])
            {
              if (Board[zLooper][yRank][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yRank][xLooper], xFile, yRank, zLevel))
                return Board[zLooper][yRank][xLooper];
            }
        }
    }

  if (xFile > 0 && yRank > 0 && zLevel < 2)
    { /* Can move down, back and left */
      for (xLooper = xFile - 1, yLooper = yRank - 1, zLooper = zLevel + 1;
           xLooper >= 0 && yLooper >= 0 && zLooper <= 2;
           --xLooper, --yLooper, ++zLooper)
        {
          if (Board[zLooper][yLooper][xLooper])
            {
              if (Board[zLooper][yLooper][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yLooper][xLooper], xFile, yRank, zLevel))
                return Board[zLooper][yLooper][xLooper];
            }
        }
    }

  if (yRank > 0 && zLevel < 2)
    { /* Can move down and back */
      for (yLooper = yRank - 1, zLooper = zLevel + 1;
           yLooper >= 0 && zLooper <= 2;
           --yLooper, ++zLooper)
        {
          if (Board[zLooper][yLooper][xFile])
            {
              if (Board[zLooper][yLooper][xFile]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yLooper][xFile], xFile, yRank, zLevel))
                return Board[zLooper][yLooper][xFile];
            }
        }
    }

  if (xFile < 7 && yRank > 0 && zLevel < 2)
    { /* Can move down, back and right */
      for (xLooper = xFile + 1, yLooper = yRank - 1, zLooper = zLevel + 1;
           xLooper <= 7 && yLooper >= 0 && zLooper <= 2;
           ++xLooper, --yLooper, ++zLooper)
        {
          if (Board[zLooper][yLooper][xLooper])
            {
              if (Board[zLooper][yLooper][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yLooper][xLooper], xFile, yRank, zLevel))
                return Board[zLooper][yLooper][xLooper];
            }
        }
    }

  if (xFile < 7 && zLevel < 2)
    { /* Can move down and right */
      for (xLooper = xFile + 1, zLooper = zLevel + 1;
           xLooper <= 7 && zLooper <= 2;
           ++xLooper,  ++zLooper)
        {
          if (Board[zLooper][yRank][xLooper])
            {
              if (Board[zLooper][yRank][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yRank][xLooper], xFile, yRank, zLevel))
                return Board[zLooper][yRank][xLooper];
            }
        }
    }

  if (xFile < 7 && yRank < 7 && zLevel < 2)
    { /* Can move down, forward and right */
      for (xLooper = xFile + 1, yLooper = yRank + 1, zLooper = zLevel + 1;
           xLooper <= 7 && yLooper <= 7 && zLooper <= 2;
           ++xLooper, ++yLooper, ++zLooper)
        {
          if (Board[zLooper][yLooper][xLooper])
            {
              if (Board[zLooper][yLooper][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yLooper][xLooper], xFile, yRank, zLevel))
                return Board[zLooper][yLooper][xLooper];
            }
        }
    }

  if (yRank < 7 && zLevel < 2)
    { /* Can move down and forward */
      for (yLooper = yRank + 1, zLooper = zLevel + 1;
           yLooper <= 7, zLooper <= 2;
           ++yLooper, ++zLooper)
        {
          if (Board[zLooper][yLooper][xFile])
            {
              if (Board[zLooper][yLooper][xFile]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yLooper][xFile], xFile, yRank, zLevel))
                return Board[zLooper][yLooper][xFile];
            }
        }
    }

  if (xFile > 0 && yRank < 7 && zLevel < 2)
    { /* Can move down, forward and left */
      for (xLooper = xFile - 1, yLooper = yRank + 1, zLooper = zLevel + 1;
           xLooper >= 0 && yLooper <= 7 && zLooper <= 2;
           --xLooper, ++yLooper, ++zLooper)
        {
          if (Board[zLooper][yLooper][xLooper])
            {
              if (Board[zLooper][yLooper][xLooper]->bwSide == bwFriend)
                break;
              else if (pieceMayMove(Board[zLooper][yLooper][xLooper], xFile, yRank, zLevel))
                return Board[zLooper][yLooper][xLooper];
            }
        }
    }

  return NULL;
}

/* Returns a legal move for the computer (or hint) to make. */
Move *genMove(const Colour bwSide)
{
  Move *genMove = NULL;
  stack *allMoves;
  Piece *moveMe = NULL, *threat, *counterMeasure;
  int pickNth, takeThreat = 1;

  /* Last-ditch attempt: make a guess! */
  srandom((int)(MoveStack->top));

  while (!genMove)
    {
      /* TODO: extend this algorithm to an exhaustive search */
      /*  If the king is in check.. */
      if ((threat = pieceThreatened(Muster[bwSide][idx(king, 0)],
                          (Muster[bwSide][idx(king, 0)])->xyzPos.xFile,
                          (Muster[bwSide][idx(king, 0)])->xyzPos.yRank,
                          (Muster[bwSide][idx(king, 0)])->xyzPos.zLevel)) != NULL)
        { /* ..and we can take the threatening piece.. */
          /* ..then we can guess a specific move.. */
          if ((counterMeasure = pieceThreatened(threat,
                                                threat->xyzPos.xFile,
                                                threat->xyzPos.yRank,
                                                threat->xyzPos.zLevel))
              != NULL &&
              takeThreat == 1) /* takeThreat is to avoid infinite loops */
            {
              genMove = (Move *)malloc(sizeof(Move));
              genMove->xyzBefore.xFile = counterMeasure->xyzPos.xFile;
              genMove->xyzBefore.yRank = counterMeasure->xyzPos.yRank;
              genMove->xyzBefore.zLevel = counterMeasure->xyzPos.zLevel;
              genMove->pVictim = threat;
              genMove->bHadMoved = counterMeasure->bHasMoved;

              takeThreat = 0;
            }
          else
            /* .. else if king is still in check, guess random move for the king */
            moveMe = Muster[bwSide][idx(king, 0)];
        }

      if (!genMove) /* genMove is not NULL if we've guessed a move already */
        {
          if (!moveMe)
            {
              while (!(moveMe = Muster[bwSide][random()%48]));
            }

          if (moveMe->bVisible)
            {
              allMoves = findAllMoves(moveMe);

              /* Pick one of the generated moves at random */
              if (allMoves)
                {
                  /* TODO: allow it detect a "no-move" scenario */
                  if (allMoves->nSize) 
                    {
                      pickNth = random()%(allMoves->nSize);

                      while (--pickNth >= 0)
                        free(stackPop(allMoves));

                      genMove = stackPop(allMoves);

                      while (allMoves->nSize > 0)
                        free(stackPop(allMoves));
                    } /* End 'did findAllMoves find any moves?' */

                  free(allMoves);
                } /* End 'did findAllMoves work?' */
            } /* End 'does piece still exist?' */
        } /* End guess random move */

      moveMe = NULL;

      /* Verify guessed move */
      /* TODO: investigate why king sometimes puts himself in check
       * despite this conditional */
      if (genMove && !pieceMayMove(Board[genMove->xyzBefore.zLevel]
                             [genMove->xyzBefore.yRank]
                             [genMove->xyzBefore.xFile],
                        genMove->xyzAfter.xFile,
                        genMove->xyzAfter.yRank,
                        genMove->xyzBefore.zLevel))
        {
          free(genMove);
          genMove = NULL; /* This means keep looping */
        }
    }

  return genMove;
}

/* Creates a stack of legal moves that this piece can take in this go */
stack *findAllMoves(const Piece *piece)
{
  stack *moves;
  Move move;
  Colour bwEnemy;
  int x, y, z;

#define CURX (piece->xyzPos.xFile)
#define CURY (piece->xyzPos.yRank)
#define CURZ (piece->xyzPos.zLevel)

  moves = (stack *)malloc(sizeof(stack));
  moves->nSize = 0;
  moves->top = NULL;

  bwEnemy = ((piece->bwSide == white) ? black : white);
  move.xyzBefore = piece->xyzPos;

  if (piece->nName == knight)
    {
      for (y = CURY -2; y <= CURY +2; y++)
        {
          if (y == CURY)
            y++;

          for (x = CURX -2; x <= CURX +2; x++)
            {
              if (x == CURX)
                x++;

              if (ABS(CURX-x) == ABS(CURY-y))
                continue;

              if (x >= 0 && x < 8 &&
                  y >= 0 && y < 8 &&
                  (Board[1][y][x] == NULL ||
                   Board[1][y][x]->bwSide == bwEnemy))
                {
                  move.xyzAfter.xFile = x;
                  move.xyzAfter.yRank = y;
                  move.xyzAfter.zLevel = 1;
                  move.pVictim = Board[1][y][x];
                  move.bHadMoved = piece->bHasMoved;

                  stackPush(moves, &move);
                } /* End valid move */
            } /* End x loop */
        } /* End y loop */
    } /* End knight */
  else if (piece->nName == cannon)
    {
      for (z = 0; z < 2; z++)
        {
          if (z == CURZ)
            z++;

          for (y = CURY -3; y <= CURY +3; y++)
            {
              if (y == CURY)
                y++;

              for (x = CURX -3; x <= CURX +3; x++)
                {
                  if (x == CURX)
                    x++;

                  if (ABS(CURX-x) == ABS(CURY-y) ||
                      ABS(CURX-x) == ABS(CURZ-z) ||
                      ABS(CURY-y) == ABS(CURZ-z))
                    continue;

                  if (x >= 0 && x < 8 &&
                      y >= 0 && y < 8 &&
                      z >= 0 && z < 3 &&
                      (Board[z][y][x] == NULL ||
                       Board[z][y][x]->bwSide == bwEnemy))
                    {
                      move.xyzAfter.xFile = x;
                      move.xyzAfter.yRank = y;
                      move.xyzAfter.zLevel = z;
                      move.pVictim = Board[z][y][x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */
                } /* End x loop */
            } /* End y loop */
        } /* End z loop */
    } /* End cannon */
  else if (piece->nName == pawn) /* Don't bother searching for en passant */
    {
      int step, side;

      step = (piece->bwSide == white ? 1 : -1);

      for (side = -1; side <= 1; side++)
        {
          if (side == 0)
            side++;

          if (((side == -1 && CURX != 0) ||
               (side ==  1 && CURX != 7)) &&
              Board[CURZ][CURY + step][CURX + side] &&
              (Board[CURZ][CURY + step][CURX + side])->bwSide == bwEnemy)
            {
              move.xyzAfter.xFile = CURX + side;
              move.xyzAfter.yRank = CURY + step;
              move.xyzAfter.zLevel = CURZ;
              move.pVictim = Board[CURZ][CURY + step][CURX + side];
              move.bHadMoved = piece->bHasMoved;

              stackPush(moves, &move);
            } /* End x = 1/-1 */
        } /* End x loop */

      if (!(Board[CURZ][CURY + step][CURX]))
        {
          move.xyzAfter.xFile = CURX;
          move.xyzAfter.yRank = CURY + step;
          move.xyzAfter.zLevel = CURZ;
          move.pVictim = NULL;
          move.bHadMoved = piece->bHasMoved;

          stackPush(moves, &move);
        } /* End move 1 forw */
      if (!(piece->bHasMoved) && !(Board[CURZ][CURY + (2*step)][CURX]))
        {
          move.xyzAfter.xFile = CURX;
          move.xyzAfter.yRank = CURY + (2*step);
          move.xyzAfter.zLevel = CURZ;
          move.pVictim = NULL;
          move.bHadMoved = False;

          stackPush(moves, &move);
        } /* End move 2 forw */
    } /* End pawn */
  else if (piece->nName == king)
    {
      for (z = MAX(CURZ -1, 0); z <= MIN(CURZ +1, 2); z++)
        {
          for (y = MAX(CURY -1, 0); y <= MIN(CURY +1, 7); y++)
            {
              for (x = MAX(CURX -1, 0); x <= MIN(CURX +1, 7); x++)
                {
                  if (x == CURX &&
                      y == CURY &&
                      z == CURZ)
                    continue;

                  if (!Board[z][y][x] ||
                      (Board[z][y][x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = x;
                      move.xyzAfter.yRank = y;
                      move.xyzAfter.zLevel = z;
                      move.pVictim = Board[z][y][x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    }
                } /* End x loop */
            } /* End y loop */
        } /* End z loop */
    } /* End king */
  else if (piece->nName == prince)
    {
      for (y = MAX(CURY -1, 0); y <= MIN(CURY +1, 7); y++)
        {
          for (x = MAX(CURX -1, 0); x <= MIN(CURX +1, 7); x++)
            {
              if (x == CURX &&
                  y == CURY)
                continue;

              if (!Board[CURZ][y][x] ||
                  (Board[CURZ][y][x])->bwSide == bwEnemy)
                {
                  move.xyzAfter.xFile = x;
                  move.xyzAfter.yRank = y;
                  move.xyzAfter.zLevel = CURZ;
                  move.pVictim = Board[CURZ][y][x];
                  move.bHadMoved = piece->bHasMoved;

                  stackPush(moves, &move);
                }
            } /* End x loop */
        } /* End y loop */
    } /* End prince */
  else
    { /* All other pieces can move `infinitely' in specific directions */
      /* Staight forw/back/side */
      if (piece->nName == galley ||
          piece->nName == rook ||
          piece->nName == princess ||
          piece->nName == queen)
        {
          /* First, forw */
          for (y = 0; y <= 7 ; y++)
            {
              if ((CURY + y) <= 7)
                {
                  if (!Board[CURZ][CURY + y][CURX] ||
                      (Board[CURZ][CURY + y][CURX])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX;
                      move.xyzAfter.yRank = CURY + y;
                      move.xyzAfter.zLevel = CURZ;
                      move.pVictim = Board[CURZ][CURY + y][CURX];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ][CURY + y][CURX])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End y loop */

          /* Now back */
          for (y = 0; y <= 7 ; y++)
            {
              if ((CURY - y) >= 0)
                {
                  if (!Board[CURZ][CURY - y][CURX] ||
                      (Board[CURZ][CURY - y][CURX])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX;
                      move.xyzAfter.yRank = CURY - y;
                      move.xyzAfter.zLevel = CURZ;
                      move.pVictim = Board[CURZ][CURY - y][CURX];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ][CURY - y][CURX])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End y loop */

          /* Now left */
          for (x = 0; x <= 7 ; x++)
            {
              if ((CURX + x) <= 7)
                {
                  if (!Board[CURZ][CURY][CURX + x] ||
                      (Board[CURZ][CURY][CURX + x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX + x;
                      move.xyzAfter.yRank = CURY;
                      move.xyzAfter.zLevel = CURZ;
                      move.pVictim = Board[CURZ][CURY][CURX + x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ][CURY][CURX + x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x loop */

          /* Now right */
          for (x = 0; x <= 7 ; x++)
            {
              if ((CURX - x) >= 0)
                {
                  if (!Board[CURZ][CURY][CURX - x] ||
                      (Board[CURZ][CURY][CURX - x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX - x;
                      move.xyzAfter.yRank = CURY;
                      move.xyzAfter.zLevel = CURZ;
                      move.pVictim = Board[CURZ][CURY][CURX - x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ][CURY][CURX + x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x loop */
        } /* End x XOR y NOT z move */

      /* Diagonally on same board */
      if (piece->nName == abbey ||
          piece->nName == bishop ||
          piece->nName == princess ||
          piece->nName == queen)
        {
          /* First, +x+y */
          for (y = 0, x = 0; x <= 7 && y <= 7 ; y++, x++)
            {
              if ((CURX + x) <= 7 && (CURY + y) <= 7)
                {
                  if (!Board[CURZ][CURY + y][CURX + x] ||
                      (Board[CURZ][CURY + y][CURX + x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX + x;
                      move.xyzAfter.yRank = CURY + y;
                      move.xyzAfter.zLevel = CURZ;
                      move.pVictim = Board[CURZ][CURY + y][CURX + x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ][CURY + y][CURX + x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/y loop */

          /* Now -x-y */
          for (y = 0, x = 0; x <= 7 && y <= 7 ; y++, x++)
            {
              if ((CURX - x) >= 0 && (CURY - y) >= 0)
                {
                  if (!Board[CURZ][CURY - y][CURX - x] ||
                      (Board[CURZ][CURY - y][CURX - x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX - x;
                      move.xyzAfter.yRank = CURY - y;
                      move.xyzAfter.zLevel = CURZ;
                      move.pVictim = Board[CURZ][CURY - y][CURX - x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ][CURY - y][CURX - x])
                    break; /* Reached a blocking piece */
                } /* End 'on board? */
            } /* End x/y loop */

          /* Now +x-y */
          for (y = 0, x = 0; x <= 7 && y <= 7 ; y++, x++)
            {
              if ((CURX + x) <= 7 && (CURY - y) >= 0)
                {
                  if (!Board[CURZ][CURY - y][CURX + x] ||
                      (Board[CURZ][CURY - y][CURX + x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX + x;
                      move.xyzAfter.yRank = CURY - y;
                      move.xyzAfter.zLevel = CURZ;
                      move.pVictim = Board[CURZ][CURY - y][CURX + x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ][CURY - y][CURX + x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/y loop */

          /* Now -x+y */
          for (y = 0, x = 0; x <= 7 && y <= 7 ; y++, x++)
            {
              if ((CURX - x) >= 0 && (CURY + y) <= 7)
                {
                  if (!Board[CURZ][CURY + y][CURX - x] ||
                      (Board[CURZ][CURY + y][CURX - x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX - x;
                      move.xyzAfter.yRank = CURY + y;
                      move.xyzAfter.zLevel = CURZ;
                      move.pVictim = Board[CURZ][CURY + y][CURX - x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ][CURY + y][CURX - x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/y loop */
        } /* End x AND y NOT z movement */

      /* Straight up/down, diagonally forw/back/side and up/down */
      if (piece->nName == rook ||
          piece->nName == queen)
        {
          /* First straight up */
          for (z = 0; z <= 2; z++)
            {
              if ((CURZ + z) <= 2)
                {
                  if (!Board[CURZ + z][CURY][CURX] ||
                      (Board[CURZ + z][CURY][CURX])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX;
                      move.xyzAfter.yRank = CURY;
                      move.xyzAfter.zLevel = CURZ + z;
                      move.pVictim = Board[CURZ + z][CURY][CURX];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ + z][CURY][CURX])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End z loop */

          /* Now straight down */
          for (z = 0; z <= 2; z++)
            {
              if ((CURZ + z) <= 2)
                {
                  if (!Board[CURZ - z][CURY][CURX] ||
                      (Board[CURZ - z][CURY][CURX])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX;
                      move.xyzAfter.yRank = CURY;
                      move.xyzAfter.zLevel = CURZ - z;
                      move.pVictim = Board[CURZ - z][CURY][CURX];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ - z][CURY][CURX])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End z loop */

          /* Now diagonally forw and up */
          for (x = 0, z = 0; x <= 7 && z <= 2; x++, z++)
            {
              if ((CURX + x) <= 7 && (CURZ + z) <= 2)
                {
                  if (!Board[CURZ + z][CURY][CURX + x] ||
                      (Board[CURZ + z][CURY][CURX + x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX + x;
                      move.xyzAfter.yRank = CURY;
                      move.xyzAfter.zLevel = CURZ + z;
                      move.pVictim = Board[CURZ + z][CURY][CURX + x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ + z][CURY][CURX + x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/z loop */

          /* Now diagonally back and up */
          for (x = 0, z = 0; x <= 7 && z <= 2; x++, z++)
            {
              if ((CURX - x) >= 0 && (CURZ + z) <= 2)
                {
                  if (!Board[CURZ + z][CURY][CURX - x] ||
                      (Board[CURZ + z][CURY][CURX - x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX - x;
                      move.xyzAfter.yRank = CURY;
                      move.xyzAfter.zLevel = CURZ + z;
                      move.pVictim = Board[CURZ + z][CURY][CURX - x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ - z][CURY][CURX - x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/z loop */

          /* Now diagonally right and up */
          for (y = 0, z = 0; y <= 7 && z <= 2; y++, z++)
            {
              if ((CURY + y) <= 7 && (CURZ + z) <= 2)
                {
                  if (!Board[CURZ + z][CURY + y][CURX] ||
                      (Board[CURZ + z][CURY + y][CURX])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX;
                      move.xyzAfter.yRank = CURY + y;
                      move.xyzAfter.zLevel = CURZ + z;
                      move.pVictim = Board[CURZ + z][CURY + y][CURX];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ + z][CURY + y][CURX])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End y/z loop */

          /* Now diagonally left and up */
          for (y = 0, z = 0; y <= 7 && z <= 2; y++, z++)
            {
              if ((CURY - y) >= 0 && (CURZ + z) <= 2)
                {
                  if (!Board[CURZ + z][CURY - y][CURX] ||
                      (Board[CURZ + z][CURY - y][CURX])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX;
                      move.xyzAfter.yRank = CURY - y;
                      move.xyzAfter.zLevel = CURZ + z;
                      move.pVictim = Board[CURZ + z][CURY - y][CURX];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ + z][CURY - y][CURX])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End y/z loop */

          /* Now diagonally right and down */
          for (x = 0, z = 0; x <= 7 && z <= 2; x++, z++)
            {
              if ((CURX + x) <= 7 && (CURZ - z) >= 0)
                {
                  if (!Board[CURZ - z][CURY][CURX + x] ||
                      (Board[CURX - z][CURY][CURZ + x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX + x;
                      move.xyzAfter.yRank = CURY;
                      move.xyzAfter.zLevel = CURZ - z;
                      move.pVictim = Board[CURZ - z][CURY][CURX + x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ - z][CURY][CURX + x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/z loop */

          /* Now diagonally left and down */
          for (x = 0, z = 0; x <= 7 && z <= 2; x++, z++)
            {
              if ((CURX - x) >= 0 && (CURZ - z) >= 0)
                {
                  if (!Board[CURZ - z][CURY][CURX - x] ||
                      (Board[CURZ - z][CURY][CURX - x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX - x;
                      move.xyzAfter.yRank = CURY;
                      move.xyzAfter.zLevel = CURZ - z;
                      move.pVictim = Board[CURZ - z][CURY][CURX - x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ + z][CURY][CURX - x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/z loop */

          /* Now diagonally forw and down */
          for (y = 0, z = 0; y <= 7 && z <= 2; y++, z++)
            {
              if ((CURY + y) <= 7 && (CURZ - z) >= 0)
                {
                  if (!Board[CURZ - z][CURY + y][CURX] ||
                      (Board[CURZ - z][CURY + y][CURX])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX;
                      move.xyzAfter.yRank = CURY + y;
                      move.xyzAfter.zLevel = CURZ - z;
                      move.pVictim = Board[CURZ - z][CURY + y][CURX];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ - z][CURY + y][CURX])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End y/z loop */

          /* Now diagonally back and down */
          for (y = 0, z = 0; y <= 7 && z <= 2; y++, z++)
            {
              if ((CURY - y) >= 0 && (CURZ - z) >= 0)
                {
                  if (!Board[CURZ - z][CURY - y][CURX] ||
                      (Board[CURZ - z][CURY - y][CURX])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX;
                      move.xyzAfter.yRank = CURY - y;
                      move.xyzAfter.zLevel = CURZ - z;
                      move.pVictim = Board[CURZ - z][CURY - y][CURX];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ - z][CURY - y][CURX])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End y/z loop */
        } /* End x XOR y AND z movement */

      /* Diagonally forw/back _and_ side and up/down */
      if (piece->nName == bishop ||
          piece->nName == queen)
        {
          /* First diagonally forw, right and up */
          for (x = 0, y = 0, z = 0; x <= 7 && y <= 7 && z <= 2; x++, y++, z++)
            {
              if ((CURX + x) <= 7 && (CURY + y) <= 7 && (CURZ + z) <= 2)
                {
                  if (!Board[CURZ + z][CURY + y][CURX + x] ||
                      (Board[CURZ + z][CURY + y][CURX + x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX + x;
                      move.xyzAfter.yRank = CURY + y;
                      move.xyzAfter.zLevel = CURZ + z;
                      move.pVictim = Board[CURZ + z][CURY + y][CURX + x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ + z][CURY + y][CURX + x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/y/z loop */

          /* Now diagonally back, right and up */
          for (x = 0, y = 0, z = 0; x <= 7 && y <= 7 && z <= 2; x++, y++, z++)
            {
              if ((CURX + x) <= 7 && (CURY - y) >= 0 && (CURZ + z) <= 2)
                {
                  if (!Board[CURZ + z][CURY - y][CURX + x] ||
                      (Board[CURZ + z][CURY - y][CURX + x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX + x;
                      move.xyzAfter.yRank = CURY - y;
                      move.xyzAfter.zLevel = CURZ + z;
                      move.pVictim = Board[CURZ + z][CURY - y][CURX + x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ + z][CURY - y][CURX + x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/y/z loop */

          /* Now diagonally forw, left and up */
          for (x = 0, y = 0, z = 0; x <= 7 && y <= 7 && z <= 2; x++, y++, z++)
            {
              if ((CURX - x) >= 0 && (CURY + y) <= 7 && (CURZ + z) <= 2)
                {
                  if (!Board[CURZ + z][CURY + y][CURX - x] ||
                      (Board[CURZ + z][CURY + y][CURX - x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX - x;
                      move.xyzAfter.yRank = CURY + y;
                      move.xyzAfter.zLevel = CURZ + z;
                      move.pVictim = Board[CURZ + z][CURY + y][CURX - x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ + z][CURY + y][CURX - x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/y/z loop */

          /* Now diagonally back, left and up */
          for (x = 0, y = 0, z = 0; x <= 7 && y <= 7 && z <= 2; x++, y++, z++)
            {
              if ((CURX - x) >= 0 && (CURY - y) >= 0 && (CURZ + z) <= 2)
                {
                  if (!Board[CURZ + z][CURY - y][CURX - x] ||
                      (Board[CURZ + z][CURY - y][CURX - x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX - x;
                      move.xyzAfter.yRank = CURY - y;
                      move.xyzAfter.zLevel = CURZ + z;
                      move.pVictim = Board[CURZ + z][CURY - y][CURX - x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ + z][CURY - y][CURX - x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/y/z loop */

          /* Now diagonally forw, right, down */
          for (x = 0, y = 0, z = 0; x <= 7 && y <= 7 && z <= 2; x++, y++, z++)
            {
              if ((CURX + x) <= 7 && (CURY + y) <= 7 && (CURZ - z) >= 0)
                {
                  if (!Board[CURZ - z][CURY + y][CURX + x] ||
                      (Board[CURZ - z][CURY + y][CURX + x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX + x;
                      move.xyzAfter.yRank = CURY + y;
                      move.xyzAfter.zLevel = CURZ - z;
                      move.pVictim = Board[CURZ - z][CURY + y][CURX + x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ - z][CURY + y][CURX + x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/y/z loop */

          /* Now diagonally back, right and down */
          for (x = 0, y = 0, z = 0; x <= 7 && y <= 7 && z <= 2; x++, y++, z++)
            {
              if ((CURX + x) <= 7 && (CURY - y) >= 0 && (CURZ - z) >= 0)
                {
                  if (!Board[CURZ - z][CURY - y][CURX + x] ||
                      (Board[CURZ - z][CURY - y][CURX + x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX + x;
                      move.xyzAfter.yRank = CURY - y;
                      move.xyzAfter.zLevel = CURZ - z;
                      move.pVictim = Board[CURZ - z][CURY - y][CURX + x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ - z][CURY - y][CURX + x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/y/z loop */

          /* Now diagonally forw, left and down */
          for (x = 0, y = 0, z = 0; x <= 7 && y <= 7 && z <= 2; x++, y++, z++)
            {
              if ((CURX - x) >= 0 && (CURY + y) <= 7 && (CURZ - z) >= 0)
                {
                  if (!Board[CURZ - z][CURY + y][CURX - x] ||
                      (Board[CURZ - z][CURY + y][CURX - x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX - x;
                      move.xyzAfter.yRank = CURY + y;
                      move.xyzAfter.zLevel = CURZ - z;
                      move.pVictim = Board[CURZ - z][CURY + y][CURX - x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ - z][CURY + y][CURX - x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/y/z loop */

          /* Now diagonally back, left and down */
          for (x = 0, y = 0, z = 0; x <= 7 && y <= 7 && z <= 2; x++, y++, z++)
            {
              if ((CURX - x) >= 0 && (CURY - y) >= 0 && (CURZ - z) >= 0)
                {
                  if (!Board[CURZ - z][CURY - y][CURX - x] ||
                      (Board[CURZ - z][CURY - y][CURX - x])->bwSide == bwEnemy)
                    {
                      move.xyzAfter.xFile = CURX - x;
                      move.xyzAfter.yRank = CURY - y;
                      move.xyzAfter.zLevel = CURZ - z;
                      move.pVictim = Board[CURZ - z][CURY - y][CURX - x];
                      move.bHadMoved = piece->bHasMoved;

                      stackPush(moves, &move);
                    } /* End valid move */

                  if (Board[CURZ - z][CURY - y][CURX - x])
                    break; /* Reached a blocking piece */
                } /* End 'on board?' */
            } /* End x/y/z loop */
        } /* End x AND y AND z movement */

    } /* End all other pieces */

  return moves;

#undef CURX
#undef CURY
#undef CURZ
}
