/*
 * The X interface for 3Dc
 */
/*

    3Dc, a game of 3-Dimensional Chess
    Copyright (C) 1995  Paul Hicks

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    E-Mail: P.Hicks@net-cs.ucd.ie
*/
#include <stdio.h>
#include <malloc.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <X11/X.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include <X11/Shell.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/List.h>
#include "DrawingA.h"

#include <X11/xpm.h>
#include "../xpm/pieces.xpm"

#include "machine.h"
#include "3Dc.h"

GfxInfo gfxInfo;
static GC gc;
static Widget board[3];
static int width[3], height[3], errX, errY;

static int initPixmaps(void);
static INLINE int initMainWindow(void);
static int initBoardWindows(void);

/* Callbacks */
static void drawBoard(Widget, XtPointer, XtPointer);
static void mouseInput(Widget, XtPointer, XtPointer);
static void quit3Dc(Widget, XtPointer, XtPointer);
static void promote(Widget, XtPointer, XtPointer);

extern int Init3DcGFX(int argc, char **argv)
{
  XtAppContext app;
  XColor col, dummy;

  app = XtCreateApplicationContext();

  XtSetLanguageProc(NULL, (XtLanguageProc)NULL, NULL);

  gfxInfo.mainWindow = XtVaAppInitialize(&app, "3Dc", NULL, 0, &argc, argv,
                                         NULL, NULL);

  gc = XDefaultGCOfScreen(XtScreen(gfxInfo.mainWindow));
  XSetFunction(XtDisplay(gfxInfo.mainWindow), gc, GXcopy);

  gfxInfo.whitePixel = WhitePixelOfScreen(XtScreen(gfxInfo.mainWindow));
  gfxInfo.blackPixel = BlackPixelOfScreen(XtScreen(gfxInfo.mainWindow));
  if (!XAllocNamedColor(XtDisplay(gfxInfo.mainWindow),
                        DefaultColormapOfScreen(XtScreen(gfxInfo.mainWindow)),
                        "grey", &col, &dummy))
    {
      if (!XAllocNamedColor(XtDisplay(gfxInfo.mainWindow),
          DefaultColormapOfScreen(XtScreen(gfxInfo.mainWindow)),
                            "light grey", &col, &dummy))
        {
          gfxInfo.greyPixel = gfxInfo.blackPixel;
        }
      else
        gfxInfo.greyPixel = col.pixel;
    }
  else
    gfxInfo.greyPixel = col.pixel;

  if (initPixmaps())
    return 1;
  if (initMainWindow())
    return 1;
  if (initBoardWindows())
    return 1;

  XtRealizeWidget(gfxInfo.mainWindow);

  return 0;
}

extern void Quit3DcGFX(void)
{
  return; /* Easy, huh? ;-) */
}

static int initPixmaps(void)
{
  Colour bwCol;
  Title nTitle;
  int ret = 0;
  XpmAttributes attrs;
  XpmColorSymbol cols[2];

  cols[0].name = strdup("foreground");
  cols[0].value = NULL;

  cols[1].name = strdup("edge");
  cols[1].value = NULL;

  attrs.valuemask = XpmColorSymbols;
  attrs.colorsymbols = cols;
  attrs.numsymbols = 2;

  for (bwCol = white; bwCol <= black; ++bwCol)
    {
      if (bwCol == white)
        {
          cols[0].pixel = gfxInfo.whitePixel;
          cols[1].pixel = gfxInfo.blackPixel;
        }
      else
        {
          cols[0].pixel = gfxInfo.blackPixel;
          cols[1].pixel = gfxInfo.whitePixel;
        }

      for (nTitle = king; nTitle <= pawn; ++nTitle)
        {
          ret |= XpmCreatePixmapFromData(XtDisplay(gfxInfo.mainWindow),
                        XRootWindowOfScreen(XtScreen(gfxInfo.mainWindow)),
                        XPMpixmaps[nTitle],
                        &(gfxInfo.face[bwCol][nTitle]),
                        &(gfxInfo.mask[bwCol][nTitle]),
                        &attrs);
        }
    }

  free(cols[0].name);
  free(cols[1].name);

  if (ret != 0)
    {
      printf("Error reading XPM images.\n");
      return 1;
    }

  return 0;
}

static INLINE int initMainWindow(void)
{
  Widget
    form,
      /*   remark,   */
      /* undo, */ quit;
  int bg;

  form = XtVaCreateManagedWidget("form", formWidgetClass, gfxInfo.mainWindow,
                                 NULL);

  gfxInfo.remark = XtVaCreateManagedWidget("remark", labelWidgetClass, form,
                                           XtNlabel, "Welcome to 3Dc",
                                           NULL);

  /* Eliminate border */
  XtVaGetValues(gfxInfo.remark, XtNbackground, &bg, NULL);
  XtVaSetValues(gfxInfo.remark, XtNborder, bg, NULL);

  gfxInfo.undo = XtVaCreateManagedWidget("Undo", commandWidgetClass, form,
                                         XtNlabel, "Undo",
                                         XtNfromVert, gfxInfo.remark,
                                         NULL);

  quit = XtVaCreateManagedWidget("Quit", commandWidgetClass, form,
                                 XtNlabel, "Quit",
                                 XtNfromVert, gfxInfo.remark,
                                 XtNfromHoriz, gfxInfo.undo,
                                 NULL);

  XtAddCallback(gfxInfo.undo, XtNcallback, undoMove, NULL);
  XtAddCallback(quit, XtNcallback, quit3Dc, NULL);

  return 0;
}

static int initBoardWindows(void)
{
  Widget curShell;
  char *boardName;
  int zCounter;

  boardName = (char *)malloc(14);
  sprintf(boardName, "3Dc board ?");

  for (zCounter = 0; zCounter < 3; ++zCounter)
    {
      boardName[10] = zCounter + 'X';
      curShell =
          XtVaCreateManagedWidget(boardName, applicationShellWidgetClass,
                                  gfxInfo.mainWindow,
                                  NULL);

      board[zCounter] = XtVaCreateManagedWidget(boardName,
                                         drawingAreaWidgetClass, curShell,
                                         XtNwidth, 256,
                                         XtNheight, 256,
                                         NULL);

      XtAddCallback(board[zCounter], XtNinputCallback, mouseInput, NULL);
      XtAddCallback(board[zCounter], XtNexposeCallback, drawBoard, NULL);
      XtAddCallback(board[zCounter], XtNresizeCallback, drawBoard, NULL);
      XtRealizeWidget(curShell);

      width[zCounter] = height[zCounter] = 32;
      XtPopup(curShell, XtGrabNone);
    }

  return 0;
}

/* Draw one level of the board */
static void drawBoard(Widget w, XtPointer client, XtPointer call)
{
  Dimension boardWidth, boardHeight;
  unsigned curX, curY;
  Level boardNum;

  for (boardNum = 0; boardNum < 3; ++boardNum)
    {
      if (w == board[boardNum])
        break;
    }

  XtVaGetValues(board[boardNum],
                XtNheight, &boardHeight,
                XtNwidth, &boardWidth,
                NULL);

  width[boardNum] = boardWidth / 8;
  errX = boardWidth % 8;
  errX /= 2;
  height[boardNum] = boardHeight / 8;
  errY = boardWidth % 8;
  errY /= 2;

  for (curY = 0; curY < 8; ++curY)
    {
      for (curX = 0; curX < 8; ++curX)
        {
          if (((curX + curY) % 2) == 0)
            {
              XSetForeground(XtDisplay(w), gc,
                             gfxInfo.whitePixel);
            }
          else
            {
              XSetForeground(XtDisplay(w), gc,
                             gfxInfo.greyPixel);
            }

          if (Board[boardNum][curY][curX] == NULL)
            {
              XRectangle rect = {0, 0, 32, 32};

              rect.width = MIN(32, width[boardNum]);
              rect.height = MIN(32, height[boardNum]);

              XSetClipRectangles(XtDisplay(gfxInfo.mainWindow), gc,
                                 errX+ width[boardNum] * curX,
                                 errY+ height[boardNum] * curY,
                                 &rect, 1, Unsorted);
                     
              XFillRectangle(XtDisplay(w), XtWindow(w), gc,
                             errX+ width[boardNum] * curX,
                             errY+ height[boardNum] * curY,
                             width[boardNum], height[boardNum]);
            }
          else /* if (Board[boardNum][curY][curX] != NULL) */
            {
              pieceDisplay(Board[boardNum][curY][curX], TRUE);
            }
        }
    }

  return;
}

extern void pieceDisplay(Piece *piece, const Boolean bDraw)
{
  XRectangle rect = {0, 0, 32, 32};

  if (!piece)
    return;

  rect.width = MIN(32, width[piece->xyzPos.zLevel]);
  rect.height = MIN(32, height[piece->xyzPos.zLevel]);

  if (((piece->xyzPos.xFile + piece->xyzPos.yRank) % 2) == 0)
    {
      XSetForeground(XtDisplay(gfxInfo.mainWindow), gc, gfxInfo.whitePixel);
    }
  else
    {
      XSetForeground(XtDisplay(gfxInfo.mainWindow), gc, gfxInfo.greyPixel);
    }

  XSetClipRectangles(XtDisplay(gfxInfo.mainWindow), gc,
                     errX+ width[piece->xyzPos.zLevel] * piece->xyzPos.xFile,
                     errY+ height[piece->xyzPos.zLevel] * piece->xyzPos.yRank,
                     &rect, 1, Unsorted);
                     
  XFillRectangle(XtDisplay(gfxInfo.mainWindow),
                 XtWindow(board[piece->xyzPos.zLevel]), gc,
                 errX + width[piece->xyzPos.zLevel] * piece->xyzPos.xFile,
                 errY + height[piece->xyzPos.zLevel] * piece->xyzPos.yRank,
                 width[piece->xyzPos.zLevel],
                 height[piece->xyzPos.zLevel]);

  if (piece->bVisible && bDraw)
    {
      XSetClipOrigin(XtDisplay(gfxInfo.mainWindow), gc,
                     errX+ width[piece->xyzPos.zLevel] * piece->xyzPos.xFile,
                     errY+ height[piece->xyzPos.zLevel] * piece->xyzPos.yRank);
      XSetClipMask(XtDisplay(gfxInfo.mainWindow), gc,
                   gfxInfo.mask[piece->bwSide][piece->nName]);
      XCopyArea(XtDisplay(gfxInfo.mainWindow),
                gfxInfo.face[piece->bwSide][piece->nName],
                XtWindow(board[piece->xyzPos.zLevel]), gc,
                0, 0,
                MIN(32, width[piece->xyzPos.zLevel]),
                MIN(32, height[piece->xyzPos.zLevel]),
                errX + width[piece->xyzPos.zLevel] * piece->xyzPos.xFile,
                errY + height[piece->xyzPos.zLevel] * piece->xyzPos.yRank);
    }

  return;
}

extern void DrawBoard(void)
{
  Level z;

  for (z = 0; z < 3; z++)
    {
      drawBoard(board[z], NULL, NULL);
    }
  return;
}

extern int Err3Dc(const char *pszLeader, const Boolean beep)
{
  char *err;

  /*
   * All strings are designed to be printed thus:
   *      printf("That piece %s.\n");
   */
  error_t ERRORS[] = {
    {E3DcSIMPLE, "may not move thus"},
    {E3DcLEVEL, "may not move vertically"},
    {E3DcCHECK, "would place your king in check"},
    {E3DcDIST, "may not move that far"},
    {E3DcINVIS, "is not currently on the board"},
    {E3DcBLOCK, "is blocked from moving there"},
    {E3DcMOVED, "has already moved"}
  };

  if (pszLeader == NULL)
    {
      XtVaSetValues(gfxInfo.remark,
                    XtNlabel, "",
                    NULL);
      return 0;
    }

  err = (char *)malloc(strlen(pszLeader) + 40);
  sprintf(err, pszLeader, ERRORS[n3DcErr].pszErrStr);

  XtVaSetValues(gfxInfo.remark,
                XtNlabel, err,
                NULL);

  if (beep == True)
    XBell(XtDisplay(gfxInfo.mainWindow), 0);

  return 0;
}

static void mouseInput(Widget w, XtPointer client, XtPointer call)
{
  static Coord src;
  XADCS *data;
  Piece *piece;
  int moved;
  Level boardNum;
  char netMove[6];
  int xWinRel, yWinRel, dummy1, dummy2, dummy3;
  Window win, dummy;
  Cursor cursor;

  data = (XADCS *)call;

  if (data->event->type == ButtonPress)
    {
      cursor = XCreateFontCursor(XtDisplay(w), XC_exchange);

      /* Getting boardNum is easy for ButtonPress */
      for (boardNum = 0; boardNum < 3; ++boardNum)
        {
          /* While we're here, let's change the cursor */
          XDefineCursor(XtDisplay(w), XtWindow(w), cursor);

          if (w == board[boardNum])
            break;
        }

      src.xFile = data->event->xbutton.x / width[boardNum];
      src.yRank = data->event->xbutton.y / height[boardNum];
      src.zLevel = boardNum;
    }
  else if (data->event->type == ButtonRelease)
    {
      /* For ButtonRelease, we have to take care of the release being
       * in another board */
      for (boardNum = 0; boardNum < 3; ++boardNum)
        {
          /* And now we have to restore the cursor */
          XUndefineCursor(XtDisplay(w), XtWindow(w));

          XQueryPointer(XtDisplay(w), XtWindow(XtParent(board[boardNum])), &dummy,
                        &win, &dummy1, &dummy2, &xWinRel, &yWinRel, &dummy3);

          if (win == XtWindow(board[boardNum]))
            break;
        }
      if (boardNum == 3)
        {
          Err3Dc("Can't move outside boards", True);
          return;
        }

     /* First - ignore clicks on the one square */
     if (src.xFile == xWinRel / width[boardNum] &&
         src.yRank == yWinRel / height[boardNum] &&
         src.zLevel == boardNum)
       {
         Err3Dc(NULL, False);
         return;
       }

      piece = Board[src.zLevel][src.yRank][src.xFile];

      if (!piece)
        {
          Err3Dc("There's no piece there!", False);
          return;
        }

      if (piece->bwSide != bwToMove)
        {
          Err3Dc("It's not your go yet!", False);
          return;
        }

      if ((piece->bwSide == white && CLIENT) ||
          (piece->bwSide == black && SERVER) ||
          (piece->bwSide == AutoPlay))
        {
          Err3Dc("That's not your piece!", False);
          return;
        }

      moved = pieceMove(piece,
                        xWinRel / width[boardNum], yWinRel / height[boardNum],
                        boardNum);

      if (moved)
        {
          pieceDisplay(piece, True);
          bwToMove = (bwToMove == white ? black : white);
          Err3Dc(NULL, False);

          if (CLIENT || SERVER)
            {
              /* Send move */
              netMove[0] = src.zLevel + 'X';
              netMove[1] = src.xFile + 'a';
              netMove[2] = src.yRank + '1';
              netMove[3] = (char)boardNum + 'X';
              netMove[4] = xWinRel / width[boardNum] + 'a';
              netMove[5] = yWinRel / height[boardNum] + '1';

              write(TCP_FD, netMove, 6);
            } /* End net stuff */
        } /* End "this was a good move" */
      else
        Err3Dc("That piece %s.", True);
    } /* End "this was a buttonrelease" */
}

void undoMove(Widget w, XtPointer client, XtPointer call)
{
  if (pieceUndo() == False)
    return;

  /* TODO: pop up dialog asking for permission to undo */
  if ((SERVER || CLIENT) &&
      w != NULL) /* w only equals NULL if this undo came from TCP_FD */
    write(TCP_FD, "!!UNDO", 6);

  /* Undo means that the same guy goes again... */
  bwToMove = (bwToMove == white ? black : white);
}

/* Prompt for piece type to promote the pawn to */
void piecePromote(Piece *piece)
{
  Widget dialog, scroll, list;
  char *types[] = {
    "Queen", "Rook", "Bishop", "Knight",
    "Princess", "Galley", "Abbey", "Cannon"
  };

  DontMove = 1; /* Suspend thinking */

  dialog = XtCreatePopupShell("Promote to what piece?",
                              transientShellWidgetClass,
                              gfxInfo.mainWindow, NULL, 0);

  scroll = XtVaCreateManagedWidget("scroll", viewportWidgetClass, dialog,
                                   XtNallowVert, True,
                                   XtNallowHoriz, True,
                                   XtNuseBottom, True,
                                   NULL);

  list = XtVaCreateManagedWidget("list", listWidgetClass, scroll,
                                 XtNlist, types,
                                 XtNverticalList, True,
                                 XtNforceColumns, True,
                                 XtNdefaultColumns, 1,
                                 XtNborderWidth, 1,
                                 NULL);

  XtAddCallback(list, XtNcallback, promote, (XtPointer)piece);
  XtManageChild(dialog);

  return;
}

/* The promotion callback */
static void promote(Widget w, XtPointer client, XtPointer call)
{
  Piece *piece;

  piece = (Piece *)client;
  pieceDisplay(piece, False);

  switch (((XawListReturnStruct *)call)->list_index)
    {
    case 0:
      piece->nName = queen;
      break;
    case 1:
      piece->nName = rook;
      break;
    case 2:
      piece->nName = bishop;
      break;
    case 3:
      piece->nName = knight;
      break;
    case 4:
      piece->nName = princess;
      break;
    case 5:
      piece->nName = galley;
      break;
    case 6:
      piece->nName = abbey;
      break;
    case 7:
      piece->nName = cannon;
      break;
    }
  
  pieceDisplay(piece, True);

  XtPopdown(XtParent(XtParent(w)));
  XtDestroyWidget(XtParent(XtParent(w)));

  DontMove = 0; /* Resume thinking */

  return;
}

static void quit3Dc(Widget w, XtPointer client, XtPointer call)
{
  if (SERVER || CLIENT)
    close(TCP_FD);

  XtDestroyApplicationContext(XtWidgetToApplicationContext(w));
  gfxInfo.mainWindow = NULL;
  return;
}
