// Filename:   color.h
// Contents:   the object definitions for the color (and terminal
//     capabilities) object
// Author: Greg Shaw
// Created:    1/29/95

#ifndef _COLOR_H_
#define _COLOR_H_

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */


// NOTE: the below defines whether it is an error for something other than
// a recognized key sequence to be valid in a string.  (e.g. anything other
// than a valid key sequence generates an error)
#define SLASH_KEYWORD

// definitions
#define ESC            0x1B     // escape code
#define NON_ANSI       -1       // non ansi
// mono, 80x25
#define ANSI_MONO_80x25        '2'
#define ANSI_COLOR_80x25   '3'  // color, 80x25
#define ANSI_NORMAL        0    // normal attributes (none set)
#define ANSI_BOLD      1        // bold
#define ANSI_UNDERLINE_MONO    4// underline (mono only)
#define ANSI_BLINK     5        // blinking
#define ANSI_INVERSE       7    // inverse
// black foreground
#define ANSI_FORE_BLACK        30
// red foreground
#define ANSI_FORE_RED          31
// green foreground
#define ANSI_FORE_GREEN        32
#define ANSI_FORE_YELLOW   33   // yellow foreground
// blue foreground
#define ANSI_FORE_BLUE         34
#define ANSI_FORE_MAGENTA  35   // magenta foreground
// cyan foreground
#define ANSI_FORE_CYAN         36
// white foreground
#define ANSI_FORE_WHITE        37
// black background
#define ANSI_BACK_BLACK        40
// red background
#define ANSI_BACK_RED          41
// green background
#define ANSI_BACK_GREEN        42
#define ANSI_BACK_YELLOW   43   // yellow background
#define ANSI_BACK_BLUE     44   // blue background
#define ANSI_BACK_MAGENTA  45   // magenta background
// cyan background
#define ANSI_BACK_CYAN         46
// white background
#define ANSI_BACK_WHITE        47
#define BLACK          1        // black
#define RED            2        // red
#define GREEN          3        // green
#define YELLOW         4        // yellow
#define BLUE           5        // blue
#define MAGENTA            6    // magenta
#define CYAN           7        // cyan
#define WHITE          8        // white

// Structure:  attributes
// Purpose:    represent the current terminal characteristics
// Author: Greg Shaw
// Created:    1/29/95

typedef struct
{
	unsigned   bold       : 1;  // high intensity (bold)
	unsigned   blink      : 1;  // blinking
	unsigned   inverse    : 1;  // inverse video
	unsigned   underline  : 1;  // underlined
	unsigned   dim        : 1;  // dim
	unsigned   normal     : 1;  // normal
	unsigned   spare      : 10; // unused
} Attributes;


// Object: color
// Purpose:    Add color and other capabilites to the output of the BBS.
// Members:    fore_c - foreground color
//     back_c - background color
//     attributes:
//         high intensity (bold)
//         blinking
//         inverse
//         underlined (mono only)
//         normal
//         dim
//     location - current cursor location
//     s_location - saved location
//     default_foreground
//     default_background
// Methods:
//     clear - clear the screen
//     foreground - set foreground color
//     background - set background color
//     bold - set bold
//     blink - set blink
//     inverse - set inverse video
//     underline - set underline
//     colorize - replace string with colorized string
//     screen_mode - set screen mode
// Notes:  This is the initial incarnation of this object.  It
//     currently only handles ANSI.  In the future, curses
//     capabilities should be added.
// Author: Greg Shaw
// Created:    1/29/95


class color: public bbsint
{
	// NOTE: most of the values are static in color.C
	int    has_ansi(void);      // has ansi
	char   color_string[50];    // current color string (scratch)
	public:
		char *ansi_color_string(int fore, int back, Attributes attr);
		// format ansi color string
		// set background color
		char *background(int val);
		char *blink(void);      // (re)set blinking
		char *bold(void);       // (re)set bold
		void check_normal(void);// set normal bit after any change
		// return length of string w/o embedded
		int  clean_len(char *str);
		void clear_scr(void);   // clear (home and clear) screen
		color();                // constructor
		char *dim(void);        // (re)set dim (!bold)
		// display file for user
		display_file(char *path, char page);
		// make a string longer
		char *expand_str(char *str, char *more);
		// set foreground color
		char *foreground(int val);
		char *inverse(void);    // (re)set inverse
		// generate cursor movement command
		char *cmove(int x, int y);
		// left justified movement
		char *rjmove(int x, int y, int len);
		char *normal(void);     // (re)set normal (default attributes)
		// process string
		char *process(char *str);
		void set_mode(void);    // set terminal mode
		int sstr_c(char *str);  // send string
		int sstrcr_c(char *str);// send string w/ cr
		char *underline(void);  // (re)set underline
		void waitcr(void);      // prompt user to hit CR
};


#endif                          // _COLOR_H_





