/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module:  tape.c                                                       */
/* Author:  Matthias Hanisch                                             */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/

#include <time.h>
#include <ntape.h>

void print_tape_popup(void);
int create_new_tape(char *);
int set_active_tape(void);

#define NO_TAPE_CELLS 4
static char *tape_cell[] = {" Create a new tape",
			    " Set active tape",
			    " Rewind tape",
			    " Tape status report",
			    " Quit"};
static int chosen_cell = 1;
static WINDOW *tape_popup;


/*************************************************************************/
/* Name:     tape_chosen                                                 */
/* Purpose:  Opens the tape popup window                                 */
/*************************************************************************/
void tape_chosen()
{
    tape_popup = newwin(NO_TAPE_CELLS+3,20,1,4);
    leaveok(tape_popup,TRUE);
    keypad(tape_popup,TRUE);
    print_tape_popup();
    popup_active = 1;
}


/*************************************************************************/
/* Name:     tape_chosen_cell                                            */
/* Purpose:  Executes a menu item.                                       */
/*************************************************************************/
void tape_chosen_cell()
{
    char name[TAPENAMELENGTH];
    switch(chosen_cell)
    {
    case 0:
	EntryField("Insert a new tape for preparation!\nATTENTION! All data on this tape will be lost!\n\nEnter tape name:\n",
		   TAPENAMELENGTH,name,1,"",WITHOUT_SPECIAL_CHARS);
	print_tape_popup();
	if (strlen(name) > 0)
	    create_new_tape(name);
	destroy_tape_popup();
	break;
    case 1:
	if ((last_tape_pos > 0) && (!offline()))
	{
	    if (last_tape_pos == do_tape_command("tell",0))
	    {
		tape->id = last_tape_id;
		print_footer("I assume that the tape hasn't changed... Knock on wood!");
		last_tape_pos = 0;
		init_list();
		destroy_tape_popup();
		break;
	    }
	}
	if (MessageBox("Insert the right tape!",2,OKCANCEL,0))
	{
	    destroy_tape_popup();
	    set_active_tape();
	}
	else
	    destroy_tape_popup();
	last_tape_pos = 0;
	break;
    case 2:
	print_footer("Rewind tape ...");
	if (offline())
	{
	    print_footer("Sorry! Tape is offline! Can't rewind!");
	    break;
	}
	if (do_tape_command("rewind",0) < 0)
	{
	    print_footer("Error on rewinding tape!");
	    break;
	}
	print_footer("Tape is rewound...");
	break;
    case 3:
	status_window();
	destroy_tape_popup();
	break;
    case 4:
	if (MessageBox("You are now exiting ntape!",2,YESNO,1))
	    nta_exit(1);
	destroy_tape_popup();
	break;
    default:
	break;
    }
}


/*************************************************************************/
/* Name:     print_tape_popup                                            */
/* Purpose:  prints the contents of the tape-popup                       */
/*************************************************************************/
void print_tape_popup()
{
    int i;
  
    wmove(tape_popup,0,0);
    for (i = 0; i <= NO_TAPE_CELLS; i++)
    {
	if (i == chosen_cell)
	    set_color_pair(tape_popup,INV_MENU);
	else
	    set_color_pair(tape_popup,MENU);
	mvwaddstr(tape_popup,i+1,0,tape_cell[i]);
	winclrtoeol(tape_popup);
    }
    set_color_pair(tape_popup,MENU);
    box(tape_popup,ACS_VLINE,ACS_HLINE);
    touchwin(tape_popup);
    wrefresh(tape_popup);
}


/*************************************************************************/
/* Name:     tape_scroll_down                                            */
/* Purpose:  scrolls down in tape-popup                                  */
/*************************************************************************/
void tape_scroll_down()
{
    if (chosen_cell == NO_TAPE_CELLS)
	chosen_cell = 0;
    else
	chosen_cell++;
    print_tape_popup();
}


/*************************************************************************/
/* Name:     tape_scroll_up                                              */
/* Purpose:  scrolls up in tape-popup                                    */
/*************************************************************************/
void tape_scroll_up()
{
    if (chosen_cell == 0)
	chosen_cell = NO_TAPE_CELLS;
    else
	chosen_cell--;
    print_tape_popup();
}


/*************************************************************************/
/* Name:     destroy_tape_popup                                          */
/* Purpose:                                                              */
/*************************************************************************/
void destroy_tape_popup()
{
    delwin(tape_popup);
    popup_active = 0;
    touchwin(tapestatus);
    wrefresh(tapestatus);
    touchwin(listbox);
    wrefresh(listbox);
}


/*************************************************************************/
/* Name:     create_new_tape                                             */
/* Purpose:  creates a new tape and makes a new entry in database        */
/*************************************************************************/
int create_new_tape(char *name)
{
    FILE *tp;
    time_t id_time;
    char msg[256];
    
    if (offline())
    {
	print_footer("Sorry! Tape is offline! No new tape created!");
	return(-1);
    }
  
    if (wr_prot())
    {
	print_footer("Sorry! Tape is write protected! No new tape created!");
	return(-1);
    }
  
    print_footer("Rewind tape ...");
    if (do_tape_command("rewind",0) < 0)
    {
	print_footer("Error on rewinding tape!");
	return(-1);
    }
  
    print_footer("Erase tape ...");
    if (do_tape_command("erase",0) < 0)
    {
	print_footer("Error on erasing tape!");
	return(-1);
    }
  
    print_footer("Writing tape header ...");
    if ((tp = fopen(tapedevice,"w")) == NULL)
    {
	print_footer("Error on accessing tape!");
	return(-1);
    }
  
    time(&id_time);
    fprintf(tp, NTAPEHEADER);
    fprintf(tp, "%ld", id_time);
    fclose(tp);

    strcpy(tape->name,name);
    tape->id = id_time;
    tape->no_archives = 0;
    backup_database(BACKUP_TAPE);

    if ((tp = fopen(tape_db,"a")) == NULL)
    {
	print_footer("Corrupt tape database! Exiting!");
	exit(-1);
    }
    fprintf(tp,"%ld\n%s\n%ld\n",tape->id,tape->name,tape->eot);
    fclose(tp);
    append_current_tape();
    init_list();
    sprintf(msg,"Active tape is now '%s'...\n",tape->name);
    print_footer(msg);
    return(1);
}


/*************************************************************************/
/* Name:     set_active_tape                                             */
/* Purpose:  reads the tape header and sets the listbox                  */
/*************************************************************************/
int set_active_tape()
{
    long id,eot;
    char msg[256];
    
    if ((id = read_header()) <= 0)
    {
	/* print_footer("Sorry! No NTAPE tape!"); */
	return(-1);
    }

#ifdef DEBUG
    fprintf(stderr,"set_active_tape: ID: %ld\n",id);
#endif

    strcpy(tape->name,get_tapename_from_id(id,&eot));
    if (!strcmp(tape->name,UNKNOWN))
    {
	print_footer("Sorry! I don't know this fucking tape!");
	return(-1);
    }
    tape->id = id;
    tape->eot = eot;
    init_list();
    sprintf(msg,"Active tape is now '%s'...\n",tape->name);
    print_footer(msg);
    return(1);
}


/*************************************************************************/
/* Copyright (C) 1994,1995 Matthias Hanisch, Wuerzburg                   */
/*************************************************************************/
