#! /users1/bert/bin/wbuild
#
# Contains the source for the Frame widget
# $Header: Frame.w,v 1.4 92/11/02 14:07:52 bert Exp $



@class XfwfFrame (XfwfCommon) @file=Frame

@ The Frame widget is a composite widget that accepts just one child.
Its only purpose is to draw a frame around widgets that do not have a
frame of their own. It always uses the size of its child, with a
little extra for the frame. There are several types of frames
available, selectable with a resource.

Widget writers can also use the Frame class as a superclass for new
widgets.  The frame is drawn by the |expose| method (which must
therefore be called by subclasses). Its width is given by
|XtNframeWidth|, the appearance by |XtNframeType|. The possible types
are:

\item{|XfwfRaised|} Gives a beveled look. The top and left borders will
be lighter, the bottom and right sides darker.

\item{|XfwfSunken|} Just the opposite.

\item{|XfwfChiseled|} The border will look as if it was made with a
chisel.

\item{|XfwfLedged|} The border will be a ledge that juts out of the
background.



@public

@ The cursor (when not |None|) is the mouse cursor that is displayed
when the mouse is over the Board widget. The default value |None|
causes the parent's cursor to be used.

	@var Cursor cursor = None

@ The |XtNframeType| determines how the border looks.

	@var FrameType frameType = XfwfRaised;

@ |XtNframeWidth| gives the width of the border. The default value
of 0 shows no border at all. The border is drawn {\it inside\/} the
widget. (See also |XtNframeOffset|.)

	@var Dimension frameWidth = 0;

@ Normally, the border is draw along the outer edge of the widget, but
it can be moved inward. |XtNframeOffset| is the number of pixels
between the edge and the frame.

	@var Dimension frameOffset = 0;

@ The colors of the top and bottom shadows can be set with the
resources |lightColor| and |darkColor|, but it is also possible to
use a stiple of foreground and background colors. This may be
preferable on workstations with limited or no color capabilities.
However, the easiest way (which is also the default) is to let the
widget determine its own shadow colors or stipples, based on the
widget's background color and the color capabilities of the screen.

The resource |shadowScheme| can be set to |XfwfColor|, |XfwfStipple| or
|XfwfAuto|. The converter for the shadow pixmap accepts the strings
|"stipple0"|\dots |"stipple8"|. which create pixmaps of the current
background and foreground colors, with |"stipple0"| entirely
background and |"stipple8"| entirely foreground. Setting pixmaps or
colors is only useful when |shadowScheme| is set to |XfwfColor| or
|XfwfStipple| respectively.

	@var ShadowScheme shadowScheme = XfwfAuto
	@var Pixel lightColor = <String> XtDefaultBackground;
	@var Pixel darkColor = <String> XtDefaultForeground;
	@var Bitmap lightStipple = NULL;
	@var Bitmap darkStipple = NULL;

@ The inherited resource |borderWidth| is given a default value of 0,
instead of 1.

	@var border_width = 0

@exports

@ A number of new types is introduced by the Common class.  The
possible types of borders are enumerated in |Frame3dType| (see the
introduction).

	@type FrameType = enum {
	    XfwfRaised, XfwfSunken, XfwfChiseled, XfwfLedged }

@ The shadow scheme can be used to choose colors, pixmaps or automatic
shadows.

	@type ShadowScheme = enum {XfwfAuto, XfwfColor, XfwfStipple}

@ The type |Bitmap| is an alias for |Pixmap|, but it is meant to
contain only bitmaps, i.e., pixmaps of depth one.

	@type Bitmap = Pixmap

@ The routine that draws the border is generally useful, so it is
exported. |t| is the thickness of the frame. The frame is drawn inside
the rectangle |(x, y, x+w-1, y+h-1)|.

@proc XfwfDrawFrame($, int x, int y, int w, int h, FrameType tp, int t,
	GC lightgc, GC darkgc)
{
    XPoint tlPoints[7], brPoints[7];

    if (t == 0) return;
    switch (tp) {
    case XfwfRaised:
    case XfwfSunken:
	tlPoints[0].x = x;		tlPoints[0].y = y;
	tlPoints[1].x = x + w;		tlPoints[1].y = y;
	tlPoints[2].x = x + w - t;	tlPoints[2].y = y + t;
	tlPoints[3].x = x + t;		tlPoints[3].y = y + t;
	tlPoints[4].x = x + t;		tlPoints[4].y = y + h - t;
	tlPoints[5].x = x;		tlPoints[5].y = y + h;
	tlPoints[6].x = x;		tlPoints[6].y = y;
	brPoints[0].x = x + w;		brPoints[0].y = y + h;
	brPoints[1].x = x;		brPoints[1].y = y + h;
	brPoints[2].x = x + t;		brPoints[2].y = y + h - t;
	brPoints[3].x = x + w - t;	brPoints[3].y = y + h - t;
	brPoints[4].x = x + w - t;	brPoints[4].y = y + t;
	brPoints[5].x = x + w;		brPoints[5].y = y;
	brPoints[6].x = x + w;		brPoints[6].y = y + h;
	if (tp == XfwfSunken) {
	    XFillPolygon(XtDisplay($), XtWindow($),
			 darkgc, tlPoints, 7, Nonconvex, CoordModeOrigin);
	    XFillPolygon(XtDisplay($), XtWindow($),
			 lightgc, brPoints, 7, Nonconvex, CoordModeOrigin);
	} else {
	    XFillPolygon(XtDisplay($), XtWindow($),
			 lightgc, tlPoints, 7, Nonconvex, CoordModeOrigin);
	    XFillPolygon(XtDisplay($), XtWindow($),
			 darkgc, brPoints, 7, Nonconvex, CoordModeOrigin);
	}
	break;
    case XfwfLedged:
	XfwfDrawFrame($, x, y, w, h, XfwfRaised, t/2, lightgc, darkgc);
	XfwfDrawFrame($, x+t/2, y+t/2, w-2*(int)(t/2), h-2*(int)(t/2),
		  XfwfSunken, t/2, lightgc, darkgc);
	break;
    case XfwfChiseled:
	XfwfDrawFrame($, x, y, w, h, XfwfSunken, t/2, lightgc, darkgc);
	XfwfDrawFrame($, x+t/2, y+t/2, w-2*(int)(t/2), h-2*(int)(t/2),
		  XfwfRaised, t/2, lightgc, darkgc);
	break;
    }

}



@private

@ The GC for drawing the light parts of the frame:

	@var GC lightgc

@ The GC for drawing the dark parts of the frame:

	@var GC darkgc

@ The |stip4| bitmap is used on screens with insufficient colors to
simulate light and dark shadows. It will be created by the
|initialize| method, whether or not it is needed. Since it is but a
small bitmap, this can't hurt much.

	@var Pixmap stip4

@ The |old_frame_type| variable is used by the |set_shadow| action
function to store the original frame type, when it is temporarily
changed.

	@var FrameType old_frame_type


@methods

@ |class_initialize| installs the type converters. The type converters
back to String are installed as a convenience, so resources can be
retrieved in readable form with |XtVaGetValues|.

@proc class_initialize
{
    XtSetTypeConverter(XtRString, XtRFrameType, cvtStringToFrameType,
		       NULL, 0, XtCacheNone, NULL);
    XtSetTypeConverter(XtRFrameType, XtRString, cvtFrameTypeToString,
		       NULL, 0, XtCacheNone, NULL);
    XtSetTypeConverter(XtRString, XtRBitmap, cvtStringToBitmap,
		       NULL, 0, XtCacheNone, NULL);
    XtSetTypeConverter(XtRString, XtRShadowScheme, cvtStringToShadowScheme,
		       NULL, 0, XtCacheNone, NULL);
    XtSetTypeConverter(XtRShadowScheme, XtRString, cvtShadowSchemeToString,
		       NULL, 0, XtCacheNone, NULL);
}

@ The |realize| method uses the inherited method, but adds the cursor
attribute.

@proc realize
{
    *mask |= CWCursor;
    attributes->cursor = $cursor;
    #realize($, mask, attributes);
}

@ The |expose| method draws the frame, for which it uses the
|XfwfDrawFrame| routine. Before it calls the routine, it sets the clip
region. Afterwards, the clip region is reset, because we don't know
which other widgets share the same GC's. As explained in {\em X
Toolkit Intrinsics Programming Manual} (Nye \& O'Reilly, Motif
Edition, 1990, p~223), the test for |XtIsRealized| is there for the
unlikely case when an expose event arrives after the widget has been
destroyed or unrealized.

@proc expose
{
    Position x, y;
    Dimension w, h;

    if (! XtIsRealized($)) return;
    if (region != NULL) {
	XSetRegion(XtDisplay($), $lightgc, region);
	XSetRegion(XtDisplay($), $darkgc, region);
    }
    #compute_inside($, &x, &y, &w, &h);
    XfwfDrawFrame($, x + $frameOffset, y + $frameOffset, w - 2*$frameOffset,
	 h - 2*$frameOffset, $frameType, $frameWidth, $lightgc, $darkgc);
    if (region != NULL) {
	XSetClipMask(XtDisplay($), $lightgc, None);
	XSetClipMask(XtDisplay($), $darkgc, None);
    }
    #expose($, event, region);
}

@ The method |compute_inside| is re-defined. The method now returns
the area inside the frame. It calls the superclass's method and then
decreases the area by the width of the frame.

@proc compute_inside
{
    #compute_inside($, x, y, w, h);
    *x += $frameOffset + $frameWidth;
    *y += $frameOffset + $frameWidth;
    *w -= 2 * ($frameOffset + $frameWidth);
    *h -= 2 * ($frameOffset + $frameWidth);
}

@ The |initialize| method has to create the GC's, for which it calls
two utility routines, since the |set_values| method has to do the same.

@proc initialize
{
    $lightgc = NULL;
    $darkgc = NULL;
    create_lightgc($);
    create_darkgc($);
    $stip4 = XCreateBitmapFromData(XtDisplay($),
        RootWindowOfScreen(XtScreen($)),
        stip4_bits, stip4_width, stip4_height);
    $old_frame_type = $frameType;
}

@ The |set_values| method has to create new GC's if the resources
change. It also makes sure that |frameWidth| is even if the frame type
is chiseled or ledged.

If the frame width was and is zero, nothing needs to be drawn,
regardless of the changes in other resources. Therefore, at the end
|need_redisplay| is set to False.

When the cursor changes, the |set_values| method uses the
|XDefineCursor| routine to set the attribute on the widget's window,
provided the widget is realized.

@proc set_values
{
    Boolean need_redisplay = False;

    if ($cursor != $old$cursor && XtIsRealized($))
	XDefineCursor(XtDisplay($), XtWindow($), $cursor);

    if ($frameType == XfwfChiseled || $frameType == XfwfLedged)
	$frameWidth = 2 * ((int) ($frameWidth / 2));

    if ($shadowScheme != $old$shadowScheme) {
	create_darkgc($);
	create_lightgc($);
	need_redisplay = True;
    } else if ($shadowScheme == XfwfColor) {
	if ($lightColor != $old$lightColor) {
	    create_lightgc($);
	    need_redisplay = True;
	}
	if ($darkColor != $old$darkColor) {
	    create_darkgc($);
	    need_redisplay = True;
	}
    } else if ($shadowScheme = XfwfStipple) {
	if ($lightStipple != $old$lightStipple) {
	    create_lightgc($);
	    need_redisplay = True;
	}
	if ($darkStipple != $old$darkStipple) {
	    create_darkgc($);
	    need_redisplay = True;
	}
    }

    if ($frameOffset != $old$frameOffset)
	need_redisplay = True;

    if ($frameType != $old$frameType) {
	$old_frame_type = $frameType;
	need_redisplay = True;
    }

    if ($frameWidth != $old$frameWidth)
	need_redisplay = True;
    else if ($frameWidth == 0)
	need_redisplay = False;
}

@ A Frame widget passes its parent's inquiry on to its (presumably)
single child. If there is no child, the proposal is accepted.
The border and position proposals are always accepted, the stacking
order and size are left to the child to decide.

@proc query_geometry
{
    XtWidgetGeometry request2, reply2;
    XtGeometryResult result;
    Dimension h;

    if ($num_children == 0) return XtGeometryYes;

    /* We're only interested in size and stacking order */
    reply->request_mode =
	(CWWidth | CWHeight | CWStackMode) & request->request_mode;

    /* If nothing of interest is left, we can return immediately */
    if (reply->request_mode == 0)
	return XtGeometryYes;

    /* Prepare a request to the child */
    h = 2 * ($frameOffset + $frameWidth);
    request2.request_mode = reply->request_mode;
    request2.width = request->width - h;
    request2.height = request->height - h;
    request2.sibling = request->sibling;
    request2.stack_mode = request->stack_mode;

    result = XtQueryGeometry($children[0], &request2, &reply2);

    /* If the child accepted its proposal, we accept ours */
    if (result == XtGeometryYes) return XtGeometryYes;

    /* If the child doesn't want any change, we don't want any, either */
    if (result == XtGeometryNo) return XtGeometryNo;

    /* Otherwise, ignore everything but size and stacking order */
    reply->request_mode &= reply2.request_mode;
    if (reply->request_mode == 0) return XtGeometryYes;

    reply->width = reply2.width + h;
    reply->height = reply2.height + h;
    reply->sibling = reply2.sibling;
    reply->stack_mode = reply2.stack_mode;
    return XtGeometryAlmost;
}

@ Requests by the child to be resized are passed on to the parent. If
the parent replies with |XtGeometryYes|, the change is accepted and
(if not |XtCWQueryOnly|) already done. In that case the Frame widget
accepts its child's request. If the parent replies with
|XtGeometryNo|, the change is denied and the denial is passed on. If
the parent replies with a different geometry, the geometry is passed
on, after compensating for the frame width.

Requests for anything other than width or height are always granted.

@proc geometry_manager
{
    XtWidgetGeometry request2, reply2;
    XtGeometryResult result;
    Position x, y;
    Dimension w, h, extraw, extrah;

    $compute_inside($, &x, &y, &w, &h);
    if (! (request->request_mode & (CWWidth|CWHeight))) return XtGeometryYes;
    extraw = $width - w;
    extrah = $height - h;
    request2.request_mode = request->request_mode & (CWWidth|CWHeight);
    request2.width = request->width + extraw;
    request2.height = request->height + extrah;
    result = XtMakeGeometryRequest($, &request2, &reply2);
    if (result == XtGeometryNo) return XtGeometryNo;
    if (result == XtGeometryYes) return XtGeometryYes;
    reply->request_mode = reply2.request_mode & (CWWidth|CWHeight);
    reply->width = reply2.width - extraw;
    reply->height = reply2.height - extrah;
    return XtGeometryAlmost;
}

@ The |resize| method doesn't have to recompute any private variables,
but it passes on the resize message to its child, after decreasing the
area by the amount needed for the frame.

@proc resize
{
    Position x, y;
    Dimension w, h;
    Widget child;

    if ($num_children == 0) return;
    $compute_inside($, &x, &y, &w, &h);
    child = $children[0];
    XtConfigureWidget(child, x, y, w, h, $child$border_width);
}

@ The |change_managed| method is called when a child becomes managed
or unmanaged. The task of the routine is enforcing the layout policy,
which in this case consists of trying to take on the size of the child
or otherwise resize the child to fit inside the frame.
If the parent of the Frame widget doesn't allow the Frame widget to be
resized, the child of the Frame widget will be resized instead.

@proc change_managed
{
    XtWidgetGeometry request2, reply2;
    XtGeometryResult result;
    Widget child;
    Position x, y;
    Dimension w, h;

    if ($num_children == 0) return;
    $compute_inside($, &x, &y, &w, &h);
    child = $children[0];
    request2.request_mode = CWWidth | CWHeight;
    request2.width = $child$width + $width - w;
    request2.height = $child$height + $height - h;
    result = XtMakeGeometryRequest($, &request2, &reply2);
    if (result != XtGeometryYes) /* we're not resized */
	XtConfigureWidget(child, x, y, w, h, $child$border_width);
}

@actions

@ Although the Frame widget has no translations, one action is
defined, that may be of use to subclasses. The action function
|set_shadow| can be used to change the shadow frame. It has zero or
one argument. Without an argument, it resets the shadow to its
original type; with an argument, it sets the shadow to the type given
in the argument.

Warning: the function uses the |XfwfDrawFrame| routine to draw the
frames directly, instead of calling the |expose| or even |set_values|
methods.  Any subclass that defines behaviour that depends on knowing
the frame type, will have to redefine the |set_shadow| action.

@proc set_shadow
{
    Position x, y;
    Dimension w, h;
    FrameType f = XfwfSunken;

    if (*num_params == 0) f = $old_frame_type;	/* Reset to old style */
    else if (strcasecmp("raised", params[0]) == 0) f = XfwfRaised;
    else if (strcasecmp("sunken", params[0]) == 0) f = XfwfSunken;
    else if (strcasecmp("chiseled", params[0]) == 0) f = XfwfChiseled;
    else if (strcasecmp("ledged", params[0]) == 0) f = XfwfLedged;
    else XtWarning("Unknown frame type in set_shadow action");

    if ($frameType != f) {
	$frameType = f;
	#compute_inside($, &x, &y, &w, &h);
	XfwfDrawFrame($, x + $frameOffset, y + $frameOffset,
		      w - 2*$frameOffset, h - 2*$frameOffset,
		      $frameType, $frameWidth, $lightgc, $darkgc);
    }
}

@utilities

@ The variable |rcsid| isn't used for anything, except tracking of
versions. The version number is that of the specification file (this
file) from which the widget's source is build.

@var char rcsid[] = "$Header: Frame.w,v 1.4 92/11/02 14:07:52 bert Exp $"

@ |cvtStringToFrameType| converts the strings `raised', `sunken',
`chiseled' and `ledged'. Case doesn't matter.

@def done(type, value) =
  do {
      if (to->addr != NULL) {
	  if (to->size < sizeof(type)) {
	      to->size = sizeof(type);
	      return False;
	  }
	  *(type*)(to->addr) = (value);
      } else {
	  static type static_val;
	  static_val = (value);
	  to->addr = (XtPointer)&static_val;
      }
      to->size = sizeof(type);
      return True;
  } while (0)

@proc Boolean cvtStringToShadowScheme(Display *display, XrmValuePtr args,
  Cardinal *num_args, XrmValuePtr from, XrmValuePtr to,
  XtPointer *converter_data)
{
    if (*num_args != 0)
	XtAppErrorMsg(XtDisplayToApplicationContext(display),
		      "cvtStringToShadowScheme", "wrongParameters",
		      "XtToolkitError",
		      "String to shadow scheme type conversion needs no arguments",
		      (String*) NULL, (Cardinal*) NULL);
    if (strcasecmp("auto", (char*) from->addr) == 0)
	done(ShadowScheme, XfwfAuto);
    else if (strcasecmp("color", (char*) from->addr) == 0)
	done(ShadowScheme, XfwfColor);
    else if (strcasecmp("stipple", (char*) from->addr) == 0)
	done(ShadowScheme, XfwfStipple);
    else
	XtDisplayStringConversionWarning(display, (char*) from->addr, 
					 XtRShadowScheme);
    done(ShadowScheme, XfwfAuto);
}

@proc Boolean cvtShadowSchemeToString(Display *display, XrmValuePtr args,
  Cardinal *num_args, XrmValuePtr from, XrmValuePtr to,
  XtPointer *converter_data)
{
    if (*num_args != 0)
	XtAppErrorMsg(XtDisplayToApplicationContext(display),
		      "cvtShadowSchemeToString", "wrongParameters",
		      "XtToolkitError",
		      "Shadow scheme to String type conversion needs no arguments",
		      (String*) NULL, (Cardinal*) NULL);
    switch (*(ShadowScheme*)from->addr) {
    case XfwfAuto: done(String, "auto");
    case XfwfColor: done(String, "color");
    case XfwfStipple: done(String, "stipple");
    default: XtError("Illegal ShadowScheme");
    }
}

@ The conversion from a file name to a bitmap is fairly simple. Most of the
code in this function only serves to give adequate error messages in case
something goes wrong.

@proc Boolean cvtStringToBitmap(Display *display, XrmValuePtr args,
				Cardinal *num_args, XrmValuePtr from,
				XrmValuePtr to, XtPointer *converter_data)
{
    int status, hotx, hoty;
    unsigned int w, h, one = 1;
    Pixmap stip;

    if (*num_args != 0)
	XtAppErrorMsg(XtDisplayToApplicationContext(display),
		      "cvtStringToBitmap", "wrongParameters",
		      "XtToolkitError",
		      "String to bitmap conversion needs no arguments",
		      (String*) NULL, (Cardinal*) NULL);

    status = XReadBitmapFile(display, DefaultRootWindow(display),
			    (String) from->addr, &w, &h, &stip, &hotx, &hoty);

    switch (status) {
    case BitmapOpenFailed:
	XtAppErrorMsg(XtDisplayToApplicationContext(display),
		      "cvtStringToBitmap", "BitmapOpenFailed",
		      "XtToolkitError",
		      "Could not open bitmap file %s",
		      (String*)&from->addr, &one);
	break;
    case BitmapFileInvalid:
	XtAppErrorMsg(XtDisplayToApplicationContext(display),
		      "cvtStringToBitmap", "BitmapFileInvalid",
		      "XtToolkitError",
		      "File %s is not a valid bitmap",
		      (String*)&from->addr, &one);
	break;
    case BitmapNoMemory:
	XtAppErrorMsg(XtDisplayToApplicationContext(display),
		      "cvtStringToBitmap", "BitmapNoMemory",
		      "XtToolkitError",
		      "Not enough memory to load bitmap %s",
		      (String*)&from->addr, &one);
	break;
    case BitmapSuccess:
	break;
    }
    if (status != BitmapSuccess)
	stip = XCreateBitmapFromData(display, DefaultRootWindow(display),
				     stip4_bits, stip4_width, stip4_height);
    done(Pixmap, stip);
}

@proc Boolean cvtStringToFrameType(Display *display, XrmValuePtr args,
  Cardinal *num_args, XrmValuePtr from, XrmValuePtr to,
  XtPointer *converter_data)
{
    if (*num_args != 0)
	XtAppErrorMsg(XtDisplayToApplicationContext(display),
		      "cvtStringToFrameType", "wrongParameters",
		      "XtToolkitError",
		      "String to frame type conversion needs no arguments",
		      (String*) NULL, (Cardinal*) NULL);
  if (strcasecmp("raised", (char*) from->addr) == 0)
      done(FrameType, XfwfRaised);
  else if (strcasecmp("sunken", (char*) from->addr) == 0)
      done(FrameType, XfwfSunken);
  else if (strcasecmp("chiseled", (char*) from->addr) == 0)
      done(FrameType, XfwfChiseled);
  else if (strcasecmp("ledged", (char*) from->addr) == 0)
      done(FrameType, XfwfLedged);
  else
      XtDisplayStringConversionWarning(display, (char*) from->addr, 
				     XtRFrameType);
    done(FrameType, XfwfRaised);
}

@proc Boolean cvtFrameTypeToString(Display *display, XrmValuePtr args,
  Cardinal *num_args, XrmValuePtr from, XrmValuePtr to,
  XtPointer *converter_data)
{
    if (*num_args != 0)
	XtAppErrorMsg(XtDisplayToApplicationContext(display),
		      "cvtFrameTypeToString", "wrongParameters",
		      "XtToolkitError",
		      "Fframe type to String conversion needs no arguments",
		      (String*) NULL, (Cardinal*) NULL);
    switch (*(FrameType*)from->addr) {
    case XfwfRaised: done(String, "raised");
    case XfwfSunken: done(String, "sunken");
    case XfwfChiseled: done(String, "chiseled");
    case XfwfLedged: done(String, "ledged");
    default: XtError("Illegal FrameType");
    }
}

@ The |create_darkgc| function creates the GC for the dark parts of
the frame. The contents of the GC depend on the resources
|shadowScheme| and possibly |background_pixel|, |darkColor|, |lightColor|,
|darkPixmap| and |lightPixmap|.

@proc create_darkgc($)
{
    XtGCMask mask;
    XGCValues values;

    if ($darkgc != NULL) XtReleaseGC($, $darkgc);
    switch ($shadowScheme) {
    case XfwfColor:
	mask = GCForeground;
	values.foreground = $darkColor;
	break;
    case XfwfStipple:
	mask = GCFillStyle | GCStipple | GCForeground | GCBackground;
	values.fill_style = FillOpaqueStippled;
	values.stipple = $darkStipple;
	values.foreground = BlackPixelOfScreen(XtScreen($));
	values.background = $background_pixel;
	break;
    case XfwfAuto:
	if (DefaultDepthOfScreen(XtScreen($)) > 4
	    && $darker_color($, $background_pixel, &values.foreground)) {
	    mask = GCForeground;
	} else {
	    mask = GCFillStyle | GCBackground | GCForeground | GCStipple;
	    values.fill_style = FillOpaqueStippled;
	    values.background = $background_pixel;
	    values.foreground = WhitePixelOfScreen(XtScreen($));
	    values.stipple =
		XCreateBitmapFromData(XtDisplay($),
				      RootWindowOfScreen(XtScreen($)),
				      stip4_bits, stip4_width, stip4_height);
	}
	break;
    }
    $darkgc = XtGetGC($, mask, &values);
}

@ |create_lightgc| does the same for the light parts of the frame.
When the |shadowScheme| resource is |XfwfAuto|, the depth of the screen
and the availability of colors determines whether colors or stipples
will be used for the frame.

@proc create_lightgc($)
{
    XtGCMask mask;
    XGCValues values;

    if ($lightgc != NULL) XtReleaseGC($, $lightgc);
    switch ($shadowScheme) {
    case XfwfColor:
	mask = GCForeground;
	values.foreground = $lightColor;
	break;
    case XfwfStipple:
	mask = GCFillStyle | GCStipple | GCForeground | GCBackground;
	values.fill_style = FillOpaqueStippled;
	values.background = $background_pixel;
	values.stipple = $lightStipple;
	values.foreground = WhitePixelOfScreen(XtScreen($));
	break;
    case XfwfAuto:
	if (DefaultDepthOfScreen(XtScreen($)) > 4
	    && $lighter_color($, $background_pixel, &values.foreground)) {
	    mask = GCForeground;
	} else {
	    mask = GCFillStyle | GCBackground | GCForeground | GCStipple;
	    values.fill_style = FillOpaqueStippled;
	    values.background = $background_pixel;
	    values.foreground = WhitePixelOfScreen(XtScreen($));
	    values.stipple =
		XCreateBitmapFromData(XtDisplay($),
				      RootWindowOfScreen(XtScreen($)),
				      stip4_bits, stip4_width, stip4_height);
	}
	break;
    }
    $lightgc = XtGetGC($, mask, &values);
}



@imports

@incl <string.h>
@incl <stdio.h>

@ The stipple for the shadows is loaded from a bitmap file.

@incl "stip4.bm"

