/* portalias.c -- Port Aliases tree management
 *
 * This file is part of TUA.
 * 
 *   Copyright (C) 1991,92,93  Lele Gaifax (lele@nautilus.sublink.org)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the license, or (at
 *   your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "tua.h"

typedef struct port_aliases
{
  char *Port;
  char *Alias;
} port_aliases_t;

static BTREE Root = NULL;

static int
DEFUN (alias_compare, (port1, port2),
       CONST port_aliases_t * port1 AND CONST port_aliases_t * port2)
{
  int ret = fstrcmp (port1->Port, port2->Port);

  return ret;
}

static PTR
DEFUN (alias_new, (data),
       CONST PTR * data)
{
  port_aliases_t * sa;
  
  sa = (port_aliases_t *) xmalloc (sizeof (port_aliases_t));
  sa->Port = savestring (((CONST port_aliases_t *) data)->Port);
  sa->Alias = NULL;

  return sa;
}
       
CONST char *
DEFUN (insert_port_alias, (port, alias),
       CONST char *port AND
       CONST char *alias)
{
  port_aliases_t * newpa, pa;
  
  if (Root == NULL)
    Root = btree_new ((compare_func_t) alias_compare, (makenew_func_t) alias_new);

  pa.Port = (char *) port;
  
  newpa = (port_aliases_t *) btree_insert (Root, &pa);
  if (! newpa->Alias)
    newpa->Alias = savestring (alias);
  
  return newpa->Alias;
}

CONST char *
DEFUN (get_possible_port_alias, (port),
       CONST char *port)
{
  port_aliases_t * pa, pa_key;

  if (Root == NULL)
    return port;
  
  pa_key.Port = (char *) port;
  pa = (port_aliases_t *) btree_search (Root, &pa_key);

#ifdef GLOB_ALIASES
  if (pa == NULL)
    return get_possible_port_glob_alias (port);
#endif   

  if (pa == NULL)
    return port;
  else
    return pa->Alias;
}

#ifdef GLOB_ALIASES

#include "globalias.h"

static glob_aliases_queue_t *PortGlobQueue = QNULL;

void
DEFUN (insert_port_glob_alias, (port, alias),
       CONST char *port AND
       CONST char *alias)
{
  globalias_insert (&PortGlobQueue, port, alias);
}

CONST char *
DEFUN (get_possible_port_glob_alias, (port),
       CONST char *port)
{
  return globalias_search (PortGlobQueue, port);
}

#endif /* ifdef GLOB_ALIASES */

#ifdef USE_TCL

int
DEFUN (tcl_insert_port_alias, (clientData, interp, argc, argv),
       ClientData clientData AND
       Tcl_Interp * interp AND
       int argc AND
       char *argv[])

{
  if (argc != 3)
    {
      sprintf (interp->result, "Wrong number of Parameters: %s needs an PortName and its Alias", argv[0]);
      return TCL_ERROR;
    }
  insert_port_alias (argv[1], argv[2]);
  return TCL_OK;
}

#ifdef GLOB_ALIASES

int
DEFUN (tcl_insert_port_glob_alias, (clientData, interp, argc, argv),
       ClientData clientData AND
       Tcl_Interp * interp AND
       int argc AND
       char *argv[])

{
  if (argc != 3)
    {
      sprintf (interp->result, "Wrong number of Parameters: %s needs an PortName and its GlobAlias", argv[0]);
      return TCL_ERROR;
    }
  insert_port_glob_alias (argv[1], argv[2]);
  return TCL_OK;
}

#endif /* ifdef GLOB_ALIASES */

#endif /* ifdef USE_TCL */
