/* process.c -- Keep status of a process
 *
 * This file is part of TUA.
 * 
 *   Copyright (C) 1991,92,93  Lele Gaifax (lele@nautilus.sublink.org)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the license, or (at
 *   your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* I keep track of a per process info, storing such info in a tree keyed on
 * the process id. Although this is not unique, it should be enough because
 * at a single time (and probably in a day) this number will be unique. */

#include "tua.h"

static BTREE Root = NULL;

static int
DEFUN (process_compare, (proc1, proc2),
       CONST process_status_t * proc1 AND CONST process_status_t * proc2)
{
  int ret;

  if (proc1->ProcessId < proc2->ProcessId)
    ret = -1;
  else if (proc1->ProcessId == proc2->ProcessId)
    ret = 0;
  else
    ret = 1;
  
  return ret;
}

static PTR
DEFUN (process_new, (data),
       CONST PTR data)
{
  process_status_t *p;

  p = (process_status_t *) xmalloc (sizeof (process_status_t));
  p->ProcessId = ((CONST process_status_t *) data)->ProcessId;
  p->Killed = FALSE;
  p->System = 0;
  p->Status = TUUS_NONE;
  p->LoginPassed = FALSE;
  
#ifdef HDB_UUCP
  p->StartTime = p->EndTime = 0;
#endif
  return (PTR) p;
}

process_status_t *
DEFUN (insert_process, (procId),
       int procId)
{
  process_status_t p, *new_proc;
  
  if (Root == NULL)
    Root = btree_new ((compare_func_t) process_compare, (makenew_func_t) process_new);

  p.ProcessId = procId;
  new_proc = (process_status_t *) btree_insert (Root, &p);
  if (new_proc->Killed)
    {
      new_proc->Killed = FALSE;
      new_proc->System = 0;
      new_proc->Status = TUUS_NONE;
#ifdef HDB_UUCP
      new_proc->StartTime = new_proc->EndTime = 0.0;
#endif
    }
  return new_proc;
}

void
DEFUN (kill_process, (procId),
       int procId)
{
  process_status_t p;

  p.ProcessId = procId;

  if (Root != NULL)
    {
      process_status_t * found = (process_status_t *) btree_search (Root, &p);

      if (found)
	found->Killed = TRUE;
    }
}

static traverse_func_t originalTraverseFunction;

static void
DEFUN (FilteredEnquiry, (proc),
       CONST process_status_t * proc)
{
  if (! proc->Killed)
    (*originalTraverseFunction) (proc);
}

void
DEFUN (enquire_processes, (funct),
       traverse_func_t funct)
{
  originalTraverseFunction = funct;
  btree_traverse (Root, (traverse_func_t) FilteredEnquiry);
}


