/*
  This is a simple program designed to set up the Cyrix 486 CPU
  by allowing configuration of the registers for maximum performance
  on a wide range of motherboards.

  It was written by Paul Gortmaker, based on information contained
  in the file CxPatch.tar.z -- by clau@acs.ucalgary.ca, which was
  a patch to earlier Linux kernels to enable the 1k on chip
  cache on Cyrix DLC / SLC processors. This program moves that
  functionality out of the kernel and into user space.

  The assembly portions were taken from Ti Kan's sysV driver. See
  the file COPYING for details.

*/

#ident "@(#)cyrix.c	1.00	03/25/94"

#define true	1
#define false	0

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>
#include <asm/io.h>

#define byte_out	outb		/* don't use outb_p(), etc */
#define byte_in		inb
#define version 	"1.00"

#define read_dlc	c_read_dlc
#define write_dlc	c_write_dlc

#define CYR_SEL		0x22		/* Select port */
#define CYR_DAT		0x23		/* Data port */

#define IOPL_HI		3		/* Screams of horror.... */
#define IOPL_LO		0		/* Sigh of relief.... */

#define PORT_IO_ON	1
#define PORT_IO_OFF	0

/*
   Function prototypes for non-K&R folks.  ;-)
*/

void c_read_dlc(void);
void c_write_dlc(void);
void show_stat(void);
void dump_regs(void);
void help_me(void);
void check_for_cyrix(void);
int hex_me (char c);
void print_addr(char ncr1, char ncr2, char ncr3);
unsigned long parse_addr(char *addr_str);
unsigned long lrotl(unsigned long value, unsigned shift);
unsigned rotl(unsigned value, unsigned shift);

/* The 4 assembly routines: probe.S, state.S, enable.S, disable.S */

extern int	cx_iscyrix(void);
/*
extern int	cx_i486cachemode(void);
extern void	cx_i486cacheon(void);
extern void	cx_i486cacheoff(void);
*/

char  ccr0,ccr1;			/* cache ctrl registers */
char ncr_a[4],ncr_b[4],ncr_c[4];	/* 4 non-cacheable areas, 3 regs ea. */
unsigned region;			/* The region we are working on */
unsigned long size;			/* The size of the region */
unsigned long start;			/* The start segment of the region */
char *comma;				/* Pointer to "," in "-xCC00,16" */
char ok_size;				/* Flag to watch for bad region size */
char override;				/* If true, skip check for 486DLC */
char enabled;				/* if true, Cx 1k cache is enabled */

int main(int argc, char *argv[]) {

int i,j;

if (argc==1) {
    fprintf(stderr,"\nNothing to do, nothing done. Use \"-h\" for help.\n\n");
    exit(0);
}

for (i=1;i<argc;i++) {			    /* help */
    if (!strncmp(argv[i],"-h",2)) {
	help_me();
	exit(0);
    }
}

for (i=1;i<argc;i++) {			    /* version # */
    if (!strncmp(argv[i],"-v",2)) {	    
	fprintf(stderr,"\nCyrix Register Setup: v%s",version);
	fprintf(stderr,"\n\nUse \"%s -h\" for more info.\n",argv[0]);
	exit(0);
    }
}

/* Check uid to see if we should continue or not. */

if (getuid()) {
	fprintf(stderr, "\ncyrix: Only root can read/alter the CPU configuration.\n\n");
	exit(EACCES);
}


/* Check to see if override was selected before doing anything else. */

for (i=1;i<argc;i++) {			    /* override DLC probe */
    if (!strncmp(argv[i],"-o",2)) {
	override=true;
    }
}

for (i=1;i<argc;i++) {			    /* register dump */
    if (!strncmp(argv[i],"-d",2)) {
	check_for_cyrix();
	iopl(IOPL_HI);
/*	enabled=cx_i486cachemode(); */
	enabled=1;
	iopl(IOPL_LO);
	read_dlc();
	dump_regs();
	exit(0);
    }
}

for (i=1;i<argc;i++) {
    if (!strncmp(argv[i],"-q",2)) {	    /* status report */
	check_for_cyrix();
	iopl(IOPL_HI);
/*	enabled=cx_i486cachemode(); */
	enabled=1;
	iopl(IOPL_LO);
	read_dlc();
	show_stat();
	exit(0);
    }
}

/*
   Ok, we made it here, so all the print and exit commands are done.
   Read the registers so that we can modify them.
*/

check_for_cyrix();
iopl(IOPL_HI);
/* enabled=cx_i486cachemode(); */
enabled=1;
iopl(IOPL_LO);
read_dlc();

for (i=1;i<argc;i++) {

    if (!strncmp(argv[i],"-a",2)) {		    /* A20M enable */
	if(!strncmp(argv[i]+2,"-",1)) ccr0&=0xfb;
	else ccr0|=0x04;
    }

    else if (!strncmp(argv[i],"-b",2)) {	    /* BARB (non DLC m/b) */
	if(!strncmp(argv[i]+2,"-",1)) ccr0&=0xdf;
	else ccr0|=0x20;
    }

    else if (!strncmp(argv[i],"-c",2)) {	    /* Cache Format */
	if (!strncmp(argv[i]+2,"d",1)) ccr0|=0x40;
	else ccr0&=0xbf;
    }

    else if (!strncmp(argv[i],"-e",2)) {	    /* Cache ctrl. via CR0 */
	if (!strncmp(argv[i]+2,"-",1)) enabled=false;
	else enabled=true;
    }

    else if (!strncmp(argv[i],"-f",2)) {	    /* FLUSH (DLC m/b) */
	if(!strncmp(argv[i]+2,"-",1)) ccr0&=0xef;
	else ccr0|=0x10;
    }

    else if (!strncmp(argv[i],"-i",2)) {
	j=atoi(argv[i]+2);
	if ((j>4)||(j<1)) {
	    fprintf(stderr,"\nRegion to inhibit must be 1, 2, 3 or 4.\n");
	    exit(EINVAL);
	}
	j--;
	ncr_a[j]=0;
	ncr_b[j]=0;
	ncr_c[j]=0;
    }

    else if (!strncmp(argv[i],"-k",2)) {	    /* KEN */
	if(!strncmp(argv[i]+2,"-",1)) ccr0&=0xf7;
	else ccr0|=0x08;
    }

    else if (!strncmp(argv[i],"-m",2)) {	    /* 1st 64kb per Mb. */
	if(!strncmp(argv[i]+2,"-",1)) ccr0|=0x01;
	else ccr0&=0xfe;
    }

    else if (!strncmp(argv[i],"-o",2)) {}	    /* Override (done above) */

    else if (!strncmp(argv[i],"-p",2)) {	    /* Power modes */
	ccr1=0;
    }

    else if (!strncmp(argv[i],"-r",2)) {	    /* 640kb -> 1Mb ROM shadow */
	if(!strncmp(argv[i]+2,"-",1)) ccr0&=0xfd;
	else ccr0|=0x02;
    }

    else if (!strncmp(argv[i],"-s",2)) {	    /* SUSP enable */
	if(!strncmp(argv[i]+2,"-",1)) ccr0&=0x7f;
	else ccr0|=0x80;
    }


    else if (!strncmp(argv[i],"-x",2)) {	    /* exclude region from cache */
	if (region==4) {
	    fprintf(stderr,"\nMaximum of 4 non-cacheable regions.\n");
	    exit(EINVAL);
	}

	comma=strchr(argv[i]+2,',');
	if (!comma) {
	    fprintf(stderr,"\nInvalid syntax: eg. use \"-xCC00,16\"\n");
	    exit(EINVAL);
	}

	*comma='\0';	    /* replace comma with string termination */

	start=parse_addr(argv[i]+2);
	size=atol(comma+1);

	if (size>32768) {	/* 32Mb = max size */
	    fprintf(stderr,"\nSize of non-cacheable region must be <= 32Mb, or \"4Gb\".\n");
	    exit(EINVAL);
	}

	if (!strncmp(comma+1,"4Gb",3)) {	    /* Deal with 4 Gigabyte option */
	    size=65536; 	/* treat as 65536, as program need not */
	}			/*know what the Cyrix does with it */

	if (start%256) {	/* div by 256, as it is a segment */
	   fprintf(stderr,"\nMemory segment must be aligned on 4k boundary.\n");
	   exit(EINVAL);
	}

	ok_size=false;

	for (j=1;j<16;j++) 
	    if (size==lrotl((unsigned long)2,j)) ok_size=true;

	if (!ok_size) {
	    fprintf(stderr,"\nSize of non-cacheable region (kb) must be one of:");
	    fprintf(stderr,"\n4, 8, 16, 32, 64, 128, 256, ... 16384, 32768, or \"4Gb\".\n");
	    exit(EINVAL);
	}
/*
	Ok, passed all the sanity checks, now put the values into ncr?_[]
	Don't forget that these are segments, not absolute addresses.
*/

	ncr_a[region]=(char)((start&0xFF00000)/0x100000);	/* set addr */
	ncr_b[region]=(char)((start&0x00FF000)/0x001000);
	ncr_c[region]=(char)((start&0x0000F00)/0x000010);

	j=0;
	do {						/* set size */
	    ncr_c[region]++;
	    j++;
	} while (size>lrotl(2,j));

	region++;		    /* Use the next region next time around */

    } /* end of "-x" option */

    else {
	fprintf(stderr,"\nUnknown option: \"%s\"",argv[i]);
	fprintf(stderr,"\nUse \"-h\" for help.\n");
	exit(EINVAL);
    }

} /* end of for i<argc */

/*
   Ok, all the fiddling is done now, so we can write the registers
   and get outta here. We turn off the internal cache first, because
   if we change some of the non-cacheable regions, or the cache format,
   the cache needs to be flushed. The on/off assembly routines do this
   as well as just turing it off and on. Note that we mask interrupts
   so that some process doesn't snarf the CPU in some undefined state.
*/

iopl(IOPL_HI);
asm("cli");
/*cx_i486cacheoff();*/			/* Turn it off so we don't crash */
iopl(IOPL_LO);

write_dlc();

iopl(IOPL_HI);
/* if (enabled) cx_i486cacheon();*/		/* and back on again, if it was on. */
asm("sti");
iopl(IOPL_LO);

return(0);

} /* end main */

/* ********************************************************************* */
void show_stat() {
/* ********************************************************************* */


/*
   Shows the detailed status of the cache control registers and the
   non-cacheable regions.
*/

fprintf(stderr,"\nInternal 1kb cache is presently %s via CR0.",
	enabled? "enabled":"disabled");

fprintf(stderr,"\nFirst 64k of each 1Mb boundary set as %scacheable.",
	(ccr0&0x01)? "non-":"");

fprintf(stderr,"\n640k --> 1Mb region set as %scacheable.",
	(ccr0&0x02)? "non-":"");

fprintf(stderr,"\nA20M input %s.",
	(ccr0&0x04)? "enabled":"disabled");

fprintf(stderr,"\nKEN input %s.",
	(ccr0&0x08)? "enabled":"disabled");

fprintf(stderr,"\nFLUSH input %s.",
	(ccr0&0x10)? "enabled":"disabled");

fprintf(stderr,"\nBARB input %s.",
	(ccr0&0x20)? "enabled":"disabled");

fprintf(stderr,"\nInternal Cache mode: %s.",
	(ccr0&0x40)? "direct-mapped":"2 way associative");

fprintf(stderr,"\nSUSP input and SUSPA output %s.",
	(ccr0&0x80)? "enabled":"disabled");

if (!ccr1) fprintf(stderr,"\nAll low power mode features disabled.");
else fprintf(stderr,"\nPower saving mode register presently set at 0x%x.",ccr1&0xff);

fprintf(stderr,"\n\nNon-cacheable region 1: ");
if (ncr_c[0]&0x0f) {
	print_addr(ncr_a[0],ncr_b[0],ncr_c[0]);
	if ((ncr_c[0]&0x0f)==0x0f) fprintf(stderr,"4Gb.");
	else fprintf(stderr,"%u kb.",rotl(2,ncr_c[0]&0x0f));
}
else fprintf(stderr,"Disabled.");

fprintf(stderr,"\nNon-cacheable region 2: ");
if (ncr_c[1]&0x0f) {
	print_addr(ncr_a[1],ncr_b[1],ncr_c[1]);
	if ((ncr_c[1]&0x0f)==0x0f) fprintf(stderr,"4Gb.");
	else fprintf(stderr,"%u kb.",rotl(2,ncr_c[1]&0x0f));
}
else fprintf(stderr,"Disabled.");

fprintf(stderr,"\nNon-cacheable region 3: ");
if (ncr_c[2]&0x0f) {
	print_addr(ncr_a[2],ncr_b[2],ncr_c[2]);
	if ((ncr_c[2]&0x0f)==0x0f) fprintf(stderr,"4Gb.");
	else fprintf(stderr,"%u kb.",rotl(2,ncr_c[2]&0x0f));
}
else fprintf(stderr,"Disabled.");

fprintf(stderr,"\nNon-cacheable region 4: ");
if (ncr_c[3]&0x0f) {
	print_addr(ncr_a[3],ncr_b[3],ncr_c[3]);
	if ((ncr_c[3]&0x0f)==0x0f) fprintf(stderr,"4Gb.");
	else fprintf(stderr,"%u kb.",rotl(2,ncr_c[3]&0x0f));
}
else fprintf(stderr,"Disabled.");

fprintf(stderr,"\n");

} /* end show_stat */

/* ********************************************************************* */
void dump_regs() {
/* ********************************************************************* */

fprintf (stderr,"\nCache enabled via CR0: %s",(enabled? "yes":"no"));
fprintf (stderr,"\nCCR0 = %x\nCCR1 = %x",(char)ccr0&0xff,(char)ccr1&0xff);

fprintf (stderr,"\nNCR1:\t%x\t%x\t%x",
(char)ncr_a[0]&0xff,(char)ncr_b[0]&0xff,(char)ncr_c[0]&0xff);

fprintf (stderr,"\nNCR2:\t%x\t%x\t%x",
(char)ncr_a[1]&0xff,(char)ncr_b[1]&0xff,(char)ncr_c[1]&0xff);

fprintf (stderr,"\nNCR3:\t%x\t%x\t%x",
(char)ncr_a[2]&0xff,(char)ncr_b[2]&0xff,(char)ncr_c[2]&0xff);

fprintf (stderr,"\nNCR4:\t%x\t%x\t%x",
(char)ncr_a[3]&0xff,(char)ncr_b[3]&0xff,(char)ncr_c[3]&0xff);

fprintf(stderr,"\n");

} /* end dump_regs */

/* ********************************************************************* */
void print_addr(char ncr1, char ncr2, char ncr3) {
/* ********************************************************************* */


/*
   A kludge to get #'s less than 0x10 printed with a leading zero.
*/

fprintf(stderr,"Start segment = 0x%s%X%s%X%X00, size = ",
	ncr1>0x0f? "":"0",(char)ncr1&0xff,
	ncr2>0x0f? "":"0",(char)ncr2&0xff,
	(char)((ncr3&0xf0)/0x10));

} /* end print_addr */

/* ********************************************************************* */
unsigned long parse_addr(char *addr_str) {
/* ********************************************************************* */

/*
   Takes a users string (supposed to be hex) and converts it into the
   hex address (stored as an unsigned int) that it is supposed to be.
*/

unsigned long addr,mult;
unsigned short i;

for (i=0;i<strlen(addr_str);i++) {
    if (!isxdigit((*(addr_str+i)))) {
	fprintf(stderr,"\nCan't get a valid memory addr from \"%s\"",addr_str);
	exit(EINVAL);
    }
}

addr=0;
mult=1;
i=strlen(addr_str);
do {
     addr+=hex_me(*(addr_str+i-1))*mult;
     mult*=16;
     i--;
} while (i);

return(addr);


} /* end parse_addr */

/* ********************************************************************* */
int hex_me(char c)
/* ********************************************************************* */

/*
   This cheezy routine expects valid 0 --> 9 or A/a --> f/F hex chars.
   Don't pass it anything else or it will break badly.
*/

{

if ((c>='0')&&(c<='9')) return(c-'0');

return(tolower(c)-'a'+10);

} /* end hex_me */

/* ********************************************************************* */
void help_me() {
/* ********************************************************************* */

fprintf(stderr,"\nCyrix 486DLC/SLC CPU Configuration Utility v%s",version);
fprintf(stderr,"\n\n\t...by Paul Gortmaker");
fprintf(stderr,"\n\nValid options are:");
fprintf(stderr,"\n\t-a[-]\t\tEnable [disable] A20M input.");
fprintf(stderr,"\n\t-b[-]\t\tEnable [disable] BARB input.");
fprintf(stderr,"\n\t-c[d]\t\tTwo way assoc. [direct-mapped] cache.");
fprintf(stderr,"\n\t-d\t\tRaw dump of control registers and exit.");
fprintf(stderr,"\n\t-e[-]\t\tEnable [disable] 1k cache via CR0.");
fprintf(stderr,"\n\t-f[-]\t\tEnable [disable] FLUSH input.");
fprintf(stderr,"\n\t-h\t\tPrint this help screen.");
fprintf(stderr,"\n\t-i<n>\t\tInhibit non-cacheable region <n>, where n={1,2,3,4}.");
fprintf(stderr,"\n\t-k[-]\t\tEnable [disable] KEN input.");
fprintf(stderr,"\n\t-m[-]\t\tEnable [disable] cache of 1st 64kb of each Mb.");
fprintf(stderr,"\n\t-o\t\tOverride check for existence of Cyrix DLC chip.");
fprintf(stderr,"\n\t-p\t\tDisable all power saving modes.");
fprintf(stderr,"\n\t-q\t\tQuery DLC configuration status.");
fprintf(stderr,"\n\t-r[-]\t\tDisable [enable] cache of 640k -> 1M ROM shadow area.");
fprintf(stderr,"\n\t-s[-]\t\tEnable [disable] SUSP i/o.");
fprintf(stderr,"\n\t-v\t\tPrint version number and exit.");
fprintf(stderr,"\n\t-x<hex>,<size>\tDon't cache <size>kb from segment <hex>.");
fprintf(stderr,"\n\n");

}

/* ********************************************************************* */
unsigned long lrotl(unsigned long value, unsigned shift) {
/* ********************************************************************* */

	while (shift) {
		value*=2;
		shift--;
	}
	return(value);
}

/* ********************************************************************* */
unsigned rotl(unsigned value, unsigned shift) {
/* ********************************************************************* */

	while (shift) {
		value*=2;
		shift--;
	}
	return(value);
}

/* ********************************************************************* */
void check_for_cyrix() {
/* ********************************************************************* */

if (!override) {
	if (!cx_iscyrix()) {
		fprintf(stderr,"\n\aThis does *not* look like a DLC/SLC chip -- exiting!\n\n");
		exit(1);
	}
}

else fprintf(stderr,"\nWarning: Check for Cyrix DLC/SLC chip *disabled*  -- <gulp>\n\n");
}	

/*
   Note that these 2 subroutines are compiled conditionally, depending
   on whether the external asm routines are being used.
*/


/* ********************************************************************* */
void c_read_dlc() {
/* ********************************************************************* */

ioperm(CYR_SEL,2,PORT_IO_ON);

byte_out(0xc0,CYR_SEL);		/* read CCR0 */
ccr0=byte_in(CYR_DAT);


byte_out(0xc1,CYR_SEL);		/* read CCR1 */
ccr1=byte_in(CYR_DAT);


byte_out(0xc4,CYR_SEL);		/* select NCR1 (1st byte) */
ncr_a[0]=byte_in(CYR_DAT);

byte_out(0xc5,CYR_SEL);		/* select NCR1 (2nd byte) */
ncr_b[0]=byte_in(CYR_DAT);

byte_out(0xc6,CYR_SEL);		/* select NCR1 (3rd byte) */
ncr_c[0]=byte_in(CYR_DAT);


byte_out(0xc7,CYR_SEL);		/* select NCR2 (1st byte) */
ncr_a[1]=byte_in(CYR_DAT);

byte_out(0xc8,CYR_SEL);		/* select NCR2 (2nd byte) */
ncr_b[1]=byte_in(CYR_DAT);

byte_out(0xc9,CYR_SEL);		/* select NCR2 (3rd byte) */
ncr_c[1]=byte_in(CYR_DAT);


byte_out(0xca,CYR_SEL);		/* select NCR3 (1st byte) */
ncr_a[2]=byte_in(CYR_DAT);

byte_out(0xcb,CYR_SEL);		/* select NCR3 (2nd byte) */
ncr_b[2]=byte_in(CYR_DAT);

byte_out(0xcc,CYR_SEL);		/* select NCR3 (3rd byte) */
ncr_c[2]=byte_in(CYR_DAT);


byte_out(0xcd,CYR_SEL);		/* select NCR4 (1st byte) */
ncr_a[3]=byte_in(CYR_DAT);

byte_out(0xce,CYR_SEL);		/* select NCR4 (2nd byte) */
ncr_b[3]=byte_in(CYR_DAT);

byte_out(0xcf,CYR_SEL);		/* select NCR4 (3rd byte) */
ncr_c[3]=byte_in(CYR_DAT);

ioperm(CYR_SEL,2,PORT_IO_OFF);

} /* end c_read_dlc */

/* ********************************************************************* */
void c_write_dlc() {
/* ********************************************************************* */

ioperm(CYR_SEL,2,PORT_IO_ON);

byte_out(0xc0,CYR_SEL);		/* write CCR0 */
byte_out(ccr0,CYR_DAT);


byte_out(0xc1,CYR_SEL);		/* write CCR1 */
byte_out(ccr1,CYR_DAT);


byte_out(0xc4,CYR_SEL);		/* select NCR1 (1st byte) */
byte_out(ncr_a[0],CYR_DAT);

byte_out(0xc5,CYR_SEL);		/* select NCR1 (2nd byte) */
byte_out(ncr_b[0],CYR_DAT);

byte_out(0xc6,CYR_SEL);		/* select NCR1 (3rd byte) */
byte_out(ncr_c[0],CYR_DAT);


byte_out(0xc7,CYR_SEL);		/* select NCR2 (1st byte) */
byte_out(ncr_a[1],CYR_DAT);

byte_out(0xc8,CYR_SEL);		/* select NCR2 (2nd byte) */
byte_out(ncr_b[1],CYR_DAT);

byte_out(0xc9,CYR_SEL);		/* select NCR2 (3rd byte) */
byte_out(ncr_c[1],CYR_DAT);


byte_out(0xca,CYR_SEL);		/* select NCR3 (1st byte) */
byte_out(ncr_a[2],CYR_DAT);

byte_out(0xcb,CYR_SEL);		/* select NCR3 (2nd byte) */
byte_out(ncr_b[2],CYR_DAT);

byte_out(0xcc,CYR_SEL);		/* select NCR3 (3rd byte) */
byte_out(ncr_c[2],CYR_DAT);


byte_out(0xcd,CYR_SEL);		/* select NCR4 (1st byte) */
byte_out(ncr_a[3],CYR_DAT);

byte_out(0xce,CYR_SEL);		/* select NCR4 (2nd byte) */
byte_out(ncr_b[3],CYR_DAT);

byte_out(0xcf,CYR_SEL);		/* select NCR4 (3rd byte) */
byte_out(ncr_c[3],CYR_DAT);

ioperm(CYR_SEL,2,PORT_IO_OFF);

} /* end c_write_dlc */

