/*

Copyright 1991-1995 Stephen S. Richardson (prefect@wpi.edu)
based off of several versions for several hardware and software platforms
from 1991-1995.

(credit is due to Arthur C. Smith, whose joystick code I studied to learn how
to write linux device drivers, and to Linus Torvalds, for being the cool kind
of person to start the whole Linux project.. yar.)

You may modify, copy, fix, change, or delete this code, as long as my
copyright remains intact.  Well, if you delete it, it doesn't have to.  I also
ask that you send me a piece of email if you make any interesting changes.
You don't have to, but I want to see if anyone does anything cool with it.

There are no warranties, expressed or implied.  If your computer blows up, or
you blow up your sibling trying to use this, I'm not responsible.  If it works,
and all your friends thing you're cool, you must tell them it is due to this
project. ;)


10/23/94    S.Richardson 0.1   Initial version
10/26/94    S.Richardson 0.2   More revisions
01/??/95    S.Richardson 0.3   More revisions again.
02/02/95    S.Richardson 0.042 "release"
07/30/95    S.Richardson 0.50  "release 2" - cleanup for newer kernels.
*/

#include <linux/mm.h>
#include "lcd.h"
#include "release.h"

static struct LCD_DATA_SAVE_TYPE LCD_DATA[LCD_MAX];	/* misc data */
static int LCD_WRITE_SEMAPHORE;	/* to prevent two processes
				 from trying to write the LCD at the same time */



static int lcd_ioctl (struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	unsigned int minor;

	minor = MINOR (inode->i_rdev);
	if (MAJOR (inode->i_rdev) != LCD_MAJOR)
		return -EINVAL;
	if (minor >= LCD_MAX)
		return -ENODEV;

	switch (cmd) {
	case LCD_SETCONTROL:	/* switch to control mode */
	        LCD_DATA[minor].mode=MODE_CONTROL;
		break;
	case LCD_SETDATA:       /* switch to data mode */
		LCD_DATA[minor].mode=MODE_DATA;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

static int lcd_open (struct inode *inode, struct file *file)
{
	unsigned int minor = MINOR (inode->i_rdev);

	if (minor >= LCD_MAX)
		return -ENODEV;	

	if (LCD_DATA[minor].BUSY)
		return -EBUSY;
	LCD_DATA[minor].BUSY = LCD_TRUE;
	MOD_INC_USE_COUNT;
	return 0;
}

static void lcd_release (struct inode *inode, struct file *file)
{
	unsigned int minor = MINOR (inode->i_rdev);
	inode->i_atime = CURRENT_TIME;
	LCD_DATA[minor].BUSY = LCD_FALSE;
	MOD_DEC_USE_COUNT;
}


static int lcd_write (struct inode *inode, struct file *file, char *buf, int count)
{
  int cnt, sz, t, i;
  struct LCD_DATA_TYPE tempdata;

  unsigned int minor, minor2;
  if (count != LCD_DATASIZE)
    return -EOVERFLOW;
  verify_area (VERIFY_WRITE, (void *) buf, sizeof (struct LCD_DATA_TYPE));
  
  minor = MINOR (inode->i_rdev);

  memcpy_fromfs (&tempdata, buf, LCD_DATASIZE);

  cli ();
  if (!LCD_WRITE_SEMAPHORE) {
    LCD_WRITE_SEMAPHORE++;
  }
  sti ();
  
  cnt=0;
  sz=strlen (tempdata.text);
  
  if (LCD_DATA[minor].mode == MODE_CONTROL) {
    while (cnt<sz) {
      outb (tempdata.text[cnt], LCD_D_PORT);
      outb (255-0, LCD_C_PORT);
      outb (255-LCD_EN, LCD_C_PORT);
      for (t=0;t<3000;t++);
      cnt++;
      outb (255-0, LCD_C_PORT);
      outb (0, LCD_D_PORT);
    }
  } else {
    while (cnt<sz) {
      outb (tempdata.text[cnt], LCD_D_PORT);
      outb (255-LCD_RS, LCD_C_PORT);
      outb (255-(LCD_EN|LCD_RS), LCD_C_PORT);
      for (t=0;t<3000;t++);
      cnt++;
      outb (255-LCD_RS, LCD_C_PORT);
      outb (0, LCD_D_PORT);
    }
  }
  
  LCD_WRITE_SEMAPHORE = 0;	/*allow other reads to progress*/
  
  return LCD_DATASIZE;
}


static struct file_operations lcd_fops =
{
	NULL,			/* lcd_lseek*/
	NULL,		        /* lcd_read */
	lcd_write,		/* lcd_write*/
	NULL,			/* lcd_readaddr*/
	NULL,			/* lcd_select */
	lcd_ioctl,		/* lcd_ioctl*/
	NULL,			/* lcd_mmap */
	lcd_open,		/* lcd_open*/
	lcd_release,		/* lcd_release*/
	NULL			/* lcd_sync */
};

#ifdef __cplusplus
extern "C" {
#endif

int init_module (void) {
	unsigned char lcd_num;
	int j;
	
	printk ("lcd_init: Hitachi HD44780 LCD driver enabled.\n");
	if (register_chrdev (LCD_MAJOR, "LCD", &lcd_fops))
		  printk ("Unable to get major=%d for LCD\n",
					LCD_MAJOR);
	for (lcd_num = 0; lcd_num < LCD_MAX; lcd_num++)
		  LCD_DATA[lcd_num].BUSY = LCD_FALSE;
	  LCD_WRITE_SEMAPHORE = 0;
	  return 0;
}

void cleanup_module (void)
{
	if (MOD_IN_USE)
		printk ("LCD: device busy, remove delayed.\n");

	if (unregister_chrdev (LCD_MAJOR, "LCD") != 0) {
		printk ("cleanup_module failed.\n");
	}
	else {
		printk ("cleanup_module succeeded.\n");
	}
}

#ifdef __cplusplus
}
#endif










