#
# This procedure graphically displays the hosts, files and dependencies of a
# project.  It allows the user to change the location of files and 
# dependencies and to add new items.
#
# Pre: hlist is the list of hosts, files, and dependencies passed from gethosts
#
# Post: hlist contains the modified list of hosts, files, and dependencies
#

proc Tree {{w .tree}} {
   global hlist list compile treecomp curcount itemcount addhere drawcount
   global modhost modfile
   global ntcol ntrow

   set list $hlist
   set treecomp $compile

#  addhere used by TreeClick
   set addhere 0
   set drawcount 0
   set type none

   catch {destroy $w}
   toplevel $w
   wm title $w "PADE Tree View"
   wm iconname $w "Tree View"
   wm minsize $w 1066 742
   wm geometry $w +200+278
   $w config -cursor {arrow red white}

   grab set .tree

   frame $w.f1 -bd 1 -relief raised
   frame $w.f2
   frame $w.f2.1 -bd 1 -relief raised
   frame $w.f2.2 -bd 1 -relief raised
   frame $w.f2.3 -bd 1 -relief raised
   frame $w.f3 -bd 1 -relief raised
   
   pack  $w.f2 $w.f3 $w.f1 -side top -fill x

#-------------------------------------------------------------------------------

#  input the name of the host
   button $w.f2.1.addhost -text "Add Host" -command {
      AddHost "" "" ""
      if {$entry1 != "CANCEL"} {
         set modhost 1
	 set list [linsert $list [llength $list] $entry1 {}]
	 set treecomp [linsert $treecomp [llength $treecomp] $entry2]
	 .tree.f1.c delete all
	 TreeDraw .tree.f1.c
	 .tree.f3.text delete 1.0 end
	 .tree.f3.text insert end "Added host: " 
	 .tree.f3.text insert end $entry1
      }
   }

   button $w.f2.1.addsrc -text "Add File" -command {
      set type file
      AddFile "" "" "ADD"
      if {$entry1 != ""} {
         set modfile 1
         set addhere 1
	 .tree.f1.c delete all
	 TreeDraw .tree.f1.c
	 .tree.f3.text delete 1.0 end
	 .tree.f3.text insert end "Click on host to Add file to" 
      }
   }

   button $w.f2.1.copy -text "Duplicate" -command {
      set type none
      .tree.f3.text delete 1.0 end
      .tree.f3.text insert end "Click on item to be Duplicated"
      set addhere -2
   }

   button $w.f2.1.modify -text "Modify" -command {
      set type none
      .tree.f3.text delete 1.0 end
      .tree.f3.text insert end "Click on item to be Modified"
      set addhere -4
   } 

   button $w.f2.1.delete -text "Delete" -command {
      set type none
      .tree.f3.text delete 1.0 end
      .tree.f3.text insert end "Click on item to be Deleted"
      set addhere -1
   }
   
   menubutton $w.f2.2.columns -text "Columns" -menu $w.f2.2.columns.menu \
      -relief raised -bd 2
   menu $w.f2.2.columns.menu 
   $w.f2.2.columns.menu add radiobutton -label 2 -variable ntcol -value 2 \
       -command TreeRedraw
   $w.f2.2.columns.menu add radiobutton -label 3 -variable ntcol -value 3 \
       -command TreeRedraw
   $w.f2.2.columns.menu add radiobutton -label 4 -variable ntcol -value 4 \
       -command TreeRedraw
   $w.f2.2.columns.menu add radiobutton -label 6 -variable ntcol -value 6 \
       -command TreeRedraw
   $w.f2.2.columns.menu add radiobutton -label 8 -variable ntcol -value 8 \
       -command TreeRedraw
   $w.f2.2.columns.menu add radiobutton -label 10 -variable ntcol -value 10 \
       -command TreeRedraw
   $w.f2.2.columns.menu add radiobutton -label 15 -variable ntcol -value 15 \
       -command TreeRedraw
   $w.f2.2.columns.menu add radiobutton -label 20 -variable ntcol -value 20 \
       -command TreeRedraw
   
   menubutton $w.f2.2.rows -text "Rows" -menu $w.f2.2.rows.menu \
      -relief raised -bd 2
   menu $w.f2.2.rows.menu 
   $w.f2.2.rows.menu add radiobutton -label 18 -variable ntrow -value 18 \
       -command TreeRedraw
   $w.f2.2.rows.menu add radiobutton -label 24 -variable ntrow -value 24 \
       -command TreeRedraw
   $w.f2.2.rows.menu add radiobutton -label 30 -variable ntrow -value 30 \
       -command TreeRedraw
   $w.f2.2.rows.menu add radiobutton -label 36 -variable ntrow -value 36 \
       -command TreeRedraw
   $w.f2.2.rows.menu add radiobutton -label 48 -variable ntrow -value 48 \
       -command TreeRedraw
   $w.f2.2.rows.menu add radiobutton -label 60 -variable ntrow -value 60 \
       -command TreeRedraw
   
   button $w.f2.2.help -text "Help" -command {
      .tree.f3.text delete 1.0 end
      .tree.f3.text insert end "To Move a File from host to host: Drag file with Left-Mouse-Button on center of host"
      .tree.f3.text insert end "\nTo Edit a File: Double-Click Left-Mouse-Button on file name"
      .tree.f3.text insert end "\nTo Scoll Canvas: Use scrollbar or Control-Left-Mouse-Button"
      .tree.f3.text insert end "\nClick on Host or File for more information about it"
   }
   
   button $w.f2.3.save -text "Save" -command {
      set hlist $list
      set compile $treecomp
      destroy .tree
   }
 
   button $w.f2.3.cancel -text "Cancel" -command {
      if {$confirm=="on" && ($modhost || $modfile)} {
         set pick [Dialog .d 230 338 {PADE Question} \
            {Are you sure you want to Cancel your changes?} \
            questhead 0 {Yes} {No}]
      } else {
         set pick 0
      }
      if {$pick==0} {destroy .tree}
   }
 
   pack $w.f2.1.addhost $w.f2.1.addsrc $w.f2.1.copy $w.f2.1.modify $w.f2.1.delete \
       -side left -expand 1 -padx 3m -pady 4m -ipadx 2m -ipady 1m 

   pack $w.f2.2.columns $w.f2.2.rows $w.f2.2.help -side left -expand 1 -padx 3m -pady 4m \
      -ipadx 2m -ipady 1m 

   pack $w.f2.3.save $w.f2.3.cancel -side left -expand 1 -padx 3m -pady 4m \
      -ipadx 2m -ipady 1m 

   pack $w.f2.1 $w.f2.2 $w.f2.3 -side left -expand 1 -fill both

#-------------------------------------------------------------------------------

   text	$w.f3.text -height 4 -font -*-helvetica-bold-r-normal--*-120-*-*-* \
      -fg red -relief sunken -bd 2
   pack $w.f3.text -anchor w -padx 2m -pady 2m -fill x

#-------------------------------------------------------------------------------
#  {left top right bottom}
   update idletasks
   set c $w.f1.c
   canvas $c -scrollregion {0c 0c 28c 15.5c} \
      -yscroll "$w.f1.yscroll set" \
      -relief sunken -height 15.5c -width 28c
   scrollbar $w.f1.yscroll -relief sunken -command "$c yview"
   pack $w.f1.yscroll -side right -fill y
   pack $c -expand 1 -fill both

   set curcount 0
   set itemcount 0
   TreeDraw $c
   set dr 0
   tkwait window $w
}

################################################################################

proc TreeRedraw {} {
   global ntcol ntrow
   .tree.f3.text delete 1.0 end
   .tree.f3.text insert end "Redrawing with $ntcol columns and $ntrow rows"
   .tree.f1.c delete all
    TreeDraw .tree.f1.c
}

################################################################################

#
# This procedure displays hosts, files, and dependencies in a canvas widget
#
# Pre: list is the list of hosts, files, and dependencies
#

proc TreeDraw {c} {
   global list curcount itemcount drawcount ntcol ntrow
   set itemcount [expr $itemcount+$curcount]
   set curcount 0
   
   set cy {}
   set cx {}
   set ytop -12
   set xleft -10
   set txscale [expr 4./$ntcol]
   set tyscale [expr 24./$ntrow]
   set deltax [expr 6.8*$txscale]
   Do i 1 $ntcol {
      lappend cy $ytop
      lappend cx $xleft
      set xleft [expr $xleft+$deltax]
   }
    
   set y [Min $cy]
   set ymax $y
   set row 0

   set colhst(1) "red"
   set colhst(2) "green"
   set colhst(3) "blue"
   set colfil(1) "yellow"
   set colfil(2) "cyan"
   set colfil(3) "magenta"
   set coldep "green"
   set colout "black"
   set ncolor 0

   if {$ntcol < 8} {
      set arrtyp last
   } else {
      set arrtyp none
   }
   if {$ntcol < 10} {
      set ptsz 120
      set nfchar 16
   } else {
      set ptsz 80
      set nfchar 25
   }
   if {$ntrow > 30 && $ptsz == 120} {
      set ptsz 80
      set nfchar 25
   }
   if {$ptsz == 120} {
      set linwid 2
   } else {
      set linwid 1
   }
   set ffont -*-courier-bold-r-normal--*-$ptsz-*-*-*
      
   set cirdia [expr 1.25*$txscale]
   set recwid [expr 4.4*$txscale]
   set recxspc [expr 0.3*$txscale]

   set recyspc [expr 0.64*$tyscale]
   set rechgt [expr $recyspc*0.75]
   
   .tree config -cursor {watch red white}

#  y is the y position of the item to be added to the tree
#  hosts are placed at x position -10c.  Files are at -7c and dependencies
#  are at -4c.  Curcount keeps track of how many items have been added.
#  This includes text, arrows, and shapes. 

   for {set i 0} {$i < [expr [llength $list]/2]} {incr i} {
      set row [Min $cy]
      set y [lindex $cy $row]
      set x [lindex $cx $row]
      if {$y!=$ytop} {
         set yinc [expr 3.*$recyspc]
         if {$yinc < $cirdia} {set yinc [expr $cirdia*1.1]}
         set y [expr $y+$yinc]
      } else {
         set y [expr $y+2.75]
      }
      set ncolor [expr $ncolor+1]
      if {$ncolor > 3} {set ncolor 1}

#  add a host circle
      set y1 [expr $y+$cirdia]
      set a [$c create oval ${x}c ${y}c [expr $x+$cirdia]c ${y1}c \
         -outline $colout -fill $colhst($ncolor) -tags tag_host -width $linwid]
      if {$y1 > $ymax} {set ymax $y1}

#  put text near the circle
      set hoststr [lindex $list [expr $i*2]]
      set nchar [expr int($nfchar*$txscale*1.5)]
      set strlen [string length $hoststr]
      if {$strlen > $nchar} {
         set hoststr "[string range $hoststr 0 [expr $nchar-3]].." 
      }
      $c create text [expr $x]c [expr $y-(0.35*$tyscale)]c \
	 -anchor w -text $hoststr -tags tag_text -font $ffont

#  put line off right side of circle
      set y2 [expr $cirdia/2. + $y]
      set x2 [expr $x+$cirdia]
      $c create line ${x2}c ${y2}c [expr $x2+$recxspc]c ${y2}c \
         -tags tag_arrow -width $linwid

      set curcount [expr 3+$curcount]  
      set list2 [lindex $list [expr $i*2+1]]   ;# The list of files for host i

#  add files
      for {set j 0} {$j < [expr [llength $list2]/3] } {incr j} {

#  don't increment y for the first file
         if {$j} {set y [expr $y+$recyspc]}  

#  add a file rectangle
         set y3 [expr $y+$recyspc]
         set x3 [expr $x+$cirdia+($recxspc*2.)]
         set a [$c create rect ${x3}c [expr ${y3}-$rechgt]c [expr $x3+$recwid]c \
            ${y3}c -outline $colout -fill $colfil($ncolor) -tags tag_file -width $linwid]
	 if {$y3 > $ymax} {set ymax $y3}

#  get the name of the file
         set y4 [expr $y3-($rechgt/2.)]
         set filestr [file tail [lindex $list2 [expr $j*3]]] 

#  add ... if it is too long, lenght depends on font and width of rectangle
         set nchar [expr int($nfchar*$txscale)]
         set strlen [string length $filestr]
         if {$strlen > $nchar} {
            set filestr "[string range $filestr 0 [expr $nchar-3]].." 
         }

#  put the filename in the rectangle
         $c create text [expr $x3+.05]c ${y4}c \
            -anchor w -text $filestr -tags tag_text -font $ffont

#  draw lines and an arrow between the file and its host
         set x4 [expr $x2+$recxspc]
         $c create line ${x4}c ${y4}c ${x3}c ${y4}c \
            -arrow $arrtyp -tags tag_arrow -width $linwid
         $c create line ${x4}c ${y2}c ${x4}c ${y4}c \
            -tags tag_arrow -width $linwid

         set curcount [expr 4+$curcount]
         set list3 [lindex $list2 [expr $j*3+2]] ;#the dependencies for file j
         set fy [expr .5+$y]

#  add dependencies
#         for {set k 0} {$k < [expr [llength $list3]/2]} {incr k} {
#
#  don't increment y for the first dependency
#            if {$k} {set y [expr $y+2]}
#
#  add a dependency oval
#            set a [$c create oval -4c ${y}c -2c [expr $y+1]c \
#              -outline $colout -fill $coldep -tags tag_dep]
#	    if {[expr $y+1] > $ymax} {set ymax [expr $y+1]}
#
#  get the name of the dependency
#            set depstr [file tail [lindex $list3 [expr $k*2]]] 
#
#  add ... if it is too long
#            set strlen [string length $depstr]
#            if {$strlen>15} {
#               set depstr "[string range $depstr 0 13]..." 
#            }
#
#  put the dependency name in the oval
#            $c create text -3c [expr $y+.5]c -anchor center \
#              -text $depstr -tags tag_text
#            $c create line -5c ${fy}c -4c [expr $y+.5]c -arrow last \
#              -tags tag_arrow
#            set curcount [expr 3+$curcount]
#         }
      } 
      set cy [lreplace $cy $row $row $y]
   }

   set sclreg {-10.5c -10c 0c}
   lappend sclreg [expr $ymax+$cirdia]c
   $c configure -scrollregion $sclreg 

   if {$list==""} {
      .tree.f2.1.addsrc configure -state disabled
      .tree.f2.1.copy configure -state disabled
      .tree.f2.1.modify configure -state disabled
      .tree.f2.1.delete configure -state disabled
   } else {
      .tree.f2.1.addsrc configure -state normal
      .tree.f2.1.copy configure -state normal
      .tree.f2.1.modify configure -state normal
      .tree.f2.1.delete configure -state normal
   }
   .tree config -cursor {arrow red white}
   update idletasks

   if {$drawcount == 0} {$c yview 0}  ;#set the view to the top of the canvas
   incr drawcount
   $c bind all <Double-1> "set type none
                            TreeClick $c %x %y -3"
   bind $c <1> "TreeClick $c %x %y \$addhere
        	set addhere 0"
   bind $c <B1-Motion> "$c scan mark %x %y
	        	TreeDrag $c %x %y"
   bind $c <ButtonRelease-1> "TreePlace $c %x %y"
   bind $c <Control-Button-1> "$c scan mark %x %y"
   bind $c <Control-B1-Motion> "
   $c scan dragto 0 %y"
}

################################################################################

#
# This procedure is invoked when the user clicks on the tree canvas.  If the
# user has just entered a new item, the procedure will add it to the canvas and 
# list if the user has clicked in a proper location.  If the user has clicked
# on a file or dependency, the procedure will record which item has been 
# selected and where it is now located for later use by the drag procedure.
#
# Pre: c is the canvas
#      x and y are the coordinates of the cursor
#      add is positive if something is to be added, negative if something is
#        to be deleted, and 0 if nothing is to be added or deleted.
#      type is the type of item to be added 
#
# Post: xpos and ypos are the current coordinates of the selected item
#       dr has the value of the selected item if the item is dragable.
#         Otherwise it contains 0.
#
 
proc TreeClick {c x y add} {
   global xpos ypos dr oldxpos oldypos type entry1 entry2 
   global itemcount list treecomp
   global modhost modfile
   set dr 0
   set xpos [$c canvasx $x]
   set ypos [$c canvasy $y]
   set oldxpos $xpos
   set oldypos $ypos

   .tree.f3.text delete 1.0 end
   .tree config -cursor {watch red white}
   update idletasks

   if {!$add} {set type none}
   set id [$c find withtag current]
   set dlist [$c find withtag tag_host]   ;#a list of all the host circles

#  if a host (or the text in the host) has been selected and a file is to be
#  added or "Delete" has been selected 
   if {(($type == "file")||($add<=0))&&(([lsearch $dlist $id]>=0)||([lsearch \
      $dlist [expr $id-1]]>=0))} {
      set whichhost 0
      set where $id

#  scan through the list until the selected host is reached
      while {$where>[expr $itemcount+2]} {
         set where [expr $where-3]  ;# each host is represented by two items
         set templist [lindex $list [expr 2*$whichhost+1]] ;#the file list
         for {set i 0} {$i<[expr [llength $templist]/3]} {set i [expr $i+1]} {

#  subtract three for each file and dependency (each has an arrow)
            set where [expr $where-4-4*[llength [lindex $templist \
            [expr $i*3+2]]]/2]
         }
         set whichhost [expr 1+$whichhost] ;#next host
      }

      set index [expr $whichhost*2+1] ;#the index of the host's list of files

      if {$add>0} {  ;#if a file is to be added
         set replist [lindex $list $index] ;# the host's list of files
#  insert the new file at the end of the list of files
         set replist [linsert $replist [llength $replist] \
            $entry1 $entry2 {}]
#  put the new file list into the main list
         set list [lreplace $list $index $index $replist]
         .tree.f3.text delete 1.0 end
         .tree.f3.text insert end "Added file to host: "
         .tree.f3.text insert end [lindex $list [expr $index-1]]
         .tree.f3.text insert end "\nLOCAL FILE: "
         .tree.f3.text insert end $entry1
         .tree.f3.text insert end "\nREMOTE FILE: "
         .tree.f3.text insert end $entry2
      }

#  if the host is to be deleted, remove it and the file list from the main list
      if {$add==-1} {
         set pick [Dialog .d 230 338 {PADE Question} \
            {Are you sure you want to Delete the host and all its commands and\
            files?} questhead 0 {Yes} {No}]
         if {$pick==0} {
            set modfile 1
            set temphost [lindex $list [expr $index-1]]
            set list [lreplace $list [expr $index-1] $index]
            set treecomp [lreplace $treecomp $whichhost $whichhost]
            .tree.f3.text delete 1.0 end
            .tree.f3.text insert end "Deleted host and its commands and files: "
            .tree.f3.text insert end $temphost 
         }
      }

#  duplicate
      if {$add==-2} {
         set modfile 1
         set temphost [lindex $list [expr $index-1]]
         set templist [lindex $list $index]
         set tempcomp [lindex $treecomp $whichhost]
         set list [linsert $list [expr 1+$index] $temphost $templist]
         set treecomp [linsert $treecomp $whichhost $tempcomp]
         .tree.f3.text delete 1.0 end
         .tree.f3.text insert end "Duplicated host and its files: "
         .tree.f3.text insert end [lindex $list [expr $index-1]] 
      }

#  modify
      if {$add == -4} {
         AddHost [lindex $list [expr $index-1]] \
            [lindex $treecomp $whichhost] ""
         if {$entry1 != "CANCEL"} {
            set modhost 1 
            set list [lreplace $list [expr $index-1] [expr $index-1] $entry1]
            set treecomp [lreplace $treecomp $whichhost $whichhost $entry2]
            .tree.f3.text delete 1.0 end
            .tree.f3.text insert end "Modified host: "
            .tree.f3.text insert end [lindex $list [expr $index-1]] 
         }
      }

#  single click on host
      if {$add == 0} {
         .tree.f3.text delete 1.0 end
         .tree.f3.text insert end "HOST: "
         .tree.f3.text insert end [lindex $list [expr $index-1]] 
         .tree.f3.text insert end "\nHOST COMMANDS: "
         .tree.f3.text insert end [lindex $treecomp $whichhost] 
      }

      if {$add} {
         $c delete all ;# delete and redraw the canvas
         TreeDraw $c
      }
   }

   set dlist [$c find withtag tag_file] ;# a list of all the files

#  if a file (or the text of a file) has been clicked on and a dependency
#  is to be added or "Delete" has been pressed
   if {(([lsearch $dlist $id] >= 0)||([lsearch $dlist \
      [expr $id-1]]>=0))&&(($type=="dependency")||($add<=0))} {
      set whichhost -1
      set whichfile 0
      set where $id

#  search the list for the file and the host that it corresponds to 
      while {$where>[expr $itemcount+2]} {
         set where [expr $where-3] ;# each host is represented by two items
         set templist [lindex $list [expr 2*$whichhost+3]] ;#the list of files
         set whichfile 0

#  while file hasn't been found and  end of list of files hasn't been reached
         while {($where>[expr $itemcount+2])&&($whichfile<[expr \
            [llength $templist]/3])} {
#  subtract 3 for each file and dependency
            set where [expr $where-4]
            set where [expr $where-4*[llength [lindex $templist \
              [expr 3*$whichfile+2]]]/2]
            set whichfile [expr $whichfile+1]
         }
         set whichhost [expr 1+$whichhost]
      }
#  the index of the file list which contains the selected file
      set index [expr 2*$whichhost+1]
#  the dependency list of the selected file
      set index2 [expr 3*$whichfile+2]
#  the file list which contains the selected file
      set delfiles [lindex $list $index]

      if {$add>0} {   ;#if a dependency is to be added
#  the dependency list to be added to
         set deldeps [lindex $delfiles $index2]
#  add the dependency and its remote name to the dependency list
         set deldeps [linsert $deldeps [llength $deldeps] $entry1 $entry2]
#  put the new dependency list into the file list
         set delfiles [lreplace $delfiles $index2 $index2 $deldeps]
      }

#  single click on file
      if {$add==0} {    ;# display the file's name and remote name
         .tree.f3.text delete 1.0 end
         .tree.f3.text insert end "HOST: "
         .tree.f3.text insert end [lindex $list [expr $index-1]] 
         .tree.f3.text insert end "\nLOCAL FILE: "
         .tree.f3.text insert end [lindex $delfiles [expr $index2-2]]
         .tree.f3.text insert end "\nREMOTE FILE: "
         .tree.f3.text insert end [lindex $delfiles [expr $index2-1]] 
      }

      if {$add==-1} {   ;# if a file is to be deleted
#  delete the file, it's remote name and the dependency list from the file list
         set tempfile [lindex $delfiles [expr $index2-2]]
         set delfiles [lreplace $delfiles [expr $index2-2] $index2]
         .tree.f3.text delete 1.0 end
         .tree.f3.text insert end "Deleted file: "
         .tree.f3.text insert end $tempfile
      }

#  duplicate
      if {$add==-2} {
         set tempfile [lindex $delfiles [expr $index2-2]]
         set tempremote [lindex $delfiles [expr $index2-1]]
         set templist [lindex $delfiles $index2]
         set delfiles [linsert $delfiles [expr 1+$index2] \
            $tempfile $tempremote $templist]
         .tree.f3.text delete 1.0 end
         .tree.f3.text insert end "Duplicated file: "
         .tree.f3.text insert end [lindex $delfiles [expr $index2-2]]
      }

#  edit file
      if {$add==-3} {
         EditFile [lindex $delfiles [expr $index2-2]]
         .tree.f3.text delete 1.0 end
         .tree.f3.text insert end "Edited file: "
         .tree.f3.text insert end [lindex $delfiles [expr $index2-2]]
      }

#  modify file
      if {$add == -4} {
         AddFile [lindex $delfiles [expr $index2-2]] \
            [lindex $delfiles [expr $index2-1]] "MOD"
         set delfiles [lreplace $delfiles [expr $index2-2] \
            [expr $index2-1] $entry1 $entry2]
         set modfile 1
         .tree.f3.text delete 1.0 end
         .tree.f3.text insert end "Modified"
         .tree.f3.text insert end "\nLOCAL FILE: "
         .tree.f3.text insert end [lindex $delfiles [expr $index2-2]]
         .tree.f3.text insert end "\nREMOTE FILE: "
         .tree.f3.text insert end [lindex $delfiles [expr $index2-1]] 
      }

#  put the new file list into the main list
      set list [lreplace $list $index $index $delfiles]
      if {$add} {
         $c delete all  ;# delete and redraw the canvas
         TreeDraw $c
      }
   }

   set dlist [$c find withtag tag_dep] ;#all the dependency ovals
#  if the item is a dependency and is to be deleted
   if {(([lsearch $dlist $id] >= 0)||([lsearch $dlist \
      [expr $id-1]]>=0))&&($add<0)} {
      set dephost -1
      set depfile -1
      set depdep 0
      set where $id

#  find the location in the list of the dependency, its file, and its host
      while {$where>[expr $itemcount+2]} {
#  subtract two from where for each host
         set where [expr $where-4] ;#each host is represented by two items
         set templist [lindex $list [expr 2*$dephost+3]] ;#the list of files
         set depfile -1
         while {($where>[expr $itemcount+2])&&($depfile<[expr \
            [llength $templist]/3 - 1])} {
            set where [expr $where-4] ;#each file represented by three items
#  the list of dependencies
            set templist2 [lindex $templist [expr 3*$depfile+5]]
            set depdep 0
            while {($where>[expr $itemcount+2])&&($depdep<[expr \
            [llength $templist2]/2])} {
              set where [expr $where-4] ;#each dependency is represented by 3
              set depdep [expr $depdep+1]
            }
            set depfile [expr $depfile+1]
         }
         set dephost [expr $dephost+1]
      }

#  the index of the file list which contains the selected dependency
      set index1 [expr $dephost*2+1] 

#  the index of the dependency list which contains the dependency 
      set index2 [expr $depfile*3+2]

#  the index of the dependency
      set index3 [expr $depdep*2]

#  the file list containing the dependency
      set delfile [lindex $list $index1]

#  the dependency list containing the dependency
      set deldep [lindex $delfile $index2]

#  remove the dependency and its remote name
      set deldep [lreplace $deldep $index3 [expr $index3+1]]

#  put the new dependency list into the file list
      set delfile [lreplace $delfile $index2 $index2 $deldep]  

#  put the new file list into the main list
      set list [lreplace $list $index1 $index1 $delfile]

      $c delete all    ;#delete and redraw the canvas
      TreeDraw $c
   }
   .tree config -cursor {arrow red white}
   update idletasks
}

################################################################################

#
# This procedure is invoked when the user holds down button 1 and moves the
# mouse.  It checks to see if what was clicked on is dragable and if so moves
# it.
#
# Pre: xpos and ypos are the current positions of the item being dragged
#
# Post: xpos and ypos are the new positions of the item being dragged
#       dr is positive if something is being dragged
#
 
proc TreeDrag {c x y} {
   global xpos ypos dr oldxpos oldypos
   set id [$c find withtag current]
   set dr 1
   if {![llength $id]} {set dr 0} ;#if nothing is selected, don't drag

   if {[llength $id]} { ;#if something is selected
      set x [$c canvasx $x]
      set y [$c canvasy $y]
      set dlist [$c find withtag tag_host] ;#all of the host circles
      set list2 [$c find withtag tag_arrow] ;#all of the arrows

#  if a host or an arrow has been selected, don't drag
      if {([lsearch $dlist $id]>=0)||([lsearch $dlist [expr $id-1]]>=0)||(\
         [lsearch $list2 $id]>=0)} {set dr 0}

#  if it's not a host or arrow (file or dependency), drag it
      if {([lsearch $dlist $id]<0)&&([lsearch $dlist [expr $id-1]]<0)&&([lsearch \
         $list2 $id]<0)} {
         set dr $id

#  move the selected item
         $c move current [expr $x-$xpos] [expr $y-$ypos]
 
#  make sure both the text and the object are moved
	 if {[lsearch [$c gettags current] tag_text] < 0} {
            $c move [expr $id+1] [expr $x-$xpos] [expr $y-$ypos]
            if {[$c gettags [expr $id+1]]=="tag_text"} {
               $c itemconfigure [expr $id] -fill white
               $c raise [expr $id]
               $c raise [expr $id+1]
	    }
	 }
	 if {[lsearch [$c gettags current] tag_text] >= 0} {
            $c move [expr $id-1] [expr $x-$xpos] [expr $y-$ypos]
            if {[$c gettags [expr $id-1]]=="tag_file"} {
               $c itemconfigure [expr $id-1] -fill white
               $c raise [expr $id-1]
               $c raise [expr $id]
	    }
         }
      }
      set xpos $x
      set ypos $y
   }
}

################################################################################

#
#This procedure is invoked when the user lets go of the mouse button.  It 
#allows the user to move a file by placing it on a host or move a dependency
#by placing it on a file.
#
#Pre: c is the canvas
#     x and y are the coordinates of the mouse
#     oldxpos and oldypos are the original positions of the object being movec
#     dr contains the id of the object being dragged or 0
#     itemcount contains the index of the first item in the canvas
#

proc TreePlace {c x y} {
   global xpos ypos dr oldxpos oldypos itemcount modfile

   if {$dr} {
      set id $dr
#  set id2 to the item if id is an object, and to an object if id is an item
      if {[lsearch [$c gettags $id] tag_text] <0} {
	set id2 [expr $id+1]
      }
      if {[lsearch [$c gettags $id] tag_text] >=0} {
	set id2 [expr $id-1]
      }

#  move id and id2 out of the way and find the closest object
      set checkx $x
      set checky $y
      $c move $id 50 50
      $c move $id2 50 50
      $c addtag insertthere closest [$c canvasx $checkx] [$c canvasy $checky]
      $c move $id -50 -50
      $c move $id2 -50 -50
      set x $checkx
      set y $checky

      set id3 [$c find withtag insertthere] ;#where the object is to be placed
      set dlist [$c find withtag tag_host]
      set test [$c find withtag tag_file]

#  if a file is being placed on a host 
      if {(([lsearch $dlist $id3] >= 0)||([lsearch $dlist [expr $id3-1]]>=0))&&(\
	 ([lsearch $test $id]>=0)||([lsearch $test [expr $id-1]]>=0))} {
	 global list
	 set whichhost 0
	 set where $id3
	 set modfile 1

#  find which host the file has been placed on
	 while {$where>[expr $itemcount+2]} {
            set where [expr $where-3]
            set templist [lindex $list [expr 2*$whichhost+1]]
            for {set i 0} {$i<[expr [llength $templist]/3]} {set i [expr $i+1]} {
               set where [expr $where-4-4*[llength [lindex $templist \
                  [expr $i*3+2]]]/2]
            } 
            set whichhost [expr 1+$whichhost]
	 }

	 set filehost -1 
	 set whichfile 0
	 set where $id

#  find the file that has been placed and its host
	 while {$where>[expr $itemcount+2]} {
            set where [expr $where-3]
            set templist [lindex $list [expr 2*$filehost+3]]
            set whichfile 0
            while {($where>[expr $itemcount+2])&&($whichfile<[expr \
               [llength $templist]/3])} {
               set where [expr $where-4]
               set where [expr $where-4*[llength [lindex $templist \
                  [expr 3*$whichfile+2]]]/2]
               set whichfile [expr $whichfile+1] 
            }
            set filehost [expr 1+$filehost]
	 }

#  the index of the file in the list
         set whichfile [expr $whichfile*3]
#  the index of the list of files to be deleted from 
         set filehost [expr $filehost*2+1]
#  the list of files
         set dellist [lindex $list $filehost]
#  the files' list of dependencies
         set deldep [lindex $dellist [expr $whichfile+2]]
#  the files' remote name
         set delrem [lindex $dellist [expr $whichfile+1]]
#  the file
         set delfile [lindex $dellist $whichfile]
#  delete the file, its remote name, and its dependency list
         set dellist [lreplace $dellist $whichfile [expr $whichfile+2]]
#  put the new file list into the main list 
         set list [lreplace $list $filehost $filehost $dellist] 

#  the index of the list of files to be added to
         set index [expr 2*$whichhost+1]
#  the list of files
         set templist [lindex $list $index]
#  insert the file, its remote name, and its dependencies at the end
         set templist [linsert $templist [llength $templist] $delfile $delrem \
            $deldep]
#  put the new file list into the main list 
         set list [lreplace $list $index $index $templist]

         $c delete all   ;#delete and redraw the canvas
         TreeDraw $c
      }

      set c .tree.f1.c
      set dlist [$c find withtag tag_file] 

#  if a dependency is being placed on a file
      if {(([lsearch $dlist $id3] >= 0)||([lsearch $dlist [expr $id3-1]]>=0))&&(\
	 $id!=$id3)&&($id2!=$id3)&&(!(([lsearch $test $id]>=0)||([lsearch $test \
	 [expr $id-1]]>=0)))} {
	 global list
	 set whichhost -1
	 set whichfile 0
	 set where $id3

#  find which file it has been placed on and the file's host
	 while {$where>[expr $itemcount+2]} {
            set where [expr $where-3]
            set templist [lindex $list [expr 2*$whichhost+3]]
            set whichfile 0
            while {($where>[expr $itemcount+2])&&($whichfile<[expr \
               [llength $templist]/3])} {
               set where [expr $where-4]
               set where [expr $where-4*[llength [lindex $templist \
               [expr 3*$whichfile+2]]]/2]
               set whichfile [expr $whichfile+1]
            }
            set whichhost [expr 1+$whichhost]
	 }

	 set dephost -1
	 set depfile -1
	 set depdep 0
	 set where $id

#  find the dependency, its file, and the file's host
	 while {$where>[expr $itemcount+2]} {
            set where [expr $where-3]
            set templist [lindex $list [expr 2*$dephost+3]]
            set depfile -1
            while {($where>[expr $itemcount+2])&&($depfile<[expr \
               [llength $templist]/3 - 1])} {
               set where [expr $where-4]
               set templist2 [lindex $templist [expr 3*$depfile+5]]
               set depdep 0
               while {($where>[expr $itemcount+2])&&($depdep<[expr \
                  [llength $templist2]/2])} {
        	  set where [expr $where-4]
        	  set depdep [expr $depdep+1]
               }
               set depfile [expr $depfile+1]
            }
            set dephost [expr $dephost+1]
	 }

#  the position of the file list containing the moved dependency
	 set dephost [expr $dephost*2+1]
#  the position of the dependency in the dependency list
	 set depdep [expr $depdep*2]
#  the position of the list of dependencys containing the moved dependency
	 set depfile [expr $depfile*3+2]
#  the file list containing the moved dependency
	 set delfile [lindex $list $dephost]
#  the dependency list containing the moved dependency
	 set dellist [lindex $delfile $depfile] 
#  the remote name of the dependency
	 set delrem [lindex $dellist [expr $depdep+1]]
#  the dependency
	 set deldep [lindex $dellist $depdep]
#  take out the dependency and its remote name
	 set dellist [lreplace $dellist $depdep [expr $depdep+1]]
#  put the new dependency list into the file list
	 set delfile [lreplace $delfile $depfile $depfile $dellist]
#  put the new file list into the main list
	 set list [lreplace $list $dephost $dephost $delfile]

#  the position of the file list containing the file which was placed upon
	 set index1 [expr $whichhost*2+1]
#  the position of the dependency list to be added to
	 set index2 [expr $whichfile*3+2]
#  the file list containing the fil
	 set newfile [lindex $list $index1]
#  the dependency list to be added to
	 set newdep [lindex $newfile $index2]
#  insert the dependency and its remote name into the dependency list
	 set newdep [linsert $newdep [llength $newdep] $deldep $delrem]
#  put the new dependency list into the file list
	 set newfile [lreplace $newfile $index2 $index2 $newdep]
#put the new file list into the main list
	 set list [lreplace $list $index1 $index1 $newfile]

	 $c delete all  ;#delete and redraw the tree
	 TreeDraw $c
      }
      set movex [expr $oldxpos-$xpos] ;#original coordinates
      set movey [expr $oldypos-$ypos]
      $c move $id $movex $movey  ;#move back the items
      $c move $id2 $movex $movey 
      $c dtag $id3 insertthere   ;#untag
   }
   set dr 0
}

