/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.2   JRR     12-Feb-1992	Fixed declaration of static routines.
*	V01.3	JRR	17-Jun-1992	Added header.
*	V01.4	JRR	11-Nov-1992	Added support for C prototypes.
*/
/*++ allodb.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.35.6/lib/RCS/allodb.c,v $
 *
 * DESCRIPTION:
 *
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	August 12, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.5 $ $Date: 1994/03/30 20:32:23 $ $State: Exp $)
 * $Log: allodb.c,v $
 * Revision 1.5  1994/03/30  20:32:23  jrroma
 * Version 3.35.6
 *
 * Revision 1.4  92/12/22  15:45:55  jrroma
 * Version 3.30
 * 
 * Revision 1.3  92/06/18  13:23:30  jrroma
 * Added gnu header
 * 
 * Revision 1.2  92/02/12  14:12:21  jrroma
 * Fixed declaration of static routines.
 * 
 * Revision 1.1  92/02/12  14:11:17  jrroma
 * Initial revision
 * 
 *
 */
#include "nqs.h"
#include <errno.h>



#ifndef __CEXTRACT__
#if __STDC__

static int asizedb ( struct gendescr *descr, int nasize );

#else /* __STDC__ */

static int asizedb (/* struct gendescr *descr, int nasize */);

#endif /* __STDC__ */
#endif /* __CEXTRACT__ */

/*** allodb
 *
 *
 *	long allodb():
 *
 *	Allocate space for, and store a new generic descriptor
 *	structure into, the specified database/configuration file.
 *
 *	The next descriptor returned on a readdb() call will be
 *	the newly created descriptor.
 *
 *	Returns:
 *		The file offset position of the allocated descriptor.
 */
long allodb (
	struct confd *file,		/* Database file */
	struct gendescr *newdescr,	/* New descriptor */
	int descrtype)			/* Descriptor type DSC_ */
{
	register struct gendescr *descr;
	register int need;		/* Total size of new descriptor */
	register int avail;		/* Available free space in entry */
	register long descrpos;
	long origblock;			/* Original block searched */
	int size;			/* Size of new descriptor contents */

	size = sizedb (newdescr, descrtype);	/* Size of contents */
	need = asizedb (newdescr, size);	/* Aligned size */
	/*
	 *  See if there exists adequate space in an existing descriptor.
	 *  Need has the number of free bytes required to hold the new
	 *  descriptor.
	 *
	 *  We first examine the block presently in the cache (provided
	 *  that the file has at least one block in it).  This speeds up
	 *  the allocation process if many sequential allodb() calls are
	 *  being made (as occurs during the NQS rebuild on startup).
	 *
	 *  If no free space can be found in the current cache block,
	 *  then a sequential search is made from the very beginning of
	 *  the file.
	 *
	 *  If no free space is found, then a new block is added onto
	 *  the end of the file.
	 *
	 *  Obviously, this algorithm assumes only one writer....
	 */
	if (file->size) {
		/*
		 *  The file has bytes in it.  Thus, the cache block
		 *  contains a block of the file.  Start searching at
		 *  the beginning of the current block.
		 */
		file->vposition = file->rposition - ATOMICBLKSIZ;
		origblock = file->vposition;		/* Original search blk*/
		do {
			descrpos = file->vposition;	/* Position of descr */
			descr = readdb (file);		/* 1st descr in block */
			if (descr->size < 0) avail = -(descr->size);
			else avail = descr->size
				   - asizedb (descr, sizedb(descr, descrtype));
			/*
			 *  Keep looping while within the current block,
			 *  and no space big enough is found....
			 */
		} while (avail < need && file->vposition < file->rposition);
		if (avail < need) {
			/*
			 *  No space was found in the current block.
			 *  Search from the very beginning of the
			 *  file.
			 */
			if (origblock != 0) {
				/*
				 *  We have not searched the very first						 *  block of the file.
				 */
				seekdbb (file, 0L);
			}
			/*
			 *  Otherwise, we have already searched the
			 *  first block of the file.  File->rposition
			 *  = file->vposition forcing read() in readdb().
			 */
			do {
				if (file->vposition == origblock) {
					/*
					 *  Skip this block, because we
					 *  have already searched it.
					 */
					seekdbb (file, origblock+ATOMICBLKSIZ);
				}
				descrpos = file->vposition;
				descr = readdb (file);	/* Get descriptor */
				if (descr != NULL) {
					if (descr->size < 0) {
						avail = -(descr->size);
					}
					else avail = descr->size
						   - asizedb (descr,
							      sizedb (
								descr,
								descrtype
							      ));
				}
			} while (avail < need && descr != NULL);
		}
	}
	else {
		descr = NULL;		/* No descriptors exist */
		descrpos = 0;		/* File offset 0 */
	}
	if (descr == NULL) {
		/*
		 *  No existing descriptor has the necessary free space.
		 *  We reached the end of the file.  Format the cache
		 *  block as a new block for later writing.
		 *
		 *  File->vposition, file->rposition, and descrpos
		 *  all equal:  file->size.
		 */
		seekdbb (file, file->size);	/* Zero fill cache */
		descr = (struct gendescr *) file->cache->v.chars;
		descr->size = ATOMICBLKSIZ;
		file->size += ATOMICBLKSIZ;	/* Increase file size */
	}
	else {
		/*
		 *  Space was located in an existing descriptor entry to
		 *  hold the new entry.  Please note that this entry is
		 *  completely contained in the cache block for the file.
		 */
		if (descr->size < 0) {
			/*
			 *  The very first entry in the block has enough
			 *  space.
			 */
			descr->size = -(descr->size);
		}
		else {
			descr->size -= avail;
			descrpos += descr->size;
			descr = (struct gendescr *)
				(((char *) descr) + descr->size);
			descr->size = avail;
		}
		/*
		 *  The contents of the read cache block will be changed
		 *  by the allocation of the descriptor space.  Seek back
		 *  to rewrite the block.
		 */
		lseek (file->fd, (file->rposition -= ATOMICBLKSIZ), 0);
	}
	/*
	 *  Copy the descriptor contents into the block.
	 */
	bytecopy ((char *) &descr->v, (char *) &newdescr->v, size);
	/*
	 *  Rewrite the modified block (or add a new formatted block).
	 */
	errno = 0;			/* In case write() works partway */
					/* writing some bytes out, then */
					/* nqs_abort() does not show spurious */
					/* error message */
	if (write (file->fd, file->cache->v.chars,
		   ATOMICBLKSIZ) != ATOMICBLKSIZ) {
		printf ("F$Write error in allodb().\n");
		nqs_abort();
	}
	file->rposition += ATOMICBLKSIZ;	/* Update real position */
	file->vposition = descrpos;		/* Descriptor just created */
	return (descrpos);			/* Return the address of the */
						/* allocated descriptor */
}


/*** asizedb
 *
 *
 *	int asizedb():
 *	Return aligned size for the specified descriptor.
 */
static int asizedb (struct gendescr *descr, int nasize)
{
	register int asize;			/* Aligned size */
	register int residue;			/* Residual */

	asize = nasize + (((char *) &descr->v) - ((char *) &descr->size));
	residue = asize % sizeof (ALIGNTYPE);
	if (residue) asize += sizeof (ALIGNTYPE) - residue;
	return (asize);
}
