/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.20  JRR     16-Jan-1992	Added support for RS6000.
*       V01.30  JRR     17-Jan-1992	Tweaked fix for RS6000.  
*       V01.4   JRR     12-Feb-1992	Fixed declaration of static routines.  
*       V01.5   JRR     28-Feb-1992	Added Cosmic V2 changes.
*	V01.6	JRR	17-Jun-1992	Added header.  
*	V01.7	JRR			Placeholder
*	V01.8	JRR	03-Mar-1994	Fixed up NMAP.
*/
/*++ grpnam.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.35.6/lib/RCS/grpnam.c,v $
 *
 * DESCRIPTION:
 *
 *	This module contains the two functions:  getgrpnam()  and
 *	endgrpnam() which return the group-name for the specified
 *	group-id, and discard the group-name cache respectively.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	August 12, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.8 $ $Date: 1994/03/30 20:32:31 $ $State: Exp $)
 * $Log: grpnam.c,v $
 * Revision 1.8  1994/03/30  20:32:31  jrroma
 * Version 3.35.6
 *
 * Revision 1.7  92/11/06  13:13:44  jrroma
 * Modified for C prototypes.
 * 
 * Revision 1.6  92/06/18  13:23:54  jrroma
 * Added gnu header
 * 
 * Revision 1.5  92/02/28  14:08:30  jrroma
 * Added Cosmic V2 changes.
 * 
 * Revision 1.4  92/02/12  12:38:04  jrroma
 * Fixed declaration of static routine newgroup.
 * 
 * Revision 1.3  92/02/12  12:37:10  jrroma
 * *** empty log message ***
 * 
 * Revision 1.2  92/01/16  15:53:02  jrroma
 * Added support for RS6000.
 * 
 * Revision 1.1  92/01/16  15:52:09  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"
#include <malloc.h>
#include <string.h>
#include "nmap.h"				/* Mid_t (all OS's) */
					/* Uid_t and gid_t (if not BSD43) */

#ifndef __CEXTRACT__
#if __STDC__

static struct g_cache *newgroup ( char *name );

#else /* __STDC__ */

static struct g_cache *newgroup (/* char *name */);

#endif /* __STDC__ */
#endif /* __CEXTRACT__ */

/*
 *	Configurable parameters.
 */
#define	MAX_G_CACHESIZE	50	/* We will remember this many group names */


/*
 *	Data structure definitions local to this module.
 */
struct g_cache {
	struct g_cache *prev;	/* Previous group name cache entry */
	struct g_cache *next;	/* Next group name cache entry */
	gid_t gid;		/* Group-id */
	char *name;		/* Group-name */
};


/*
 *	Variables which are global to this module but not public.
 */
static struct g_cache *g_set = (struct g_cache *) 0;
					/* Group-id/name cache */
static int g_count = 0;			/* # of group-id/name cache entries */


/*** getgrpnam
 *
 *
 *	char *getgrpnam():
 *
 *	Return the groupname for the specified group-id on the local
 *	machine.
 */
char *getgrpnam (gid)
gid_t gid;				/* Group-id */
{

	register struct g_cache *scan;	/* Current group cache entry */
	register struct g_cache *prev;	/* Previous group cache entry */
	register char *groupname;	/* Ptr to groupname for local gid */

	prev = (struct g_cache *) 0;
	scan = g_set;			/* Scan group cache */
	while (scan != (struct g_cache *) 0 && scan->gid != gid) {
		prev = scan;
		scan = scan->next;
	}
	if (scan == (struct g_cache *) 0) {
		/*
		 *  The group-id/name was not in the cache.
		 */
		groupname = fmtgidname (gid);	/* Format groupname for gid */
		if (g_count < MAX_G_CACHESIZE) scan = newgroup (groupname);
		while (scan == (struct g_cache *) 0 &&
		       prev != (struct g_cache *) 0) {
			/*
			 *  Discard the least recently used mapping and
			 *  try to add the new mapping to the head of
			 *  the mapping set.
			 */
			free (prev->name);	/* Dispose of LRU name part */
			scan = prev;
			prev = prev->prev;	/* Backup one entry */
			free ((char *) scan);	/* Dispose of LRU entry */
			g_count--;		/* One less entry */
			if (prev != (struct g_cache *) 0) {	/* Disconnect */
				prev->next = (struct g_cache *) 0;
			}
			else {			/* No more entries left */
				g_set = (struct g_cache *) 0;
			}
			scan = newgroup (groupname);
						/* Try to allocate new entry */
		}
		if (scan == (struct g_cache *) 0) {
			/*
			 *  Insufficient memory existed to add the mapping
			 *  cache entry.  g_set points to nothing.
			 */
			return (groupname);
		}
		/*
		 *  Add the new mapping to the head of the mapping cache.
		 */
		if (g_set != (struct g_cache *) 0) g_set->prev = scan;
		scan->prev = (struct g_cache *) 0;
		scan->next = g_set;
		g_set = scan;
		scan->gid = gid;			/* Save group-id */
		strcpy (scan->name, groupname);		/* Save group-name */
	}
	else {
		/*
		 *  The group-id/name pair has been found in the cache.
		 *  Move the entry to the front of the cache to keep track
		 *  of the least-recently used order of the cache entries.
		 */
		if (scan != g_set) {	/* If not already as the front.... */
			if (prev != (struct g_cache *) 0) {
				prev->next = scan->next;
			}
			if (scan->next != (struct g_cache *) 0) {
				scan->next->prev = prev;
			}
			scan->prev = (struct g_cache *) 0;
			scan->next = g_set;
			g_set = scan;
		}
	}
	return (scan->name);	/* Return ptr to groupname */
}


/*** endgrpnam
 *
 *
 *	void endgrpnam():
 *	Clear the group-id/name cache.
 */
void endgrpnam ()
{
	register struct g_cache *walk;
	register struct g_cache *next;

	walk = g_set;
	while (walk != (struct g_cache *) 0) {
		next = walk->next;
		free (walk->name);
		free ((char *) walk);
		walk = next;
	}
	g_count = 0;			/* Zero group-id/name cache entries */
	g_set = (struct g_cache *) 0;
	closegrdb();			/* Close the group database */
}


/*** newgroup
 *
 *
 *	struct g_cache *newgroup():
 *	Returns:
 *		A pointer to a new group-id/name cache entry if
 *		adequate heap space exists; otherwise a null ptr
 *		is returned.
 */
static struct g_cache *newgroup (name)
char *name;					/* Groupname */
{
	register struct g_cache *new;

	if ((new = (struct g_cache *)
		   malloc (sizeof (struct g_cache))) != (struct g_cache *) 0) {
		/*
		 *  We successfully got a new cache entry.
		 *  Now try to allocate sufficient name space.
		 */
		if ((new->name = malloc (strlen (name) + 1)) == (char *) 0) {
			/*
			 *  Insufficient heap space for name.
			 */
			free ((char *) new);
			new = (struct g_cache *) 0;
		}
		else g_count++;	/* One more entry */
	}
	return (new);
}
