/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*++ readhdr.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.30/lib/RCS/readhdr.c,v $
 *
 * DESCRIPTION:
 *
 *	Read and verify the common header portion of a request
 *	control file.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	August 12, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.3 $ $Date: 1992/12/22 15:46:23 $ $State: Exp $)
 * $Log: readhdr.c,v $
 * Revision 1.3  1992/12/22  15:46:23  jrroma
 * Version 3.30
 *
 * Revision 1.2  92/06/18  13:24:32  jrroma
 * Added gnu header
 * 
 * Revision 1.1  92/06/18  12:45:56  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"
#include <errno.h>
#include <unistd.h>

/*** readhdr
 *
 *
 *	int readhdr():
 *
 *	Read and verify the common header portion of a request
 *	control file.
 *
 *	If the return value of this function is 0, then the lseek()
 *	position of the control file descriptor upon return, is
 *	ALWAYS left at the first byte of the varying portion of the
 *	control file.
 *
 *
 *	Returns:
 *		0: if successful, and the request header is valid;
 *	       -1: otherwise.
 *
 *			Note:  errno will be returned 0 by this function
 *			       unless a system call error has occurred.
 */
int readhdr (fd, rawreq)
int fd;					/* Control file descriptor */
struct rawreq *rawreq;			/* Raw request info to be returned */
{
	register int size;		/* Size of common request header */

	errno = 0;			/* Clear errno */
	if (lseek (fd, 0L, 0) == -1) {	/* Seek to the beginning of the file */
		return (-1);		/* Report EBADF */
	}
	/*
	 *  We read the common header of the control file in a single
	 *  read() system call.
	 *
	 *  We do this for reasons of efficiency, and also so that the
	 *  first block of the control file is read in one atomic I/O
	 *  operation so that concurrent Qmod updates (being done by the
	 *  careful indivisible rewrite of the first control file block),
	 *  and multiple read accesses do not step on each other.
	 */
	size = ((char *) &rawreq->v) - ((char *) rawreq);
	if (read (fd, (char *) rawreq, size) != size) {
		/*
		 *  Something bad happened trying to read the control
		 *  file header.
		 */
		return (-1);
	}
	if (verifyhdr (rawreq) < 0) return (-1);
	/*
	 *  Leave the file pointer pointing at the varying part
	 *  of the request control file.
	 */
	if (rawreq->type == RTYPE_DEVICE) {
		/*
		 *  We are [re]writing the control file common
		 *  header of a device-oriented request.
		 */
		size = sizeof (struct rawdevreq);
	}
	else {
		/*
		 *  We are [re]writing the control file common
		 *  header of a non device-oriented request.
		 */
		size = sizeof (struct rawbatreq);
	}
	lseek (fd, (long) size, 1);	/* Relative seek */
	errno = 0;			/* Extreme paranoia */
	return (0);
}
