/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.20  JRR     05-Dec-1991	Added support for remote deletes.
*       V01.30  JRR     16-Jan-1992	Added support for RS6000.
*	V01.4	JRR	17-Jun-1992	Added header.
*	V01.5	JRR	07-Dec-1992	Modified for C prototypes.
*	V01.6	JRR	02-Mar-1994	Fixed up NETDB.
*	V01.7	JRR	18-Aug-1994	No machine spec before the target 
*					spec should use local machine. 
*/
/*++ reqspec.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.36/lib/RCS/reqspec.c,v $
 *
 * DESCRIPTION:
 *
 *	Return the sequence number, and explicit or implied machine
 *	specification as given in the request-id text parameter.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	October 11, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.7 $ $Date: 1994/09/02 17:37:57 $ $State: Exp $)
 * $Log: reqspec.c,v $
 * Revision 1.7  1994/09/02  17:37:57  jrroma
 * Version 3.36
 *
 * Revision 1.6  1994/03/30  20:32:41  jrroma
 * Version 3.35.6
 *
 * Revision 1.5  92/12/22  15:46:24  jrroma
 * Version 3.30
 * 
 * Revision 1.4  92/06/18  13:24:33  jrroma
 * Added gnu header
 * 
 * Revision 1.3  92/01/16  17:10:23  jrroma
 * Added support for RS6000.
 * 
 * Revision 1.2  91/12/05  16:48:59  jrroma
 * Completed fixes for remote deletes.
 * 
 * Revision 1.1  91/11/19  11:33:42  jrroma
 * Initial revision
 * 
 *
 */

#include <string.h>
#include <netdb.h>			/* Network database header file; */
#include "nmap.h"			/* Network mapping codes */


#ifndef __CEXTRACT__
#if __STDC__

static char *reqspec_machine ( char *text, Mid_t *machine_id, int *status );

#else /* __STDC__ */

static char *reqspec_machine (/* char *text, Mid_t *machine_id, int *status */);

#endif /* __STDC__ */
#endif /* __CEXTRACT__ */

/*** reqspec
 *
 *
 *	int reqspec():
 *
 *	Return the sequence number, and explicit or implied machine
 *	specifications as given in the request-id text parameter.
 *
 *	A request-id is always of the form:
 *
 *		<seqno>		OR
 *		<seqno>.<machine-specification> OR
 *		<seqno>.<machine-specification>@<remote-machine-specification>
 *
 *	If no <machine-specification> is given, then the request is
 *	assumed to have been submitted from the local machine.
 *
 *	If no <remote-machine.specification> is given, then the request
 *	is assumed to exist on the local machine.
 *
 *	The form of an explicit machine specification (when present
 *	in a request-id specification), is:
 *
 *		.machine-name	OR
 *		.[integer#]
 *
 *	where the "machine-name" form represents the principal name
 *	or alias for the machine from whence the request was submitted,
 *	and the "[integer#]" form is supported so that explicit
 *	machine-ids can be specified in place of a name or alias.
 *
 *	Returns:
 *		0: if successful, in which case the orig_seqno,
 *		   and machine-id parameters are properly updated.
 *
 *	       -1: if an invalid request-id syntax is encountered.
 *
 *	       -2: if the explicit or implied machine-specification
 *		   is not recognized by the local system (NMAP_ENOMAP).
 *
 *	       -3: if the Network Mapping Procedures (NMAP_)
 *		   deny access to the caller (NMAP_ENOPRIV).
 *
 *	       -4: if some other general NMAP_ error occurs.
 */
int reqspec (text, orig_seqno, machine_id, target_mid)
register char *text;			/* Literal text */
long *orig_seqno;			/* Req sequence number */
Mid_t *machine_id;			/* Pointer of where to return */
					/* the proper machine-id */
Mid_t *target_mid;			/* Pointer of where to return */
					/* the target machine-id */
{
	register long seqno;		/* Sequence# of request */
	int	status;			/* status integer */

	seqno = 0;
	while (*text >= '0' && *text <= '9') {
		seqno *= 10;
		seqno += *text - '0';
		text++;			/* Advance to the next character. */
	}
	if (*text == '\0') {
		/*
		 *  No explicit machine-specification is present.
		 *  We assume the local machine in this case.
		 */
		*orig_seqno = seqno;
		if ( status = localmid (machine_id) ) return ( status );
		return (localmid (target_mid));
	}
	else if ((*text != '.') && (*text != '@') ) {
		/*
		 *  Invalid request-id syntax.
		 */
		return (-1);		/* Report bad request-id syntax */
	}
	if (*text == '.') {
	    text++;				/* Scan past '.' */
	    /*
 	     *  A machine-specification is present.
	     */
	    text = reqspec_machine(text, machine_id, &status);
	    if (status) {
	      return (status);
            }
        } else {
            /* 
	     * No machine spec before the target spec should use local mach 
             */
            if ( status = localmid (machine_id) ) return ( status );
        }
	if (*text == '\0') {		/* if we have no target_mid ... */
		*orig_seqno = seqno;
		return (localmid (target_mid));
	}
	if ( *text != '@') {
		/*
		 *  Invalid request-id syntax.
		 */
		return (-1);
	}
	text++;
	text = reqspec_machine(text, target_mid, &status);
	if (status) return (status);
	*orig_seqno = seqno;	/* Return request sequence# */
	return (0);
}
	
/*
 *  This routine takes a pointer to a machine specification string and
 *  returns the machine_id and the status.  If successful, status is
 *  0 and it returns a pointer the character after the specification string.
 *  On failure, status is negative and it returns 0.
 */
static char *reqspec_machine(text, machine_id, status)
char	*text;				/* buffer holding machine spec */
Mid_t	*machine_id;			/* return the machine id here */
int	*status;			/* return the status here */
{
	register Mid_t mid_spec;	/* Machine-id specification */
	struct hostent *ent;		/* Host table entry structure */
	char   buffer[64];		/* temporary buffer */
	char   *cp;

	if (*text == '\0') {
		/*
		 *  Missing machine-specification.
		 */
		*status = -1;
		return (0);		/* Bad request-id syntax */
	}
	/*
	 *  Determine the machine-id of the machine specification
	 *  as present in the request-id text.
	 */
	if (*text == '[') {
		/*
		 *  Could be an explicit machine specification by machine-id.
		 */
		text++;
		mid_spec = 0;
		while (*text >= '0' && *text <= '9') {
			mid_spec *= 10;
			mid_spec += *text - '0';
			*text++;
		}
		if (*text == ']') {
			/*
			 *	[ <digit-sequence> ]
			 */
			if (*++text) {		/* Invalid machine-id spec */
				*status = -1;
				return (0);	/* Bad syntax */
			}
			*machine_id = mid_spec;
			*status = 0;
			return (text);		/* Success */
		}
		*status = -1;
		return (0);			/* Invalid syntax */
	}
	/*
	 *  The machine-specification portion of the request-id has been
	 *  specified as a name.
	 *
	 *  First remove any extra stuff at the end, if present.
	 */
	strcpy( buffer, text);
	if (  cp = strchr(buffer, '@')  ) {
		*cp = '\0';
		text = strchr(text, '@');
	} else {
	        text = strchr(text, '\0');
	}
	if ((ent = gethostbyname (buffer)) == (struct hostent *) 0) {
		*status = -2;
		return (0);			/* This machine is not */
						/* known to us */
	}
	switch (nmap_get_mid (ent, machine_id)) {
	case NMAP_SUCCESS:		/* Successfully got local machine-id */
		*status = 0;
		return (text);
	case NMAP_ENOMAP:		/* What?  No local mid defined! */
		*status = -2;
		return (0);
	case NMAP_ENOPRIV:		/* No privilege */
		*status = -3;
		return (0);
	}
	*status = -4;
	return (0);			/* General NMAP_ error */
}
