/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.20  JRR     16-Jan-1992	Added support for RS6000.
*       V01.3   JRR     12-Feb-1992	Fixed declaration of static routines.
*	V01.4	JRR	17-Jun-1992	Added header.
*	V01.5	JRR	20-Nov-1992	Modified for C prototypes.
*	V01.6	JRR	22-Oct-1993	validmgr is static.	
*/
/*++ nqs_upm.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.35.3/src/RCS/nqs_upm.c,v $
 *
 * DESCRIPTION:
 *
 *	NQS manager set update module.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	June 16, 1986.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.6 $ $Date: 1994/02/24 21:30:57 $ $State: Exp $)
 * $Log: nqs_upm.c,v $
 * Revision 1.6  1994/02/24  21:30:57  jrroma
 * Version 3.35.3
 *
 * Revision 1.5  92/12/22  15:41:36  jrroma
 * Version 3.30
 * 
 * Revision 1.4  92/06/18  17:31:28  jrroma
 * Added gnu header
 * 
 * Revision 1.3  92/02/12  14:00:57  jrroma
 * Fixed declaration of static routines.
 * 
 * Revision 1.2  92/01/17  11:17:51  jrroma
 * Added support for RS6000.
 * 
 * Revision 1.1  92/01/17  11:17:10  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"
#include "nqsxvars.h"
#include "transactcc.h"		/* Transaction completion codes */

#ifndef __CEXTRACT__
#if __STDC__

static int validmgr ( uid_t account_uid, Mid_t account_mid );

#else /* __STDC__ */

static int validmgr (/* uid_t account_uid, Mid_t account_mid */);

#endif /* __STDC__ */
#endif /* __CEXTRACT__ */

/*** upm_addnqsman
 *
 *
 *	long upm_addnqsman():
 *	Add an NQS manager account to the NQS manager set.
 *
 *	Returns:
 *		TCML_COMPLETE:	if successful;
 *		TCML_ALREADEXI:	if the specified account is already
 *				present in the NQS manager set.
 */
long upm_addnqsman (account_uid, account_mid, privilege_bits)
uid_t account_uid;			/* NQS manager account user-id */
Mid_t account_mid;			/* NQS manager account machine-id */
int privilege_bits;			/* Privilege bits */
{
	if (validmgr (account_uid, account_mid)) {
		/*
		 *  The specified account to add to the NQS manager
		 *  set is already present in the set.
		 */
		return (TCML_ALREADEXI);
	}
	udb_addnqsman (account_uid, account_mid, privilege_bits);
	return (TCML_COMPLETE);		/* Add the account to the */
					/* NQS manager set */
}


/*** upm_delnqsman
 *
 *
 *	long upm_delnqsman():
 *	Delete an NQS manager account from the NQS manager set.
 *
 *	Returns:
 *		TCML_COMPLETE:	if successful;
 *		TCML_NOSUCHMAN:	if the specified manager account was not
 *				present at the correct privilege level
 *				in the NQS manager set.
 *		TCML_ROOTINDEL:	if an attempt is made to delete "root"
 *				from the NQS manager set.
 */
long upm_delnqsman (account_uid, account_mid, privilege_bits)
uid_t account_uid;			/* NQS manager account user-id */
Mid_t account_mid;			/* NQS manager account machine-id */
int privilege_bits;			/* Privilege bits */
{
	if (account_uid == 0 && account_mid == Locmid) {
		/*
		 *  The "root" account can never be deleted from the NQS
		 *  manager set.
		 */
		return (TCML_ROOTINDEL);
	}
	if (validmgr (account_uid, account_mid) == privilege_bits) {
		/*
		 *  The specified account to delete from the NQS manager
		 *  set is present at the specified privilege level.
		 */
		udb_delnqsman (account_uid, account_mid);
		return (TCML_COMPLETE);	/* Delete the account from the */
					/* NQS manager set */
	}
	return (TCML_NOSUCHMAN);	/* No such manager account */
}


/*** validmgr
 *
 *
 *	validmgr():
 *
 *	Return the (non-zero) privilege bits if the specified
 *	manager account is defined in the NQS manager set.
 *	Otherwise return 0.
 */
static
int validmgr (account_uid, account_mid)
uid_t account_uid;			/* Manager account user-id */
Mid_t account_mid;			/* Manager account machine-id */
{
	register struct gendescr *descr;

	seekdbb (Mgrfile, 0L);		/* Seek to the beginning of the NQS */
					/* manager list file */
	/*
	 *  Search for the entry to be added, to see if it is already
	 *  present.
	 */
	descr = nextdb (Mgrfile);
	while (descr != (struct gendescr *) 0) {
		if (descr->v.mgr.manager_uid == account_uid &&
		    descr->v.mgr.manager_mid == account_mid) {
			/*
			 *  He or she is a manager.
			 */
			return (descr->v.mgr.privileges);
		}
		else descr = nextdb (Mgrfile);
	}
	return (0);			/* He or she is not a manager. */
}
