/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.20  JRR     16-Jan-1992	Added support for RS6000.
*       V01.3   JRR     28-Feb-1992	Added Cosmic V2 changes.
*	V01.4	JRR	17-Jun-1992	Added header.
*	V01.5	JRR	06-Nov-1992	Added support for HPUX.
*	V01.6	JRR	06-Apr-1993	Added support for DECOSF
*					(by removing conditionals).
*	V01.7	JRR	11-Apr-1994	Ranking compute servers.
*/
/*++ opendb.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.36/lib/RCS/opendb.c,v $
 *
 * DESCRIPTION:
 *
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	August 12, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.7 $ $Date: 1994/09/02 17:37:56 $ $State: Exp $)
 * $Log: opendb.c,v $
 * Revision 1.7  1994/09/02  17:37:56  jrroma
 * Version 3.36
 *
 * Revision 1.6  93/07/13  21:31:31  jrroma
 * Version 3.34
 * 
 * Revision 1.5  92/11/06  13:48:45  jrroma
 * Added support for HPUX.
 * 
 * Revision 1.4  92/06/18  13:24:20  jrroma
 * Added gnu header
 * 
 * Revision 1.3  92/02/28  15:18:30  jrroma
 * Added Cosmic V2 changes.
 * 
 * Revision 1.2  92/01/16  16:59:43  jrroma
 * Added support for RS6000.
 * 
 * Revision 1.1  92/01/16  16:58:56  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"
#include <errno.h>
#include <sys/stat.h>
#include <malloc.h>
#define	MAX_CONFIGFILES	10		/* Maximum number of open configur- */
					/* ation files */


/*** opendb
 *
 *
 *	struct confd *opendb():
 *
 *	Open the named file of the NQS database.
 *	Returns:
 *		A pointer to the allocated NQS database file access
 *		structure if successful;
 *		Otherwise NULL is returned (errno has reason).
 */
struct confd *opendb (name, flags)
register char *name;			/* Name of file */
register int flags;			/* file access flags */
{
	static short initialized = 0;	/* Tables not initialized */
	static struct confd confiles [MAX_CONFIGFILES];

	struct stat statbuf;		/* Fstat() buffer */
	register struct confd *file;	/* NQS database file descriptor */
	register int descr;

	if (!initialized) {
		/*
		 *  We have not yet initialized the configuration
		 *  file descriptor table.
		 */
		initialized = 1;	/* Set initialized flag */
		for (descr = 0; descr < MAX_CONFIGFILES; descr++) {
			confiles [descr].fd = -1;	/* Mark slot as free */
		}
	}
	descr = 0;		/* Search for a free confd descriptor */
	while (descr < MAX_CONFIGFILES && confiles [descr].fd != -1) {
		descr++;
	}
	if (descr >= MAX_CONFIGFILES) {
		/*
		 *  There is not a free confd file descriptor
		 *  to be found.
		 */
		printf ("E$Too many files in opendb().\n");
		fflush (stdout);
		errno = 0;		/* Not a system call error */
		return (NULL);
	}
	file = &confiles [descr];	/* Get address of free descriptor */
	if (flags & O_CREAT) file->fd = open (name, flags & ~O_EXCL, 0644);
	else file->fd = open (name, flags);
	if (file->fd == -1) return (NULL);	/* The open failed */
	/*
	 *  The open succeeded.  Initialize confd file descriptor.
	 */
	if ((file->cache = (struct rawblock *)
			    malloc (sizeof (struct rawblock))) == NULL) {
		/*
		 *  Insufficient heap space to open database file.
		 */
		printf ("E$No heap space in opendb().\n");
		fflush (stdout);
		close (file->fd);	/* Close the file */
		file->fd = -1;		/* Free the descriptor */
		return (NULL);
	}
	file->vposition = 0;		/* Virtual file position */
	file->rposition = 0;		/* Real file position */
	file->lastread = -1;		/* No descriptor has even been read */
	fstat (file->fd, &statbuf);	/* Fstat() file */
	file->size = statbuf.st_size;	/* File size */
	if (file->size % ATOMICBLKSIZ) {
		/*
		 *  The file size is not a multiple of ATOMICBLKSIZ bytes.
		 */
		printf ("E$Invalid file size in opendb().\n");
		fflush (stdout);
		close (file->fd);	/* Close the file */
		file->fd = -1;		/* Free the descriptor */
		return (NULL);
	}
	/*
	 *  Nature abhors a vacuum.  Load the first block of the file
	 *  into the cache (if the file has 1 or more blocks).
	 */
	if (file->size) {
		errno = 0;		/* No system call error on partial */
					/* reads please */
		if (read (file->fd, file->cache->v.chars,
			  ATOMICBLKSIZ) != ATOMICBLKSIZ) {
			printf ("E$Read error in opendb().\n");
			fflush (stdout);
			close (file->fd);	/* Close the file */
			file->fd = -1;		/* Free the descriptor */
			return (NULL);
		}
		file->rposition += ATOMICBLKSIZ;
	}
	return (file);
}
