/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*++ getsockch.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.35/lib/RCS/getsockch.c,v $
 *
 * DESCRIPTION:
 *
 *	Routines for reading from a socket.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	April 21, 1986.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.3 $ $Date: 1993/09/10 13:55:11 $ $State: Exp $)
 * $Log: getsockch.c,v $
 * Revision 1.3  1993/09/10  13:55:11  jrroma
 * Version 3.35
 *
 * Revision 1.2  92/06/18  13:23:53  jrroma
 * Added gnu header
 * 
 * Revision 1.1  92/06/18  10:18:43  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"			/* NQS types and definitions */
					/* + sys/types.h + nmap.h + quolim.h */
#include <errno.h>

/*
 *	External variables.
 */
extern int errno;			/* System call error codes */

/*
 *	Internal variables.
 */
static int sockfd;			/* Socket file descriptor */
static char buffer [MAX_PACKET];	/* Receive buffer */
static int bufptr = 0;			/* Index of next char in buffer */
static int n_read = 0;			/* Index of 1st empty slot in buffer */


/*** bufsockch
 *
 *
 *	char *bufsockch():
 *
 *	Return a pointer to the characters that
 *	1) this module has read from the socket, and
 *	2) have not yet been turned over with getsockch().
 *
 */
char *bufsockch ()
{
	return (&buffer[bufptr]);
}


/*** extrasockch
 *
 *
 *	int extrasockch():
 *
 *	Report the number of characters that
 *	1) this module has read from the socket, and
 *	2) have not yet been turned over with getsockch().
 */
int extrasockch ()
{
	return (n_read - bufptr);
}


/*** getsockch
 *
 *
 *	int getsockch():
 *
 *	Return a character from a socket.
 */
int getsockch ()
{
	register int no_packet_yet;

	if (bufptr < n_read) {
		return (buffer [bufptr++] & 0377);
	}
	bufptr = 1;			/* Return buffer [1] NEXT time */
	no_packet_yet = 1;
	do {
		if ((n_read = read (sockfd, buffer, MAX_PACKET)) == -1) {
			if (errno != EINTR) {
				return (-1);
			}
		}
		else no_packet_yet = 0;
	} while (no_packet_yet);
	if (n_read == 0) {
		return (-1);
	}
	return (buffer [0] & 0377);
}


/*** setsockfd
 *
 *
 *	void setsockfd():
 *
 *	Notify this module which socket it is that it should read on.
 */
void setsockfd (int fd)
{
	sockfd = fd;
}
