/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.2   JRR     12-Feb-1992	Fixed static declaration of routines.
*       V01.3   JRR     14-Feb-1992	Made boundpreamble into tcm_bound....
*	V01.4	JRR	17-Jun-1992	Added header.
*	V01.5	JRR	20-Nov-1992	Modified for C prototypes.
*/
/*++ analyzetcm.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.30/lib/RCS/analyzetcm.c,v $
 *
 * DESCRIPTION:
 *
 *
 *	Completely analyze and diagnose a transaction completion
 *	code.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	April 28, 1986.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.5 $ $Date: 1992/12/22 15:45:56 $ $State: Exp $)
 * $Log: analyzetcm.c,v $
 * Revision 1.5  1992/12/22  15:45:56  jrroma
 * Version 3.30
 *
 * Revision 1.4  92/06/18  13:23:32  jrroma
 * Added gnu header
 * 
 * Revision 1.3  92/02/14  10:22:26  jrroma
 * Finally fix up static routines.
 * 
 * Revision 1.2  92/02/12  12:44:59  jrroma
 * Fixed static declaration of routines.
 * 
 * Revision 1.1  92/02/12  12:43:52  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"		/* Everyone gets this */
#include "informcc.h"		/* NQS information completion codes */
#include "transactcc.h"		/* NQS transaction completion codes */


#ifndef __CEXTRACT__
#if __STDC__

static void exceedpreamble ( FILE *stream, char *prefix );
static void tcm_boundpreamble ( FILE *stream, char *prefix );

#else /* __STDC__ */

static void exceedpreamble (/* FILE *stream, char *prefix */);
static void tcm_boundpreamble (/* FILE *stream, char *prefix */);

#endif /* __STDC__ */
#endif /* __CEXTRACT__ */

/*** analyzetcm
 *
 *
 *	void analyzetcm():
 *
 *	Completely analyze and diagnose a transaction completion
 *	code.
 */
void analyzetcm (
	long code,		/* Completion code */
	FILE *stream,		/* Output stream */
	char *prefix)		/* Output prefix */
{

	register short reason;		/* Reason bits of completion code */

	reason = (code & XCI_REASON_MASK);
	if (reason < 0 || reason > (TCM_MAXTCM & XCI_REASON_MASK) ||
	   (code & XCI_TRANSA_MASK) == 0) {
		/*
		 *  The transaction code is invalid.
		 */
		reason = (TCML_UNDEFINED & XCI_REASON_MASK);
		code = TCML_UNDEFINED;
	}
	else if (reason == (TCML_UNDEFINED & XCI_REASON_MASK)) {
		/*
		 *  Clear information bits if undefined transaction code.
		 */
		code &= ~XCI_INFORM_MASK;
	}
	/*
	 *  Analyze and describe a specific transaction code.
	 */
	tcmmsgs (code, stream, prefix);
	/*
	 *  Analyze and describe any additional information bits
	 *  set in the transaction completion code.
	 */
	if (reason == (TCML_QUOTALIMIT & XCI_REASON_MASK)) {
		tcimsgs (code, stream, prefix, exceedpreamble);
	}
	else if (reason == (TCML_SUBMITTED & XCI_REASON_MASK)) {
		tcimsgs (code, stream, prefix, tcm_boundpreamble);
	}
}


/*** tcm_boundpreamble
 *
 *
 *	void tcm_boundpreamble():
 *	Display bounded limit preamble.
 */
static void tcm_boundpreamble (FILE *stream, char *prefix)
{
	fputc ('\n', stream);
	fputs (prefix, stream);
	fputs ("In order to successfully queue or deliver the request,\n",
		stream);
	fputs (prefix, stream);
	fputs ("it was necessary to alter the following explicit request\n",
		stream);
	fputs (prefix, stream);
	fputs ("quota limit(s), binding them to values within the ranges\n",
		stream);
	fputs (prefix, stream);
	fputs ("enforceable at the execution machine:\n\n", stream);
	fflush (stream);		/* For output to logfile */
}


/*** exceedpreamble
 *
 *
 *	void exceedpreamble():
 *	Display limit too large to queue preamble.
 */
static void exceedpreamble (FILE *stream, char *prefix)
{
	fputc ('\n', stream);
	fputs (prefix, stream);
	fputs ("The request could not be queued or delivered because the\n",
		stream);
	fputs (prefix, stream);
	fputs ("following explicit request quota limit(s) exceeded the\n",
		stream);
	fputs (prefix, stream);
	fputs ("corresponding limits of the target queue:\n\n", stream);
	fflush (stream);		/* For output to logfile */
}
