/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.20  JRR     16-Jan-1992	Added support for RS6000.
*       V01.30  JRR     17-Jan-1992	Tweaked ubove fix.        
*       V01.4   JRR     12-Feb-1992	Fixed static declaration of newuser.
*       V01.5   JRR     28-Feb-1992	Added Cosmic V2 changes.
*	V01.6	JRR	17-Jun-1992	Added header.
*	V01.7	JRR	06-Nov-1992	Modified for C prototypes.
*	V01.8	JRR	03-Mar-1994	Fixed up NMAP.
*/
/*++ usenam.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.35.6/lib/RCS/usenam.c,v $
 *
 * DESCRIPTION:
 *
 *	This module contains the two functions:  getusenam()  and
 *	endusenam() which return the user-name account for the specified
 *	user-id, and discard the user-name cache respectively.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	August 12, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.8 $ $Date: 1994/03/30 20:32:48 $ $State: Exp $)
 * $Log: usenam.c,v $
 * Revision 1.8  1994/03/30  20:32:48  jrroma
 * Version 3.35.6
 *
 * Revision 1.7  92/11/06  15:39:05  jrroma
 * Modified for C prototypes.
 * 
 * Revision 1.6  92/06/18  13:24:59  jrroma
 * Added gnu header
 * 
 * Revision 1.5  92/02/28  15:44:28  jrroma
 * Added Cosmic V2 changes.
 * 
 * Revision 1.4  92/02/12  12:40:44  jrroma
 * Fixed static declaration of newuser.
 * 
 * Revision 1.3  92/01/17  16:40:37  jrroma
 * Tweak to fix for RS6000.
 * 
 * Revision 1.2  92/01/16  17:23:49  jrroma
 * Added support for RS6000.
 * 
 * Revision 1.1  92/01/16  17:23:04  jrroma
 * Initial revision
 * 
 *
 */


#include "nqs.h"
#include <malloc.h>
#include <string.h>
#include "nmap.h"			/* Mid_t (all OS's) */
					/* Uid_t and gid_t (if not BSD43) */

static struct u_cache *newuser ( char *name );

/*
 *	Configurable parameters.
 */
#define	MAX_U_CACHESIZE	400	/* We will remember this many user names */


/*
 *	Data structure definitions local to this module.
 */
struct u_cache {
	struct u_cache *prev;	/* Previous user name cache entry */
	struct u_cache *next;	/* Next user name cache entry */
	uid_t uid;		/* User-id */
	char *name;		/* User-name */
};


/*
 *	Variables which are global to this module but not public.
 */
static struct u_cache *u_set = (struct u_cache *) 0;
					/* User-id/name cache */
static int u_count = 0;			/* # of user-id/name cache entries */


/*** getusenam
 *
 *
 *	char *getusenam():
 *
 *	Return the username for the specified user-id on the local
 *	machine.
 */
char *getusenam (uid_t uid)
{
	register struct u_cache *scan;	/* Current user cache entry */
	register struct u_cache *prev;	/* Previous user cache entry */
	register char *name;		/* Name for local user-id */

	prev = (struct u_cache *) 0;
	scan = u_set;			/* Scan user cache */
	while (scan != (struct u_cache *) 0 && scan->uid != uid) {
		prev = scan;
		scan = scan->next;
	}
	if (scan == (struct u_cache *) 0) {
		/*
		 *  The user-id/name was not in the cache.
		 */
		name = fmtuidname (uid);
		if (u_count < MAX_U_CACHESIZE) scan = newuser (name);
		while (scan == (struct u_cache *) 0 &&
		       prev != (struct u_cache *) 0) {
			/*
			 *  Discard the least recently used mapping and
			 *  try to add the new mapping to the head of
			 *  the mapping set.
			 */
			free (prev->name);	/* Dispose of LRU name part */
			scan = prev;
			prev = prev->prev;	/* Backup one entry */
			free ((char *) scan);	/* Dispose of LRU entry */
			u_count--;		/* One less entry */
			if (prev != (struct u_cache *) 0) {	/* Disconnect */
				prev->next = (struct u_cache *) 0;
			}
			else {			/* No more entries left */
				u_set = (struct u_cache *) 0;
			}
			scan = newuser (name);
						/* Try to allocate new entry */
		}
		if (scan == (struct u_cache *) 0) {
			/*
			 *  Insufficient memory existed to add the mapping
			 *  cache entry.  u_set points to nothing.
			 */
			return (name);
		}
		/*
		 *  Add the new mapping to the head of the mapping cache.
		 */
		if (u_set != (struct u_cache *) 0) u_set->prev = scan;
		scan->prev = (struct u_cache *) 0;
		scan->next = u_set;
		u_set = scan;
		scan->uid = uid;		/* Save user-id */
		strcpy (scan->name, name);	/* Save user-name */
	}
	else {
		/*
		 *  The user-id/name pair has been found in the cache.
		 *  Move the entry to the front of the cache to keep track
		 *  of the least-recently used order of the cache entries.
		 */
		if (scan != u_set) {	/* If not already as the front.... */
			if (prev != (struct u_cache *) 0) {
				prev->next = scan->next;
			}
			if (scan->next != (struct u_cache *) 0) {
				scan->next->prev = prev;
			}
			scan->prev = (struct u_cache *) 0;
			scan->next = u_set;
			u_set = scan;
		}
	}
	return (scan->name);	/* Return ptr to username */
}


/*** endusenam
 *
 *
 *	void endusenam():
 *	Clear the user-id/name cache.
 */
void endusenam ()
{
	register struct u_cache *walk;
	register struct u_cache *next;

	walk = u_set;
	while (walk != (struct u_cache *) 0) {
		next = walk->next;
		free (walk->name);
		free ((char *) walk);
		walk = next;
	}
	u_count = 0;		/* Zero user-id/name cache entries */
	u_set = (struct u_cache *) 0;
	closepwdb();		/* Close the account/password database */
}


/*** newuser
 *
 *
 *	struct u_cache *newuser():
 *	Returns:
 *		A pointer to a new user-id/name cache entry if
 *		adequate heap space exists; otherwise a null ptr
 *		is returned.
 */
static struct u_cache *newuser (char *name)
{
	register struct u_cache *new;

	if ((new = (struct u_cache *)
		   malloc (sizeof (struct u_cache))) != (struct u_cache *) 0) {
		/*
		 *  We successfully got a new cache entry.
		 *  Now try to allocate sufficient name space.
		 */
		if ((new->name = (char *)malloc (strlen (name) + 1)) == (char *) 0) {
			/*
			 *  Insufficient heap space for name.
			 */
			free ((char *) new);
			new = (struct u_cache *) 0;
		}
		else u_count++;	/* One more entry */
	}
	return (new);
}
