/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.2   JRR     12-Feb-1992	Fixed declaration of static routines.
*       V01.3   JRR     09-Apr-1992     Added CERN enhancements.
*	V01.4	JRR	17-Jun-1992	Added header.
*	V01.5	JRR	21-Sep-1992	Reformat source.
*					If RQF_AFTER not set, assume can
*					run right away.	
*	V01.6	JRR	18-Aug-1993	Miniscule modification.
*	V01.7	JRR	01-Mar-1994	Added support for SOLARIS.
*/
/*++ nqs_wakreq.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.35.6/src/RCS/nqs_wakreq.c,v $
 *
 * DESCRIPTION:
 *
 *	Wakeup requests that are being held by -a times which have now
 *	occurred.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	August 12, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.7 $ $Date: 1994/03/30 20:37:01 $ $State: Exp $)
 * $Log: nqs_wakreq.c,v $
 * Revision 1.7  1994/03/30  20:37:01  jrroma
 * Version 3.35.6
 *
 * Revision 1.6  93/09/10  13:57:24  jrroma
 * Version 3.35
 * 
 * Revision 1.5  92/12/22  15:41:41  jrroma
 * Version 3.30
 * 
 * Revision 1.4  92/06/18  17:31:35  jrroma
 * Added gnu header
 * 
 * Revision 1.3  92/05/06  10:44:57  jrroma
 *  Version 3.20
 * 
 * Revision 1.2  92/02/12  14:08:56  jrroma
 * Fixed declaration of static routines.
 * 
 * Revision 1.1  92/02/12  14:07:23  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"
#include "nqsxvars.h"			/* Global vars and directories */

static void wakeupset ( struct nqsqueue *queueset );

/*** nqs_wakreq
 *
 *
 *	void nqs_wakreq():
 *
 *	Move all requests in all queue waiting sets whose -a times have
 *	occurred, to the eligible to run request sets in their respective
 *	queues.
 */
void nqs_wakreq()
{

	wakeupset (Nonnet_queueset);	/* Wakeup requests in non-network */
					/* queues */
	wakeupset (Net_queueset);	/* Wakeup requests in network queues*/
}


/*** wakeupset
 *
 *
 *	void wakeupset():
 *
 *	Move all requests in all waiting sets whose -a times have occurred,
 *	to the eligible to run request sets in their respective queues, as
 *	present in the given queue set.
 */
static void wakeupset (struct nqsqueue *queueset)
{
	struct nqsqueue *queue;
	struct request *prevreq;
	struct request *req;
	time_t timenow;			/* Current GMT time */

	time (&timenow);		/* Get current time */
	queue = queueset;
	while (queue != (struct nqsqueue *) 0) {
	    prevreq = (struct request *)0;
	    req = queue->waitset;
	    while (req != (struct request *)0) {
		if ( (req->start_time > timenow) && (req->status & RQF_AFTER )){
		    /*
		     *  This request has a start time in the
		     *  future.  Set the virtual alarm to wake
		     *  it up later.
		     */
		    nqs_vtimer (&req->start_time, nqs_wakreq);
		    prevreq = req;
		    req = req->next;
		} else {
		    if (Debug > 4) {
			printf("D$nqs_wakreq: Request %d.[%d] is eligible\n", 
				req->v1.req.orig_seqno, req->v1.req.orig_mid);
			fflush(stdout);
		    }		
		    /*
		     *  This request is now eligible for execution.
		     *  Remove the request from the waiting set and
		     *  place it in the eligible queue set.
		     */
		    if (prevreq == (struct request *)0) {
			queue->waitset = req->next;
		    }
		    else prevreq->next = req->next;
		    queue->q.waitcount--;
		    /*
		     *  Add the request to the eligible to run
		     *  set.
		     */
		    a2s_a2qset (req, queue);
		    /*
		     *  Continue scanning the waiting set of
		     *  requests for the queue.
		     */
		    if (prevreq == (struct request *)0) {
			req = queue->waitset;
		    }
		    else req = prevreq->next;
		} 
	    }
	    queue = queue->next;
	}
	/*
	 *  Request(s) may be eligible for spawning.
	 */
	bsc_spawn();			/* Maybe spawn batch reqs */
	dsc_spawn();			/* Maybe spawn device reqs */
	psc_spawn();			/* Maybe spawn pipe reqs */
	/*
	 *  Update queue states whose images are out of date in the NQS
	 *  database.
	 */
	queue = queueset;
	while (queue != (struct nqsqueue *)0) {
	   if (queue->q.status & QUE_UPDATE) {
	       /*
		*  Update queue image and clear the QUE_UPDATE
	        *  bit.
	        */
		udb_qorder (queue);
	    }
	    queue = queue->next;
	}
}
