/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.20  JRR     16-Jan-1992	Added support for RS6000.
*       V01.3   JRR     02-Mar-1992	Added Cosmic V2 changes.
*       V01.4   JRR     08-Apr-1992     Added CERN enhancements.
*	V01.5	JRR	17-Jun-1992	Added header.
*	V01.6	JRR	10-NOv-1992	Added support for HPUX.
*	V01.7	JRR	06-Apr-1993	Added support for DECOSF.
*	V01.8	JRR	18-Aug-1993	Trying to go with kill and not killpg.
*	V01.9	JRR	01-Mar-1994	Added support for SOLARIS.
*/
/*++ nqs_family.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.35.6/src/RCS/nqs_family.c,v $
 *
 * DESCRIPTION:
 *
 *	Process a "report process family" packet from a shepherd process.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	August 12, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.9 $ $Date: 1994/03/30 20:36:37 $ $State: Exp $)
 * $Log: nqs_family.c,v $
 * Revision 1.9  1994/03/30  20:36:37  jrroma
 * Version 3.35.6
 *
 * Revision 1.8  93/09/10  13:57:14  jrroma
 * Version 3.35
 * 
 * Revision 1.7  93/07/13  21:33:50  jrroma
 * Version 3.34
 * 
 * Revision 1.6  92/12/22  15:39:55  jrroma
 * Version 3.30
 * 
 * Revision 1.5  92/06/18  17:31:04  jrroma
 * Added gnu header
 * 
 * Revision 1.4  92/05/06  10:39:44  jrroma
 *  Version 3.20
 * 
 * Revision 1.3  92/03/02  14:04:56  jrroma
 * Added Cosmic V2 changes.
 * 
 * Revision 1.2  92/01/17  10:56:16  jrroma
 * Added support for RS6000.
 * 
 * Revision 1.1  92/01/17  10:55:23  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"			/* NQS constants and data types */
#include "nqsxvars.h"			/* NQS global variables */

/*** nqs_family
 *
 *
 *	void nqs_family():
 *	Process a "report process family" packet from a shepherd process.
 */
void nqs_family (
	int shepherd_pid,	/* Process-id of shepherd process */
	int process_family)	/* "Process family" of server */
{
	struct nqsqueue *serverque;	/* Server was handling this queue */
	struct nqsqueue *queue;		/* Ptr to queue structure */
	int runcount;			/* Reqs running in a queue */
	struct request *req;		/* Request structure */
	short reqindex;			/* Index in Runvars of the running */
					/* structure allocated for the */
					/* request under inspection */

	queue = Nonnet_queueset;		/* First queue in queue set */
	serverque = (struct nqsqueue *)0;	/* Request queue not found */
	while (queue != (struct nqsqueue *)0 && serverque == (struct nqsqueue *)0) {
		runcount = queue->q.runcount;	/* # of requests running */
		req = queue->runset;		/* First run request */
		while (runcount--) {
			reqindex = req->reqindex;
			if ( (Runvars+reqindex)->shepherd_pid == shepherd_pid) {
				/*
				 *  We have found the request!
				 */
				serverque = queue;
				runcount = 0;	/* Exit inner loop */
			}
			else req = req->next;
		}
		queue = queue->next;		/* Get next queue */
	}
	if (serverque == (struct nqsqueue *) 0) {
		/*
		 *  The request was not located in the non-network
		 *  queue set.  Scan the network queue set for the
		 *  request.
		 */
		queue = Net_queueset;		/* First queue in queue set */
		serverque = (struct nqsqueue *)0;	/* Request queue not found */
		while (queue != (struct nqsqueue *) 0 &&
		       serverque == (struct nqsqueue *) 0) {
			runcount = queue->q.runcount;
						/* # of requests running */
			req = queue->runset;	/* First run request */
			while (runcount--) {
				reqindex = req->reqindex;
				if ( (Runvars+reqindex)->shepherd_pid
					== shepherd_pid) {
					/*
					 *  We have found the request!
					 */
					serverque = queue;
					runcount = 0;
						/* Exit inner loop */
				}
				else req = req->next;
			}
			queue = queue->next;	/* Get next queue */
		}
	}
	/*
	 *  Determine whether or not we found the request, and if it's
	 *  valid.
	 */
	if (serverque == (struct nqsqueue *) 0 ||	/* Request not found */
	    (Runvars+reqindex)->process_family != 0 || /*Family known already!*/
	    process_family <= 0) {
		/*
		 *  Bad family reported.
		 */
		printf ("E$Bad process family reported.\n");
		printf ("E$process family = %d\n", process_family);
		printf ("E$known family =%d\n", (Runvars+reqindex)->process_family);
		if (serverque == (struct nqsqueue *) 0) printf ("E$req not found\n");
		fflush (stdout);
		return;
	}
	/*
	 *  Req points to the request structure for the request whose
	 *  "process family" is being reported.
	 *
	 *  Serverque points to the queue structure for the queue
	 *  containing the request.
	 */
	if (req->status & RQF_SIGQUEUED) {
		/*
		 *  We have a queued "kill -<signal> request".
		 *  Kill the server process group/family.
		 */
		kill (-process_family, (Runvars+reqindex)->queued_signal);
		req->status &= ~RQF_SIGQUEUED;	/* Queued signal sent */
	}
	else {
		(Runvars+reqindex)->process_family = process_family;
						/* Store process-family */
		udb_reqpgrp (serverque, req);	/* Update the proper qentry */
	}					/* to show process group */
}



