/*
 * nqs_generic/all-systems/getfilnam.c
 * Build filenames for certain systems.
 */

#include <nqs_generic/license.h>
  /* NQS license information */
#include <nqs_generic/debug.h>
  /* debugging support */
#include <nqs_generic/proto.h>
  /* ANSI C prototypes */

/*++ getfilnam.c - Network Queueing System
 *
 * DESCRIPTION:
 *
 *	Function for building file names in directory independent installation.
 *	Function getfilnam checks the environment from specific variables.
 *	If set, the fully qualified name returned becomes the concatenation
 *	of the string defined by the environment variable and a suffix
 *	supplied by the calling function.  If the environment is not set,
 *	the fully qualified name returned becomes the compiled in default
 *	string concatenated with the suffix.
 *
 *	CAUTION:  This function allocates memory for the fully qualified
 *		  file name string.  It is the responsibility of the calling
 *		  function to release this memory when no longer needed.
 *
 *	Author:
 *	-------
 *	Chuck Keagle, Boeing Computer Services
 *	August 11, 1992
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/param.h>
#include <string.h>
#include <nqs_generic/nqs_generic.h>
#include <nqs_generic/nqsdirs.h>
#include <nqs_generic/autoconf.h>

char *getfilnam (char *szSuffix, short siDirtype)
{
  char *szHomedir;
  char *szName;
  char *szPrefix;

  ENTER_FUNCTION("nqs_generic/all-systems/getfilnam");
    /* debugging information */
  TEST_ARG(szSuffix  != NULL, 1);
  TEST_ARG(siDirtype >  0,    2);
    /* ensure we are not working with garbage.
     * FIXME:
     *   The test on siDirType should check both the upper AND the lower
     *   bounds ...
     */
  
  /*
   *  Find the appropriate environment variable.  Also determine the
   *  compiled in default which is used if the environment variable is
   *  undefined.
   */
  switch (siDirtype)
  {
   case LIBDIR:
    szHomedir = getenv (Nqs_libdir);
    szPrefix  = CONFIG_NQS_LIBEXE;
    break;
   case MAPDIR:
    szHomedir = getenv (Nqs_nmapdir);
    szPrefix  = CONFIG_NQS_NMAP;
    break;
   case SPOOLDIR:
    szHomedir = getenv (Nqs_spooldir);
    szPrefix  = CONFIG_NQS_SPOOL;
    break;
   default:
    EXIT_FUNCTION;
    return NULL;
    break;
  }
  /*
   *  If homedir is non-null, concatenate it to the supplied suffix.
   *  If it is null, use the compiled in prefix.  Return a pointer to
   *  the fully qualified name.
   */
  if (szHomedir == NULL)
  {
    szName = (char *)malloc((unsigned)(strlen(szPrefix) + strlen(szSuffix) + 2));
    sprintf (szName, "%s/%s", szPrefix, szSuffix);
  }
  else
  {
    szName = (char *)malloc ((unsigned)(strlen(szHomedir) + strlen(szSuffix) + 2));
    sprintf (szName, "%s/%s", szHomedir, szSuffix);
  }
  EXIT_FUNCTION;
  return (szName);
}

/*++ relfilnam
 *
 * DESCRIPTION:
 *
 *	Function relfilnam frees the memory that was allocated by getfilnam.
 *	Author:
 *	-------
 *	Chuck Keagle, Boeing Computer Services
 *	August 11, 1992
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 */
void relfilnam (char *szString)
{
  ENTER_FUNCTION("nqs_generic/all-systems/relfilnam");
  
  TEST_ARG(szString != NULL, 1);
    /* ensure we are not working with garbage */
  free (szString);
  
  EXIT_FUNCTION;
}

/*++ getenviron
 *
 * DESCRIPTION:
 *
 *	Function getenviron obtains a pointer to the supplied environment
 *	variable.  The pointer points to the beginning of the name=value
 *	string.  By contrast, getenv returns a pointer to the value
 *	portion of the string.  If the environment does not exist,
 *	getenviron returns a null pointer.
 *
 *	Author:
 *	-------
 *	Chuck Keagle, Boeing Computer Services
 *	August 11, 1992
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 */
#if 0
char *getenviron (char *szName)
{
  extern char **environ;
  /* FIXME:
   * Is this variable POSIX-compliant?
   */
  
  char **szTable;
  
  ENTER_FUNCTION("nqs_generic/all-systems/getenviron");
  /*
   *  Find the specified variable (Uh-huh).
   *  Environment table is terminated by a null pointer.
   *  Be careful not to smash the current environment.
   */
  
  szTable = environ;
  while ((*szTable != NULL) && strncmp (szName, *szTable, strlen (szName)))
    szTable++;
  
  EXIT_FUNCTION;
  return (*szTable);
}
#endif

/*++ buildenv
 *
 * DESCRIPTION:
 *
 *	Function buildenv builds the NQS directory independent environment
 *	from the contents of the nqs.config file in the directory pointed
 *	to by the NQS_HOME environment variable.  If the putenv call fails,
 *	buildenv returns an error.  It is not an error for the NQS_HOME
 *	environment variable to be undefined.
 *
 *	Author:
 *	-------
 *	Chuck Keagle, Boeing Computer Services
 *	August 11, 1992
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 */
int buildenv(void)
{
  FILE *pstFp;
  char *szEnvp;
  char *szHomedir;
  char szString [MAXPATHLEN];

  ENTER_FUNCTION("nqs_generic/all-systems/buildenv");
    /* debugging info */
  
  /*
   *  Find the nqs.config file in the directory pointed to by the
   *  $NQS_HOME environment variable.  If the $NQS_HOME environment
   *  variable does not exist, there is no error.  The administrator
   *  has installed NQS in the default directories.
   */
  
  if ((szHomedir = getenv ("NQS_HOME")) != NULL)
  {
    sprintf (szString, "%s/%s", szHomedir, "nqs.config");
    if ((pstFp = fopen (szString, "r")) != NULL)
    {
      /*
       *  When reading the config file, ignore lines that begin
       *  with either a # sign or a newline character.  All other
       *  lines are placed into the environment after the
       *  newline character is deleted.
       */
      while (fgets (szString, sizeof (szString), pstFp) != NULL)
      {
	if (szString [0] != '#' && szString [0] != '\n')
	{
	  szEnvp = (char *)malloc (strlen (szString) + 1);
	  strncpy(szEnvp, szString, strlen (szString));
	  szEnvp [strlen (szString)] = '\0';
	  if (putenv (szEnvp))
	  {
	    EXIT_FUNCTION;
	    return (0);
	  }
	}
      }
      fclose (pstFp);
    }
  }
  EXIT_FUNCTION;
  return (1);
}