/**********************************************************
 * dda.c - Analyzer for the accounting log of diald
 *
 * 0.10 02-May-1995 TB  First version
 *
 *********************************************************/

#include <stdio.h>
#include <strings.h>
#include <getopt.h>

/* External variables */
extern char                             *optarg;
extern int                              optind, opterr;

/* Modules defines */
#define PROGNAME                        "dda.c"
#define VERSION                         "0.1, 02-May-1995"
#define COPYRIGHT                       "(c) Copyright 1995 Thomas Bullinger"

#define STAT_FILE_NAME                  "dda.stats"

#define INP_LINE_LENGTH                 128
#define START_OF_CONN                   "Calling site "
#define SUCCESS_OF_CONN                 "Connected to site "
#define END_OF_CONN                     "Disconnected. Call duration "
#define XFER_OF_CONN                    "IP transmit"
typedef struct {
    char                                MonthName[4];
    unsigned int                        NumberOfConnections,
                                        SuccessfulConnections;
    float                               Duration,   /* in seconds */
                                        Xmit,       /* in KB */
                                        Rcv;        /* in KB */
} CONNECTION_INFO;
#define NUMBER_OF_MONTHS                13

/* Global variables */
CONNECTION_INFO                         ConnectionInfo[NUMBER_OF_MONTHS] = {0};
CONNECTION_INFO                         CurrentInfo = {0};
int                                     CurrentMonth;

/*---------------------------------------------------------
 * ProcessTimeString
 *-------------------------------------------------------*/
int
ProcessTimeString (char *TimeString)
{
    /* Local variables */
    char                                DoW[4],
                                        Month[4],
                                        Index;

    /* Extract the month string */
    sscanf (TimeString, "%s %s", DoW, Month);

    /* Convert the string into a number */
    if (!strcmp (Month, "Jan"))
        Index = 1;
    else if (!strcmp (Month, "Feb"))
        Index = 2;
    else if (!strcmp (Month, "Mar"))
        Index = 3;
    else if (!strcmp (Month, "Apr"))
        Index = 4;
    else if (!strcmp (Month, "May"))
        Index = 5;
    else if (!strcmp (Month, "Jun"))
        Index = 6;
    else if (!strcmp (Month, "Jul"))
        Index = 7;
    else if (!strcmp (Month, "Aug"))
        Index = 8;
    else if (!strcmp (Month, "Sep"))
        Index = 9;
    else if (!strcmp (Month, "Oct"))
        Index = 10;
    else if (!strcmp (Month, "Nov"))
        Index = 11;
    else if (!strcmp (Month, "Dec"))
        Index = 12;
    else
        Index = 0;

    return (Index);
}   /* end ProcessTimeString */

/*---------------------------------------------------------
 * ProcessStartLine
 *-------------------------------------------------------*/
void
ProcessStartLine (char *StartLine, char UpdateStats)
{
    /* Get the month */
    CurrentMonth = ProcessTimeString (StartLine);

    /* We have one more connection since last time */
    (CurrentInfo.NumberOfConnections)++;

    if (UpdateStats)
    {
        /* We have another connection for that month */
        (ConnectionInfo[CurrentMonth].NumberOfConnections)++;
    }

}   /* end ProcessStartLine */

/*---------------------------------------------------------
 * ProcessSuccessLine
 *-------------------------------------------------------*/
void
ProcessSuccessLine (char *SuccessLine, char UpdateStats)
{
    /* Local variables */
    int                                 Month;

    /* Get the month */
    Month = ProcessTimeString (SuccessLine);

    /* We have one more sucessful connection since last time */
    (CurrentInfo.SuccessfulConnections)++;

    if (UpdateStats)
    {
        /* We have another successful connection for that month */
        (ConnectionInfo[Month].SuccessfulConnections)++;
    }

}   /* end ProcessSuccessLine */

/*---------------------------------------------------------
 * ProcessStopLine
 *-------------------------------------------------------*/
void
ProcessStopLine (char *StopLine, char UpdateStats)
{
    /* Local variables */
    int                                 Month,
                                        Seconds;
    char                                Disc[15],
                                        Call[5],
                                        Durat[9],
                                       *p;

    /* Get the month */
    Month = ProcessTimeString (StopLine);

    /* Extract the duration string */
    p = strstr (StopLine, END_OF_CONN);
    sscanf (p, "%s %s %s %d", Disc, Call, Durat, &Seconds);

    /* We have spent time ... */
    CurrentInfo.Duration += (Seconds * 1.0);

    if (UpdateStats)
    {
        /* We also spent it in that month */
        ConnectionInfo[Month].Duration += (Seconds * 1.0);
    }

}   /* end ProcessStopLine */

/*---------------------------------------------------------
 * ProcessXferLine
 *-------------------------------------------------------*/
void
ProcessXferLine (char *XferLine, char UpdateStats)
{
    /* Local variables */
    char                                IP[3],
                                        Transm[12],
                                        Bytes1[6],
                                        And[4],
                                        Received[9];
    int                                 Xmit,
                                        Rcv;

    /* Extract the xmit and rcv values string */
    sscanf (XferLine, "%s %s %d %s %s %s %d",
            IP, Transm, &Xmit, Bytes1, And, Received, &Rcv);

    /* We have transmitted ... */
    CurrentInfo.Xmit += (Xmit / 1024.0);


    /* We have received ... */
    CurrentInfo.Rcv += (Rcv / 1024.0);

    if (UpdateStats)
    {
        /* We also trasmitted it in that month */
        ConnectionInfo[CurrentMonth].Xmit += (Xmit / 1024.0);

        /* We also received it in that month */
        ConnectionInfo[CurrentMonth].Rcv += (Rcv / 1024.0);
    }

}   /* end ProcessXferLine */

/*---------------------------------------------------------
 * ShowConnectionInfo
 *-------------------------------------------------------*/
void
ShowConnectionInfo (CONNECTION_INFO *pConnectionInfo)
{
    printf ("Number of connections: %9d", pConnectionInfo->NumberOfConnections);
    printf (" (success: %d)", pConnectionInfo->SuccessfulConnections);
    printf (" (no success: %d)\n", pConnectionInfo->NumberOfConnections -
                                        pConnectionInfo->SuccessfulConnections);
    printf ("Connect time (sec):    %9.0f", pConnectionInfo->Duration);
    printf (" (min: %.2f)", pConnectionInfo->Duration / 60.0);
    printf (" (hrs: %.2f)\n", pConnectionInfo->Duration / 3600.0);
    if (pConnectionInfo->Xmit > 1024.0)
        printf ("Data: %.4f MB xmit", pConnectionInfo->Xmit / 1024.0);
    else
        printf ("Data: %.4f KB xmit", pConnectionInfo->Xmit);
    if (pConnectionInfo->Rcv > 1024.0)
        printf (", %.4f MB rcvd\n\n", pConnectionInfo->Rcv / 1024.0);
    else
        printf (", %.4f KB rcvd\n\n", pConnectionInfo->Rcv);
}   /* end ShowConnectionInfo */

/*---------------------------------------------------------
 * ProcessLogFile
 *-------------------------------------------------------*/
void
ProcessLogFile (char *LogFileName, char UpdateStats, FILE *StatFile)
{
    /* Local variables */
    char                                InpLine[INP_LINE_LENGTH + 1];
    FILE                                *LogFile;

    if ((LogFile = fopen (LogFileName, "r")) != NULL)
    {
        /* Read line by line from the log file */
        while (fgets (InpLine, sizeof (InpLine), LogFile) != NULL)
        {
            /* Is it a start ? */
            if (strstr (InpLine, START_OF_CONN) != NULL)
            {
                /* Yes -> Process it */
                ProcessStartLine (InpLine, UpdateStats);
            }
            /* Is it a success line ? */
            else if (strstr (InpLine, SUCCESS_OF_CONN) != NULL)
            {
                /* Yes -> Process it */
                ProcessSuccessLine (InpLine, UpdateStats);
            } 
            /* Is it a stop line ? */
            else if (strstr (InpLine, END_OF_CONN) != NULL)
            {
                /* Yes -> Process it */
                ProcessStopLine (InpLine, UpdateStats);
            } 
            /* Is it a xmit/rcv line ? */
            else if (strstr (InpLine, XFER_OF_CONN) != NULL)
            {
                /* Yes -> Process it */
                ProcessXferLine (InpLine, UpdateStats);
            } 
        }   /* end while */

        /* Should we update the statitistics file ? */
        if (UpdateStats)
        {
            /* Yes -> Do it */
            fwrite (&(ConnectionInfo[0]), sizeof (CONNECTION_INFO),
                    NUMBER_OF_MONTHS, StatFile);
            fclose (StatFile);
        }   /* end if UpdateStats */

        /* Show the infos for this time */
        printf (">----------------------------------------------------------------\n");
        printf ("Connection infos for %s:\n\n", LogFileName);
        ShowConnectionInfo (&CurrentInfo);
    }
    else
    {
        perror ("Log file not found");
    }   /* end if LogFile */

}   /* end ProcessLogFile */

/*---------------------------------------------------------
 * ShowAllInfos
 *-------------------------------------------------------*/
void
ShowAllInfos (CONNECTION_INFO ConnInfo[])
{
    /* First the first 6 months */
    printf ("Month      %10s %10s %10s %10s %10s %10s\n", ConnInfo[1].MonthName,
            ConnInfo[2].MonthName, ConnInfo[3].MonthName,
            ConnInfo[4].MonthName, ConnInfo[5].MonthName,
            ConnInfo[6].MonthName);
    printf ("----------------------------------------------------------------------------\n");
    printf ("Connect    %10d %10d %10d %10d %10d %10d\n",
            ConnInfo[1].NumberOfConnections,
            ConnInfo[2].NumberOfConnections,
            ConnInfo[3].NumberOfConnections,
            ConnInfo[4].NumberOfConnections,
            ConnInfo[5].NumberOfConnections,
            ConnInfo[6].NumberOfConnections);
    printf (" success    %9d %10d %10d %10d %10d %10d\n",
            ConnInfo[1].SuccessfulConnections,
            ConnInfo[2].SuccessfulConnections,
            ConnInfo[3].SuccessfulConnections,
            ConnInfo[4].SuccessfulConnections,
            ConnInfo[5].SuccessfulConnections,
            ConnInfo[6].SuccessfulConnections);
    printf (" no success %9d %10d %10d %10d %10d %10d\n",
            ConnInfo[1].NumberOfConnections - ConnInfo[1].SuccessfulConnections, 
            ConnInfo[2].NumberOfConnections - ConnInfo[2].SuccessfulConnections,
            ConnInfo[3].NumberOfConnections - ConnInfo[3].SuccessfulConnections,
            ConnInfo[4].NumberOfConnections - ConnInfo[4].SuccessfulConnections,
            ConnInfo[5].NumberOfConnections - ConnInfo[5].SuccessfulConnections,
            ConnInfo[6].NumberOfConnections - ConnInfo[6].SuccessfulConnections);
    printf ("Duration   %10.0f %10.0f %10.0f %10.0f %10.0f %10.0f\n",
            ConnInfo[1].Duration, ConnInfo[2].Duration,
            ConnInfo[3].Duration, ConnInfo[4].Duration,
            ConnInfo[5].Duration, ConnInfo[6].Duration);
    printf (" in min    %10.2f %10.2f %10.2f %10.2f %10.2f %10.2f\n",
            ConnInfo[1].Duration / 60.0,
            ConnInfo[2].Duration / 60.0,
            ConnInfo[3].Duration / 60.0,
            ConnInfo[4].Duration / 60.0,
            ConnInfo[5].Duration / 60.0,
            ConnInfo[6].Duration / 60.0);
    printf (" in hrs    %10.2f %10.2f %10.2f %10.2f %10.2f %10.2f\n",
            ConnInfo[1].Duration / 3600.0,
            ConnInfo[2].Duration / 3600.0,
            ConnInfo[3].Duration / 3600.0,
            ConnInfo[4].Duration / 3600.0,
            ConnInfo[5].Duration / 3600.0,
            ConnInfo[6].Duration / 3600.0);
    printf ("Xmit (MB)  %10.2f %10.2f %10.2f %10.2f %10.2f %10.2f\n",
            ConnInfo[1].Xmit / 1024.0, ConnInfo[2].Xmit / 1024.0,
            ConnInfo[3].Xmit / 1024.0, ConnInfo[4].Xmit / 1024.0,
            ConnInfo[5].Xmit / 1024.0, ConnInfo[6].Xmit / 1024.0);
    printf ("Rcvd (MB)  %10.2f %10.2f %10.2f %10.2f %10.2f %10.2f\n",
            ConnInfo[1].Rcv / 1024.0, ConnInfo[2].Rcv / 1024.0,
            ConnInfo[3].Rcv / 1024.0, ConnInfo[4].Rcv / 1024.0,
            ConnInfo[5].Rcv / 1024.0, ConnInfo[6].Rcv / 1024.0);
    /* Then the second half */
    printf ("============================================================================\n");
    printf ("Month      %10s %10s %10s %10s %10s %10s\n", ConnInfo[7].MonthName,
            ConnInfo[8].MonthName, ConnInfo[9].MonthName,
            ConnInfo[10].MonthName, ConnInfo[11].MonthName,
            ConnInfo[12].MonthName);
    printf ("----------------------------------------------------------------------------\n");
    printf ("Connect    %10d %10d %10d %10d %10d %10d\n",
            ConnInfo[7].NumberOfConnections,
            ConnInfo[8].NumberOfConnections,
            ConnInfo[9].NumberOfConnections,
            ConnInfo[10].NumberOfConnections,
            ConnInfo[11].NumberOfConnections,
            ConnInfo[12].NumberOfConnections);
    printf ("success    %10d %10d %10d %10d %10d %10d\n",
            ConnInfo[7].SuccessfulConnections,
            ConnInfo[8].SuccessfulConnections,
            ConnInfo[9].SuccessfulConnections,
            ConnInfo[10].SuccessfulConnections,
            ConnInfo[11].SuccessfulConnections,
            ConnInfo[12].SuccessfulConnections);
    printf ("no success %10d %10d %10d %10d %10d %10d\n",
            ConnInfo[7].NumberOfConnections - ConnInfo[7].SuccessfulConnections, 
            ConnInfo[8].NumberOfConnections - ConnInfo[8].SuccessfulConnections,
            ConnInfo[9].NumberOfConnections - ConnInfo[9].SuccessfulConnections,
            ConnInfo[10].NumberOfConnections - ConnInfo[10].SuccessfulConnections,
            ConnInfo[11].NumberOfConnections - ConnInfo[11].SuccessfulConnections,
            ConnInfo[12].NumberOfConnections - ConnInfo[12].SuccessfulConnections);
    printf ("Duration   %10.0f %10.0f %10.0f %10.0f %10.0f %10.0f\n",
            ConnInfo[7].Duration, ConnInfo[8].Duration,
            ConnInfo[9].Duration, ConnInfo[10].Duration,
            ConnInfo[11].Duration, ConnInfo[12].Duration);
    printf (" in min    %10.2f %10.2f %10.2f %10.2f %10.2f %10.2f\n",
            ConnInfo[7].Duration / 60.0,
            ConnInfo[8].Duration / 60.0,
            ConnInfo[9].Duration / 60.0,
            ConnInfo[10].Duration / 60.0,
            ConnInfo[11].Duration / 60.0,
            ConnInfo[12].Duration / 60.0);
    printf (" in hrs    %10.2f %10.2f %10.2f %10.2f %10.2f %10.2f\n",
            ConnInfo[7].Duration / 3600.0,
            ConnInfo[8].Duration / 3600.0,
            ConnInfo[9].Duration / 3600.0,
            ConnInfo[10].Duration / 3600.0,
            ConnInfo[11].Duration / 3600.0,
            ConnInfo[12].Duration / 3600.0);
    printf ("Xmit (MB)  %10.2f %10.2f %10.2f %10.2f %10.2f %10.2f\n",
            ConnInfo[7].Xmit / 1024.0, ConnInfo[8].Xmit / 1024.0,
            ConnInfo[9].Xmit / 1024.0, ConnInfo[10].Xmit / 1024.0,
            ConnInfo[11].Xmit / 1024.0, ConnInfo[12].Xmit / 1024.0);
    printf ("Rcvd (MB)  %10.2f %10.2f %10.2f %10.2f %10.2f %10.2f\n",
            ConnInfo[7].Rcv / 1024.0, ConnInfo[8].Rcv / 1024.0,
            ConnInfo[9].Rcv / 1024.0, ConnInfo[10].Rcv / 1024.0,
            ConnInfo[11].Rcv / 1024.0, ConnInfo[12].Rcv / 1024.0);
}   /* end ShowAllInfos */

/*---------------------------------------------------------
 * Usage
 *-------------------------------------------------------*/
void
Usage (void)
{
    printf (" %s (Version %s) [options]\n", PROGNAME, VERSION);
    printf ("   -f <statistics file> Specify the file where the\n");
    printf ("                        statistics are stored\n");
    printf ("   -a                   Show all months\n");
    printf ("   -m <month>           Show statistics for <month>\n");
    printf ("   -l <log file>        Process this log file\n");
    printf ("   -u                   Update statistics file\n");
    printf ("   -h                   Show this help\n");
    exit (1);
}   /* end Usage */

/*---------------------------------------------------------
 * main
 *-------------------------------------------------------*/
int
main (int argc, char *argv[])
{
    /* Local variables */
    char                                InpLine[INP_LINE_LENGTH + 1],
                                        StatFileName[2 * INP_LINE_LENGTH + 1] = {0},
                                        LogFileName[2 * INP_LINE_LENGTH + 1] = {0},
                                        ShowAllMonths = 0,
                                        ShowMonthOnly = 0,
                                        UpdateStats = 0;
    int                                 OSRet;
    FILE                                *StatFile = NULL;

    /* Pass any options */
    while ((OSRet = getopt (argc, argv, "ahf:l:m:u")) != -1)
    {
        switch (OSRet)
        {
        case 'a':   /* Show all infos */
            ShowAllMonths = 1;
            break;

        case 'l':   /* Log file given */
            strcpy (LogFileName, optarg);
            break;

        case 'f':   /* Statistics file given */
            strcpy (StatFileName, optarg);
            break;

        case 'm':   /* Show statistics for month */
            sprintf (InpLine, "Mon %s", optarg);
            CurrentMonth = ProcessTimeString (InpLine);
            ShowMonthOnly = 1;
            break;

        case 'h':   /* Help requested */
        case '?':   /* Unknown option */
            Usage ();
            break;

        case 'u':   /* Update statistics file */
            UpdateStats = 1;
            break;

        default:
            break;
        }   /* end switch OSRet */
    }   /* end while getopt */

    /* Show some greeting */
    printf ("\n%s Version %s\n%s\n\n", PROGNAME, VERSION, COPYRIGHT);

    /* Did the user specify a log file ? */
    if (LogFileName[0] == '\0')
    {
        /* No -> Don't update the stats */
        UpdateStats = 0;
    }

    /* Initialize the month names */
    strcpy (ConnectionInfo[0].MonthName, "???");
    strcpy (ConnectionInfo[1].MonthName, "Jan");
    strcpy (ConnectionInfo[2].MonthName, "Feb");
    strcpy (ConnectionInfo[3].MonthName, "Mar");
    strcpy (ConnectionInfo[4].MonthName, "Apr");
    strcpy (ConnectionInfo[5].MonthName, "May");
    strcpy (ConnectionInfo[6].MonthName, "Jun");
    strcpy (ConnectionInfo[7].MonthName, "Jul");
    strcpy (ConnectionInfo[8].MonthName, "Aug");
    strcpy (ConnectionInfo[9].MonthName, "Sep");
    strcpy (ConnectionInfo[10].MonthName, "Oct");
    strcpy (ConnectionInfo[11].MonthName, "Nov");
    strcpy (ConnectionInfo[12].MonthName, "Dec");

    /* Did the user specify a statistics file ? */
    if (StatFileName[0] == '\0')
    {
        /* No -> Use a precompiled name */
        strcpy (StatFileName, STAT_FILE_NAME);
    }

    /* Does the statistics file exist ? */
    if ((StatFile = fopen (StatFileName, "r")) != NULL)
    {
        /* Yes -> Can we read it's contents ? */
        OSRet = fread (&ConnectionInfo[0], sizeof (CONNECTION_INFO),
                       NUMBER_OF_MONTHS, StatFile);
        if (OSRet == NUMBER_OF_MONTHS)
        {
            /* Yes -> Do we have to update the stats ? */
            if (UpdateStats)
            {
                /* Yes -> Make sure that we write from the start ... */
                fclose (StatFile);
                StatFile = fopen (StatFileName, "w");
            }
        }
        else
        {
            /* No -> Overwrite it ... */
            fclose (StatFile);
            StatFile = fopen (StatFileName, "w");
        }   /* end if OSRet */
    }
    else
    {
        /* No -> Create it */
        StatFile = fopen (StatFileName, "w");
    }   /* end if fopen */

    /* Did the user specify a log file ? */
    if (LogFileName[0] != '\0')
    {
        /* Yes -> Process it */
        ProcessLogFile (LogFileName, UpdateStats, StatFile);
    }   /* end if LogFileName */

    /* Should we show all months ? */
    if (ShowAllMonths)
    {
        printf (">----------------------------------------------------------------\n");
        printf ("Stored connection infos:\n\n");
        ShowAllInfos (ConnectionInfo);
    }   /* end if ShowAllMonths */

    /* Do we have to show the month only ? */
    if (ShowMonthOnly)
    {
        /* Yes -> Show the infos for this month */
        printf (">----------------------------------------------------------------\n");
        printf ("Connection infos for %s:\n\n",
                ConnectionInfo[CurrentMonth].MonthName);
        ShowConnectionInfo (&(ConnectionInfo[CurrentMonth]));
    }   /* end if ShowMonthOnly */

    return (0);
}   /* end main */
