/*
 * renamed pppdfncs.c and modified for use with SLiRP
 * by Juha Pirkola 8-May-1995
 */

/*
 * main.c - Point-to-Point Protocol main module
 *
 * Copyright (c) 1989 Carnegie Mellon University.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by Carnegie Mellon University.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#define SETSID

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <syslog.h>
#include <netdb.h>
#include <utmp.h>
#include <pwd.h>
#include <sys/wait.h>

/*
 * If REQ_SYSOPTIONS is defined to 1, pppd will not run unless
 * /etc/ppp/options exists.
 */
#ifndef	REQ_SYSOPTIONS
#define REQ_SYSOPTIONS	1
#endif

#ifdef SGTTY
#include <sgtty.h>
#else
#if !defined(sun) && !defined(hcx)
#include <sys/ioctl.h>
#endif
#include <termios.h>
#endif

#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <net/if.h>

#include "callout.h"
#include "ppp.h"
#include "magic.h"
#include "fsm.h"
#include "lcp.h"
#include "ipcp.h"
#include "upap.h"
#include "chap.h"

#include "pppd.h"
#include "pathnames.h"
#include "patchlevel.h"


#ifndef TRUE
#define TRUE (1)
#endif /*TRUE*/

#ifndef FALSE
#define FALSE (0)
#endif /*FALSE*/

/*
#ifdef PIDPATH
static char *pidpath = PIDPATH;
#else
static char *pidpath = _PATH_PIDFILE;
#endif
 */

/* interface vars */
char ifname[IFNAMSIZ];		/* Interface name */
int ifunit;			/* Interface unit number */

char *progname;			/* Name of this program */
char hostname[MAXNAMELEN];	/* Our hostname */
char our_name[MAXNAMELEN];
char remote_name[MAXNAMELEN];
/* static char pidfilename[MAXPATHLEN]; */

/* static pid_t	pid;	*/	/* Our pid */
/* static pid_t	pgrpid;	*/	/* Process Group ID */
uid_t uid;			/* Our real user-id */

char pppdevname[MAXPATHLEN] = "/dev/tty";	/* Device name */
int default_device = TRUE;	/* use default device (stdin/out) */

int fd = -1;			/* Device file descriptor */
int s;				/* Socket file descriptor */

int phase;			/* where the link is at */

/*
#ifdef SGTTY
static struct sgttyb initsgttyb;
#else
static struct termios inittermios;
#endif
 */

/* static int initfdflags = -1; */	/* Initial file descriptor flags */

/* static int restore_term; */	/* 1 => we've munged the terminal */

u_char outpacket_buf[MTU+DLLHEADERLEN]; /* buffer for outgoing packet */
/* static u_char inpacket_buf[MTU+DLLHEADERLEN]; */ /* buffer for incoming packet */

int hungup;			/* terminal has been hung up */
/* static int n_children;	*/	/* # child processes still running */

/* configured variables */

int debug = 0;		        /* Debug flag */
int kdebugflag = 0;		/* Kernel debugging flag */
char user[MAXNAMELEN];		/* username for PAP */
char passwd[MAXSECRETLEN];	/* password for PAP */
char *connector = NULL;		/* "connect" command */
char *disconnector = NULL;	/* "disconnect" command */
int inspeed = 0;		/* Input/Output speed requested */
int baud_rate;			/* bits/sec currently used */
u_long netmask = 0;		/* netmask to use on ppp interface */
int crtscts = 0;		/* use h/w flow control */
int nodetach = 0;		/* don't fork */
int modem = 0;			/* use modem control lines */
int auth_required = 0;		/* require peer to authenticate */
int defaultroute = 0;		/* assign default route through interface */
int proxyarp = 0;		/* set entry in arp table */
int persist = 0;		/* re-initiate on termination */
int answer = 0;			/* wait for incoming call */
int uselogin = 0;		/* check PAP info against /etc/passwd */
int lockflag = 0;		/* lock the serial device */


/* prototypes */

void alrm __ARGS((int));

void novm __ARGS((char *));


extern	char	*ttyname __ARGS((int));
extern	char	*getlogin __ARGS((void));

/*
 * PPP Data Link Layer "protocol" table.
 * One entry per supported protocol.
 */
struct protent {				/* static removed by JP */		
    u_short protocol;
    void (*init)();
    void (*input)();
    void (*protrej)();
    int  (*printpkt)();
    char *name;
} prottbl[] = {
    { LCP, lcp_init, lcp_input, lcp_protrej, lcp_printpkt, "LCP" },
    { IPCP, ipcp_init, ipcp_input, ipcp_protrej, ipcp_printpkt, "IPCP" },
    { UPAP, upap_init, upap_input, upap_protrej, upap_printpkt, "PAP" },
    { CHAP, ChapInit, ChapInput, ChapProtocolReject, ChapPrintPkt, "CHAP" },
};

#define N_PROTO		(sizeof(prottbl) / sizeof(prottbl[0]))


/*
 * quit - Clean up state and exit.
 */
void 
quit()
{
    die(0);
}



static struct callout *callout = NULL;		/* Callout list */
static struct timeval schedtime;		/* Time last timeout was set */

/*
 * timeout - Schedule a timeout.
 *
 * Note that this timeout takes the number of seconds, NOT hz (as in
 * the kernel).
 */
void
timeout(func, arg, time)
    void (*func)();
    caddr_t arg;
    int time;
{
    struct itimerval itv;
    struct callout *newp, **oldpp;
  
    MAINDEBUG((LOG_DEBUG, "Timeout %x:%x in %d seconds.",
	       (int) func, (int) arg, time));
  
    /*
     * Allocate timeout.
     */
    if ((newp = (struct callout *) malloc(sizeof(struct callout))) == NULL) {
	syslog(LOG_ERR, "Out of memory in timeout()!");
	die(1);
    }
    newp->c_arg = arg;
    newp->c_func = func;
  
    /*
     * Find correct place to link it in and decrement its time by the
     * amount of time used by preceding timeouts.
     */
    for (oldpp = &callout;
	 *oldpp && (*oldpp)->c_time <= time;
	 oldpp = &(*oldpp)->c_next)
	time -= (*oldpp)->c_time;
    newp->c_time = time;
    newp->c_next = *oldpp;
    if (*oldpp)
	(*oldpp)->c_time -= time;
    *oldpp = newp;
  
    /*
     * If this is now the first callout then we have to set a new
     * itimer.
     */
    if (callout == newp) {
	itv.it_interval.tv_sec = itv.it_interval.tv_usec =
	    itv.it_value.tv_usec = 0;
	itv.it_value.tv_sec = callout->c_time;
	MAINDEBUG((LOG_DEBUG, "Setting itimer for %d seconds in timeout.",
		   itv.it_value.tv_sec));
	if (setitimer(ITIMER_REAL, &itv, NULL)) {
	    syslog(LOG_ERR, "setitimer(ITIMER_REAL): %m");
	    die(1);
	}
	if (gettimeofday(&schedtime, NULL)) {
	    syslog(LOG_ERR, "gettimeofday: %m");
	    die(1);
	}
    }
}


/*
 * untimeout - Unschedule a timeout.
 */
void
untimeout(func, arg)
    void (*func)();
    caddr_t arg;
{
    struct itimerval itv;
    struct callout **copp, *freep;
    int reschedule = 0;
  
    MAINDEBUG((LOG_DEBUG, "Untimeout %x:%x.", (int) func, (int) arg));
  
    /*
     * If the first callout is unscheduled then we have to set a new
     * itimer.
     */
    if (callout &&
	callout->c_func == func &&
	callout->c_arg == arg)
	reschedule = 1;
  
    /*
     * Find first matching timeout.  Add its time to the next timeouts
     * time.
     */
    for (copp = &callout; *copp; copp = &(*copp)->c_next)
	if ((*copp)->c_func == func &&
	    (*copp)->c_arg == arg) {
	    freep = *copp;
	    *copp = freep->c_next;
	    if (*copp)
		(*copp)->c_time += freep->c_time;
	    (void) free((char *) freep);
	    break;
	}
  
    if (reschedule) {
	itv.it_interval.tv_sec = itv.it_interval.tv_usec =
	    itv.it_value.tv_usec = 0;
	itv.it_value.tv_sec = callout ? callout->c_time : 0;
	MAINDEBUG((LOG_DEBUG, "Setting itimer for %d seconds in untimeout.",
		   itv.it_value.tv_sec));
	if (setitimer(ITIMER_REAL, &itv, NULL)) {
	    syslog(LOG_ERR, "setitimer(ITIMER_REAL): %m");
	    die(1);
	}
	if (gettimeofday(&schedtime, NULL)) {
	    syslog(LOG_ERR, "gettimeofday: %m");
	    die(1);
	}
    }
}


/*
 * adjtimeout - Decrement the first timeout by the amount of time since
 * it was scheduled.
 */
void
adjtimeout()
{
    struct timeval tv;
    int timediff;
  
    if (callout == NULL)
	return;
    /*
     * Make sure that the clock hasn't been warped dramatically.
     * Account for recently expired, but blocked timer by adding
     * small fudge factor.
     */
    if (gettimeofday(&tv, NULL)) {
	syslog(LOG_ERR, "gettimeofday: %m");
	die(1);
    }
    timediff = tv.tv_sec - schedtime.tv_sec;
    if (timediff < 0 ||
	timediff > callout->c_time + 1)
	return;
  
    callout->c_time -= timediff;	/* OK, Adjust time */
}


/*
 * alrm - Catch SIGALRM signal.
 *
 * Indicates a timeout.
 */
void							/* static removed by JP */
alrm(sig)
    int sig;
{
    struct itimerval itv;
    struct callout *freep, *list, *last;

    MAINDEBUG((LOG_DEBUG, "Alarm"));

    if (callout == NULL)
	return;
    /*
     * Get the first scheduled timeout and any that were scheduled
     * for the same time as a list, and remove them all from callout
     * list.
     */
    list = last = callout;
    while (last->c_next != NULL && last->c_next->c_time == 0)
	last = last->c_next;
    callout = last->c_next;
    last->c_next = NULL;

    /*
     * Set a new itimer if there are more timeouts scheduled.
     */
    if (callout) {
	itv.it_interval.tv_sec = itv.it_interval.tv_usec = 0;
	itv.it_value.tv_usec = 0;
	itv.it_value.tv_sec = callout->c_time;
	MAINDEBUG((LOG_DEBUG, "Setting itimer for %d seconds in alrm.",
		   itv.it_value.tv_sec));
	if (setitimer(ITIMER_REAL, &itv, NULL)) {
	    syslog(LOG_ERR, "setitimer(ITIMER_REAL): %m");
	    die(1);
	}
	if (gettimeofday(&schedtime, NULL)) {
	    syslog(LOG_ERR, "gettimeofday: %m");
	    die(1);
	}
    }

    /*
     * Now call all the timeout routines scheduled for this time.
     */
    while (list) {
	(*list->c_func)(list->c_arg);
	freep = list;
	list = list->c_next;
	(void) free((char *) freep);
    }
  
}


/*
 * demuxprotrej - Demultiplex a Protocol-Reject.
 */
void
demuxprotrej(unit, protocol)
    int unit;
    u_short protocol;
{
    int i;

    /*
     * Upcall the proper Protocol-Reject routine.
     */
    for (i = 0; i < sizeof (prottbl) / sizeof (struct protent); i++)
	if (prottbl[i].protocol == protocol) {
	    (*prottbl[i].protrej)(unit);
	    return;
	}

    syslog(LOG_WARNING,
	   "demuxprotrej: Unrecognized Protocol-Reject for protocol %d!",
	   protocol);
}


/*
 * novm - log an error message saying we ran out of memory, and die.
 */
void
novm(msg)
    char *msg;
{
    syslog(LOG_ERR, "Virtual memory exhausted allocating %s\n", msg);
    die(1);
}
