/*
 * Copyright (c) 1995 Danny Gasparovski.
 * 
 * Please read the file COPYRIGHT for the 
 * terms and conditions of the copyright.
 */

#include "h/common.h"
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#include <sys/socket.h>
#include <sys/uio.h>
#ifdef HAVE_SYS_FILIO_H
# include <sys/filio.h>
#endif
#ifdef HAVE_SYS_IOCTL_H
# include <sys/ioctl.h>
#endif
#include "h/ip.h"
#include "h/tcp.h"
#include "h/udp.h"
#include "h/socket.h"
#include "h/main.h"

void
so_init() {
	/* Nothing yet */
}


struct socket *
solookup(head, laddr, lport, faddr, fport)
        struct socket *head;
        struct in_addr faddr, laddr;
        u_int fport, lport;
{
	struct socket *so;

#ifdef DEBUG
	debug_call(dfd,"solookup(so, head) called ...\n");
	debug_call(dfd,"    (original endian, as stored in vars)\n");
	debug_call(dfd,"    head = %d\n", (int)head);
	debug_call(dfd,"    laddr = %x\n", (u_int)laddr.s_addr);
	debug_call(dfd,"    lport = %d\n", lport);
	debug_call(dfd,"    faddr = %x\n", (u_int)faddr.s_addr);
	debug_call(dfd,"    fport = %d\n", fport);
	fflush_call(dfd);
#endif
	        
	        
	for (so = head->so_next; so != head; so = so->so_next) {
		if (so->so_lport == lport && 
		    so->so_laddr.s_addr == laddr.s_addr &&
		    so->so_faddr.s_addr == faddr.s_addr &&
		    so->so_fport == fport)
		   break;
	}
	
	if (so == head)
	   return (struct socket *)NULL;
	return so;
	
}

/*
 * Create a new socket, initialise the fields
 * It is the responsibility of the caller to
 * insque() it into the correct linked-list
 */
struct socket *
socreate()
{
	struct socket *so;

#ifdef DEBUG
	debug_call(dfd,"socreate() called ...\n");
	fflush_call(dfd);
#endif
	
	so = (struct socket *)malloc(sizeof (struct socket));
	if (so == NULL)
	   return ((struct socket *)NULL);
	
	memset(so, 0, sizeof (struct socket));
	so->so_state = SS_NOFDREF;
	so->s = -1;
	
	return so;
}

/*
 * remque and free a socket, clobber cache
 */
void
sofree(so)
	struct socket *so;
{
	
#ifdef DEBUG
	debug_call(dfd,"sofree(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	if (so == tcp_last_so)
	   tcp_last_so = &tcb;
	else if (so == udp_last_so)
	   udp_last_so = &udb;
	
	if (so->so_m)
	   m_free(so->so_m);
	
	remque(so);
	free(so);
}

/*
 * Read from so's socket into sb_snd, updating all relevant sbuf fields
 * NOTE: This will only be called if it is select()ed for reading, so
 * a read() of 0 (or less) means it's disconnected
 */
int
soread(so)
	struct socket *so;
{
	int n, nn;
	struct sbuf *sb = &so->so_snd;
	int len = sb->sb_datalen - sb->sb_cc;
	struct iovec iov[2];
	
#ifdef DEBUG
	debug_call(dfd,"soread(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	
	/* 
	 * No need to check if there's enough room to read.
	 * soread wouldn't have been called if there weren't
	 */
	
	len = sb->sb_datalen - sb->sb_cc;
	
	iov[0].iov_base = sb->sb_wptr;
	if (sb->sb_wptr < sb->sb_rptr) {
		iov[0].iov_len = sb->sb_rptr - sb->sb_wptr;
		/* Should never succeed, but... */
		if (iov[0].iov_len > len) iov[0].iov_len = len;
		n = 1;
	} else  {
		iov[0].iov_len = (sb->sb_data + sb->sb_datalen) - sb->sb_wptr;
		/* Should never succeed, but... */
		if (iov[0].iov_len > len) iov[0].iov_len = len;
		len -= iov[0].iov_len;
		if (len)  {
			iov[1].iov_base = sb->sb_data;
			iov[1].iov_len = sb->sb_rptr - sb->sb_data;
			if(iov[1].iov_len > len) iov[1].iov_len = len;
			n = 2;
		} else
		   n = 1;
	}
#ifdef DEBUG
	errno = 0;
#endif

#ifdef HAVE_READV
	nn = readv(so->s, (struct iovec *)iov, n);
# ifdef DEBUG
	debug_misc(dfd," ... read nn = %d bytes\n", nn);
# endif
#else
	nn = read(so->s, iov[0].iov_base, iov[0].iov_len);
#endif	
	/* This should never happen, but people tell me it does *shrug* */
	if (nn < 0 && (errno == EINTR || errno == EAGAIN))
		return 0;
	
	if (nn <= 0)  {
#ifdef DEBUG
		debug_misc(dfd," --- soread() disconnected, nn = %d, errno = %d\n", nn, errno);
#endif
		sofcantrcvmore(so);
		tcp_sockclosed(sototcpcb(so));
		return -1;
	}
	
#ifndef HAVE_READV
	/*
	 * If there was no error, try and read the second time round
	 * We read again if n = 2 (ie, there's another part of the buffer)
	 * and we read as much as we could in the first read
	 * We don't test for <= 0 this time, because there legitimately
	 * might not be any more data (since the socket is non-blocking),
	 * a close will be detected on next iteration.
	 * A return of -1 wont (shouldn't) happen, since it didn't happen above
	 */
	if (n == 2 && nn == iov[0].iov_len)
	   nn += read(so->s, iov[1].iov_base, iov[1].iov_len);
	
#ifdef DEBUG
	debug_misc(dfd," ... read nn = %d bytes\n", nn);
#endif
#endif
	
	/* Update fields */
	sb->sb_cc += nn;
	sb->sb_wptr += nn;
	if (sb->sb_wptr >= (sb->sb_data + sb->sb_datalen))
		sb->sb_wptr -= sb->sb_datalen;
	return nn;
}
	
/*
 * Get urgent data
 * 
 * When the socket is created, we set it SO_OOBINLINE,
 * so when OOB data arrives, we soread() it and everything
 * in the send buffer is sent as urgent data
 */
void
sorecvoob(so)
	struct socket *so;
{
	struct tcpcb *tp = sototcpcb(so);

#ifdef DEBUG
	debug_call(dfd,"sorecvoob(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	
	/*
	 * We take a guess at how much urgent data has arrived.
	 * In most situations, when urgent data arrives, the next
	 * read() should get all the urgent data.  This guess will
	 * be wrong however if more data arrives just after the
	 * urgent data, or the read() doesn't return all the 
	 * urgent data.
	 */
	soread(so);
	tp->snd_up = tp->snd_una + so->so_snd.sb_cc;
	tp->t_force = 1;
	tcp_output(tp);
	tp->t_force = 0;
}

/*
 * Send urgent data
 * There's a lot duplicated code here, but...
 */
int
sosendoob(so)
	struct socket *so;
{
	struct sbuf *sb = &so->so_rcv;
	/* XXX I don't like this */
#ifdef __GNUC__
	char buff[sb->sb_cc]; /* XXX? */
#else
	char *buff = (char *) malloc(sb->sb_cc);
#endif
	int n, len;

#ifdef DEBUG
	debug_call(dfd,"sosendoob(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	debug_call(dfd,"sb->sb_cc = %d, so->so_urgc = %d\n", sb->sb_cc, so->so_urgc);
	fflush_call(dfd);
#endif
	
	if (sb->sb_rptr < sb->sb_wptr) {
		/* We can send it directly */
		n = send(so->s, sb->sb_rptr, so->so_urgc, (MSG_OOB)); /* |MSG_DONTWAIT)); */
		so->so_urgc -= n;
#ifdef DEBUG
		debug_misc(dfd," --- sent %d bytes urgent data, %d urgent bytes left\n", n, so->so_urgc);
#endif
	} else {
		/* 
		 * Since there's no sendv or sendtov like writev,
		 * we must copy all data to a linear buffer then
		 * send it all
		 */
		len = (sb->sb_data + sb->sb_datalen) - sb->sb_rptr;
		if (len > so->so_urgc) len = so->so_urgc;
		memcpy(buff, sb->sb_rptr, len);
		so->so_urgc -= len;
		if (so->so_urgc) {
			n = sb->sb_wptr - sb->sb_data;
			if (n > so->so_urgc) n = so->so_urgc;
			memcpy((buff + len), sb->sb_data, n);
			so->so_urgc -= n;
			len += n;
		}
		n = send(so->s, buff, len, (MSG_OOB)); /* |MSG_DONTWAIT)); */
#ifdef DEBUG
		debug_misc(dfd," ---2 sent %d bytes urgent data, %d urgent bytes left\n", n, so->so_urgc);
#endif
	}
	
	sb->sb_cc -= n;
	sb->sb_rptr += n;
	if (sb->sb_rptr >= (sb->sb_data + sb->sb_datalen))
		sb->sb_rptr -= sb->sb_datalen;
	
#ifdef DEBUG
	if (so->so_urgc) {
		debug_error(dfd,"there's still urgent data!\n");
	}
#endif
	
#ifndef __GNUC__
	free(buff);
#endif
	
	return n;
}

/*
 * Write data from so_rcv to so's socket, 
 * updating all sbuf field as necessary
 */
int
sowrite(so)
	struct socket *so;
{
	int  n,nn;
	struct sbuf *sb = &so->so_rcv;
	int len = sb->sb_cc;
	struct iovec iov[2];
	
#ifdef DEBUG
	debug_call(dfd,"sowrite(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	
	if (so->so_urgc) {
		sosendoob(so);
		if (sb->sb_cc == 0)
			return 0;
	}

	/*
	 * No need to check if there's something to write,
	 * sowrite wouldn't have been called otherwise
	 */
	
        len = sb->sb_cc;
	
	iov[0].iov_base = sb->sb_rptr;
	if (sb->sb_rptr < sb->sb_wptr)  {
		iov[0].iov_len = sb->sb_wptr - sb->sb_rptr;
		/* Should never succeed, but... */
		if (iov[0].iov_len > len) iov[0].iov_len = len;
		n = 1;
	} else  {
		iov[0].iov_len = (sb->sb_data + sb->sb_datalen) - sb->sb_rptr;
		if (iov[0].iov_len > len) iov[0].iov_len = len;
		len -= iov[0].iov_len;
		if (len)  {
			iov[1].iov_base = sb->sb_data;
			iov[1].iov_len = sb->sb_wptr - sb->sb_data;
			if (iov[1].iov_len > len) iov[1].iov_len = len;
			n = 2;
		} else
			n = 1;
	}
	/* Check if there's urgent data to send, and if so, send it */

#ifdef HAVE_READV
	nn = writev(so->s, (const struct iovec *)iov, n);
	
# ifdef DEBUG
	debug_misc(dfd,"  ... wrote nn = %d bytes\n", nn);
# endif
#else
	nn = write(so->s, iov[0].iov_base, iov[0].iov_len);
#endif
	/* This should never happen, but people tell me it does *shrug* */
	if (nn < 0 && (errno == EAGAIN || errno == EINTR))
		return 0;
	
	if (nn <= 0) {
#ifdef DEBUG
		debug_misc(dfd," --- sowrite disconnected, so->so_state = %x, errno = %d\n",
			so->so_state, errno);
#endif
		sofcantsendmore(so);
		tcp_sockclosed(sototcpcb(so));
		return -1;
	}
	
#ifndef HAVE_READV
	if (n == 2 && nn == iov[0].iov_len)
	   nn += write(so->s, iov[1].iov_base, iov[1].iov_len);
#ifdef DEBUG
        debug_misc(dfd,"  ... wrote nn = %d bytes\n", nn);
#endif
#endif
	
	/* Update sbuf */
	sb->sb_cc -= nn;
	sb->sb_rptr += nn;
	if (sb->sb_rptr >= (sb->sb_data + sb->sb_datalen))
		sb->sb_rptr -= sb->sb_datalen;
	
	/*
	 * If in DRAIN mode, and there's no more data, set
	 * it CANTSENDMORE
	 */
	if ((so->so_state & SS_FWDRAIN) && sb->sb_cc == 0)
		sofcantsendmore(so);
	
	return nn;
}

/*
 * recvfrom() a UDP socket
 */
void
sorecvfrom(so)
	struct socket *so;
{
	struct sockaddr_in addr;
	int addrlen = sizeof(struct sockaddr_in);
	struct mbuf *m;
	int len, n;

#ifdef DEBUG
	debug_call(dfd,"sorecvfrom(so) called\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	
	if ((m = m_get()) == NULL)
		return;
	m->m_data += ifp.ifp_maxlinkhdr;
	
	/* 
	 * XXX Shouldn't FIONREAD packets destined for port 53,
	 * but I don't know the max packet size for DNS lookups
	 */
	len = M_ROOM(m);
/*	if (so->so_fport != htons(53)) {
 *		int n;
 */		
	ioctl(so->s, FIONREAD, &n);
	
	if (n > len) {
		n = (m->m_data - m->m_dat) + m->m_len + n + 1;
		m_inc(m, n);
		len = M_ROOM(m);
	}
/*	} */

	m->m_len = recvfrom(so->s, m->m_data, len, 0,
		     (struct sockaddr *)&addr, &addrlen);
#ifdef DEBUG
	debug_misc(dfd," did recvfrom %d, errno = %d\n", m->m_len, errno);	
#endif

/*	if (m->m_len == len) {
 *		m_inc(m, MINCSIZE);
 *		m->m_len = 0;
 *	}
 */	
	/*
	 * Hack: domain name lookup will be used the most for UDP,
	 * and since they'll only be used once there's no need
	 * for the 4 minute (or whatever) timeout... So we time them
	 * out much quicker (10 seconds  for now...)
	 */
	if (so->so_expire) {
		if (so->so_fport == htons(53))
			so->so_expire = curtime + SO_EXPIREFAST;
		else
			so->so_expire = curtime + SO_EXPIRE;
	}
	
	if (m->m_len > 0)
		udp_output(so, m, &addr);
}

/*
 * sendto() a socket
 */
int
sosendto(so, m)
	struct socket *so;
	struct mbuf *m;
{
	int ret;
	struct sockaddr_in addr;

#ifdef DEBUG
	debug_call(dfd,"sosendto() called ...\n");
	debug_call(dfd,"  so = %d\n    m = %d\n", (int)so, (int)m);
	fflush_call(dfd);
#endif

        addr.sin_family = AF_INET;
	addr.sin_port = so->so_fport;
	addr.sin_addr = so->so_faddr;

#ifdef DEBUG
	debug_misc(dfd," sendto()ing, addr.sin_port = %d, addr.sin_addr.s_addr %x\n", addr.sin_port,
		(int)addr.sin_addr.s_addr);
#endif
	/* Don't care what port we get */
	ret = sendto(so->s, m->m_data, m->m_len, 0,
		     (struct sockaddr *)&addr, sizeof (struct sockaddr));
	if (ret < 0)
		return -1;
	
	/*
	 * Kill the socket if there's no reply in 4 minutes,
	 * but only if it's an expirable socket
	 */
	if (so->so_expire)
		so->so_expire = curtime + SO_EXPIRE;
	so->so_state = SS_ISFCONNECTED; /* So that it gets select()ed */
	return 0;
}

/*
 * XXX This should really be tcp_listen
 */
struct socket *
solisten(port, laddr, lport, flags)
	u_int port;
	u_int32_t laddr;
	u_int lport;
	int flags;
{
	struct sockaddr_in addr;
	struct socket *so;
	int s, addrlen = sizeof(addr), opt = 1;

#ifdef DEBUG
	debug_call(dfd,"solisten(port,laddr,lport,flags) called ...\n");
	debug_call(dfd,"    port = %d\n    laddr = %x\n    lport = %d\n    flags = %d\n",
		port, (u_int)laddr, lport, flags);
	fflush_call(dfd);
#endif
	
	if ((so = socreate()) == NULL)  {
		free(so); /* Not sofree() */
		return NULL;
	}
	
	/* Don't tcp_attach... we don't need so_snd nor so_rcv */
	if ((so->so_tcpcb = tcp_newtcpcb(so)) == NULL) {
		free(so);
		return NULL;
	}
	insque(so,&tcb);
	
	/* 
	 * SS_FACCEPTONCE sockets must time out.
	 */
	if (flags & SS_FACCEPTONCE)
	   so->so_tcpcb->t_timer[TCPT_KEEP] = TCPTV_KEEP_INIT;
	
	so->so_state = (SS_FACCEPTCONN|flags);
	so->so_lport = lport; /* Kept in network format */
	so->so_laddr.s_addr = laddr; /* Ditto */
	
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = port;
	
	if (((s = socket(AF_INET,SOCK_STREAM,0)) < 0) ||
	    (bind(s,(struct sockaddr *)&addr, sizeof(addr)) < 0) ||
	    (listen(s,1) < 0)) {
		int tmperrno = errno; /* Don't clobber the real reason we failed */
		
		close(so->s);
		sofree(so);
		/* Restore the real errno */
		errno = tmperrno;
		return NULL;
	}
	setsockopt(s,SOL_SOCKET,SO_REUSEADDR,(char *)&opt,sizeof(int));
	setsockopt(s,SOL_SOCKET,SO_OOBINLINE,(char *)&opt,sizeof(int));
	
	getsockname(s,(struct sockaddr *)&addr,&addrlen);
	so->so_fport = addr.sin_port;
	if (addr.sin_addr.s_addr == 0 || addr.sin_addr.s_addr == inet_addr("127.0.0.1"))
	   so->so_faddr.s_addr = our_addr;
	else
	   so->so_faddr = addr.sin_addr;

	so->s = s;
	return so;
}

/* 
 * Data is available in so_rcv
 * Just write() the data to the socket
 * XXX not yet...
 */
void
sorwakeup(so)
	struct socket *so;
{

#ifdef DEBUG
	debug_call(dfd,"sorwakeup(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	
/*	sowrite(so); */
/*	FD_CLR(so->s,&writefds); */
}
	
/*
 * Data has been freed in so_snd
 * We have room for a read() if we want to
 * For now, don't read, it'll be done in the main loop
 */
void
sowwakeup(so)
	struct socket *so;
{
#ifdef DEBUG
	debug_call(dfd,"sowwakeup(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	
	/* Nothing, yet */
}

/*
 * Various session state calls
 * XXX Should be #define's
 * The socket state stuff needs work, these often get call 2 or 3
 * times each when only 1 was needed
 */
void
soisfconnecting(so)
	register struct socket *so;
{
#ifdef DEBUG
	debug_call(dfd,"soisfconnecting(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	so->so_state &= ~(SS_NOFDREF|SS_ISFCONNECTED|SS_FCANTRCVMORE|
			  SS_FCANTSENDMORE|SS_FWDRAIN);
	so->so_state |= SS_ISFCONNECTING; /* Clobber other states */
}

void
soisfconnected(so)
        register struct socket *so;
{
#ifdef DEBUG
	debug_call(dfd,"soisfconnected(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	so->so_state &= ~(SS_ISFCONNECTING|SS_FWDRAIN|SS_NOFDREF);
	so->so_state |= SS_ISFCONNECTED; /* Clobber other states */
}

void
sofcantrcvmore(so)
	struct  socket *so;
{
#ifdef DEBUG
	debug_call(dfd,"sofcantrcvmore(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	if ((so->so_state & SS_NOFDREF) == 0) {
		shutdown(so->s,0);
		FD_CLR(so->s,&writefds);
	}
	so->so_state &= ~(SS_ISFCONNECTING);
	if (so->so_state & SS_FCANTSENDMORE)
	   so->so_state = SS_NOFDREF; /* Don't select it */ /* XXX close() here as well? */
	else
	   so->so_state |= SS_FCANTRCVMORE;
}

void
sofcantsendmore(so)
	struct socket *so;
{
#ifdef DEBUG
	debug_call(dfd,"sofcantsendmore(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	if ((so->so_state & SS_NOFDREF) == 0) {
		shutdown(so->s,1);
		FD_CLR(so->s,&readfds);
	}
	so->so_state &= ~(SS_ISFCONNECTING);
	if (so->so_state & SS_FCANTRCVMORE)
	   so->so_state = SS_NOFDREF; /* as above */
	else
	   so->so_state |= SS_FCANTSENDMORE;
}

void
soisfdisconnected(so)
	struct socket *so;
{
#ifdef DEBUG
	debug_call(dfd,"soisfdisconnected(so) called ...\n");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
/*	so->so_state &= ~(SS_ISFCONNECTING|SS_ISFCONNECTED); */
/*	close(so->s); */
/*	so->so_state = SS_ISFDISCONNECTED; */
	/*
	 * XXX Do nothing ... ?
	 */
}

/*
 * Set write drain mode
 * Set CANTSENDMORE once all data has been write()n
 */
void
sofwdrain(so)
	struct socket *so;
{

#ifdef DEBUG
	debug_call(dfd,"sofwdrain(so) called ...");
	debug_call(dfd,"    so = %d\n", (int)so);
	fflush_call(dfd);
#endif
	if (so->so_rcv.sb_cc)
		so->so_state |= SS_FWDRAIN;
	else
		sofcantsendmore(so);
}

