/* DSP support routines for my programs.

*/


#include <stdio.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>

#include "dsp.h"


/* Audio buffer */
unsigned char * audiobuffer=NULL;
int audioindex=0;

/* Set verbose mode */
int verbose=1,abuf_size;

/* Allocate buffers */
void dsp_init(int audio) {

	ioctl(audio, SNDCTL_DSP_GETBLKSIZE, &abuf_size);
	if (abuf_size < 4096 || abuf_size > 65536) {
		if (abuf_size == -1)
			perror (AUDIO);
		else
			fprintf (stderr, "Invalid audio buffers size %d\n", abuf_size);
		exit (-1);
	}

	if(verbose)
		printf("Buffer size: %d\n",abuf_size);
 
	if((audiobuffer=(char*)malloc(abuf_size))==NULL) {
		fprintf(stderr,"Not able to allocate memory\n");
		perror(AUDIO);
		exit(1);
	}

}



/* if need a SYNC, 
   (is needed if we plan to change speed, stereo ... during output)
*/
void sync_dsp(int audio) {
	if (ioctl (audio, SNDCTL_DSP_SYNC, NULL) < 0) {
		fprintf (stderr, "Unable to sync audio\n");
		perror(AUDIO);
		exit (-1);
	}
}

/* setting the speed for output */
void set_dsp_speed (int audio, int dsp_speed) {
	if (ioctl(audio, SNDCTL_DSP_SPEED, &dsp_speed) < 0) {
		fprintf (stderr, "Unable to set audio speed\n");
		perror (AUDIO);
		exit (-1);
	}
	if(verbose)
		printf("Samplerate: %d\n",dsp_speed);

}

/* Set sampling size */
void set_dsp_samplesize(int audio, int samplesize) {
	int dsp_stereo;

	if(ioctl(audio, SNDCTL_DSP_SAMPLESIZE, &samplesize)<0) {
		fprintf(stderr,"Unable to set sample size\n");
		perror(AUDIO);
		exit(-1);
	}
	if(verbose)
		printf("Samplesize: %d\n",samplesize);
}



/* and there are MONO by default */
void set_dsp_mode(int audio, int dsp_stereo ) {
	dsp_stereo = MODE_MONO;
	if(ioctl(audio, SNDCTL_DSP_STEREO, &dsp_stereo)<0) {
		fprintf(stderr,"Unable to set mode\n");
		perror(AUDIO);
		exit(-1);
	}
	if(verbose)
		printf("Mode: %s\n", dsp_stereo==MODE_MONO ? "Mono" : "Stereo");
}

/* DSP buffer */


/* Write a sample to the dsp */
void write_dsp_byte(int audio, int sample) {
	if(audiobuffer==NULL) {
		fprintf(stderr,"Buffer not allocated\n");
		perror(AUDIO);
		exit(1);
	}

	/* Write to buffer first, then flush it */
	audiobuffer[audioindex++]=sample;

	/* Check if buffer is full, then flush */
	if(audioindex>=abuf_size) {
		if(write(audio,audiobuffer,abuf_size)<0) {
			fprintf(stderr,"Unable to write samples\n");
			perror(AUDIO);
			exit(-1);
		}
		audioindex=0;
	}
}

/* Read a sample from the DSP */
void read_dsp_byte(int audio, long * sample) {

	if(audiobuffer==NULL) {
		fprintf(stderr,"Buffer not allocated\n");
		perror(AUDIO);
		exit(-1);
	}

	if(audioindex>=abuf_size || audioindex==0) {
		if(read(audio,audiobuffer,abuf_size)<0) {
			fprintf(stderr,"Unable to read samples\n");
			perror(AUDIO);
			exit(-1);
		}
		audioindex=0;
	}
	*sample=audiobuffer[audioindex++];
}


/* Close the dsp */
void close_dsp(int audio) {
	if(write(audio,audiobuffer,audioindex+1)<0) {
		fprintf(stderr,"Unable to write samples\n");
		perror(AUDIO);
		exit(-1);
	}
	audioindex=0;
	audiobuffer=NULL;
	close(audio);
}

