/*
**
**	 #########	   #	   # ######### #
**		#	    #	  #  #	       #
**	      ##   #	 #   #	 #   #	       #
**	     #	   #	 #    ###    #######   #
**	   ##	   #	 #   #	 #   #	       #
**	  #	   #	 #  #	  #  #	       #
**	 #########  ###### #	   # ######### #########
**			 #
**			 #
**		    #####
**
**
**	A control program for the ZyXEL U-1496 series MODEM for use with the
**	Linux operating system (and other UNIX-like systems)
**
**	This program manages the use of the ZyXEL MODEM for:
**
**	- dial-in, for use by "uucp" and other programs started from
**	  a login
**
**	- dial-in security (checking username and password)
**
**	- dial-back security
**
**	- dial-out, e.g. using "cu", "uucp" or "seyon"
**
**	- receiving FAXes
**
**	- sending FAXes
**
**	- polling for FAXes
**
**	- handling incoming FAX polls
**
**	- answering machine (voice mailbox)
**
**	Everything is done in a single program, to keep things simple
**	and fast.  Multi-program solutions are available from others.
**	(e.g. modgetty, mgetty+sendfax)
**
**	Copyright 1993-1995 by Rob Janssen, PE1CHL
**
**	This program uses parts of "getty_ps", written by Paul Sutcliffe:
**	Copyright 1989,1990 by Paul Sutcliffe Jr.
**
**	Permission is hereby granted to copy, reproduce, redistribute,
**	or otherwise use this software as long as: there is no monetary
**	profit gained specifically from the use or reproduction or this
**	software, it is not sold, rented, traded or otherwise marketed,
**	and this copyright notice is included prominently in any copy
**	made.
**
**	The author make no claims as to the fitness or correctness of
**	this software for any use whatsoever, and it is provided as is.
**	Any use of this software is at the user's own risk.
*/

/*************************************************************************
**	User configuration section					**
*************************************************************************/


/*  Feature selection
 */

#undef	ZFAX				/* use AT# FAX commands (vs CLASS 2) */
#define ASCIIPID			/* PID stored in ASCII in lock file */
#define BOTHPID				/* ... or perhaps not */
#define DOUNAME				/* use uname() to get hostname */
#define FIDO				/* allow fido logins */
#define LOGFILE				/* log to file */
#define LOGUTMP				/* need to update utmp/wtmp files */
#define SETTERM				/* need to set TERM in environment */
#undef	TRYMAIL				/* mail logmesgs if CONSOLE unavailable */
#define ULAW_AUDIO			/* SYNTH uses u-law vs. unsigned data */

#define CONSOLE		"/dev/console"	/* place to print fatal problems */

#ifdef LOGFILE
#undef LOGFILE
#define LOGFILE		"/var/adm/%s.log"	/* place to log events and errors */
						/* %s gets invocation name */
#else
#define LOGFILE		CONSOLE
#endif

#ifdef SVR4LOCK
#define LOCK		"/usr/spool/locks/LK.%03d.%03d.%03d" /* SysV R4 */
#else
#define LOCK		"/usr/spool/uucp/LCK..%s" /* others */
#endif

#define DEFAULTS	"/etc/default/%s"	/* name of defaults file */
#define ISSUE		"/etc/issue"		/* name of the issue file */
#define LOGIN		"/etc/login"		/* name of login program */
#define PHOSTNAME	"/bin/hostname"		/* prints our hostname */
#define TTYTYPE		"/etc/ttytype"		/* terminal types */
#define UUCPNAME	"uucp"			/* uucp user in passwd file */

#define MAILER		"/bin/mail"
#define NOTIFY		"root"


/* const is buggy in SCO XENIX */

#ifdef M_XENIX
# define const
#endif

/*************************************************************************
**	Include files...  Just about everything you can get		**
*************************************************************************/

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include <sys/types.h>
#ifdef __linux__
# include <sys/ioctl.h>
#endif
#ifndef M_XENIX
# include <sys/mman.h>
# include <sys/time.h>
#endif
#include <sys/stat.h>
#include <sys/utsname.h>
#include <sys/wait.h>
#ifdef SVR4LOCK
# include <sys/mkdev.h>
#endif

#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <malloc.h>
#include <pwd.h>
#include <setjmp.h>
#include <signal.h>
#include <termio.h>
#include <time.h>
#include <utmp.h>
#ifdef _GNU_SOURCE
# include <getopt.h>
#endif

#ifdef __linux__				/* for highspeed serial stuff */
# ifdef OLD_TTY_DRIVER
#  include <linux/fs.h>				/* these are for the */
#  include <linux/tty.h>			/* ... old tty driver */
# else
#  include <linux/serial.h>			/* this is for new tty driver */
# endif
#endif

#ifndef __USE_BSD
/* directory scan functions not available in SysV */
extern int scandir(const char *__dir,
		   struct dirent ***__namelist,
		   int (*__select) (struct dirent *),
		   int (*__cmp) (const void *, const void *));
extern int alphasort(const void *, const void *);
#endif

/* version/whatstring */

static const char *Id = "@(#)ZyXEL 1.5 (" __DATE__ " " __TIME__ ")   (c) 1993,1994 by Rob Janssen, PE1CHL";

/* macro definitions */

#define FALSE		(0)
#define TRUE		(1)

#define OK		(0)

#define SUCCESS		(0)	/* normal return */
#define FAIL		(-1)	/* error return */

#define STDIN		0
#define STDOUT		1
#define STDERR		2

				/* modemlines to be monitored and state */
#define MODEMLINES	(TIOCM_DTR|TIOCM_RTS|TIOCM_CTS|TIOCM_CAR|TIOCM_DSR)
#define MODEMLN_OK	(TIOCM_DTR|TIOCM_RTS|TIOCM_CTS)

#ifndef CRTSCTS
#define CRTSCTS		0
#endif

#define SHOWMSG_AA	1	/* show messages present by turning AA on */

#define EXPFAIL		30	/* default num seconds to wait for expected input */

#define EXPECT		0	/* states for chat() */
#define SEND		1

#define INITIALIZE	0	/* indicators for settermio() */
#define INITIAL		1
#define DROPDTR		2
#define SHORTTIME	3
#define SHORT255	4
#define MEDMTIME	5
#define LONGTIME	6
#define XONXOFF		7
#define IMMEDIATE	8
#define WATCHDCD	9
#define SEVEN_E		10
#define FINAL		11

#define ONLINE		0	/* indicators for datamode(),faxmode(),voicemode() */
#define ANSWER		1
#define ANSWERDATA	2
#define FAXPOLL		3

#define NO1496		1	/* Return values for initmodem() */

#define INITFAIL	1	/* Return values for wait_work() */
#define LOCKED		2
#define FIFO		3
#define CONNECT		4
#define NOCARRIER	5
#define RING		6
#define VCON		7

#define INCORRECT	11	/* Return values for protect() */

#define DATAFAX		21	/* Return values for datamode(),voicemode() */
#define DATA		22
#define FAX		23
#define VOICE		24
#define DIALBACK	25

#define EXIT		31	/* Return values for voicecommand() */
#define COMMAND		32
#define UNKNOWN		33
#define ERROR		34

#define DIALFAIL	41	/* Return values for faxpoll(),faxsend() */
#define CONNECTFAIL	42
#define NOPOLL		43
#define XFERFAIL	44	/* also from faxmode() */

#define BADSPEED	51	/* Return values for getlogname() */
#define BADCASE		52
#define NONAME		53
#define FIDOCALL	54
#define BBSCALL		55

#define MAXDEF		64	/* max # lines in defaults file */
#define MAXLINE		256	/* max # chars in a line */
#define MAXBUF		1024	/* buffer size */
#define NARG		20	/* max # of args on a command line */

#define TSYNC		0xAE	/* Fido stuff (chars recognized at login:) */
#define YOOHOO		0xF1
#define ESC		0x1B
#define BS		0x08
#define DEL		0x7F

#define ETX		003	/* ZyXEL voice data stuff */
#define DLE		020
#define XON		021
#define DC2		022
#define XOFF		023
#define DC4		024
#define CAN		030
#undef FS
#define FS		034

#ifndef CEOF
#define CEOF		004	/* ^D */
#define CERASE		010	/* ^H */
#define CINTR		003	/* ^C */
#define CKILL		030	/* ^X */
#define CQUIT		034	/* ^\ */
#define CSTART		021	/* ^Q */
#define CSTOP		023	/* ^S */
#define CSUSP		032	/* ^Z */
#define CRPRNT		022	/* ^R */
#define CWERASE		027	/* ^W */
#define CLNEXT		026	/* ^V */
#endif

#define strncopy(s1,s2) (strncpy(s1,s2,sizeof(s1)))
#define getch()		(InPtr >= InLen && (InLen = read(STDIN,InBuf,BUFSIZ)) <= (InPtr = 0)? EOF : (int)(unsigned char)InBuf[InPtr++])
#define getapi()	(ApiPtr >= ApiLen && (ApiLen = read(FifoFd,ApiBuf,BUFSIZ)) <= (ApiPtr = 0)? EOF : (int)(unsigned char)ApiBuf[ApiPtr++])

/* debug levels
 */
#define D_OPT		0001	/* option settings */
#define D_DEF		0002	/* defaults file processing */
#define D_UTMP		0004	/* utmp/wtmp processing */
#define D_CHAT		0010	/* MODEM chat (send/expect/waitfor) */
#define D_FAX		0020	/* FAX responses */
#define D_GETL		0040	/* get login name routine */
#define D_RUN		0100	/* other runtime diagnostics */
#define D_LOCK		0200	/* lockfile processing */

#ifdef	DEBUG

/* debug defs
 */
#define debug1(a,b)		dprint(a,b)
#define debug2(a,b)		debug(a,b)
#define debug3(a,b,c)		debug(a,b,c)
#define debug4(a,b,c,d)		debug(a,b,c,d)
#define debug5(a,b,c,d,e)	debug(a,b,c,d,e)
#define debug6(a,b,c,d,e,f)	debug(a,b,c,d,e,f)

#else	/* DEBUG */

#define debug1(a,b)		/* define to nothing, disables debugging */
#define debug2(a,b)
#define debug3(a,b,c)
#define debug4(a,b,c,d)
#define debug5(a,b,c,d,e)
#define debug6(a,b,c,d,e,f)

#endif	/* DEBUG */

/* data definitions */

typedef int	boolean;	/* useless crap */
typedef void	sig_t;		/* type of a signal handler */

/* lines in defaults file are in the form "NAME=value"
 */

typedef struct Default {
	char	*name;		/* name of the default */
	char	*value;		/* value of the default */
} DEF;

#ifdef ZFAX
/* header of .fax (ZyXEL FAX) files
 * this assumes lo-hi byte ordering and 16-bit shorts :-(
 */

typedef struct fax_header {
	char zyxel[6];		/* magic string */
#define FAX_MAGIC	"ZyXEL" /* includes \0 */
	short factory;		/* always 'Z' */
	short version;		/* always 2 */
	short rec_width;	/* recording width 1728/2048/2432 */
#define FAX_R0		1728
#define FAX_R1		2048
#define FAX_R2		2432
	short pages;		/* number of pages */
	short flags;
#define FAX_V1		0x0001	/* high resolution */
#define FAX_T1		0x0002	/* 2-D coding */
} FAX_HEADER;

const char FaxExt[] = ".fax";	/* the extension */
#else
/* CLASS 2 FAX information received in +F responses
 */

typedef struct fax_info {
	int cfr;		/* +FCFR received */
	int et;			/* +FET post message response */
	int hng;		/* +FHNG hangup response */
	int poll;		/* +FPOLL received */
	int dtc;		/* +FDTC: received (poll request) */

	/* info from +FPTS */
	int pts;		/* page transfer status */
	int lc;			/* line count */
	int blc;		/* bad line count */
	int cblc;		/* consecutive bad line count */
	int lbc;		/* lost byte count */

	/* info from +FDIS, +FDCS, +FDTC */
	int vr;			/* vertical resolution */
	int br;			/* bitrate */
	int wd;			/* page width */
	int ln;			/* page length */
	int df;			/* data format */
	int ec;			/* error correcting mode */
	int bf;			/* binary file transfer */
	int st;			/* scan time */

	/* info from +FCSI, +FTSI, +FCIG */
	char id[30];		/* identification string */

	/* info from +FNSC */
	int nstyp;		/* type of nonstandard frame (F, S, C) */
	int nslen;		/* length of nonstandard frame */
	unsigned char ns[64];	/* nonstandard frame */
} FAX_INFO;

const char FaxExt[] = ".g3";	/* the extension */
#endif

/* header of .zad (ZyXEL audio) files
 * this assumes lo-hi byte ordering and 16-bit shorts :-(
 */

typedef struct zad_header {
	char zyxel[6];		/* magic string */
#define ZAD_MAGIC	"ZyXEL\2"
	short version;
	short reserved1;
	short enc;		/* audio encoding (compression) scheme */
#define ENC_CELP	0x00
#define ENC_ADPCM2	0x01
#define ENC_ADPCM3	0x02
#define ENC_ADPCM4	0x03	/* AT+VSM=4 (from 6.13) */
#define ENC_RESYNC	0x04	/* this bit indicates resync is used */
#define ENC_ADPCM3N	0x08	/* AT+VSM=30 (from 6.13) */
	short reserved2;
	short reserved3;
} ZAD_HEADER;

const char AudioExt[] = ".zad"; /* the extension */



/* variables */

#ifdef	DEBUG
int	Debug;			/* debug value from command line */
FILE	*Dfp;			/* debugging file pointer */
#endif

boolean FinalXoff = TRUE;	/* set IXON and IXANY after logging in */
boolean Is_ZyXEL;		/* MODEM seems to be a ZyXEL U-1496 */
boolean LockPoll;		/* poll for lockfiles while IDLE? */
boolean LoginEnv = TRUE;	/* pass environment as args to LOGIN? */
boolean ModemPoll;		/* poll the MODEM while IDLE? */
boolean NoHangUp;		/* don't hangup line before setting speed */
boolean No_nonblock;		/* Don't use O_NDELAY */
boolean No_ZyXEL;		/* Don't enforce ZyXEL MODEM */
boolean WarnCase;		/* controls display of bad case message */
char	*AnswerTone;		/* VOICE mode answer tone */
char	*Archive;		/* archiving directory */
#ifndef ZFAX
char	*Buffer;		/* Pathname of buffering program */
#endif
char	*CndName;		/* last received CND NAME */
char	*CndNumber;		/* last received CND NUMBER */
char	*Connect;		/* last received CONNECT message from MODEM */
char	*DefName;		/* defaults file name */
char	*Device;		/* controlling line (minus "/dev/") */
char	*DevName;		/* controlling line including "/dev/" */
char	*DialPrefix;		/* sent before each phone number */
char	*Error;			/* error beep for voice mode */
char	*FaxDir;		/* Incoming FAX directory */
char	*FaxFile = "test.g3";	/* Incoming FAX file */
char	*FaxId;			/* FAX ID (phone number) */
char	*FaxMail;		/* Address to send 'fax arrived' mail msgs */
#ifndef ZFAX
char	*FaxPoll;		/* file to send when polled for FAX */
#endif
#ifdef	FIDO
char	*Fido;			/* fido protocol program */
#endif	/* FIDO */
char	*Fifo;			/* API FIFO configured name */
char	*FifoFile;		/* API FIFO full pathname */
#ifndef ZFAX
char	*GhostScript;		/* Pathname of Ghostscript program */
#endif
char	*Greeting = "greeting"; /* Greeting for VOICE mode */
#ifndef ZFAX
char	*Gunzip;		/* Pathname of Gunzip program */
#endif
char	*Init;			/* value of INIT */
char	*Issue = ISSUE;		/* default issue file */
char	*Lock;			/* lockfile name */
char	*Login = LOGIN;		/* default login program */
char	*LoginFlags;		/* flags for login program */
char	*Match;			/* location in ChatBuf where waitfor matched */
char	*Mute = "M1";		/* mute command (Lx/Mx/Nx) sent in DATA/FAX mode */
char	*MyName;		/* this program name */
char	*PagerNum;		/* telephone number of PAGER */
char	*PlayDir;		/* directory of voice play files */
char	*ProdInfo;		/* MODEM product info (ATI) */
char	*Prompt1;		/* prompt beep for DTMF digit entry */
char	*Prompt2;		/* prompt beep for DTMF code entry */
char	*Protect;		/* name of "protected login" file */
#ifndef ZFAX
char	*PsText;		/* Pathname of PsText program */
#endif
char	*Ready;			/* ready beep (after startup) */
char	*RecAttn;		/* attention beep */
char	*RecDir;		/* recording directory */
char	*RecDone;		/* recording done beep */
char	*RecFile = "test";	/* recording file */
char	*Silence;		/* silence-detect parameters */
char	*StateFile = "/tmp/ZyXEL";	/* file holding current state */
char	*Synth;			/* voice synthesizer executable program */
char	*SynthFlags;		/* flags for synthesizer program */
char	*SysName;		/* nodename of system */
char	*Version;		/* value of VERSION */
char	*VoiceCmd;		/* Voice expert command file */
char	*VoiceKey;		/* Voice DTMF keys file */
char	ApiBuf[BUFSIZ];		/* read() buffer for API FIFO */
char	ChatBuf[MAXBUF];	/* result from last chat() */
char	CurVSD[20];		/* current setting of VSD= */
char	InBuf[BUFSIZ];		/* read() buffer for STDIN */
char	MsgBuf[MAXLINE];	/* message buffer */
char	Number[80];		/* from readnumber command */
char	Recordfile[MAXLINE];	/* 'current recording file' for voice mailbox */
#ifdef	SETTERM
char	Term[16];		/* terminal type */
#endif	/* SETTERM */
int	ApiLen;			/* ApiBuf current length */
int	ApiPtr;			/* ApiBuf character ptr */
int	ApiUserLevel;		/* default userlevel for users on API FIFO */
int	Cbaud;			/* decoded baudrate field */
int	CurLS;			/* currently set LS */
int	CurSync;		/* currently set resync mode (voice) */
int	CurVSM;			/* currently set VSM */
int	DropMsg;		/* drop messages at or below this duration */
#ifndef ZFAX
int	FaxComp = 1;		/* supported FAX compression */
#endif
int	FifoFd = -1;		/* API FIFO file descriptor */
int	Fw_plus;		/* Firmware indicates PLUS version? */
int	Fw_version;		/* Firmware version number */
int	InLen;			/* InBuf current length */
int	InPtr;			/* InBuf character ptr */
int	Mode = DATAFAX;		/* answer mode */
int	Nrings = 4;		/* required number of rings to answer */
int	Nusers;			/* number of users currently logged in */
int	PagerType;		/* type of PAGER (dial algorithm) */
int	RecLS;			/* recording LS (0 = current) */
int	RecMode = ENC_CELP;	/* recording mode */
int	RingBack;		/* ringback datamode enabled */
int	ShowMsg;		/* show "messages recorded" */
int	Speed;			/* current baud rate */
int	TimeOut;		/* timeout value from command line */
int	UserLevel;		/* userlevel for current VOICE user */
int	WFclass = 8;		/* FCLASS while waiting for work */
long	Samples;		/* samples read from pipe in say() */
long	Signal;			/* last caught signals */
long	Leakage;		/* for ADPCM coder */
short	Delta;			/* for ADPCM coder */
short	EstMax;			/* for ADPCM coder */
struct termios Termio;		/* termios used for the tty */
struct timeval MatchTime;	/* time of last matched input */
time_t	HoldTime;		/* holdtime for selected state (mode) */
uid_t	UuUid;			/* UUCP's user-id */
gid_t	UuGid;			/* UUCP's group-id */
#ifdef __linux__
struct serial_struct Serial;	/* high-speed and other serial details */
#endif

#ifdef ZFAX
/* RTC strings (end-of-page) for 1-D and 2-D FAX reception */

const unsigned char fax_rtc_1d[] = {0x00,0x08,0x80,0x00,0x08,0x80,0x00,0x08,0x80};
#define RTC_1D_LEN	9
const unsigned char fax_rtc_2d[] = {0x00,0x18,0x00,0x03,0x00,0x06,0x00,0x0c,0x80,0x01,0x30,0x00};
#define RTC_2D_LEN	12
#else				/* Class 2 */
FAX_INFO FaxInfo;		/* info about current FAX connection */

/* RTC string (end-of-page) */

const char fax_rtc[] = {0x00,0x10,0x01,0x00,0x10,0x01,0x00,0x10};
#define RTC_LEN		8

/* blank FAX_INFO structure used as a startoff */

const FAX_INFO blank_info = {
	0, -1, -1, 0, 0,	/* cfr, et, hng, poll, dtc */
	-1, -1, -1, -1, -1,	/* pts, lc, blc, cblc, lbc */
	-1, -1, -1, -1, -1, -1, -1, -1, /* dcs params */
	"",			/* ID */
	0, 0, ""		/* nstyp, nslen, ns */
};

/* nonstandard facility used to send polling ID when polling for FAXes */

const unsigned char poll_nsc[] = {
	0xad,0x00,0x12,0x00,0x00,0x50,0x4f,0x4c,0x4c,0x2d,0x49,0x44,0x3a
};

const char poll_hex[] = "AD00120000504F4C4C2D49443A";

/* bit reverse table (some firmware revisions can't do it...) */

const unsigned char rev_bits[] = {
	0x00,0x80,0x40,0xc0,0x20,0xa0,0x60,0xe0,0x10,0x90,0x50,0xd0,0x30,0xb0,0x70,0xf0,
	0x08,0x88,0x48,0xc8,0x28,0xa8,0x68,0xe8,0x18,0x98,0x58,0xd8,0x38,0xb8,0x78,0xf8,
	0x04,0x84,0x44,0xc4,0x24,0xa4,0x64,0xe4,0x14,0x94,0x54,0xd4,0x34,0xb4,0x74,0xf4,
	0x0c,0x8c,0x4c,0xcc,0x2c,0xac,0x6c,0xec,0x1c,0x9c,0x5c,0xdc,0x3c,0xbc,0x7c,0xfc,
	0x02,0x82,0x42,0xc2,0x22,0xa2,0x62,0xe2,0x12,0x92,0x52,0xd2,0x32,0xb2,0x72,0xf2,
	0x0a,0x8a,0x4a,0xca,0x2a,0xaa,0x6a,0xea,0x1a,0x9a,0x5a,0xda,0x3a,0xba,0x7a,0xfa,
	0x06,0x86,0x46,0xc6,0x26,0xa6,0x66,0xe6,0x16,0x96,0x56,0xd6,0x36,0xb6,0x76,0xf6,
	0x0e,0x8e,0x4e,0xce,0x2e,0xae,0x6e,0xee,0x1e,0x9e,0x5e,0xde,0x3e,0xbe,0x7e,0xfe,
	0x01,0x81,0x41,0xc1,0x21,0xa1,0x61,0xe1,0x11,0x91,0x51,0xd1,0x31,0xb1,0x71,0xf1,
	0x09,0x89,0x49,0xc9,0x29,0xa9,0x69,0xe9,0x19,0x99,0x59,0xd9,0x39,0xb9,0x79,0xf9,
	0x05,0x85,0x45,0xc5,0x25,0xa5,0x65,0xe5,0x15,0x95,0x55,0xd5,0x35,0xb5,0x75,0xf5,
	0x0d,0x8d,0x4d,0xcd,0x2d,0xad,0x6d,0xed,0x1d,0x9d,0x5d,0xdd,0x3d,0xbd,0x7d,0xfd,
	0x03,0x83,0x43,0xc3,0x23,0xa3,0x63,0xe3,0x13,0x93,0x53,0xd3,0x33,0xb3,0x73,0xf3,
	0x0b,0x8b,0x4b,0xcb,0x2b,0xab,0x6b,0xeb,0x1b,0x9b,0x5b,0xdb,0x3b,0xbb,0x7b,0xfb,
	0x07,0x87,0x47,0xc7,0x27,0xa7,0x67,0xe7,0x17,0x97,0x57,0xd7,0x37,0xb7,0x77,0xf7,
	0x0f,0x8f,0x4f,0xcf,0x2f,0xaf,0x6f,0xef,0x1f,0x9f,0x5f,0xdf,0x3f,0xbf,0x7f,0xff
};
#endif

const char AudioEnd[] = { DLE, ETX, '\0' };		/* end audio */

char AudioAbort[] = { ETX, DLE, ETX, '\r', '\0' };	/* abort audio */
/*				^^^  this byte overwritten with DC4 when >= 6.11 */

const short MaxTbl[][8] =				/* used in ADPCM conversion */
{
	/* dummy entry to make ENC_XXX the index into this table */
	{ 0 },

	/* ADPCM2, 6.10 or later */
	{ 0x3800, 0x5600 },

	/* ADPCM3, 6.10 or later */
	{ 0x399A, 0x3A9F, 0x4D14, 0x6607 },

	/* ADPCM4, 6.13 or later */
	{ 0x3556, 0x3556, 0x399A, 0x3A9F, 0x4200, 0x4D14, 0x6607, 0x6607 },
};

#ifdef ULAW_AUDIO
const short Ulaw[] =					/* u-law -> linear */
{
	0x1f5f, 0x1e5f, 0x1d5f, 0x1c5f, 0x1b5f, 0x1a5f, 0x195f, 0x185f,
	0x175f, 0x165f, 0x155f, 0x145f, 0x135f, 0x125f, 0x115f, 0x105f,
	0x0f9f, 0x0f1f, 0x0e9f, 0x0e1f, 0x0d9f, 0x0d1f, 0x0c9f, 0x0c1f,
	0x0b9f, 0x0b1f, 0x0a9f, 0x0a1f, 0x099f, 0x091f, 0x089f, 0x081f,
	0x07bf, 0x077f, 0x073f, 0x06ff, 0x06bf, 0x067f, 0x063f, 0x05ff,
	0x05bf, 0x057f, 0x053f, 0x04ff, 0x04bf, 0x047f, 0x043f, 0x03ff,
	0x03cf, 0x03af, 0x038f, 0x036f, 0x034f, 0x032f, 0x030f, 0x02ef,
	0x02cf, 0x02af, 0x028f, 0x026f, 0x024f, 0x022f, 0x020f, 0x01ef,
	0x01d7, 0x01c7, 0x01b7, 0x01a7, 0x0197, 0x0187, 0x0177, 0x0167,
	0x0157, 0x0147, 0x0137, 0x0127, 0x0117, 0x0107, 0x00f7, 0x00e7,
	0x00db, 0x00d3, 0x00cb, 0x00c3, 0x00bb, 0x00b3, 0x00ab, 0x00a3,
	0x009b, 0x0093, 0x008b, 0x0083, 0x007b, 0x0073, 0x006b, 0x0063,
	0x005d, 0x0059, 0x0055, 0x0051, 0x004d, 0x0049, 0x0045, 0x0041,
	0x003d, 0x0039, 0x0035, 0x0031, 0x002d, 0x0029, 0x0025, 0x0021,
	0x001e, 0x001c, 0x001a, 0x0018, 0x0016, 0x0014, 0x0012, 0x0010,
	0x000e, 0x000c, 0x000a, 0x0008, 0x0006, 0x0004, 0x0002, 0x0000
};
#endif

const char *bad_case[] = {			/* UPPER-CASE-LOGIN warning */
	"\n",
	"If your terminal supports lower case letters, please\n",
	"use them.  Login again, using lower case if possible.\n",
	NULL
};


/* function prototypes */

DEF	**defbuild(char *filename);
DEF	*defread(FILE *fp);
FILE	*defopen(char *filename);
boolean checklock(char *name);
boolean expmatch(const char *got,const char *exp);
char	*apigets(void);
char	*defvalue(DEF **deflist,const char *name);
char	*dtmfstring(void);
char	*getuname(void);
const char *unquote(const char *s,char *c);
int	Fputs(const char *s,int fd);
int	adpcm(int enc,int edata);
int	api_eol(void);
int	api_mode(void);
int	apigetc(void);
int	auplay(char *filename);
int	auread(int fd,char *buf,size_t size);
int	ausend(int fd,int (*rdf)(int fd,char *buf,size_t size));
int	beep(const char *s);
int	chat(char *s);
int	datamode(int answer);
int	defclose(FILE *fp);
int	dialup(const char *mode,int fclass,const char *number);
int	dir_select(struct dirent *dirent);
int	docommand(char *line,int (*inputdigit)(void),char *(*inputstring)(void));
int	dtmfcode(void);
int	dtmfdigit(void);
int	dtmfrecord(const char *end,int maxtime);
int	execlogin(char *login,char *flags,char *username);
int	expect(const char *s);
int	fax2rsp(int logfd);
int	faxcpoll(int logfd);
int	faxmode(int answer);
int	faxpoll(char *number,char *pollid,int logfd);
int	faxrecv(int answer,int logfd);
int	faxsend(char *number,char *filename,int logfd);
int	filespresent(const char *dirname);
int	getlogname(struct termios *termio,char *name,int size);
int	getpasswd(char *passwd,int size);
int	initmodem(int dropdtr);
int	login_user(void);
int	main(int argc,char *argv[]);
int	makelock(char *name);
int	modemstat(void);
int	openline(void);
int	pager(int type,const char *number,const char *info);
int	play(const char *filename,const char *end);
int	playback(char *direc,int (*inputdigit)(void));
int	push(pid_t *pid,int *fd,char *program,char *argv[]);
int	protect(char *username);
int	read8k(int fd,char *buf,size_t size);
int	read_state(int ringtype,int altcount);
int	readlock(char *name);
int	record(char *filename,const char *end,const char *vsd,int enc,int maxtime);
int	rip(char *string);
int	run(char *argv[]);
int	say(const char *string);
int	send(const char *s);
int	set_vls(int ls);
int	set_vsm(int enc);
int	startsynth(pid_t *pid,int *ifd,int *ofd);
int	voicecommand(char *filename,int code,int (*inputdigit)(void),char *(*inputstring)(void));
int	voicekey(int firstdigit);
int	voicemode(int answer);
int	wait_work(int fclass);
int	waitfor(const char *word,int state);
sig_t	expalarm(int);
sig_t	huphandler(int);
sig_t	inthandler(int);
sig_t	sighandler(int);
sig_t	timeout(int);
void	closeline(void);
void	debug(int lvl,const char *fmt,...);
void	dprint(int lvl,const char *word);
void	exit_usage(int code);
void	getvalues(void);
void	hold_state(int duration,int mode,int nrings);
void	logmesg(const char *message);
void	rmlocks(void);
void	settermio(int state);
#ifdef ZFAX
void	faxheader(FAX_HEADER *header,char *connect);
#endif
#ifdef SA_SIGINFO
sig_t	sigcatcher(int,siginfo_t *);
void	catchsig(int);
#endif


/*************************************************************************
**	Functions...							**
*************************************************************************/

/*
** main: where it all starts
**
** we are called from "init" as: /etc/ZyXEL [options] device speed [term]
** example: /etc/ZyXEL ttyS1 38400
*/

int
main(argc,argv)
int argc;
char *argv[];
{
	FILE *fp;
	int c;
	pid_t pid;
	struct passwd *pwd;
	struct utmp *utmp;
	struct stat st;
	char buf[MAXLINE+1];

	/* startup
	 */
	nice(-9);				/* real-time execution */
	srand((int)time(NULL));

#ifdef SA_SIGINFO
	for (c = 0; c < NSIG; c++)		/* catch all signals... */
	    catchsig(c);			/* so that we can log info */
#else
	signal(SIGQUIT, SIG_DFL);		/* make sure we can quit */
	signal(SIGTERM, SIG_DFL);
#endif

#ifdef	SETTERM
	strcpy(Term, "unknown");
#endif	/* SETTERM */

	/* who am I?  ZyXEL, of course...
	 */
	if ((MyName = strrchr(argv[0],'/')) != NULL)
	    MyName++;
	else
	    MyName = argv[0];

	/* process the command line
	 */

	while ((c = getopt(argc, argv, "D:d:hlmnt:Z?")) != EOF) {
		switch (c) {
		case 'D':
#ifdef	DEBUG
			sscanf(optarg, "%o", &Debug);
#else	/* DEBUG */
			logmesg("DEBUG not compiled in");
#endif	/* DEBUG */
			break;
		case 'd':
			DefName = optarg;
			break;
		case 'h':
			NoHangUp = TRUE;
			break;
		case 'l':
			LockPoll = TRUE;
			break;
		case 'm':
			ModemPoll = TRUE;
			break;
		case 'n':
			No_nonblock = TRUE;
			break;
		case 't':
			TimeOut = atoi(optarg);
			break;
		case 'Z':
			No_ZyXEL = TRUE;
			break;
		case '?':
			exit_usage(2);
			break;
		}
	}

	/* normal System V init options handling
	 */

	if (optind < argc)
	    Device = argv[optind++];
	if (optind < argc)
	    Speed = atoi(argv[optind++]);
#ifdef	SETTERM
	if (optind < argc)
	    strncpy(Term, argv[optind++], sizeof(Term));
#endif	/* SETTERM */

	if (!Device) {
	    logmesg("no line given");
	    exit_usage(2);
	}

#ifdef	SETTERM
	if (!strcmp(Term, "unknown")) {
	    if ((fp = fopen(TTYTYPE, "r")) == NULL) {
		sprintf(MsgBuf, "cannot open %s", TTYTYPE);
		logmesg(MsgBuf);
	    } else {
		char device[MAXLINE+1];

		while ((fscanf(fp, "%s %s", buf, device)) != EOF) {
		    if (!strcmp(device, Device)) {
			strncpy(Term,buf,sizeof(Term));
			break;
		    }
		}
		fclose(fp);
	    }
	}
#endif	/* SETTERM */

	/* need full name of the device
	 */
	sprintf(buf, "/dev/%s", Device);
	DevName = strdup(buf);

	/* command line parsed, now build the list of
	 * runtime defaults; this may override things set above.
	 */

	getvalues();			/* read defaults from file */

	/* generate the FIFO filename (when configured)
	 * create the FIFO when it doesn't exist yet
	 */
	if (Fifo != NULL) {
	    sprintf(buf, Fifo, Device);
	    FifoFile = strdup(buf);

	    if ((stat(FifoFile, &st) && errno == ENOENT) ||
		!S_ISFIFO(st.st_mode))
	    {
		unlink(FifoFile);
		if (mkfifo(FifoFile,0200) != 0) {
		    logmesg("cannot make FIFO");
		    return EXIT_FAILURE;
		}
	    }

	    chmod(FifoFile,0200);	/* make sure it is accessible... */
	    chown(FifoFile,0,0);	/* ... to root only */
	}

#ifdef	DEBUG

	if (Debug) {
	    sprintf(buf, "/tmp/%s:%s", MyName, Device);
	    if ((Dfp = fopen(buf, "a")) == NULL) {
		logmesg("cannot open debug file");
		return EXIT_FAILURE;
	    } else {
		time_t tim;
		int fd;

		if (fileno(Dfp) < 3) {
		    if ((fd = dup2(fileno(Dfp),3)) > 2) {
			fclose(Dfp);
			Dfp = fdopen(fd, "a");
		    }
		}
		tim = time(NULL);
		fprintf(Dfp, "%s Started: %s", MyName, ctime(&tim));
		fflush(Dfp);
	    }
	}

	debug(D_OPT, " debug    = (%o)\n", Debug);
	debug(D_OPT, " defname  = (%s)\n", DefName);
	debug(D_OPT, " NoHangUp = (%s)\n", (NoHangUp) ? "TRUE" : "FALSE");
	debug(D_OPT, " LockPoll = (%s)\n", (LockPoll) ? "TRUE" : "FALSE");
	debug(D_OPT, " TimeOut  = (%d)\n", TimeOut);
	debug(D_OPT, " line     = (%s)\n", Device);
#ifdef	SETTERM
	debug(D_OPT, " type     = (%s)\n", Term);
#endif	/* SETTERM */

#endif	/* DEBUG */

	/* get uid/gid of UUCP user to use for lockfiles and line.
	 * when not found, 0/0 will be used instead.
	 */
	if ((pwd = getpwnam(UUCPNAME)) != NULL) {
	    UuUid = pwd->pw_uid;
	    UuGid = pwd->pw_gid;
	}

	/* close the stdio descriptors, we will not use them
	 * because there are problems with re-opening
	 */
	fclose(stdin);
	fclose(stdout);
	fclose(stderr);

	debug2(D_RUN, "check for lockfiles\n");

	/* deal with the lockfiles; we don't want to charge
	 * ahead if uucp, kermit or something is already
	 * using the line.
	 */

	/* name the lock file(s)
	 */
#ifdef SVR4LOCK
	if (stat(DevName,&st)) {
	    logmesg("cannot stat line");
	    return EXIT_FAILURE;
	}
	sprintf(buf, LOCK, major(st.st_dev), major(st.st_rdev), minor(st.st_rdev));
#else
	sprintf(buf, LOCK, Device);
#endif
	Lock = strdup(buf);
	debug3(D_LOCK, "Lock = (%s)\n", Lock);

	atexit(rmlocks);			/* remove lockfile when exit */

	/* check for existing lock file(s)
	 */
	if (checklock(Lock) == TRUE) {
	    do
	    {
		sleep(15);
	    }
	    while (checklock(Lock) == TRUE);	/* continue when lock gone */
	}

	/* the line is mine now ...
	 */

	if (openline() != SUCCESS) {
	    if (getppid() == 1)			/* called from INIT? */
		sleep(60);			/* prevent rapid respawn */

	    return EXIT_FAILURE;
	}

	/* branch back here when a call has been handled and
	 * we don't need to restart the program for some reason
	 */
re_init:
	/* reset and initialize the MODEM
	 */
	if (initmodem(!NoHangUp)) {
	    send("ATZ\r");
	    closeline();

	    if (getppid() == 1) {		/* called from INIT? */
		makelock(Lock);			/* holdoff other users */
		sleep(300);			/* prevent rapid respawn */
	    }

	    return EXIT_FAILURE;
	}

	/* allow uucp to access the device
	 */
	debug2(D_RUN, "chmod and chown to uucp\n");
	chmod(DevName, 0666);
	chown(DevName, UuUid, UuGid);

#ifdef	LOGUTMP

	/* create UTMP entry when we were started by INIT
	 */
	if (getppid() == 1) {			/* only when from INIT */
	    debug2(D_RUN, "update utmp/wtmp files\n");

	    pid = getpid();
	    while ((utmp = getutent()) != NULL)
		if (utmp->ut_type == INIT_PROCESS && utmp->ut_pid == pid)
		{
		    debug2(D_UTMP, "logutmp entry made\n");
		    /* show login process in utmp
		     */
		    strncopy(utmp->ut_line, Device);
		    strncopy(utmp->ut_user, "LOGIN");
		    utmp->ut_type = LOGIN_PROCESS;
		    utmp->ut_time = time(NULL);
		    pututline(utmp);

		    /* write same record to end of wtmp
		     * if wtmp file exists
		     */
		    if (stat(WTMP_FILE, &st) && errno == ENOENT)
			break;

		    if ((fp = fopen(WTMP_FILE, "a")) != NULL) {
			fseek(fp,0L,SEEK_END);
			fwrite((char *)utmp,sizeof(*utmp),1,fp);
			fclose(fp);
		    }
		}
	    endutent();
	}

#endif	/* LOGUTMP */

	/* sound a short beep to indicate we are ready
	 */
	beep(Ready);

	signal(SIGINT, inthandler);

	/* wait for incoming calls or data on application FIFO
	 */
	switch (wait_work(WFclass))
	{
	case FIFO:				/* data on application FIFO */
	    if ((c = api_mode()) != LOCKED)	/* handle it */
	    {
		close(FifoFd);

		if (c == SUCCESS)
		    break;			/* OK, just loop around */

		send("ATZ\r");
		return EXIT_FAILURE;		/* some mishap, restart */
	    }

	    close(FifoFd);
	    /* it was LOCKED */
	case LOCKED:				/* someone else locked it */
	    closeline();			/* close the line */
	    debug2(D_RUN, "line locked now\n");

	    /* closing the line still leaves it as controlling terminal :-( */
	    /* it seems best to just exit, and take care of the lock at */
	    /* startup only, where we are in a new process group */
#if 0
#ifndef SA_SIGINFO
	    /* wait until the lock has been released
	     */
	    do
	    {
		sleep(15);
	    }
	    while (checklock(Lock) == TRUE);
	    debug2(D_RUN, "lock gone, stopping\n");
#endif
#endif
	    return EXIT_SUCCESS;		/* INIT will respawn us */

	case INITFAIL:				/* initialization failed */
	    send("ATZ\r");
	    closeline();

	    if (getppid() == 1)			/* called from INIT? */
		sleep(300);			/* prevent rapid respawn */
	    /* just as FAIL */
	case FAIL:				/* got some garbage */
	    return EXIT_FAILURE;		/* re-init from start */

	case VCON:				/* DATA/VOICE in FCLASS=8 */
	    if (read_state(0,0) == FAIL)
		Mode = DATAFAX;			/* default to DATA/FAX mode */
	    /* continue as if a RING */

	case RING:
	    if (!Is_ZyXEL)
		usleep(500000);			/* some small delay after RING */

	    switch (Mode)			/* state file already read! */
	    {
	    case DATAFAX:
		switch (datamode(ANSWER))	/* answer in DATA/FAX mode */
		{
		case VOICE:			/* was a VOICE call? */
		    hold_state(120,VOICE,2);
		    goto voice;

		case FAX:			/* was a FAX call */
		    if (faxmode(ONLINE) != SUCCESS)
			hold_state(120,FAX,2);
		    break;

		case FAIL:			/* some real failure? */
		    return EXIT_FAILURE;	/* play safe and restart it */
		}
		break;

	    case DATA:
		switch (datamode(ANSWERDATA))	/* answer in DATA-only mode */
		{
		case VOICE:			/* was a VOICE call? */
		    hold_state(120,VOICE,2);
		    goto voice;

		case FAX:			/* was a FAX call */
		    if (faxmode(ONLINE) != SUCCESS)
			hold_state(120,FAX,2);
		    break;

		case FAIL:			/* some real failure? */
		    return EXIT_FAILURE;	/* play safe and restart it */
		}
		break;

	    case FAX:
		if (faxmode(ANSWER) == DATA)	/* FAX-only answer */
		    if (datamode(ONLINE) == FAIL) /* it still connected DATA! */
			return EXIT_FAILURE;
		break;

	    case VOICE:
voice:
		switch (voicemode(ANSWER))	/* try voicemode */
		{
		case DATA:
		    /* DATA requested, continue in datamode */
		    /* no FAX in this case */

		    switch (datamode(ANSWERDATA)) /* answer in DATA-only mode */
		    {
		    case SUCCESS:		/* worked? */
			break;

		    case FAIL:			/* some real failure? */
			return EXIT_FAILURE;	/* play safe and restart it */

		    default:			/* NO CARRIER etc */
			hold_state(120,DATA,2); /* remain in DATA mode */
			break;
		    }
		    break;

		case FAX:
		    if (faxmode(ANSWER) != SUCCESS)
			hold_state(120,FAX,2);
		    break;

		case DIALBACK:			/* have dialled back to user */
		    if (voicemode(ONLINE) != FAIL) /* handle voice */
			break;			/* but ignore DATA and FAX */

		case FAIL:			/* some real failure? */
		    return EXIT_FAILURE;	/* play safe and restart it */
		}
		break;
	    }
	    break;

	case CONNECT:
	   /* we can enter here when DATA/VOICE is pressed
	    *	while MODEM is in FCLASS other than 8
	    */
	    switch (datamode(ONLINE))		/* already CONNECTed */
	    {
	    case FAX:				/* was a FAX call */
		if (faxmode(ONLINE) == FAIL)
		    return EXIT_FAILURE;
		break;

	    case FAIL:				/* some real failure? */
		return EXIT_FAILURE;		/* play safe and restart it */
	    }
	    break;
	}

	rmlocks();				/* remove lock if mine */
	goto re_init;				/* and do it all over again */
}


/*
**	exit_usage() - exit with usage display
*/

void
exit_usage(code)
int code;
{
	FILE *fp = stderr;

	/* when not started from INIT, just write to stderr */

	if (getppid() != 1 || (fp = fopen(CONSOLE, "w")) != NULL) {
		fprintf(fp, "\n%s\n\n",Id + 4);
		fprintf(fp, "This program is copyrighted by:\n");
		fprintf(fp, "\tR.E. Janssen\n");
		fprintf(fp, "\tProf L Fuchslaan 8\n");
		fprintf(fp, "\t3571 HC  UTRECHT, NETHERLANDS\n\n");
		fprintf(fp, "Permission is granted only for noncommercial use.  Any commercial use or\n");
		fprintf(fp, "distribution via commercial services without the author's permission is\n");
		fprintf(fp, "prohibited.\n\n");
#ifdef	SETTERM
		fprintf(fp, "Usage: %s [options] line speed [term]\n",MyName);
#else
		fprintf(fp, "Usage: %s [options] line speed\n",MyName);
#endif	/* SETTERM */
		fprintf(fp, "Options:\n");
		fprintf(fp, "\t-D<dbglvl>   set debugging level (octal)\n");
		fprintf(fp, "\t-d<defaults> set defaults file\n");
		fprintf(fp, "\t-h           don't hangup (DTR-toggle) during init\n");
		fprintf(fp, "\t-l           check for lockfiles even when no input\n");
		fprintf(fp, "\t-n           don't open line with NDELAY (require DCD)\n");
		fprintf(fp, "\t-t<timeout>  timeout (seconds) at login: prompt\n");
		fclose(fp);
	}
	exit(code);
}



/*
**	getvalues() - read all runtime values from /etc/defaults
*/

void
getvalues()

{
	FILE *fp;
	register DEF **def;
	char *p;
	char buf[MAXLINE+1];

	def = defbuild(DefName);

#ifdef	DEBUG

	/* debugging on?
	 */
	if ((p = defvalue(def, "DEBUG")) != NULL)
		sscanf(p, "%o", &Debug);

#endif	/* DEBUG */

	Init = defvalue(def, "INIT");
	if ((p = defvalue(def, "HANGUP")) != NULL)
	    NoHangUp = !strcmp(p, "NO");
	if ((p = defvalue(def, "LOCKPOLL")) != NULL)
	    LockPoll = !strcmp(p, "YES");
	if ((p = defvalue(def, "MODEMPOLL")) != NULL)
	    ModemPoll = !strcmp(p, "YES");
	if ((p = defvalue(def, "WFCLASS")) != NULL)
	    WFclass = atoi(p);
	if ((p = defvalue(def, "MUTE")) != NULL)
	    Mute = p;
	DialPrefix = defvalue(def, "DIALPREFIX");
	Fifo = defvalue(def, "FIFO");
	if ((p = defvalue(def, "STATEFILE")) != NULL)
	    StateFile = p;

	if ((SysName = defvalue(def, "SYSTEM")) == NULL)
		SysName = getuname();

	if ((Version = defvalue(def, "VERSION")) != NULL)
	    if (*Version == '/') {
		if ((fp = fopen(Version, "r")) != NULL) {
		    if (fgets(buf, sizeof(buf), fp) != NULL)
		       buf[strlen(buf)-1] = '\0';
		    else
		       buf[0] = '\0';
		    fclose(fp);
		    Version = strdup(buf);
		}
	    }

	if ((p = defvalue(def, "XOFF")) != NULL && !strcmp(p,"NO"))
		FinalXoff = FALSE;
	if ((p = defvalue(def, "LOGIN")) != NULL)
		Login = p;
	if ((p = defvalue(def, "LOGINENV")) != NULL && !strcmp(p,"NO"))
		LoginEnv = FALSE;
	LoginFlags = defvalue(def, "LOGINFLAGS");
#ifdef	FIDO
	Fido = defvalue(def, "FIDO");
#endif	/* FIDO */
	Protect = defvalue(def, "PROTECT");

	if ((p = defvalue(def, "ISSUE")) != NULL)
	    Issue = p;
	if ((p = defvalue(def, "TIMEOUT")) != NULL)
	    TimeOut = atoi(p);

	FaxDir = defvalue(def, "FAXDIR");
	if ((p = defvalue(def, "FAXFILE")) != NULL)
	    FaxFile = p;
	FaxId = defvalue(def, "FAXID");
	FaxMail = defvalue(def, "FAXMAIL");
#ifndef ZFAX
	if ((p = defvalue(def, "FAXCOMP")) != NULL)
	    FaxComp = atoi(p);
	if ((p = defvalue(def, "FAXPOLL")) != NULL)
	    FaxPoll = p;
	if ((p = defvalue(def, "GHOSTSCRIPT")) != NULL)
	    GhostScript = p;
	if ((p = defvalue(def, "GUNZIP")) != NULL)
	    Gunzip = p;
	if ((p = defvalue(def, "BUFFER")) != NULL)
	    Buffer = p;
	if ((p = defvalue(def, "PSTEXT")) != NULL)
	    PsText = p;
#endif

	if ((p = defvalue(def, "APIUSERLEVEL")) != NULL)
	    ApiUserLevel = atoi(p);
	Archive = defvalue(def, "ARCHIVE");
	if ((p = defvalue(def, "DROPMSG")) != NULL)
	    DropMsg = atoi(p);
	AnswerTone = defvalue(def, "ANSWERTONE");
	Error = defvalue(def, "ERROR");
	PlayDir = defvalue(def, "PLAYDIR");
	Prompt1 = defvalue(def, "PROMPT1");
	Prompt2 = defvalue(def, "PROMPT2");
	Ready = defvalue(def, "READY");
	RecAttn = defvalue(def, "RECATTN");
	RecDir = defvalue(def, "RECDIR");
	RecDone = defvalue(def, "RECDONE");
	if ((p = defvalue(def, "RECFILE")) != NULL)
	    RecFile = p;
	if ((p = defvalue(def, "RECMODE")) != NULL)
	    RecMode = atoi(p);
	if ((p = defvalue(def, "SHOWMSG")) != NULL)
	    if (!strcmp(p,"AA"))
		ShowMsg = SHOWMSG_AA;
	Silence = defvalue(def, "SILENCE");
	Synth = defvalue(def, "SYNTH");
	SynthFlags = defvalue(def, "SYNTHFLAGS");
	VoiceCmd = defvalue(def, "VOICECMD");
	VoiceKey = defvalue(def, "VOICEKEY");

	PagerNum = defvalue(def, "PAGERNUM");
	if ((p = defvalue(def, "PAGERTYPE")) != NULL)
	    PagerType = atoi(p);
}

/*
**	initmodem() - initialize MODEM for use by this program
*/

int
initmodem(dropdtr)
int dropdtr;

{
	char *p,*q;
#ifdef TIOCMGET
	int mstat;
#endif

	debug2(D_RUN, "perform MODEM initialization\n");

	signal(SIGINT, SIG_IGN);		/* it will trip when switched */

	/* toggle DTR
	 * when AT&D3 is still active this will reset the MODEM
	 */
	if (dropdtr) {
	    usleep(100000);			/* keep DTR on a while */
#ifdef CLOSE_OPEN
	    closeline();
	    usleep(500000);			/* keep DTR off a while */
	    if (openline() != SUCCESS)
		return FAIL;
#else
	    settermio(DROPDTR);
	    usleep(500000);			/* keep DTR off a while */
#endif
	}

	/* make sure the line is properly initialized
	 */
	settermio(INITIAL);

	/* flush any pending garbage & allow MODEM to recover from DTR drop
	 */
	waitfor("FLUSH",SHORTTIME);

	/* now try to RESET the MODEM
	 */
	send("ATZ\r");

	if (waitfor("OK",MEDMTIME)) {
	    /* no reaction to ATZ
	     * first attempt to end voice mode (DLE-ETX)
	     */
	    send(AudioAbort);
	    waitfor("FLUSH",SHORTTIME);

	    /* exit from data mode, when it is stuck in a mode that does
	     * not honour the DTR drop
	     */
	    send("\\d+++\\dATZ\r");

#ifdef TIOCMGET
	    /* make sure the MODEM is present
	     * do this by checking CTS, the only line which is
	     * guaranteed to be active
	     */
	    if ((mstat = modemstat()) != -1 &&
		!(mstat & TIOCM_CTS) &&
		sleep(3),!(modemstat() & TIOCM_CTS))
	    {
		waitfor("FLUSH",SHORTTIME);	/* flush OK */

		debug2(D_RUN, "CTS is OFF\n");
		logmesg("MODEM not switched ON, waiting");

		while (!(modemstat() & TIOCM_CTS))
		    sleep(5);

		debug2(D_RUN, "CTS is ON\n");
		sleep(5);
		logmesg("MODEM switched ON, continuing");

		send("ATZ\r");
	    }
#endif

	    if (waitfor("OK",MEDMTIME)) {
		logmesg("MODEM RESET (ATZ) failed");
		debug2(D_RUN, "MODEM RESET (ATZ) failed -- aborting\n");
		return FAIL;
	    }
	}

	waitfor("FLUSH",SHORTTIME);			/* skip all garbage */

	if (ProdInfo == NULL) {
	    /* check that the MODEM is a 1496
	     * don't want to be responsible for destroying other MODEM's
	     * config..
	     */
	    send("ATI\r");
	    if (waitfor("OK",MEDMTIME)) {
		logmesg("MODEM Product Info inquiry (ATI) failed");
		debug2(D_RUN, "MODEM Product Info inquiry (ATI) failed -- aborting\n");
		return FAIL;
	    }

	    p = ChatBuf;

	    if (!strncmp(p,"ATI",3)) {			/* skip the echo */
		if ((p = strchr(p,'\n')) == NULL)
		    p = ChatBuf;
		else
		    while (*p == '\n')
			p++;				/* skip to next line */
	    }

	    rip(p);					/* discard CR etc */
	    ProdInfo = strdup(p);			/* save prod. info */

	    if (!strncmp(p,"U1496",5)) {		/* really old ZyXEL? */
		Is_ZyXEL = TRUE;
		goto prodinfo;				/* evaluate as ATI1 */
	    }

	    if (atoi(p) == 1496)
		Is_ZyXEL = TRUE;			/* looks like ZyXEL */
	    else
		if (!No_ZyXEL) {
		    sprintf(MsgBuf, "MODEM returns Product Info \"%s\", \"1496\" required", p);
		    logmesg(MsgBuf);
		    debug3(D_RUN, "%s -- aborting\n",MsgBuf);
		    return NO1496;
		}
	}

	/* get the firmware version, when not yet known
	 */
	if (Is_ZyXEL && Fw_version == 0) {
	    send("ATI1\r");
	    if (waitfor("OK",MEDMTIME)) {
		logmesg("MODEM Version inquiry (ATI1) failed");
		debug2(D_RUN, "MODEM Version inquiry (ATI1) failed -- aborting\n");
		return FAIL;
	    }

	    p = ChatBuf;

	    if (!strncmp(p,"ATI",3)) {			/* skip the echo */
		if ((p = strchr(p,'\n')) == NULL)
		    p = ChatBuf;
		else
		    while (*p == '\n')
			p++;				/* skip to next line */
	    }

	    while (*p != '\0' && *p != 'U') {		/* find U1496... */
		if ((p = strchr(p,'\n')) == NULL)
		    break;
		else
		    while (*p == '\n')
			p++;				/* skip to next line */
	    }

prodinfo:
	    if (p != NULL && *p == 'U') {
		if ((q = strchr(p,'\n')) != NULL)
		    *q = '\0';				/* take 1 line */

		if ((q = strchr(p,' ')) != NULL) {	/* get product info */
		    *q = '\0';
		    free(ProdInfo);
		    ProdInfo = strdup(p);		/* more accurate info */
		    *q = ' ';
		}

		if ((q = strchr(p,'V')) != NULL) {	/* find V x.xx [P] */
		    while (*q != '\0' && !isdigit(*q))
			q++;

		    Fw_version = atoi(q) * 100;		/* major version */

		    while (isdigit(*q))
			q++;

		    if (*q == '.')
			Fw_version += atoi(q + 1);	/* add minor version */

		    if (strchr(q,'P') != NULL)
			Fw_plus = 1;			/* PLUS version */
		}
	    }
	}

	/* handle init sequence if requested
	 */
	CurVSD[0] = '\0';			/* VSD setting gets reset */
	CurLS = CurSync = CurVSM = 0;		/* VLS/VSM setting as well */
	if (Init != NULL) {
	    if (chat(Init) == FAIL){
		logmesg("INIT sequence failed");
		debug2(D_RUN, "Modem initialization failed -- aborting\n");
		return FAIL;
	    }
	    waitfor(NULL,SHORTTIME);		/* eat all remaining chars */
	}

#ifdef TIOCMGET
	/* check the MODEM line status to make sure INIT has worked
	 */
	if ((mstat = modemstat()) != -1 && (mstat & MODEMLINES) != MODEMLN_OK) {
	    logmesg("Invalid line status after INIT sequence");
	    debug2(D_RUN, "Invalid line status after INIT sequence -- aborting\n");
	    return FAIL;
	}
#endif

#ifndef ZFAX
	/* send default CLASS 2 FAX parameters
	 */
	sprintf(ChatBuf,"AT+FCLASS=2;+FBOR=0;+FCR=1;+FDCC=1,,0,2,%d,,,0\r",
		FaxComp);
	send(ChatBuf);
	if (waitfor("OK",MEDMTIME)) {
	    logmesg("Set FAX params failed");
	    debug2(D_RUN, "Set FAX params failed -- aborting\n");
	    return FAIL;
	}

	/* not sure if all versions accept this one, so send it separately...
	 */
	send("AT+FCQ=2\r");
	waitfor("OK",MEDMTIME);
#endif
	/* send FAX ID when defined
	 */
	if (FaxId != NULL) {
#ifdef ZFAX
	    sprintf(ChatBuf,"AT#P%.25s\r",FaxId);
#else
	    sprintf(ChatBuf,"AT+FLID=\"%.20s\"\r",FaxId);
#endif
	    send(ChatBuf);
	    if (waitfor("OK",MEDMTIME)) {
		logmesg("Set FAX ID failed");
		debug2(D_RUN, "Set FAX ID failed -- aborting\n");
		return FAIL;
	    }
	}

	/* override "abort audio" sequence to react quicker on modems */
	/* that support the DLE-DC4 command to clear transmit buffer */
	if (Fw_version >= 611)
	    AudioAbort[2] = DC4;

	/* when possible with this version, enable RTS/CTS flow
	 * control for VOICE mode (and FAX mode as well)
	 */
	if (Fw_version >= 613) {
	    send("AT+FLO=2\r");
	    if (waitfor("OK",MEDMTIME)) {
		logmesg("Set VOICE flow control failed");
		debug2(D_RUN, "Set VOICE flow control failed -- aborting\n");
		return FAIL;
	    }
	}

	return SUCCESS;
}

/*
**	openline() - open line to STDIN/STDOUT/STDERR
*/

int
openline(void)

{
	int fd;
	int mode;

	debug2(D_RUN, "open stdin, stdout and stderr\n");

	/* open the line; don't wait around for carrier-detect
	 */
	mode = O_RDWR;
	if (!No_nonblock)
	    mode |= O_NDELAY;
	if ((fd = open(DevName,mode)) != STDIN) {
	    logmesg("cannot open line");
	    return FAIL;
	}

	/* make stdout and stderr, too
	 */
	if ((fd = dup2(STDIN,STDOUT)) != STDOUT) {
	    debug4(D_RUN, "dup stdout: %d %d\n", fd, errno);
	    logmesg("cannot open stdout");
	    return FAIL;
	}
	if ((fd = dup2(STDIN,STDERR)) != STDERR) {
	    debug4(D_RUN, "dup stderr: %d %d\n", fd, errno);
	    logmesg("cannot open stderr");
	    return FAIL;
	}

	/* setup tty characteristics
	 */
	debug2(D_RUN, "setup tty characteristics\n");
#ifdef __linux__
	mode = N_TTY;				/* select TTY discipline */
	ioctl(STDIN,TIOCSETD,&mode);
#endif
	settermio(INITIALIZE);

	/* clear O_NDELAY flag now, we need it for blocking reads
	 */
	if (!No_nonblock) {
	    int flags;

	    flags = fcntl(STDIN, F_GETFL, 0);
	    fcntl(STDIN, F_SETFL, flags & ~O_NDELAY);
	}

	/* create a new process group (maybe INIT does it?)
	 */
	if (getppid() == 1) {			/* only when from INIT */
#ifdef TIOCSPGRP
	    int rv;
	    pid_t pid;
#endif

	    setpgrp();				/* new process group */

#ifdef TIOCSPGRP
	    pid = getpid();
	    if ((rv = ioctl(STDIN, TIOCSPGRP, &pid)) < 0) {
		debug4(D_RUN, "TIOCSPGRP returns %d pid=%d\n", rv, pid);
		logmesg("cannot set process group");
	    }
#endif
	}

	return SUCCESS;
}

/*
**	closeline() - close the line (STDIN/STDOUT/STDERR)
*/

void
closeline(void)

{
	signal(SIGHUP, SIG_IGN);
	signal(SIGINT, SIG_IGN);
	close(STDIN);
	close(STDOUT);
	close(STDERR);
#ifdef SA_SIGINFO
	catchsig(SIGHUP);
	catchsig(SIGINT);
#endif
}


/*
**	wait_work() - wait for data from MODEM
*/

int
wait_work(fclass)
int fclass;					/* which +FCLASS to use */

{
	int sel,res;
	char *p;
#ifdef DEBUG
	time_t now;
#endif
	int messages;				/* messages indicated on AA? */
#ifndef ZFAX
	int faxpoll = 0;			/* fax available for poll? */
#endif
	int maxfd;
	fd_set select_set;			/* which fd's we have to select() on */
	fd_set read_set;			/* copy to be used for read */
	struct timeval timeval;			/* timeout for select() */
	struct termios termio;

	messages = filespresent(RecDir);	/* messages recorded? */

#ifndef ZFAX
	if (FaxPoll != NULL) {
	    sprintf(ChatBuf,FaxPoll,"0000");	/* create default doc filename */
	    if (!access(ChatBuf,R_OK))
		faxpoll = 1;			/* we can respond to poll */
	}
#endif

all_over_again:
	free(CndName);				/* release CND data */
	CndNumber = NULL;
	free(CndNumber);
	CndName = NULL;

	if (ShowMsg && messages) {
	    debug2(D_RUN, "messages present in RecDir!");

	    switch (ShowMsg)
	    {
	    case SHOWMSG_AA:
		send("ATS0=255\r");		/* AA on, but no answering! */
		if (waitfor("OK",MEDMTIME))
		    return INITFAIL;
		break;
	    }
	}

	debug2(D_RUN, "waiting for character from MODEM ...\n");

	/* now prepare the MODEM for incoming calls:
	 * ATH		hangup the line (seems to be required after beeps etc)
	 * +FCLASS=0	so DATA/VOICE attempts DATA connection and sends
	 *		CONNECT when successful (ORG/ANS also works)
	 * +FCLASS=8	so DATA/VOICE button immediately sends VCON, but
	 *		ORG/ANS cannot be checked, always ANSWER mode
	 */
	strcpy(ChatBuf,"ATH");
	if (fclass >= 0) {
#ifndef ZFAX
	    sprintf(ChatBuf + strlen(ChatBuf),"+FLPL=%d;",faxpoll);
#endif
	    sprintf(ChatBuf + strlen(ChatBuf),"+FCLASS=%d",fclass);
	}
	strcat(ChatBuf,"\r");
	send(ChatBuf);
	if (waitfor("OK",MEDMTIME))
	    return INITFAIL;

	/* setup API FIFO and select() set
	 */
	FD_ZERO(&select_set);			/* all bits off */
	FD_SET(STDIN,&select_set);		/* stdin */
	maxfd = 1;

	if (FifoFile != NULL) {
	    if ((FifoFd = open(FifoFile,O_RDONLY|O_NONBLOCK)) < 0) {
		logmesg("cannot open FIFO");
		debug2(D_RUN, "cannot open FIFO...\n");
	    } else {
		int flags = fcntl(FifoFd, F_GETFL, 0);
		fcntl(FifoFd, F_SETFL, flags & ~O_NONBLOCK);
		FD_SET(FifoFd,&select_set);	/* the API FIFO */
		maxfd = FifoFd + 1;
	    }
	}

	ioctl(STDIN, TCFLSH, 0);		/* flush input */
	InPtr = InLen = 0;

	/* wait for incoming character on MODEM and/or FIFO...
	 */
	do
	{
	    Match = NULL;			/* nothing Matched... */

	    read_set = select_set;
	    timeval.tv_sec = 15;		/* check locks every 15 sec? */
	    timeval.tv_usec = 0;

	    if ((sel = select(maxfd,&read_set,NULL,NULL,
			      ((LockPoll || ModemPoll)? &timeval : NULL))) < 0)
	    {
		logmesg("select failed");
		debug2(D_RUN, "select failed...\n");
		return FAIL;
	    }

	    /* check for locks from others
	     */
	    if ((LockPoll || sel > 0) && checklock(Lock)) {
		close(FifoFd);
		return LOCKED;
	    }

	    /* when data available on FIFO, return FIFO and leave it OPEN
	     */
	    if (FifoFd > 0 && FD_ISSET(FifoFd,&read_set))
		return FIFO;

	    ioctl(STDIN, TCGETS, &termio);	/* check if others changed line */

	    if (memcmp(&termio,&Termio,sizeof(struct termios))) {
		if (!(LockPoll || sel > 0) && checklock(Lock)) {
		    close(FifoFd);
		    return LOCKED;
		}

		logmesg("someone changed termios");
		debug2(D_RUN, "someone changed termios...\n");
		close(FifoFd);
		return FAIL;
	    }

#ifdef TIOCMGET
	    /* make sure the handshake lines are still valid
	     * when not, the MODEM probably was powered off or disconnected...
	     */
	    if ((res = modemstat()) != -1 && (res & MODEMLINES) != MODEMLN_OK)
	    {
		if (!(LockPoll || sel > 0) && checklock(Lock)) {
		    close(FifoFd);
		    return LOCKED;
		}

		logmesg("lost MODEM while waiting for work");
		debug3(D_RUN, "lost MODEM while waiting for work\n", MsgBuf);
		close(FifoFd);
		return FAIL;
	    }
#endif

	    /* when requested, check if AT still results in a response
	     * (not merely an echo of the AT command)
	     * we can't do a check for OK, because RING or other events
	     * may come in just at this moment...
	     */
	    if (ModemPoll && !FD_ISSET(STDIN,&read_set))
	    {
		send("AT\r");

		if (waitfor(NULL,MEDMTIME) || !strcmp(Match,"AT")){
		    logmesg("MODEM died while waiting for work");
		    debug3(D_RUN, "MODEM died while waiting for work\n", MsgBuf);
		    close(FifoFd);
		    return FAIL;
		}
	    }
	}
	while (!FD_ISSET(STDIN,&read_set));	/* continue until MODEM data */

	close(FifoFd);				/* close the FIFO */

#ifdef DEBUG
	now = time(NULL);
	debug3(D_RUN, "... got one! - %s", ctime(&now));
#endif

	/* try to lock the line, we don't want to
	 * read more chars if it is locked by someone else, and neither
	 * do we want anyone else to pickup the MODEM while the phone
	 * is ringing...
	 */
	if (makelock(Lock) == FAIL)
	    return LOCKED;

	/* read the first keyword (RING, VCON, NO CARRIER...) */

	if (Match == NULL && waitfor(NULL,SHORTTIME))
	    goto garbage;

	/* save the reply, remove trailing CR/LF garbage and spaces */

	rip(Match);
	strcpy(MsgBuf,Match);

	/* see what it is... CONNECT checked first because it changes mode */

	if (!strncmp(MsgBuf,"CONNECT",7))
	    return CONNECT;

#ifndef ZFAX
	if (!strcmp(Match,"+FCON"))
	    return CONNECT;
#endif

	/* turn-off AA to be really sure (cannot be done when CONNECT) */

	if (ShowMsg && messages)
	    switch (ShowMsg)
	    {
	    case SHOWMSG_AA:
		send("ATS0=0\r");		/* AA off */
		if (waitfor("OK",MEDMTIME))
		    return FAIL;
		break;
	    }

	/* check for remaining result strings */

	if (!strcmp(MsgBuf,"NO CARRIER"))
	    return NOCARRIER;

	if (!strcmp(MsgBuf,"VCON"))
	    return VCON;

	if (!strncmp(MsgBuf,"RING",4))
	{
	    int ringtype;
	    int rings;

	    /* distinctive ring may result in messages RING1..RING4 */

	    ringtype = atoi(MsgBuf + 4);	/* get type of ringing */

	    /* find out how many rings we need (for this type of ringing) */
	    /* 'messages present' indicator is passed to select tollsaver */

	    if (read_state(ringtype,messages) == FAIL)
		Nrings = 4;

	    for (rings = 1; rings < Nrings; rings++)
	    {
		/* wait for the next ring, or CND message from MODEM */

		if (waitfor(NULL, LONGTIME))
		{
		    sprintf(MsgBuf, "%d RING%s", rings, (rings == 1? "":"s"));
		    if (ringtype != 0)
			sprintf(MsgBuf + strlen(MsgBuf)," (%d)", ringtype);
		    sprintf(MsgBuf + strlen(MsgBuf), ", %d required", Nrings);
		    if (RingBack != 0 && rings <= RingBack) {
			strcat(MsgBuf,", DATA mode for 60 sec");
			hold_state(60,DATA,2);
		    }
		    logmesg(MsgBuf);
		    debug3(D_RUN, "%s\n", MsgBuf);
		    rmlocks();
		    goto all_over_again;
		}

		/* remove trailing CR/LF garbage and spaces */

		if (rip(Match) > 80)
		    goto garbage;

		/* again check for these statuses, it may be that a human */
		/* answered the phone after some RINGs, and then pressed */
		/* DATA/VOICE because she detected it was a FAX or DATA call */

		if (!strncmp(Match,"CONNECT",7))
		    return CONNECT;

		if (!strcmp(Match,"NO CARRIER"))
		    return NOCARRIER;

		if (!strcmp(Match,"VCON"))
		    return VCON;

		/* check if it was indeed a RING that we received */

		if (!strncmp(Match,"RING",4))
		    continue;

		/* it most likely is a CND message...  */

		if (!strncmp(Match,"TIME:",5))
		    continue;

		if ((p = strchr(Match,':')) != NULL && p > (Match + 6) && p[1] == ' ')
		{
		    if (!strncmp(p - 4,"NAME",4)) {
			free(CndName);
			CndName = strdup(p + 2);
		    }
		    if (!strncmp(p - 6,"NUMBER",6)) {
			free(CndNumber);
			CndNumber = strdup(p + 2);
		    }
		}

		if (isdigit(Match[0]) && isdigit(Match[1]) && Match[2] == '-')
		{
		    free(CndNumber);
		    CndNumber = strdup(Match + 12);
		}

		/* log it in the user logfile */

		logmesg(Match);
	    }

	    /* we have got the requested rings */

	    debug4(D_RUN, "%d RING%s, SUCCESS\n", rings, (rings == 1? "":"s"));
	    return RING;
	}

garbage:
	strcpy(MsgBuf,"garbage from MODEM: ");
	for (p = ChatBuf; *p != '\0' && strlen(MsgBuf) < sizeof(MsgBuf) - 3; p++)
	    sprintf(MsgBuf + strlen(MsgBuf),
		    (((unsigned char)*p < ' ') ? "^%c" : "%c"),
		    (((unsigned char)*p < ' ') ? *p | 0100 : *p));
	logmesg(MsgBuf);
	debug3(D_RUN, "%s\n", MsgBuf);
	return FAIL;
}


/*
**	dialup() - dial a number and wait for connection
**
**	will return RING when a call comes in while trying to dial
*/

int
dialup(mode,fclass,number)
const char *mode;
int fclass;					/* which +FCLASS to use */
const char *number;

{
	const char *tomatch;
	int n;
#ifdef TIOCMGET
	int mstat;
#endif

	sprintf(MsgBuf, "Dialing FCLASS=%d (%s)", fclass, number);
	logmesg(MsgBuf);
	debug3(D_RUN, "dialup() - %s\n", MsgBuf);

#ifdef TIOCMGET
	if ((mstat = modemstat()) != -1 && (mstat & TIOCM_RNG))
	    return RING;
#endif

	if (waitfor(NULL,SHORTTIME) && !strncmp(Match,"RING",4))
	    return RING;

	/* put MODEM in the correct mode and FCLASS
	 */
	if (fclass >= 0)
	    sprintf(ChatBuf,"ATH%s+FCLASS=%d\r",mode,fclass);
	else
	    sprintf(ChatBuf,"ATH%s\r",mode);
	send(ChatBuf);

	if (waitfor(NULL,MEDMTIME))
	    return FAIL;

	rip(Match);

	if (!strncmp(Match,"RING",4))
	    return RING;

	if (strcmp(Match,"OK"))
	    return FAIL;

#ifdef TIOCMGET
	if (mstat != -1 && (mstat = modemstat()) != -1 && (mstat & TIOCM_RNG))
	    return RING;
#endif

	/* now dial the number...
	 */
	sprintf(ChatBuf,"AT%sD%s%s\r",
			(fclass > 0? "":Mute),
			(DialPrefix != NULL? DialPrefix : ""),
			number);
	send(ChatBuf);

	switch (fclass)
	{
#ifndef ZFAX
	case 2:					/* CLASS2 FAX, expect +FCON */
	    tomatch = "+FCON";
	    break;
#endif
	case 8:					/* VOICE class, expect VCON */
	    tomatch = "VCON";
	    break;

	default:
	    tomatch = "CONNECT";
	    break;
	}

	/* now wait for connection, may take some time
	 */
	for (n = 5; n > 0; n--)
	    if (!waitfor(NULL,LONGTIME)) {
		rip(Match);

		if (!strncmp(Match,tomatch,strlen(tomatch))) /* prefix match only */
		    break;

		/* got something else than expected */

		logmesg(Match);			/* log what we got (NO CARRIER?) */
		debug3(D_RUN, "dialup() (%s) - FAIL\n", Match);

		if (!strncmp(Match,"RING",4)) {
		    send("ATH\r");
		    return RING;
		}

#ifdef TIOCMGET
		if (mstat != -1 && (mstat = modemstat()) != -1 && (mstat & TIOCM_RNG)) {
		    send("ATH\r");
		    return RING;
		}
#endif

		return FAIL;
	    }

	if (!n) {				/* nothing received */
	    send("ATH\r");
	    return FAIL;
	}

	if (fclass != 2)			/* +FCON not so interesting */
	    logmesg(Match);			/* log the CONNECT we got! */

	free(Connect);
	Connect = strdup(Match);		/* and save it */

	debug3(D_RUN, "dialup() (%s) - SUCCESS\n", Match);
	return SUCCESS;
}


/*
**	read_state() - read the statefile to find mode, number of rings etc
**
**	ringtype selects how many lines are skipped in the file, to
**	support distinctive ringing.  (RING, RING1, RING2 etc)
*/

int
read_state(ringtype,altcount)
int ringtype;
int altcount;

{
	FILE *fp;
	char *p;
	char buf[MAXLINE];

	/* check if we a still holding a previous state
	 * (e.g. after manual entry into datamode, the phone is
	 *  answered in datamode for some time)
	 */
	if (HoldTime != 0 && time(NULL) < HoldTime) {
	    debug4(D_RUN, "Held state: Mode=%d Nrings=%d\n", Mode, Nrings);
	    return SUCCESS;
	}

	HoldTime = 0;
	RingBack = 0;

	sprintf(buf, StateFile, Device);

	if ((fp = fopen(buf,"r")) == NULL) {
	    debug3(D_RUN, "no state file (%s) ", buf);
	    return FAIL;
	}

	debug4(D_RUN, "reading state(%d) from (%s) ", ringtype, buf);

	do
	{
	    if (fgets(buf,sizeof(buf),fp) != NULL) {
		switch (buf[0])
		{
		case 'D':			/* DATA/FAX mode */
		    Mode = DATAFAX;
		    break;

		case 'F':			/* FAX-only mode */
		    Mode = FAX;
		    break;

		case 'V':			/* VOICE mode */
		    Mode = VOICE;
		    break;
		}

		p = buf;

		while (*p != '\0' && !isspace(*p))
		    p++;

		while (*p != '\0' && isspace(*p))
		    p++;

		Nrings = atoi(p);		/* number of rings before answer */

		if (altcount) {			/* try to get 2nd (tollsaver) count? */
		    while (*p != '\0' && !isspace(*p))
			p++;

		    while (*p != '\0' && isspace(*p))
			p++;

		    if (isdigit(*p))
			Nrings = atoi(p);
		}

		if ((p = strchr(p,'R')) != NULL) /* ringback specified? */
		    RingBack = atoi(p + 1);	/* get number of rings */
	    }
	} while (ringtype--);

	debug5(D_RUN, "Mode=%d Nrings=%d RingBack=%d\n", Mode, Nrings, RingBack);
	fclose(fp);
	return SUCCESS;
}

/*
**	hold_state() - lock the state for specified duration
*/

void
hold_state(duration,mode,nrings)
int duration;
int mode;
int nrings;

{
	HoldTime = time(NULL) + duration;	/* end-time for this state */

	Mode = mode;
	Nrings = nrings;
}


/*************************************************************************
**	API mode handling						**
*************************************************************************/

int
api_mode()

{
	int ch,quit,out,playLS,result;
	char *p;
	char buf[MAXLINE],buf2[MAXLINE];
	sig_t (*oldint)();

	/* try to lock the line, we don't want others to use it
	 */
	if (makelock(Lock) == FAIL)
	    return LOCKED;

	/* ignore sig_int during api_mode, the interface
	 * uses this signal to stop playing/recording, and
	 * we don't want this program to quit if the signal
	 * comes a little to late and playing/recording already ended
	 */
	oldint = signal(SIGINT,SIG_IGN);

	ApiPtr = ApiLen = 0;			/* flush buffer */
	quit = 0;
	out = -1;
	playLS = 16;

	while (!quit && (ch = getapi()) != EOF) {
	    switch (ch)
	    {
	    case '\n':				/* newline, give menu */
		Fputs(Id + 4,out);		/* my name and version */
		Fputs("\n\nCommands are:\n",out);
		Fputs("!                 Print ZyXEL pid\n",out);
		Fputs("#                 Print program version\n",out);
		Fputs("$                 Print MODEM version\n",out);
		Fputs(">                 Set output fifo/file/tty\n",out);
		Fputs("S                 Use SPEAKER for playback (default)\n",out);
		Fputs("L                 Use LINE for playback\n",out);
		Fputs("V                 Access VoiceMailBox\n",out);
		Fputs("C<command>        Execute VoiceMailBox command\n",out);
		Fputs("P<file>           Playback\n",out);
		Fputs("R<n>              Record over line <n>\n",out);
		Fputs("F<number> <file>  Send a FAX\n",out);
		Fputs("Q<number> <id>    Poll for FAX\n",out);
		Fputs("D                 Hold DATA/FAX mode for 5 minutes\n",out);
		Fputs("AT<cmd>           Issue MODEM command\n",out);
		Fputs("\n",out);
		break;

	    case '!':				/* report pid */
		quit = api_eol();
		sprintf(buf,"%d\n",getpid());
		Fputs(buf,out);
		break;

	    case '>':				/* output */
		if ((p = apigets()) != NULL) {
		    close(out);
		    out = open(p,O_WRONLY);
		} else
		    quit = 1;
		break;

	    case '#':				/* program version */
		quit = api_eol();
		Fputs(Id + 4,out);		/* my name and version */
		Fputs("\n",out);
		break;

	    case '$':				/* MODEM version */
		quit = api_eol();
		sprintf(buf,"%s%s V %d.%02d%s\n",
			(Is_ZyXEL? "ZyXEL " : ""),
			ProdInfo,
			Fw_version / 100,Fw_version % 100,
			(Fw_plus? " P" : ""));
		Fputs(buf,out);
		break;

	    case 'A':				/* issue MODEM command */
	    case 'a':
		p = apigets();
		sprintf(ChatBuf,"%c%s\r",ch,p);
		send(ChatBuf);
		waitfor("OK",MEDMTIME);
		Fputs(ChatBuf,out);
		break;

	    case 'L':				/* line */
	    case 'l':
		quit = api_eol();
		Fputs("LINE playback\n",out);
		playLS = 2;
		break;

	    case 'S':				/* speaker */
	    case 's':
		quit = api_eol();
		Fputs("SPEAKER playback\n",out);
		playLS = 16;
		break;

	    case 'V':				/* voice mailbox */
	    case 'v':
		quit = api_eol();

		send("AT+FCLASS=8\r");		/* select VOICE mode */
		if (waitfor("OK",MEDMTIME)) {
		    Fputs("FAIL\n",out);
		    return FAIL;
		}

		if (set_vls(playLS)) {		/* select speaker */
		    Fputs("FAIL\n",out);
		    return FAIL;
		}

		Fputs("VoiceMailBox, enter numeric command\n",out);
		Fputs("  0: leave VoiceMailBox\n",out);
		Fputs("\nIn Playback mode, commands are:\n",out);
		Fputs("  0: leave playback mode\n",out);
		Fputs("  2: tell message date/time\n",out);
		Fputs("  3: next message\n",out);
		Fputs("  4: delete message\n",out);
		Fputs("  5: archive message\n",out);
		Fputs("  7: first message\n",out);
		Fputs("  8: repeat message\n",out);
		Fputs("  9: last message\n",out);

		UserLevel = ApiUserLevel;	/* API default user level */
		Number[0] = '\0';

		while (Fputs("\nVMB> ",out), (p = apigets()) != NULL)
		{
		    char *file;

		    if (p[0] == '.') {
			file = VoiceKey;
			p++;
		    } else
			file = VoiceCmd;

		    switch (voicecommand(file,atoi(p),apigetc,apigets))
		    {
		    case DIALBACK:
			voicemode(ONLINE);
			break;

		    case EXIT:
			goto endvmb;

		    case UNKNOWN:
			Fputs("?\n",out);
			break;

		    case ERROR:
			Fputs("ERROR\n",out);
			break;

		    case FAIL:
			Fputs("FAIL\n",out);
			return FAIL;
		    }
		}
endvmb:
		if (set_vls(0)) {
		    Fputs("FAIL\n",out);
		    return FAIL;
		}

		Fputs("\nexit from VMB\n",out);
		break;

	    case 'C':				/* VoiceMailBox command */
	    case 'c':
		p = buf;			/* read command into buf */

		while (p < (buf + MAXLINE - 1) && (ch = getapi()) != EOF &&
			ch != '\n' && ch != '\r')
		    if (p != buf || ch != ' ')
			*p++ = ch;

		*p = '\0';

		if (p != buf)			/* actually something entered? */
		{
		    send("AT+FCLASS=8\r");	/* select VOICE mode */
		    if (waitfor("OK",MEDMTIME)) {
			Fputs("FAIL\n",out);
			return FAIL;
		    }

		    if (set_vls(playLS)) {	/* select speaker */
			Fputs("FAIL\n",out);
			return FAIL;
		    }

		    if (docommand(buf,apigetc,apigets) == SUCCESS)
			Fputs("OK\n",out);
		    else
			Fputs("ERROR\n",out);

		    if (set_vls(0)) {
			Fputs("FAIL\n",out);
			return FAIL;
		    }
		}
		break;

	    case 'P':				/* play back */
	    case 'p':
		p = apigets();			/* read filename */
		strcpy(buf,p);

		send("AT+FCLASS=8\r");		/* select VOICE mode */
		if (waitfor("OK",MEDMTIME)) {
		    Fputs("FAIL\n",out);
		    return FAIL;
		}

		if (set_vls(playLS)) {		/* select speaker */
		    Fputs("FAIL\n",out);
		    return FAIL;
		}

		if (play(buf,NULL) != SUCCESS)
		    Fputs("ERROR invalid file\n",out);
		else
		    if (play(buf,"#bd") != SUCCESS)
			Fputs("ERROR play\n",out);
		    else
			if (strchr(MsgBuf,'b') != NULL || /* 'busy' */
			    strchr(MsgBuf,'d') != NULL) /* ... or 'dialtone' */
			    Fputs("ERROR busy or dialtone\n",out);
			else
			    Fputs("OK\n",out);

		if (set_vls(0)) {
		    Fputs("FAIL\n",out);
		    return FAIL;
		}
		break;

	    case 'R':				/* record */
	    case 'r':
		RecLS = atoi(apigets());
		strcpy(buf,RecFile);
		switch (result = record(buf,"0123456789*#bcdeq",Silence,RecMode,120))
		{
		case 0:
		    Fputs("ERROR silence\n",out);
		    break;

		case FAIL:
		    Fputs("ERROR in record\n",out);
		    break;

		default:
		    sprintf(ChatBuf,"%dsec message recorded (%s)\n",result,MsgBuf);
		    Fputs(ChatBuf,out);
		    Fputs("OK\n",out);
		    break;
		}
		RecLS = 0;
		break;

	    case 'F':				/* fax sending */
	    case 'f':
		if ((p = apigets()) == NULL)	/* read number */
		    break;
		strcpy(buf,p);

		if ((p = apigets()) == NULL)	/* read filename */
		    break;

		strcpy(buf2,p);
		Fputs("Sending FAX to ",out);
		Fputs(buf,out);
		Fputs(", file ",out);
		Fputs(buf2,out);
		Fputs("\n",out);

		switch (faxsend(buf,buf2,out))
		{
		case SUCCESS:
		    Fputs("OK\n",out);
		    break;

		case DIALFAIL:
		    Fputs("DIAL Failed\n",out);
		    break;

		case CONNECTFAIL:
		    Fputs("CONNECT Failed or incompatible\n",out);
		    break;

		case XFERFAIL:
		    Fputs("Transfer failed\n",out);
		    break;

		default:
		    Fputs("Failed\n",out);
		    break;
		}
		break;

	    case 'Q':				/* fax polling */
	    case 'q':
		if ((p = apigets()) == NULL)	/* read number */
		    break;
		strcpy(buf,p);

		p = buf2;			/* read POLL ID into buf2 */

		while (p < (buf2 + MAXLINE - 1) && (ch = getapi()) != EOF &&
			ch != '\n' && ch != '\r')
		    if (p != buf2 || ch != ' ')
			*p++ = ch;

		if (p == buf2)
		    strcpy(buf2,"0000");
		else
		    *p = '\0';

		Fputs("Polling ",out);
		Fputs(buf,out);
		Fputs(" ID ",out);
		Fputs(buf2,out);
		Fputs("\n",out);

		switch (faxpoll(buf,buf2,out))
		{
		case SUCCESS:
		    Fputs("OK\n",out);
		    break;

		case DIALFAIL:
		    Fputs("DIAL Failed\n",out);
		    break;

		case NOPOLL:
		    Fputs("No FAX POLLING available\n",out);
		    break;

		case XFERFAIL:
		    Fputs("Transfer failed\n",out);
		    break;

		default:
		    Fputs("Failed\n",out);
		    break;
		}
		break;

	    case 'D':				/* hold DATA/FAX mode */
	    case 'd':
		quit = api_eol();
		hold_state(300,DATAFAX,2);
		Fputs("DATA/FAX mode enabled for 5 minutes\n",out);
		break;

	    default:				/* unknown command */
		quit = api_eol();
		Fputs("?\n",out);
		break;
	    }
	}

	close(out);				/* close output channel */
	signal(SIGINT,oldint);
#ifdef SA_SIGINFO
	catchsig(SIGINT);
#endif
	rmlocks();				/* release the lock */
	return SUCCESS;
}

/*
**	apigets() - get space-separated string from API FIFO
**
**	Will put the string in MsgBuf and return pointer to it
**	Reads ahead one character (which is lost)
*/

char *
apigets()

{
	int ch;
	char *p;

	/* skip initial spaces
	 */
	while ((ch = getapi()) != EOF && isspace(ch))
	    ;

	p = MsgBuf;
	*p++ = ch;

	/* read characters until EOF, space, or buffer full
	 */
	while ((ch = getapi()) != EOF && !isspace(ch) &&
		p < (MsgBuf + sizeof(MsgBuf) - 1))
	    *p++ = ch;

	*p = 0;

	return (ch == EOF? NULL : MsgBuf);
}

/*
**	apigetc() - get a single char from the API FIFO (with beep prompt)
*/

int
apigetc()

{
	beep(Prompt1);				/* issue a prompting beep */
	return getapi();
}

/*
**	api_eol() - skip until newline, return 1 when EOF
*/

int
api_eol()

{
	int ch;

	while ((ch = getapi()) != EOF && ch != '\n' && ch != '\r')
	    ;

	return (ch == EOF);
}


/*************************************************************************
**	DATA mode handling						**
*************************************************************************/


/*
**	datamode() - answer an incoming call in DATA mode
**
**	'answer' can be one of:
**
**	ONLINE		the CONNECT is assumed to be already
**			received, and stored in the ChatBuf pointed by Match.
**	ANSWER		answer the call first, wait for CONNECT
**	ANSWERDATA	same, but don't allow FAX calls
*/

int
datamode(answer)
int answer;					/* answer the phone? */

{
	int n;
	sig_t (*oldhup)(),(*oldalarm)() = SIG_DFL;

	/* BREAK should not trip inthandler, getlogname returns BADSPEED.
	 * actually we cannot have a bad speed as the MODEM does the
	 * interspeeding, so we effectively ignore BREAK but re-issue
	 * the login prompt to ease life for UUCP chat scripts...
	 */
	signal(SIGINT, SIG_IGN);

	if (answer != ONLINE) {
	    debug2(D_RUN, "DATA mode, perform connect sequence\n");

	    if (WFclass > 0) {
		/* prepare MODEM for answer
		 */
		if (answer == ANSWERDATA) {
#ifdef ZFAX
		    send("ATS38.4=1+FCLASS=0\r"); /* no-FAX answer, DATA mode */
#else
		    send("ATS38.4=1+FCLASS=0;+FAA=0\r"); /* go to DATA mode */
#endif
		    if (waitfor("OK",MEDMTIME))
			return FAIL;
		} else {
#ifdef ZFAX
		    send("AT#B0+FCLASS=6\r"); /* go to DATA/FAX mode */
#else
		    send("AT+FCLASS=0;+FAA=1\r"); /* go to DATA/FAX mode */
#endif
		    if (waitfor("OK",MEDMTIME))
			return FAIL;
		}
	    }

	    /* +VNH=1	no hangup when connect attempt fails (ZyXEL) */
	    /* Mute	sets muting for answer (e.g. M0) */

	    sprintf(ChatBuf,"AT%s%sA\r",(Is_ZyXEL? "+VNH=1;" : ""),Mute);
	    send(ChatBuf);

	    /* wait some time to get the connect, 60 seconds set by MODEM */
	    /* read next answer from MODEM and make sure it is CONNECT */
	    /* (done this way to proceed quickly on other responses) */

	    for (n = 5; n > 0; n--)
		if (!waitfor(NULL,LONGTIME)) {
		    if (rip(Match) > 80) {	/* garbage? */
			logmesg("garbage from MODEM in DATA mode answer");
			n = 0;			/* terminate loop, hangup */
			break;
		    }

		    if (!strncmp(Match,"CONNECT",7))
			break;

#ifndef ZFAX
		    if (!strcmp(Match,"+FCON")) /* is it a FAX connect? */
			break;
#endif

		    /* got something else than CONNECT */

		    logmesg(Match);		/* log what we got */

		    if (!strcmp(Match,"NO CARRIER"))
			return VOICE;		/* go try VOICE mode */

		    n = 0;			/* break from loop */
		    break;
		}

	    if (!n)				/* nothing or error received */
		return FAIL;
	}

	logmesg(Match);				/* log the CONNECT we got! */
	free(Connect);
	Connect = strdup(Match);		/* and save it */

#ifdef ZFAX
	if (!strncmp(Match + 8, "FAX", 3))	/* is it a FAX connect? */
	    return FAX;
#else
	if (!strcmp(Match,"+FCON"))		/* is it a FAX connect? */
	    return FAX;
#endif

	usleep(1500000);			/* allow line to settle */

	/* from now, watch for loss of DCD
	 */
	oldhup = signal(SIGHUP, huphandler);
	settermio(WATCHDCD);

#ifdef TIOCMGET
	/* check if DCD already dropped before we got here...
	 */
	if ((n = modemstat()) != -1 && !(n & TIOCM_CAR)) {
	    logmesg("CARRIER lost within 1 second");
	    settermio(INITIAL);
	    signal(SIGHUP, oldhup);
	    waitfor(NULL,SHORTTIME);		/* flush "NO CARRIER" */
	    waitfor(NULL,SHORTTIME);		/* and other pending garbage */
	    return VOICE;			/* could be a whistle... */
	}
#endif

	/* start timer, if required
	 */
	if (TimeOut > 0) {
	    oldalarm = signal(SIGALRM, timeout);
	    alarm((unsigned) TimeOut);
	}

	/* attempt to get someone logged in
	 */
	n = login_user();

	/* stop alarm clock
	 */
	if (TimeOut > 0) {
	    alarm((unsigned) 0);
	    signal(SIGALRM, oldalarm);
	}

	signal(SIGHUP, oldhup);
	settermio(INITIAL);
	return n;
}


/*
**	login_user() - attempt to get someone logged in (DATA mode)
*/

int
login_user()

{
	struct utmp *utmp;
	FILE *fp;
	char *login;
	char *flags;
	int logintry = 3;
	int i;
	struct termios termio;
#ifdef TIOCGWINSZ
	struct winsize winsize;
#endif
	char buf[MAXLINE+1];

	debug2(D_RUN, "entering login loop\n");
	WarnCase = TRUE;

	settermio(INITIAL);			/* select normal handling */

	/* loop until a successful login is made
	 */
	for (;;)
	{
	    /* set Nusers value
	     */
	    Nusers = 0;
	    setutent();
	    while ((utmp = getutent()) != NULL) {
#ifdef	USER_PROCESS
		if (utmp->ut_type == USER_PROCESS)
#endif	/* USER_PROCESS */
		{
		    Nusers++;
		    debug3(D_UTMP, "utmp entry (%s)\n",
				    utmp->ut_name);
		}
	    }
	    endutent();
	    debug3(D_UTMP, "Nusers=%d\n", Nusers);

	    Fputs("\r",STDOUT);		/* just in case */

	    /* display ISSUE, if present
	     */
	    if (*Issue != '/') {
		Fputs(Issue,STDOUT);
		Fputs("\n",STDOUT);
	    } else {
		if ((fp = fopen(Issue, "r")) != NULL) {
		    while (fgets(buf, sizeof(buf), fp) != NULL)
			Fputs(buf,STDOUT);

		    fclose(fp);
		}
	    }

login_prompt:
	    /* display login prompt
	     */
	    Fputs("login: ",STDOUT);

	    /* eat any chars from line noise
	     */
	    ioctl(STDIN, TCFLSH, 0);
	    InPtr = InLen = 0;

	    /* take current termios to modify during logname entry
	     */
	    termio = Termio;
	    login = Login;
	    flags = LoginFlags;

	    /* handle the login name
	     */
	    switch (getlogname(&termio, buf, MAXLINE))
	    {
#ifdef FIDO
	    case FIDOCALL:		/* buf holds the type of FIDO */
		if (Fido != NULL) {
		    login = Fido;
		    flags = NULL;
		}

		logmesg("Fido Call Detected!");
#endif
	    case SUCCESS:
	    case BBSCALL:
		/* screen the username and optionally ask for password
		 * and/or dial back
		 */
		if ((i = protect(buf)) != SUCCESS) {
		    if (i == INCORRECT) {
			Fputs("Login incorrect\n",STDOUT);
			sleep(5);

			if (logintry--)
			    goto login_prompt;	/* try again */
		    }

		    return i;
		}

		/* setup terminal
		 */
		Termio = termio;	/* take suggestions by getlogname */
		settermio(FINAL);
#ifdef TIOCGWINSZ
		if (ioctl(STDIN,TIOCGWINSZ,&winsize) == 0) {
		    if (winsize.ws_row == 0)
			winsize.ws_row = 25;

		    if (winsize.ws_col == 0)
			winsize.ws_col = 80;

		    ioctl(STDIN,TIOCSWINSZ,&winsize);
		}
#endif

		/* change terminal modes/owner for login purposes
		 */
		debug2(D_RUN, "chmod and chown to root\n");
		chmod(DevName, 0622);
		chown(DevName, 0, 0);

		/* make sure all signals are set to SIG_DFL
		*/
		for (i = 0; i < NSIG; i++)
		    signal(i, SIG_DFL);

		/* exec the login program
		 */
		nice(9);
		return execlogin(login,flags,buf);

	    case BADSPEED:
		goto login_prompt;

	    case BADCASE:
		/* first try was all uppercase
		 */
		for (i=0; bad_case[i] != NULL; i++)
		    Fputs(bad_case[i],STDOUT);
		goto login_prompt;

	    case NONAME:
		/* no login name entered
		 * re-display the issue and try again
		 */
		break;

	    case FAIL:
		return FAIL;
	    }
	}
}

/*
**	execlogin() - run the login program
*/

int
execlogin(login,flags,username)
char *login;
char *flags;
char *username;

{
	int argc;
	char *argv[NARG];
	char *lbase;
	char flagbuf[80];

	/* guard against bogus username that may confuse "login"
	 */
	if (strchr(username,'-') != NULL) {
	    sprintf(MsgBuf, "bogus username %s", username);
	    logmesg(MsgBuf);
	    return FAIL;
	}

	/* log the login attempt
	 */
	if (flags != NULL)
	    sprintf(MsgBuf, "exec %s %s %s", login, flags, username);
	else
	    sprintf(MsgBuf, "exec %s %s", login, username);

	logmesg(MsgBuf);

	/* get basename of login program
	 */
	if ((lbase = strrchr(login,'/')) != NULL)
	    lbase++;
	else
	    lbase = login;

	/* build the argument vector
	 */
	argc = 0;
	memset(argv,0,sizeof(argv));

	argv[argc++] = lbase;

	/* handle multiple flags passed in the LOGINFLAGS value.
	 * the great John F. Haugh II couldn't get the parsing right
	 * in his shadow login, and so we have to pass -p and -f as
	 * separate flags, instead of -pf.  sigh.
	 */
	if (flags != NULL) {
	    char *p,*q;

	    strcpy(q = flagbuf,flags);

	    while ((p = strchr(q,' ')) != NULL) {
		*p++ = '\0';
		argv[argc++] = q;
		q = p;
	    }

	    argv[argc++] = q;
	}

	argv[argc++] = username;

	/* add strings to environment as required
	 * some "init" programs export "HOME=/", remove it
	 */
	putenv("HOME");

#ifdef	SETTERM
	sprintf(MsgBuf, "TERM=%s", Term);
	putenv(strdup(MsgBuf));
	if (LoginEnv)
	    argv[argc++] = strdup(MsgBuf);
#endif	/* SETTERM */

	if (Connect != NULL) {
	    sprintf(MsgBuf, "CONNECT=%s", Connect);
	    putenv(strdup(MsgBuf));
	    if (LoginEnv)
		argv[argc++] = strdup(MsgBuf);
	}

	if (CndName != NULL) {
	    sprintf(MsgBuf, "CNDNAME=%s", CndName);
	    putenv(strdup(MsgBuf));
	    if (LoginEnv)
		argv[argc++] = strdup(MsgBuf);
	}

	if (CndNumber != NULL) {
	    sprintf(MsgBuf, "CNDNUMBER=%s", CndNumber);
	    putenv(strdup(MsgBuf));
	    if (LoginEnv)
		argv[argc++] = strdup(MsgBuf);
	}

#ifdef	DEBUG
	if (Dfp != NULL) {
	    char **p;

	    for (p = environ; *p != NULL; p++)
		debug3(D_RUN, "environment: %s\n", *p);

	    for (p = argv; *p != NULL; p++)
		debug3(D_RUN, "argv: %s\n", *p);

	    debug2(D_RUN, "exec login program...\n");
	    fclose(Dfp);
	    Dfp = NULL;
	}
#endif	/* DEBUG */

	/* hand off to login, which can be a shell script!
	 * some systems cannot execv shell scripts directly...
	 */
	execv(login,argv);
#ifndef __linux__
	strcpy(ChatBuf,login);
	for (argc = 1; argv[argc] != NULL; argc++)
	    sprintf(ChatBuf + strlen(ChatBuf)," '%s'",argv[argc]);
	execl("/bin/sh", "sh", "-c", ChatBuf, NULL);
#endif

	/* when we get here, exec has failed
	 */
	sprintf(MsgBuf, "cannot execute %s", login);
	logmesg(MsgBuf);
	return FAIL;
}


/*
**	protect() - check username and optionally ask for dialup password
**		    and/or dial back
*/

int
protect(username)
char *username;

{
	FILE *fp;
	int match;
	int i;
	struct termios termio;
	char name[MAXLINE+1],passwd[MAXLINE+1],dialback[MAXLINE+1];
	char buf[MAXLINE+1];

	/* when PROTECT not defined, no validation
	 */
	if (Protect == NULL)
	    return SUCCESS;

	/* find logged-in username in PROTECT file
	 */
	if ((fp = fopen(Protect,"r")) == NULL) {
	    sprintf(MsgBuf, "cannot open %s", Protect);
	    logmesg(MsgBuf);
	    Fputs(MsgBuf,STDOUT);
	    Fputs("\n",STDOUT);
	    return FAIL;
	}

	while (match = -1, fgets(buf,MAXLINE,fp) != NULL &&
	       (buf[0] == '#' || (match = sscanf(buf,"%[^:\n]:%[^:\n]:%[^:\n]",
						 name,passwd,dialback)) > 0))
	{
	    if (match > 0) {
		debug5(D_GETL, "protect file: match %d name=(%s)\n",
			    match, name, passwd);

		if (!strcmp(name,username)) {
		    debug2(D_GETL, "got it!\n");
		    break;
		}
	    }
	}

	fclose(fp);

	if (match <= 0) {
	    debug2(D_GETL, "not found, ask for passwd anyway...\n");

	    Fputs("Password:",STDOUT);		/* ask for password */
	    getpasswd(buf, MAXLINE);

	    sprintf(MsgBuf, "login \"%s\" attempted, unknown", username);
	    logmesg(MsgBuf);
	    return INCORRECT;
	}

	/* we have found a matching entry
	 * see if a password has to be entered for it
	 */
	if (match > 1 && strcmp(passwd,"*")) {
passwd_prompt:
	    Fputs("Password:",STDOUT);

	    switch (getpasswd(buf, MAXLINE))
	    {
	    case NONAME:
		goto passwd_prompt;

	    case SUCCESS:
		if (strcmp(passwd,buf)) {
		    debug3(D_GETL, "incorrect passwd (%s)\n", buf);
		    sprintf(MsgBuf, "login \"%s\" attempted, incorrect passwd \"%s\"",
				username,buf);
		    logmesg(MsgBuf);
		    return INCORRECT;
		}
		break;

	    default:
		return FAIL;
	    }
	}

	/* now the entered password was okay (or there wasn't any)
	 * check if we need to dial back to this user...
	 */
	if (match > 2 && strcmp(dialback,"UUCP_U")) {
	    Fputs("Hangup the phone, you will be called back\n",STDOUT);
	    sleep(3);

	    /* hangup and init the MODEM
	     */
	    if (initmodem(TRUE) != SUCCESS)
		return FAIL;

	    /* make sure hangup is complete and defeat hackers who try
	     * to collide with the dialback
	     */
	    sleep(5 + ((unsigned)rand() % 30));

	    /* dial back in datamode
	     */
	    if ((i = dialup("",(WFclass < 0? -1 : 0),dialback)) != SUCCESS)
		return i;

	    /* now ask again for a username, as he may want to login
	     * under another name
	     */
	    WarnCase = TRUE;		/* again warn for uppercase */
	    settermio(INITIAL);		/* back to normal handling */

login_prompt:
	    Fputs("@S!login: ",STDOUT);
	    termio = Termio;
	    switch (getlogname(&termio, username, MAXLINE))
	    {
	    case SUCCESS:
	    case BBSCALL:
		break;			/* okay, continue with this name */

	    case BADSPEED:
	    case NONAME:
		goto login_prompt;

	    case BADCASE:
		/* first try was all uppercase
		 */
		for (i=0; bad_case[i] != NULL; i++)
			Fputs(bad_case[i],STDOUT);
		goto login_prompt;

	    default:
		return FAIL;		/* some problem, give up */
	    }
	}

	return SUCCESS;
}


/*
**	timeout() - handles SIGALRM while in login loop
*/

sig_t
timeout(sig)
int sig;
{
	/* log it
	 */
	sprintf(MsgBuf, "Timed out after %d seconds", TimeOut);
	logmesg(MsgBuf);

	/* say bye-bye
	 */
	Fputs("\n",STDOUT);
	Fputs(MsgBuf,STDOUT);
	Fputs(".\nBye Bye.\n",STDOUT);
	sleep(3);

	/* force a hangup
	 */
	settermio(DROPDTR);
	closeline();

	exit(EXIT_FAILURE);
}


/*
**	getlogname() - get the users login response
**
**	Returns int value indicating success.
**	If a fido-protocol is detected, return the name of the protocol.
**	Modify passed termio struct according to input conventions used.
*/

int
getlogname(termio, name, size)
struct termios *termio;
char *name;
int size;
{
	register int ch,count;
	int lower = 0;
	int upper = 0;
	char c,*p;

	debug2(D_GETL, "getlogname() called\n");

	p = name;	/* point to beginning of buffer */
	count = 0;	/* nothing entered yet */

	do
	{
	    ch = getch();

	    if ((ch & 0200) && ch != TSYNC && ch != YOOHOO) {
		termio->c_iflag |= ISTRIP;	/* better strip his bytes... */
		ch &= 0177;
	    }

	    switch (ch)
	    {
	    case EOF:				/* nobody home */
		return FAIL;

	    case CQUIT:				/* user wanted out, i guess */
		return FAIL;

	    case '\0':				/* BREAK? */
	    case CINTR:
		debug2(D_GETL, "returned (BADSPEED)\n");
		return BADSPEED;

	    case TSYNC:
		strcpy(name,"tsync");
		return FIDOCALL;

	    case YOOHOO:
		strcpy(name,"yoohoo");
		return FIDOCALL;

	    case ESC:				/* for frontdoor addicts */
		termio->c_iflag |= ICRNL;	/* turn on cr/nl xlate */
		termio->c_oflag |= ONLCR;
		Fputs("bbs\n",STDOUT);
		strcpy(name,"bbs");
		return BBSCALL;

	    case BS:
#if CINTR != DEL
	    case DEL:				/* alternate way to backspace */
#endif
#if CERASE != BS && CERASE != DEL
	    case CERASE:
#endif
		termio->c_cc[VERASE] = ch;
		if (count) {
		    if (!(termio->c_lflag & ECHOE))
			    Fputs("\b \b",STDOUT);
		    --p;
		    --count;
		}
		break;

	    case CKILL:
	    case CWERASE:
		if (!(termio->c_lflag & ECHOK))
		    Fputs("\n",STDOUT);

		p = name;
		count = 0;
		break;

	    case '\r':
		termio->c_iflag |= ICRNL;	/* turn on cr/nl xlate */
		termio->c_oflag |= ONLCR;
		break;

	    case '\n':
		break;

	    case CEOF:
		if (p == name)			/* ctrl-d was first char */
		    return FAIL;
		/* fall through... */

	    default:				/* any normal character... */
		if (isspace(ch) || !isprint(ch))
		    break;			/* ignore dirty stuff */

		if (!(termio->c_lflag & ECHO)) {
		    c = ch;
		    write(STDOUT, &c, 1);
		}

		*p++ = ch;
		count++;
		if (islower(ch))
		    lower++;
		if (isupper(ch))
		    upper++;
		break;
	    }
	} while ((ch != '\n') && (ch != '\r') && (count < size));

	*p = '\0';				/* terminate buffer */
	Fputs("\n",STDOUT);

	if (name[0] == '\0') {
		debug2(D_GETL, "returned NONAME\n");
		return NONAME;
	}

	if (upper && !lower) {
		if (WarnCase) {
			WarnCase = FALSE;
			debug3(D_GETL, "returned BADCASE (%s)\n", name);
			return BADCASE;
		}

		termio->c_iflag |= IUCLC;
		termio->c_oflag |= OLCUC;
		termio->c_lflag |= XCASE;
	}

	/* the ~?~?~? garbage often results when the modems have not */
	/* negotiated error correction... */

	if (count == size &&
	    (!strncmp(name,"~?~?~?~?",8) || !strncmp(name,"?~?~?~?~",8)))
		return FAIL;

	debug3(D_GETL, "returned SUCCESS, name=(%s)\n", name);
	return SUCCESS;
}


/*
**	getpasswd() - get the users password response
**		      it is not echoed, but erase/kill are handled
*/

int
getpasswd(passwd, size)
char *passwd;
int size;
{
	register int ch,count;
	char *p;

	debug2(D_GETL, "getpasswd() called\n");

	p = passwd;	/* point to beginning of buffer */
	count = 0;	/* nothing entered yet */

	do
	{
	    switch (ch = getch() & 0177)
	    {
	    case EOF:				/* nobody home */
		return FAIL;

	    case CQUIT:				/* user wanted out, i guess */
		return FAIL;

	    case '\0':				/* BREAK? */
	    case CINTR:
		debug2(D_GETL, "returned (BADSPEED)\n");
		return BADSPEED;

	    case BS:
#if CINTR != DEL
	    case DEL:
#endif
#if CERASE != BS && CERASE != DEL
	    case CERASE:
#endif
		if (count) {
		    --p;
		    --count;
		}
		break;

	    case CKILL:
	    case CWERASE:
		p = passwd;
		count = 0;
		break;

	    case '\r':
	    case '\n':
		break;

	    case CEOF:
		if (p == passwd)		/* ctrl-d was first char */
		    return FAIL;
		/* fall through... */

	    default:				/* any normal character... */
		if (isspace(ch) || !isprint(ch))
		    break;			/* ignore dirty stuff */

		*p++ = ch;
		count++;
		break;
	    }
	} while ((ch != '\n') && (ch != '\r') && (count < size));

	*p = '\0';				/* terminate buffer */
	Fputs("\n",STDOUT);

	if (passwd[0] == '\0') {
		debug2(D_GETL, "returned NONAME\n");
		return NONAME;
	}

	debug2(D_GETL, "returned SUCCESS\n");
	return SUCCESS;
}


#ifdef ZFAX
/*************************************************************************
**	ZFAX mode handling						**
*************************************************************************/

/*
**	faxmode() - main handling of FAX calls (ZFAX mode)
**
**	'answer' can be one of:
**
**	ONLINE		the CONNECT is assumed to be already received
**	ANSWER		answer the call first, wait for CONNECT
*/

int
faxmode(answer)
int answer;

{
	debug2(D_RUN, "FAX mode\n");

	return faxrecv(answer,-1);
}

/*
**	faxpoll() - poll a FAX and receive any data it may have (ZFAX mode)
*/

int
faxpoll(number,pollid,logfd)
char *number;
char *pollid;
int logfd;

{
	if (!Is_ZyXEL)				/* ZFAX requires ZyXEL */
	    return FAIL;

	/* dialup in FAX POLLING mode
	 */
	if (dialup("#F#B2#V1#T1#L2#C7",6,number) != SUCCESS) {
	    if (!strncmp(Match,"DISCONNECT1",11))
		return NOPOLL;

	    return DIALFAIL;
	}

	return faxrecv(FAXPOLL,logfd);
}

/*
**	faxrecv() - receive a FAX and write it to a file (ZFAX mode)
*/

int
faxrecv(answer,logfd)
int answer;
int logfd;

{
	register FILE *fp;
	register int ch;
	const unsigned char *rtc;
	int ematch,rmatch,rtclen;
	FAX_HEADER header;
	char filename[MAXLINE];
	static const char end[] = "\r\nDISCONNECT";	/* result code */
#define NMATCH	12

	if (!Is_ZyXEL)				/* ZFAX requires ZyXEL */
	    return FAIL;

	if (answer == ANSWER) {
	    int n;

	    /* select FAX mode
	     */
	    send("ATS38.4=0#F#B0+FCLASS=6\r");
	    if (waitfor("OK",MEDMTIME))
		return FAIL;

	    /* answer the call
	     * +VNH=1	no hangup when connect attempt fails (ZyXEL)
	     * Mute	sets muting for answer (e.g. M0)
	     */
	    sprintf(ChatBuf,"AT%s%sA\r",(Is_ZyXEL? "+VNH=1;" : ""),Mute);
	    send(ChatBuf);

	    /* now wait for connection, may take some time
	     */
	    for (n = 5; n > 0; n--)
		if (!waitfor(NULL,LONGTIME)) {
		    rip(Match);

		    if (!strncmp(Match,"CONNECT",7)) /* prefix match only */
			break;

		    /* got something else than expected */

		    logmesg(Match);		/* log what we got (NO CARRIER?) */
		    debug3(D_RUN, "faxmode() - answer failed (%s)\n", Match);
		    return FAIL;
		}

	    if (!n)				/* nothing received */
		return FAIL;

	    logmesg(Match);			/* log the CONNECT we got! */
	    free(Connect);
	    Connect = strdup(Match);		/* and save it */
	}

	/* check if we have a FAX connection */

	if (strncmp(Connect + 8, "FAX", 3))	/* is it a FAX connect? */
	    return DATA;			/* no, forget FAX receive */

	if (logfd >= 0) {
	    sprintf(ChatBuf,"\t%s\r\n",Connect);
	    write(logfd,ChatBuf,strlen(ChatBuf));
	}

	/* create filename for recording, add extension */

	if (FaxDir != NULL && FaxFile[0] != '/') {
	    sprintf(filename,"%s/%s",FaxDir,FaxFile);

	    if (strcmp(filename + strlen(filename) - 4,FaxExt))
		strcat(filename,FaxExt);
	} else
	    strcpy(filename,FaxFile);

	/* %-macro's will be expanded using strftime, to generate */
	/* a time-dependent recording file */

	if (strchr(filename,'%') != NULL) {
	    time_t now = time(NULL);
	    char buf[MAXLINE];

	    strftime(buf,sizeof(buf),filename,localtime(&now));
	    strcpy(filename,buf);
	}

	if ((fp = fopen(filename,"w")) == NULL) {
	    sprintf(MsgBuf, "cannot create faxfile (%s)", filename);
	    logmesg(MsgBuf);
	    return FAIL;
	}

	faxheader(&header,Connect);		/* prepare a FAX file header */
	fwrite(&header,sizeof(header),1,fp);

	settermio(LONGTIME);			/* prepare for 15-sec timeout */
	ematch = rmatch = 0;

	if (header.flags & FAX_T1) {
	    rtc = fax_rtc_2d;
	    rtclen = RTC_2D_LEN;
	} else {
	    rtc = fax_rtc_1d;
	    rtclen = RTC_1D_LEN;
	}

	if (logfd >= 0)
	    write(logfd,"\t[page data]\r\n",14);

	/* copy data to outputfile until timeout
	 * match the RTC string, and increment pagecount when it is found
	 * match the end string, and terminate when it is found
	 */
	while ((ch = getch()) != EOF) {
	    if (ch != rtc[rmatch])
		rmatch = 0;
	    else
		if (++rmatch == rtclen) {
		    /* matched entire RTC string, end of page */

		    header.pages++;
		    rmatch = 0;
		}

	    if (ch != end[ematch]) {
		/* mismatch, write possibly-matched data
		 */
		if (ematch != 0) {
		    fwrite(end,1,ematch,fp);
		    ematch = 0;
		}

		putc(ch,fp);
	    } else {
		/* match of endstring, check if we have matched entire
		 * string
		 */
		if (++ematch == NMATCH) {
		    char *p;

		    /* matched entire end string, assume FAX has ended
		     * transfer end string to MsgBuf (up to CR)
		     */
		    strcpy(MsgBuf,end + 2);	/* copy except CR/LF */
		    p = MsgBuf + NMATCH - 2;

		    while ((ch = getch()) != EOF && ch != '\r')
			if (p < (MsgBuf + sizeof(MsgBuf) - 2))
			    *p++ = ch;

		    *p = '\0';

		    while (--p >= MsgBuf && *p == ' ')
			*p = '\0';

		    logmesg(MsgBuf);		/* log the end string */

		    if (logfd >= 0) {
			sprintf(ChatBuf,"\t%s\r\n",MsgBuf);
			write(logfd,ChatBuf,strlen(ChatBuf));
		    }

		    waitfor(NULL,SHORTTIME);	/* flush CR/LF */
		    break;
		}
	    }
	}

	fflush(fp);
	ch = ftell(fp) - sizeof(header);	/* see how much we wrote */

	fseek(fp,0L,SEEK_SET);			/* re-write (modified) header */
	fwrite(&header,sizeof(header),1,fp);

	if (!!ferror(fp) | fclose(fp) == EOF)
	    logmesg("Error writing FAX file");

	if (ch == 0) {
	    unlink(filename);
	    if (answer != FAXPOLL)
		logmesg("FAX call, no data recorded");
	} else {
	    sprintf(ChatBuf,"FAX %s, %d pages received",
		    (answer == FAXPOLL? "poll":"call"),header.pages);
	    logmesg(ChatBuf);
	}

	send("ATH\r");
	waitfor("OK",MEDMTIME);

	if (ch > 0 && FaxMail != NULL) {
	    sprintf(ChatBuf, "%s %s", MAILER, FaxMail);
	    if ((fp = popen(ChatBuf, "w")) != NULL) {
		fprintf(fp, "From: %s %s\n", MyName, Device);
		fprintf(fp, "To: %s\n", FaxMail);
		fprintf(fp, "Subject: FAX arrived\n\n");
		fprintf(fp, "A %d-page FAX has arrived from: %s\n",
				header.pages, Connect + 28);
		if (CndNumber != NULL)
		    fprintf(fp, "NUMBER: %s\n", CndNumber);
		if (CndName != NULL)
		    fprintf(fp, "NAME: %s\n", CndName);
		fprintf(fp, "\n%d bytes have been recorded to file %s\n",
				ch, filename);
		pclose(fp);
	    }
	}

	return (MsgBuf[10] == '0')? SUCCESS : XFERFAIL;
}

/*
**	faxsend() - send a FAX (ZFAX mode)
*/

int
faxsend(number,filename,logfd)
char *number;
char *filename;
int logfd;

{
	FILE *fp;
	int res,num;
	long arg;
	FAX_HEADER header,connect;
	char buf[BUFSIZ];

	if (!Is_ZyXEL)				/* ZFAX requires ZyXEL */
	    return FAIL;

	/* open the file */

	if ((fp = fopen(filename,"r")) == NULL) {
	    sprintf(MsgBuf, "cannot open faxfile (%s)", filename);
	    logmesg(MsgBuf);
	    return FAIL;
	}

	/* read and check the file header */

	if (fread(&header,sizeof(header),1,fp) != 1 ||
	    memcmp(header.zyxel,FAX_MAGIC,sizeof(header.zyxel)))
	{
	    fclose(fp);
	    return FAIL;
	}

	/* prepare the mode string from FAX header data */

	switch (header.rec_width)
	{
	case FAX_R1:
	    res = 1;
	    break;

	case FAX_R2:
	    res = 2;
	    break;

	default:
	    res = 0;
	    break;
	}

	sprintf(buf,"#F#B0#V%d#T%d#R%d#L2#C7",
		((header.flags & FAX_V1)? 1 : 0),
		((header.flags & FAX_T1)? 1 : 0),
		res);

	/* dialup in FAX SENDING mode
	 */
	if (dialup(buf,6,number) != SUCCESS)
	    return DIALFAIL;

	if (logfd >= 0) {
	    sprintf(ChatBuf,"\t%s\r\n",Connect);
	    write(logfd,ChatBuf,strlen(ChatBuf));
	}

	/* analyze the CONNECT to see if it is compatible with our header
	 */
	faxheader(&connect,Connect);

	if (header.rec_width > connect.rec_width ||
	    header.flags != connect.flags)
	{
	    fclose(fp);
	    logmesg("FAX CONNECT incompatible with file");
	    return CONNECTFAIL;
	}

	/* everything OK, start sending the FAX
	 */
	settermio(IMMEDIATE);			/* immediate return on read */

	if (logfd >= 0)
	    write(logfd,"\t[page data]\r\n",14);

	while ((num = fread(buf,1,sizeof(buf),fp)) > 0) {
	    if (write(STDOUT,buf,num) != num)
		break;

	    switch (getch())			/* something returned? */
	    {
	    case EOF:				/* nothing */
	    case XON:				/* leaked flow control char */
	    case XOFF:
		break;				/* ignore it and go on */

	    default:				/* something else */
		debug2(D_RUN, "faxsend() terminated by unexpected char\n");
		InPtr--;			/* push it back */
		waitfor(NULL,SHORTTIME);	/* and flush the input */
		goto quit;			/* stop sending */
	    }
	}
quit:
	fclose(fp);

	/* end of file, send an extra RTC in case the file did not
	 * contain one (hope this does not result in an empty page...)
	 */
	if (header.flags & FAX_T1)
	    write(STDOUT,fax_rtc_2d,RTC_2D_LEN);
	else
	    write(STDOUT,fax_rtc_1d,RTC_1D_LEN);

	/* all sent, now we must lower RTS to signal the end of the FAX
	 * it is better to first wait until all data sent...  I'm afraid
	 * we'll interrupt the transfer when RTS is made low while sending.
	 */

	settermio(IMMEDIATE);			/* wait for data to be sent */
	settermio(XONXOFF);			/* no RTS/CTS for now... */
	usleep(20000);
	arg = TIOCM_RTS;			/* RTS OFF */
	ioctl(STDIN,TIOCMBIC,&arg);

	/* now wait for the result, and see what it is
	 */
	for (res = 0; res < 5; res++)
	    if (waitfor(NULL,LONGTIME) != FAIL) /* read result */
		break;

	arg = TIOCM_RTS;			/* RTS ON after response */
	ioctl(STDIN,TIOCMBIS,&arg);
	settermio(INITIAL);			/* enable RTS/CTS handshake */

	rip(Match);
	strcpy(MsgBuf,Match);			/* save result */
	logmesg(MsgBuf);			/* log the result string */

	if (logfd >= 0) {
	    sprintf(ChatBuf,"\t%s\r\n",MsgBuf);
	    write(logfd,ChatBuf,strlen(ChatBuf));
	}

	send("ATH\r");				/* just to make sure... */
	waitfor("OK",MEDMTIME);

	return ((int)strlen(MsgBuf) > 10 && MsgBuf[10] == '0')? SUCCESS : XFERFAIL;
}


/*
**	faxheader() - convert CONNECT info to FAX file header (ZFAX mode)
**
**	page count will be left at zero
*/

void
faxheader(header,connect)
FAX_HEADER *header;
char *connect;

{
	char *p;

	/* clear header and put fixed values */

	memset(header,0,sizeof(FAX_HEADER));
	memcpy(header->zyxel,FAX_MAGIC,sizeof(header->zyxel));
	header->factory = 'Z';
	header->version = 2;

	if (connect != NULL) {
	    if ((p = strchr(connect + 11,'V')) != NULL)
		if (p[1] != '0')
		    header->flags |= FAX_V1;	/* high resolution */

	    if ((p = strchr(connect + 11,'T')) != NULL)
		if (p[1] != '0')
		    header->flags |= FAX_T1;	/* 2-D encoding */

	    if ((p = strchr(connect + 11,'R')) != NULL)
		switch (p[1])
		{
		case '0':
		    header->rec_width = FAX_R0; /* 1728 pixels */
		    break;

		case '1':
		    header->rec_width = FAX_R1; /* 2048 pixels */
		    break;

		case '2':
		    header->rec_width = FAX_R2; /* 2432 pixels */
		    break;
		}
	}
}
#else
/*************************************************************************
**	CLASS2 FAX mode handling					**
*************************************************************************/

/*
**	faxmode() - main handling of FAX calls (CLASS 2 mode)
**
**	'answer' can be one of:
**
**	ONLINE		the +FCON is assumed to be already received
**	ANSWER		answer the call first, wait for +FCON
**	FAXPOLL		used when polling
*/

int
faxmode(answer)
int answer;

{
	char *name;
	int logfd = -1;
	int rv;
	char tmpname[MAXLINE];

	debug2(D_RUN, "FAX mode\n");

	/* create a logfile for the FAX mode responses */

	sprintf(tmpname,"%s/%sXXXXXX",P_tmpdir,MyName);

	if ((name = mktemp(tmpname)) != NULL && name[0] != '\0')
	    logfd = open(name,O_RDWR|O_CREAT|O_TRUNC,0600);

	/* receive the FAX and remove the logfile */

	rv = faxrecv(answer,logfd);

	if (logfd >= 0) {
	    close(logfd);
	    unlink(name);
	}

	return rv;
}

/*
**	faxpoll() - poll a FAX and receive any data it may have (CLASS 2 mode)
*/

int
faxpoll(number,pollid,logfd)
char *number;
char *pollid;
int logfd;

{
	int res;

	send("AT+FCLASS=2\r");
	if (waitfor("OK",MEDMTIME))
	    return FAIL;

	/* set polling station ID
	 */
	if (pollid != NULL) {
	    send("AT+FNS=\"\"\r");		/* reset to null string */
	    if (waitfor("OK",MEDMTIME))
		return FAIL;

	    sprintf(ChatBuf,"AT+FNS=\"%s",poll_hex);
	    while (*pollid != '\0')
		sprintf(ChatBuf + strlen(ChatBuf),"%02X",*pollid++);
	    strcat(ChatBuf,"\"\r");

	    send(ChatBuf);			/* send nonstandard string */
	    if (waitfor("OK",MEDMTIME))
		return FAIL;
	}

	/* set local ID for polling (necessary?)
	 */
	if (FaxId != NULL) {
	    sprintf(ChatBuf,"AT+FCIG=\"%.20s\"\r",FaxId);
	    send(ChatBuf);
	    if (waitfor("OK",MEDMTIME))
		return FAIL;
	}

	/* dialup in FAX POLLING mode
	 */
	if (dialup("+FSPL=1;",2,number) != SUCCESS)
	    return DIALFAIL;

	res = faxrecv(FAXPOLL,logfd);

	send("AT+FSPL=0\r");
	waitfor("OK",MEDMTIME);

	return res;
}

/*
**	faxrecv() - receive a FAX and write it to a file (CLASS2 mode)
*/

int
faxrecv(answer,logfd)
int answer;
int logfd;

{
	register FILE *fp;
	register int ch;
	int pages,bytes;
	char dirname[MAXLINE],filename[MAXLINE];

	if (answer == ANSWER) {
	    int n;

	    /* select FAX mode
	     */
	    sprintf(ChatBuf,"AT%s+FCLASS=2;+FAA=0\r",(Is_ZyXEL? "S38.4=0":""));
	    send(ChatBuf);
	    if (waitfor("OK",MEDMTIME))
		return FAIL;

	    /* answer the call
	     * +VNH=1	no hangup when connect attempt fails (ZyXEL)
	     * Mute	sets muting for answer (e.g. M0)
	     */
	    sprintf(ChatBuf,"AT%s%sA\r",(Is_ZyXEL? "+VNH=1;":""),Mute);
	    send(ChatBuf);

	    /* now wait for connection, may take some time
	     */
	    for (n = 5; n > 0; n--)
		if (!waitfor(NULL,LONGTIME)) {
		    rip(Match);

		    if (logfd >= 0) {
			sprintf(MsgBuf,"\t%s\r\n",Match);
			write(logfd,MsgBuf,strlen(MsgBuf));
		    }

		    if (!strcmp(Match,"+FCON"))
			break;

		    /* got something else than expected */

		    logmesg(Match);		/* log what we got (NO CARRIER?) */
		    debug3(D_RUN, "faxmode() - answer failed (%s)\n", Match);
		    return FAIL;
		}

	    if (!n)				/* nothing received */
		return FAIL;
	}

	/* get the connection info
	 */
	FaxInfo = blank_info;			/* set everything to unknown */

	if (fax2rsp(logfd) != SUCCESS || strcmp(MsgBuf,"OK"))
	    return FAIL;

	/* log it for later reference
	 */
	sprintf(MsgBuf,"FAX connect: DCS=%d,%d,%d,%d,%d,%d,%d,%d ID=%s",
		FaxInfo.vr,FaxInfo.br,FaxInfo.wd,FaxInfo.ln,
		FaxInfo.df,FaxInfo.ec,FaxInfo.bf,FaxInfo.st,
		FaxInfo.id);
	logmesg(MsgBuf);

	/* are we polling?
	 */
	if (answer == FAXPOLL) {
	    if (!FaxInfo.poll) {
		send("ATH\r");
		fax2rsp(logfd);
		return NOPOLL;			/* no fax available for polling */
	    }
	} else {
	    int rv;

	    /* we are not polling, check if remote is polling us */

	    if ((rv = faxcpoll(logfd)) != NOPOLL)
		return rv;
	}

	/* create directory name for storing recorded pages */

	if (FaxDir != NULL && FaxFile[0] != '/')
	    sprintf(dirname,"%s/%s",FaxDir,FaxFile);
	else
	    strcpy(dirname,FaxFile);

	/* %-macro's will be expanded using strftime, to generate */
	/* a time-dependent recording directory */

	if (strchr(dirname,'%') != NULL) {
	    time_t now = time(NULL);
	    char buf[MAXLINE];

	    strftime(buf,sizeof(buf),dirname,localtime(&now));
	    strcpy(dirname,buf);
	}

	/* make the directory to store the pages
	 */
	if (mkdir(dirname,0755) != 0) {
	    sprintf(MsgBuf, "cannot create faxdir (%s)", dirname);
	    logmesg(MsgBuf);
	    return FAIL;
	}

	pages = bytes = 0;

	/* loop to receive all pages in the FAX message...
	 */
	do
	{
	    send("AT+FDR\r");			/* ready to receive a page */

	    if (logfd >= 0)
		write(logfd,"\tAT+FDR\r\n",9);

	    if (fax2rsp(logfd) != SUCCESS || strcmp(MsgBuf,"CONNECT"))
		break;

	    if (faxcpoll(logfd) != NOPOLL)
		break;				/* fax polling detected */

	    if (!FaxInfo.cfr && pages == 0)
		logmesg("FAX receive: no +FCFR (trying to continue)");

	    sprintf(filename,"%s/p%02d_%c%d%s",
		dirname,pages + 1,(FaxInfo.vr? 'f':'n'),FaxInfo.df + 1,FaxExt);

	    if ((fp = fopen(filename,"w")) == NULL) {
		sprintf(MsgBuf, "cannot create faxfile (%s)", filename);
		logmesg(MsgBuf);
		break;
	    }

	    if (logfd >= 0)
		write(logfd,"\t[page data]\r\n",14);

	    /* don't switch to XON/XOFF mode here.  we are receiving */
	    /* binary data and eating these characters is not a good idea... */
	    /* let's simply assume we are fast enough to keep up */

	    send("\022");			/* send DC2 */
	    settermio(SHORT255);
	    debug2(D_RUN, "started FAX data receive\n");

	    /* read the FAX data, it is terminated by DLE-ETX */
	    /* or a timeout on receiving FAX data */

	    for (;;)
	    {
		switch (ch = getch())
		{
		case EOF:
		    debug2(D_RUN, "FAX data receive timeout\n");
		    goto done;

		case DLE:
		    switch (ch = getch())
		    {
		    case EOF:
			debug2(D_RUN, "FAX data receive timeout\n");
			goto done;

		    case ETX:			/* end of message! */
			goto done;

		    case DLE:			/* doubled DLE, just write */
			putc(ch,fp);
			break;

		    default:			/* something else... */
			debug3(D_FAX, "Received DLE-%02x in FAX data\n", ch);
			break;
		    }
		    break;

		default:
		    putc(ch,fp);		/* normal FAX data */
		    break;
		}
	    }
done:

	    ch = ftell(fp);			/* bytes received */

	    if (fclose(fp) == EOF) {
		logmesg("Error writing FAX file");
		ch = 0;
	    }

	    if (ch == 0)
		unlink(filename);
	    else {
		bytes += ch;
		pages++;
	    }

	    if (fax2rsp(logfd) != SUCCESS || strcmp(MsgBuf,"OK"))
		break;

	    switch (FaxInfo.et)
	    {
	    case 1:				/* end of doc, another follows */
		FaxInfo.et = 0;			/* ignore, just continue */
		break;

	    case 2:				/* end of doc, no more follows */
		break;				/* stop */

	    case 4:				/* procedure interrupt, another page */
	    case 5:				/* procedure interrupt, end of doc, another follows */
	    case 6:				/* procedure interrupt, end of doc */
		sprintf(MsgBuf,"Procedure interrupt %d",FaxInfo.et);
		logmesg(MsgBuf);
		break;
	    }
	} while (FaxInfo.hng < 0 && FaxInfo.pts < 4 && FaxInfo.et == 0);

	/* see if they are polling us now...
	 */
	faxcpoll(logfd);

	/* when call not yet terminated, request completion status
	 */
	if (FaxInfo.hng < 0) {
	    send("AT+FDR\r");

	    if (logfd >= 0)
		write(logfd,"\tAT+FDR\r\n",9);

	    fax2rsp(logfd);
	}

	sprintf(ChatBuf,"FAX %s, %d pages received, result=%d",
		(answer == FAXPOLL? "poll":"call"),pages,FaxInfo.hng);
	logmesg(ChatBuf);

	if (bytes == 0)
	    rmdir(dirname);

	/* end of call, hangup
	 */
	send("ATH\r");
	fax2rsp(logfd);

	/* generate mail when requested
	 */
	if (bytes > 0 && FaxMail != NULL) {
	    sprintf(ChatBuf, "%s %s", MAILER, FaxMail);
	    if ((fp = popen(ChatBuf, "w")) != NULL) {
		fprintf(fp, "From: %s %s\n", MyName, Device);
		fprintf(fp, "To: %s\n", FaxMail);
		fprintf(fp, "Subject: FAX arrived\n\n");
		fprintf(fp, "A %d-page FAX has arrived from: %s\n",
				pages, FaxInfo.id);
		if (CndNumber != NULL)
		    fprintf(fp, "NUMBER: %s\n", CndNumber);
		if (CndName != NULL)
		    fprintf(fp, "NAME: %s\n", CndName);
		fprintf(fp, "\n%d bytes have been recorded into directory %s\n",
				bytes, dirname);

		if (logfd >= 0 && !isatty(logfd) && lseek(logfd,0L,SEEK_SET) == 0)
		{
		    int n;

		    fprintf(fp,"\nTransfer log:\n");

		    while ((bytes = read(logfd,ChatBuf,sizeof(ChatBuf))) > 0)
			for (n = 0; n < bytes; n++)
			    if (ChatBuf[n] != '\r')
				fputc(ChatBuf[n],fp);
		}

		pclose(fp);
	    }
	}

	return (FaxInfo.hng == 0)? SUCCESS : XFERFAIL;
}

/*
**	faxcpoll() - check if remote is polling for a FAX (CLASS 2 mode)
*/

int
faxcpoll(logfd)
int logfd;

{
	int pass,rv;
	char pollid[5],orgpoll[5];
	char filename[MAXLINE];

	if (FaxInfo.hng >= 0 || !FaxInfo.dtc)
	    return NOPOLL;			/* no poll request received */

	if (FaxPoll == NULL)
	    return NOPOLL;			/* poll request, but no doc */

	strcpy(pollid,"0000");			/* default poll-id */
	strcpy(MsgBuf,"FAX poll");
	orgpoll[0] = '\0';

	/* check for optional 4-digit poll id, which is passed as a
	 * nonstandard command.  keep default 0000 when not found.
	 * the format of the nonstandard command is unclear. there is
	 * an example in a ZyXEL doc which uses the string in poll_nsc,
	 * but I have seen at least one FAX which sent something completely
	 * different.  in an attempt to decode many possibilities, a
	 * binary value 04 followed by 4 ASCII digits is also recognized
	 * as a POLL-ID.  The 4 digits are sent least-significant-first...
	 */

	if (FaxInfo.nstyp == 'C')
	{
	    unsigned char *p;
	    int idset = 0;

	    if (FaxInfo.nslen > sizeof(poll_nsc) && !memcmp(FaxInfo.ns,poll_nsc,sizeof(poll_nsc))) {
		memcpy(pollid,FaxInfo.ns + sizeof(poll_nsc),4);
		idset = 1;
	    } else
		for (p = FaxInfo.ns; p < FaxInfo.ns + FaxInfo.nslen - 4; p++)
		    if (p[0] == 0x04 &&
			isdigit(p[1]) && isdigit(p[2]) &&
			isdigit(p[3]) && isdigit(p[4]))
		    {
			pollid[0] = p[4];
			pollid[1] = p[3];
			pollid[2] = p[2];
			pollid[3] = p[1];
			idset = 1;
			break;
		    }

	    if (idset)
	    {
		strcpy(orgpoll,pollid);
		sprintf(MsgBuf + strlen(MsgBuf),", POLL-ID: %s",pollid);
	    }
	}

	logmesg(MsgBuf);

	if (logfd >= 0) {
	    sprintf(ChatBuf,"\t[%s]\r\n",MsgBuf);
	    write(logfd,ChatBuf,strlen(ChatBuf));
	}

	for (pass = 0; pass < 2; pass++)
	{
	    sprintf(filename,FaxPoll,pollid);	/* generate filename */

	    if (!access(filename,R_OK))
	    {
		rv = faxsend(NULL,filename,logfd);

		/* generate mail when requested
		 */
		if (FaxMail != NULL) {
		    FILE *fp;

		    sprintf(ChatBuf, "%s %s", MAILER, FaxMail);
		    if ((fp = popen(ChatBuf, "w")) != NULL) {
			fprintf(fp, "From: %s %s\n", MyName, Device);
			fprintf(fp, "To: %s\n", FaxMail);
			fprintf(fp, "Subject: FAX polled\n\n");
			fprintf(fp, "FAX has been polled by: %s\n", FaxInfo.id);
			if (orgpoll[0] != '\0')
			    fprintf(fp, "POLL-ID: %s\n", orgpoll);
			if (CndNumber != NULL)
			    fprintf(fp, "NUMBER: %s\n", CndNumber);
			if (CndName != NULL)
			    fprintf(fp, "NAME: %s\n", CndName);

			if (logfd >= 0 && !isatty(logfd) && lseek(logfd,0L,SEEK_SET) == 0)
			{
			    int n,bytes;

			    fprintf(fp,"\nTransfer log:\n");

			    while ((bytes = read(logfd,ChatBuf,sizeof(ChatBuf))) > 0)
				for (n = 0; n < bytes; n++)
				    if (ChatBuf[n] != '\r')
					fputc(ChatBuf[n],fp);
			}

			pclose(fp);
		    }
		}

		return rv;
	    }

	    strcpy(pollid,"0000");		/* try default poll-id */
	}

	return NOPOLL;				/* no pollfile found */
}

/*
**	faxsend() - send a FAX (CLASS 2 mode)
*/

int
faxsend(number,name,logfd)
char *number;
char *name;
int logfd;

{
	char *p,*q;
	int fd,ptr,num,rmatch,file,nfiles,vr,df,retry,fet;
	int rv = FAIL;
	int skip = 0;
	pid_t gpid,bpid,cpid;
	off_t pagepos = 0;
	struct dirent **namelist = NULL;
	struct stat st;
	char filename[MAXLINE];
	char buf[BUFSIZ];

	/* check if the passed name exists, and
	 * if it is a directory (in that case it contains the
	 * files that have to be sent)
	 */
	if (stat(name,&st) != 0) {
	    sprintf(MsgBuf, "cannot open faxfile (%s)", name);
	    logmesg(MsgBuf);
	    return FAIL;
	}

	if (S_ISDIR(st.st_mode)) {
	    /* get the file list in the directory passed to us */

	    /* this line may generate a warning for arg #3, depending on */
	    /* the version of <dirent.h> */
	    nfiles = scandir(name,&namelist,dir_select,alphasort);

	    if (nfiles == 0)
		return FAIL;
	} else {
	    strcpy(filename,name);
	    nfiles = -1;
	}

	if (number != NULL) {
	    /* dialup in FAX SENDING mode
	     */
	    if (dialup("",2,number) != SUCCESS) {
		rv = DIALFAIL;
		goto fail;
	    }

	    /* get the connection info
	     */
	    FaxInfo = blank_info;		/* set everything to unknown */

	    if (fax2rsp(logfd) != SUCCESS || strcmp(MsgBuf,"OK"))
		goto fail;

	    /* log it for later reference
	     */
	    sprintf(MsgBuf,"FAX connect: DCS=%d,%d,%d,%d,%d,%d,%d,%d ID=%s",
		    FaxInfo.vr,FaxInfo.br,FaxInfo.wd,FaxInfo.ln,
		    FaxInfo.df,FaxInfo.ec,FaxInfo.bf,FaxInfo.st,
		    FaxInfo.id);
	    logmesg(MsgBuf);
	}

	file = 0;

	do
	{
	    /* see what file we should send */

	    if (nfiles > 0) {
		sprintf(filename,"%s/%.*s",name,namelist[file]->d_reclen,
						namelist[file]->d_name);
		file++;
	    }

	    /* open the file */

	    if ((fd = open(filename,O_RDONLY)) < 0) {
		sprintf(MsgBuf, "cannot open faxfile (%s)", filename);
		logmesg(MsgBuf);
		goto fail;
	    }

	    /* read first chunk of data */

	    num = read(fd,buf,sizeof(buf));
	    ptr = 0;
	    gpid = bpid = cpid = 0;

	    /* determine file type from first data byte(s) */

	    switch (buf[0])
	    {
	    case 0:
		/* looks like a raw G3 file */

		if (logfd >= 0) {
		    sprintf(MsgBuf,"\tG3 file: %s\r\n",filename);
		    write(logfd,MsgBuf,strlen(MsgBuf));
		}

		/* guess vertical resolution and data format from name */

		if ((p = strrchr(filename,'_')) != NULL &&
		    (p[1] == 'n' || p[1] == 'f') && isdigit(p[2]))
		{
		    vr = (p[1] == 'f');
		    df = p[2] - '1';
		}
		else
		    vr = df = 0;
		break;

	    case 0x1f:
		if (Gunzip != NULL && GhostScript != NULL &&
		    ((unsigned char)buf[1] == 0x8b || (unsigned char)buf[1] == 0x9d))
		{
		    char *argv[2];

		    /* looks like a gzipped or compressed file */

		    if (logfd >= 0) {
			sprintf(MsgBuf,"\tCompressed file: %s\r\n",filename);
			write(logfd,MsgBuf,strlen(MsgBuf));
		    }

		    lseek(fd,0L,SEEK_SET);	/* rewind the file */
		    num = 0;
		    memset(argv,0,sizeof(argv)); /* build argvector for GUNZIP */

		    if (push(&cpid,&fd,Gunzip,argv) != SUCCESS) {
			sprintf(MsgBuf,"Cannot start %s",Gunzip);
			logmesg(MsgBuf);
			close(fd);
			goto fail;
		    } else {
			/* as we cannot rewind the pipe, we have to assume */
			/* that compressed files are always of the same type */

			goto postscript;
		    }
		} else
		    goto othertype;

		break;

	    case '%':
		if (GhostScript != NULL && !strncmp(buf,"%!PS-Adobe-",11)) {
		    int argc;
		    char *argv[10];

		    /* looks like a PS file */

		    if (logfd >= 0) {
			sprintf(MsgBuf,"\tPostscript file: %s\r\n",filename);
			write(logfd,MsgBuf,strlen(MsgBuf));
		    }

		    lseek(fd,0L,SEEK_SET);	/* rewind the file */

postscript:
		    memset(argv,0,sizeof(argv)); /* build argvector for GS */
		    argc = 1;			
		    argv[argc++] = (FaxInfo.vr? "-sDEVICE=dfaxhigh" : "-sDEVICE=dfaxlow");
		    argv[argc++] = "-sOutputFile=-";
		    argv[argc++] = "-q";
		    argv[argc++] = "-dNOPAUSE";
		    argv[argc++] = "-dSAFER";
		    argv[argc++] = "-";

		    /* start GhostScript process */

		    if (push(&gpid,&fd,GhostScript,argv) != SUCCESS) {
			sprintf(MsgBuf,"Cannot start %s",GhostScript);
			logmesg(MsgBuf);

			if (logfd >= 0) {
			    sprintf(ChatBuf,"\t%s\r\n",MsgBuf);
			    write(logfd,ChatBuf,strlen(ChatBuf));
			}

			close(fd);
			goto fail;
		    }

		    /* when Buffer is defined, start the buffering program */

		    if (Buffer != NULL) {
			memset(argv,0,sizeof(argv)); /* empty argvector */

			if (push(&bpid,&fd,Buffer,argv) != SUCCESS) {
			    sprintf(MsgBuf,"Cannot start %s",Buffer);
			    logmesg(MsgBuf);
			}
		    }

		    /* Let the above programs run for a while */

		    usleep(500000L);

		    /* cheat a bit, by forcing the page send loop below to
		     * start negotiating before doing the first read from
		     * the pipe.  that gains us some time for the GS process
		     * to format the first page while the modem negotiates
		     * the parameters with the remote FAX...  the 64-byte
		     * DigiFAX header will be skipped later.
		     */

		    skip = 64;
		    num = 0;
		    vr = FaxInfo.vr;
		    df = 0;
		} else
		    goto othertype;

		break;

	    case '#':
		if (GhostScript != NULL && buf[1] == '!' &&
		    !access(filename,X_OK))
		{
		    char *argv[2];

		    /* looks like a shell script */
		    /* we will execute it, and assume the output is Postscript */
		    /* data.  this can be achieved by including "pstext" as */
		    /* part of the script, if necessary */

		    if (logfd >= 0) {
			sprintf(MsgBuf,"\tShell script: %s\r\n",filename);
			write(logfd,MsgBuf,strlen(MsgBuf));
		    }

		    close(fd);
		    fd = open("/dev/null",O_RDWR); /* /dev/null as input */
		    num = 0;
		    memset(argv,0,sizeof(argv)); /* build argvector to exec */

		    if (push(&cpid,&fd,filename,argv) != SUCCESS) {
			sprintf(MsgBuf,"Cannot start %s",filename);
			logmesg(MsgBuf);
			close(fd);
			goto fail;
		    } else {
			/* as we cannot rewind the pipe, we have to assume */
			/* that the output is always the same type */

			goto postscript;
		    }
		}
		/* when it did not qualify as a shell script, fall through */

	    default:
othertype:
		if (PsText != NULL && GhostScript != NULL &&
		    buf[0] > 0 && buf[1] > 0 && buf[2] > 0 && buf[3] > 0)
		{
		    char *argv[2];

		    /* it probably is a text file, setup for conversion to PS */

		    if (logfd >= 0) {
			sprintf(MsgBuf,"\tText file: %s\r\n",filename);
			write(logfd,MsgBuf,strlen(MsgBuf));
		    }

		    lseek(fd,0L,SEEK_SET);	/* rewind the file */
		    num = 0;
		    memset(argv,0,sizeof(argv)); /* build argvector for PSTEXT */

		    if (push(&cpid,&fd,PsText,argv) != SUCCESS) {
			sprintf(MsgBuf,"Cannot start %s",PsText);
			logmesg(MsgBuf);
			close(fd);
			goto fail;
		    } else
			goto postscript;	/* output will be PS */
		} else {
		    sprintf(MsgBuf,"Unrecognized file: %s",filename);
		    logmesg(MsgBuf);

		    if (logfd >= 0) {
			sprintf(ChatBuf,"\t%s\r\n",MsgBuf);
			write(logfd,ChatBuf,strlen(ChatBuf));
		    }

		    close(fd);
		    goto fail;
		}
		break;
	    }

	    retry = 3;

	    while (FaxInfo.hng < 0)
	    {
		if (!gpid && !cpid)
		    pagepos = lseek(fd,0L,SEEK_CUR) - num;

		if (!skip)
		{
		    if (num < 64)		/* see if we need to read */
		    {
			memcpy(buf,buf + ptr,num); /* yes, keep data & fill buf */
			ptr = 0;

			if ((num += read(fd,buf + num,sizeof(buf) - num)) < 64)
			    break;		/* nothing left to read */
		    }

		    /* see if it has a DigiFAX header (Ghostscript) */

		    if (!strcmp(buf + ptr + 1,"PC Research, Inc")) {
			vr = buf[ptr + 29];	/* yes, get vert. resolution */
			ptr += 64;		/* skip header */
			num -= 64;
		    }
		}

		/* check if vertical resolution and data format match */
		/* negotiated values, re-negotiate when required */

		if (vr != FaxInfo.vr || df != FaxInfo.df) {
		    sprintf(ChatBuf,"AT+FDIS=%d,,,,%d\r",vr,df);
		    send(ChatBuf);

		    if (logfd >= 0) {
			sprintf(MsgBuf,"\t%s\n",ChatBuf);
			write(logfd,MsgBuf,strlen(MsgBuf));
		    }

		    if (fax2rsp(logfd) != SUCCESS || strcmp(MsgBuf,"OK"))
			goto fail;
		}

		/* tell it we have a FAX page */

		send("AT+FDT\r");

		if (logfd >= 0)
		    write(logfd,"\tAT+FDT\r\n",9);

		if (fax2rsp(logfd) != SUCCESS || strcmp(MsgBuf,"CONNECT"))
		    goto fail;

		/* re-check transfer parameters (result of negotiation) */

		if (vr != FaxInfo.vr || df != FaxInfo.df) {
		    sprintf(MsgBuf,"Parameter mismatch: vr=%d/%d df=%d/%d",
			    vr,FaxInfo.vr,df,FaxInfo.df);
		    logmesg(MsgBuf);

		    if (logfd >= 0) {
			sprintf(MsgBuf,"\tvr=%d/%d df=%d/%d\r\n",
				    vr,FaxInfo.vr,df,FaxInfo.df);
			write(logfd,MsgBuf,strlen(MsgBuf));
		    }

		    send(AudioEnd);
		    fax2rsp(logfd);
		    goto fail;
		}

		/* everything OK, start sending the FAX page */

		debug2(D_FAX, "faxsend() sending FAX data\n");

		if (logfd >= 0)
		    write(logfd,"\t[page data]\r\n",14);

		if (Fw_version >= 613)
		    settermio(IMMEDIATE);	/* >= 6.13: hw flow ctl */
		else
		    settermio(XONXOFF);		/* XON/XOFF flow ctl */

		send("\"\"");			/* send nothing (delay) */
		rmatch = 0;

		for (;;) {
		    /* refill the buffer when required */

		    if (num <= 0) {
#ifdef DEBUG
			errno = 0;
#endif
			num = read(fd,buf,sizeof(buf));
			debug4(D_CHAT, "read: %d bytes, errno=%d\n",num,errno);

			if (num <= 0)
			    break;

			ptr = 0;
		    }

		    /* skip the DigiFAX header we faked when running GS */

		    if (skip) {
			if (num < skip) {
			    skip -= num;
			    num = 0;
			} else {
			    ptr += skip;
			    num -= skip;
			    skip = 0;
			}

			continue;
		    }

		    /* send this block of FAX data, doubling DLE as needed */

		    p = q = buf + ptr;

		    for (;;)
		    {
			while (--num >= 0) {
			    if (*p != fax_rtc[rmatch]) /* match with RTC */
				rmatch = 0;
			    else
				rmatch++;

			    *p = rev_bits[(unsigned char)*p]; /* swap the bits in the file data */

			    if (*p++ == DLE ||	/* find DLE as we have to double it */
				rmatch == RTC_LEN) /* also quit when RTC seen */
				break;
			}

			if (write(STDOUT,q,p - q) <= 0) /* write data */
			    break;

			if (rmatch == RTC_LEN)	/* did we see RTC? */
			    goto quit;

			if (num < 0)		/* all sent? */
			    break;

			q = p - 1;		/* next time around, DLE again! */
		    }

		    switch (getch())		/* something returned? */
		    {
		    case EOF:			/* nothing */
		    case XON:			/* leaked flow control char */
		    case XOFF:
			break;			/* ignore it and go on */

		    default:			/* something else */
			debug2(D_RUN, "faxsend() terminated by unexpected char\n");
			InPtr--;		/* push it back */
			waitfor(NULL,SHORTTIME); /* and flush the input */
			goto quit;		/* stop sending */
		    }
		}
quit:
		debug2(D_FAX, "faxsend() FAX data sent\n");

		if (logfd >= 0)
		    write(logfd,"\t[DLE-ETX]\r\n",12);

		/* end of page, send DLE-ETX */

		send(AudioEnd);

		if (waitfor("OK",LONGTIME) == FAIL)
		    goto fail;

		if (logfd >= 0)
		    write(logfd,"\tOK\r\n",5);

		settermio(INITIAL);		/* enable RTS/CTS handshake */

		/* now we have to tell it if more follows.  inconvenient. */

		if (num < 64)			/* see if we need to read */
		{
		    int n;

		    memcpy(buf,buf + ptr,num);	/* yes, keep data & fill buf */
		    ptr = 0;
#ifdef DEBUG
		    errno = 0;
#endif
		    n = read(fd,buf + num,sizeof(buf) - num);
		    debug4(D_CHAT, "read: %d bytes, errno=%d\n",n,errno);
		    num += n;
		}

		if (num >= 64 || (nfiles > 0 && file != nfiles))
		    fet = 0;			/* more pages follow */
		else
		    fet = 2;			/* end of document */

		sprintf(ChatBuf,"AT+FET=%d\r",fet);
		send(ChatBuf);

		if (logfd >= 0) {
		    sprintf(MsgBuf,"\t%s\n",ChatBuf);
		    write(logfd,MsgBuf,strlen(MsgBuf));
		}

		/* get the page transfer status */

		if (fax2rsp(logfd) != SUCCESS || strcmp(MsgBuf,"OK"))
		    goto fail;

		/* when page bad, retry it.  but not too often. */

		if (!gpid && !cpid && (FaxInfo.pts == 2 || FaxInfo.pts == 4) && retry--) {
		    lseek(fd,pagepos,SEEK_SET);
		    num = 0;
		}
	    }

	    close(fd);
	    alarm(10);
	    if (cpid)
		waitpid(cpid,&fd,0);
	    if (bpid)
		waitpid(bpid,&fd,0);
	    if (gpid)
		waitpid(gpid,&fd,0);
	    alarm(0);
	} while (FaxInfo.hng < 0 && nfiles > 0 && file != nfiles);

	if (FaxInfo.hng == 0)
	    rv = SUCCESS;

fail:
	send("ATH\r");
	waitfor("OK",MEDMTIME);

	/* free the storage associated with file list */

	if (nfiles > 0) {
	    for (file = 0; file < nfiles; file++)
		free(namelist[file]);

	    free(namelist);
	}

	sprintf(MsgBuf,"FAX send \"%s\" %s",
		name,(rv == SUCCESS? "okay":"failed"));
	if (FaxInfo.hng > 0)
	    sprintf(MsgBuf + strlen(MsgBuf),", code: %d",FaxInfo.hng);
	logmesg(MsgBuf);
	return rv;
}


/*
**	fax2rsp() - parse CLASS 2 FAX messages and save interesting data
*/

int
fax2rsp(logfd)
int logfd;

{
	char *rsp;

	FaxInfo.dtc = 0;			/* assume no DTC */

	while (waitfor(NULL,LONGTIME) == SUCCESS ||
	       waitfor(NULL,LONGTIME) == SUCCESS)
	{
	    /* save the reply, remove trailing CR/LF garbage and spaces */

	    rip(Match);
	    strcpy(MsgBuf,Match);
	    debug3(D_FAX, "fax2rsp: \"%s\"\n", MsgBuf);

	    if (logfd >= 0) {
		sprintf(ChatBuf,"\t%s\r\n",MsgBuf);
		write(logfd,ChatBuf,strlen(ChatBuf));
	    }

	    if (strncmp(MsgBuf,"+F",2))		/* no more CLASS 2 messages? */
		return SUCCESS;

	    rsp = MsgBuf + 2;			/* point to response type */

	    if (!strncmp(rsp,"CFR",3)) {	/* confirmation to receive */
		FaxInfo.cfr = 1;
		continue;
	    }

	    if (!strncmp(rsp,"CSI:",4) ||	/* called station ID */
		!strncmp(rsp,"TSI:",4) ||	/* transmit station ID */
		!strncmp(rsp,"CIG:",4))		/* polling station ID */
	    {
		rsp += 4;
		while (*rsp == ' ')		/* skip leading spaces */
		    rsp++;

		strncpy(FaxInfo.id,rsp,sizeof(FaxInfo.id));
		continue;
	    }

	    if (!strncmp(rsp,"DIS:",4) ||	/* remote capability */
		!strncmp(rsp,"DCS:",4) ||	/* current session capability */
		!strncmp(rsp,"DTC:",4))		/* request for polling */
	    {
		if (rsp[1] == 'T')
		    FaxInfo.dtc = 1;		/* request for polling rcvd */

		sscanf(rsp + 4,"%d,%d,%d,%d,%d,%d,%d,%d",
			&FaxInfo.vr,&FaxInfo.br,&FaxInfo.wd,&FaxInfo.ln,
			&FaxInfo.df,&FaxInfo.ec,&FaxInfo.bf,&FaxInfo.st);
		continue;
	    }

	    if (!strncmp(rsp,"ET:",3))	{	/* post page message response */
		FaxInfo.et = atoi(rsp + 3);
		continue;
	    }

	    if (!strncmp(rsp,"HNG:",4)) {	/* hangup */
		FaxInfo.hng = atoi(rsp + 4);
		continue;
	    }

	    if (!strncmp(rsp,"NSF:",4) ||	/* nonstandard facilities */
		!strncmp(rsp,"NSS:",4) ||	/* nonstandard setup */
		!strncmp(rsp,"NSC:",4))		/* nonstandard commands */
	    {
		int n;

		FaxInfo.nstyp = rsp[2];		/* F, S or C */
		FaxInfo.nslen = 0;
		rsp += 4;

		/* get hex digits and save into ns array */

		while (FaxInfo.nslen < sizeof(FaxInfo.ns)) {
		    if (sscanf(rsp,"%x",&n) != 1)
			break;

		    FaxInfo.ns[FaxInfo.nslen++] = n;
		    rsp += 3;
		}

		continue;
	    }

	    if (!strncmp(rsp,"POLL",4)) {	/* has document to poll */
		FaxInfo.poll = 1;
		continue;
	    }

	    if (!strncmp(rsp,"PTS:",4)) {	/* page transfer status */
		sscanf(rsp + 4,"%d,%d,%d,%d,%d",
			&FaxInfo.pts,&FaxInfo.lc,&FaxInfo.blc,&FaxInfo.cblc,
			&FaxInfo.lbc);
		continue;
	    }

	    logmesg(MsgBuf);			/* nonmatched response, log it */
	}

	debug2(D_FAX, "fax2rsp: timeout\n");
	return FAIL;				/* timeout */
}
#endif

/*************************************************************************
**	VOICE mode handling						**
*************************************************************************/

/*
**	voicemode() - main handling of VOICE calls
*/

int
voicemode(answer)
int answer;

{
	int result = FAIL;
	int rectime = 0;
	const char *action;

	debug2(D_RUN, "VOICE mode\n");

	if (answer != ONLINE) {
	    send("AT+FCLASS=8;+VNH=1;M2A\r"); /* select VOICE mode */
					/* and answer (no hangup, no mute) */
	    if (waitfor("VCON",MEDMTIME))
		return FAIL;
	}

	UserLevel = 0;			/* initial user level */
	Number[0] = '\0';		/* no user-entered number yet */

	if (set_vls(2))			/* select telephone line */
	    return FAIL;

	/* send optional answertone */
	/* (could try to use this to trick calling modems) */

	if (answer != ONLINE)
	    beep(AnswerTone);

	/* play the greeting */

	if (Greeting != NULL)
	    if (play(Greeting,"*#bcde") == SUCCESS)
	    {
		/* check for 'b' (BUSY) or 'd' (DIALTONE, only >= 6.11 firmware) */

		if (strchr(MsgBuf,'b') != NULL || strchr(MsgBuf,'d') != NULL)
		{
		    sprintf(ChatBuf,"VOICE call, hangup during play (%s)",MsgBuf);
		    logmesg(ChatBuf);
		    goto done;
		}

		/* check for 'c' (FAX CNG) */

		if (strchr(MsgBuf,'c') != NULL)
		{
		    logmesg("VOICE call, T.30 FAX tone detected during play");
		    result = FAX;
		    goto done;
		}

		/* check for 'e' (DATA CNG, only >= 6.11 firmware) */

		if (strchr(MsgBuf,'e') != NULL)
		{
		    logmesg("VOICE call, DATA tone detected during play");
		    result = DATA;
		    goto done;
		}
	    }

	/* record incoming message */

	strcpy(Recordfile,RecFile);

	switch (rectime = record(Recordfile,"0123456789*#bcdeq",Silence,RecMode,120))
	{
	case 0:
	    logmesg("VOICE call, silence");
	    result = DATA;		/* try DATA mode */
	    break;

	case FAIL:
	    logmesg("VOICE call, error in record");
	    result = FAIL;
	    break;

	default:
	    /* message too short to keep? */

	    if (rectime <= DropMsg && unlink(Recordfile) == 0)
		action = ", erased";
	    else
		action = " recorded";

	    /* check for FAX/DATA tones */

	    if (strchr(MsgBuf,'c') != NULL)
	    {
		logmesg("VOICE call, T.30 FAX tone detected during record");
		result = FAX;
		break;
	    }

	    if (strchr(MsgBuf,'e') != NULL)
	    {
		logmesg("VOICE call, DATA tone detected during record");
		result = DATA;
		break;
	    }

	    /* not FAX or DATA, assume a human caller */

	    sprintf(ChatBuf,"VOICE call, %dsec message%s (%s)",rectime,action,MsgBuf);
	    logmesg(ChatBuf);

	    result = SUCCESS;			/* be optimistic */

	    if (strchr(MsgBuf,'b') == NULL && strchr(MsgBuf,'d') == NULL) {
		if (strchr(MsgBuf,'*') != NULL)
		    result = voicekey(0);	/* '*' keyed */
		else
		    if (isdigit(MsgBuf[0]))	/* number keyed */
			result = voicekey(MsgBuf[0]);

		RecLS = 0;			/* reset current LS */

		if (result == SUCCESS)
		    play("goodbye","#bd");
	    }
	    break;
	}

done:
	/* when terminating call, reset LS and hangup
	 * when switching to other mode, keep the line active
	 */
	if (result == SUCCESS || result == FAIL) {
	    if (set_vls(0))
		result = FAIL;

	    send("ATH\r");
	    if (waitfor("OK",MEDMTIME))
		result = FAIL;
	}

	/* check if we have a recorded message (i.e. it was not too
	 * short and wasn't deleted by the caller) and perform optional
	 * action if so.
	 */

	if (rectime > 0 && !access(Recordfile,R_OK))
	    voicecommand(VoiceKey,20,dtmfdigit,dtmfstring);

	debug3(D_RUN, "VOICE mode returns %d\n",result);
	return result;
}


/*
**	voicekey() - handling of * or number entry in VOICE mode
*/

int
voicekey(firstdigit)
int firstdigit;				/* first digit to be handled */

{
	int code;
	int result;

	debug2(D_RUN, "DTMF key pressed\n");

	for (;;)
	{
	    /* get a digit */

	    code = firstdigit? firstdigit : dtmfdigit();

	    /* check for numerics, '*' and '#' */

	    if (isdigit(code))
		code -= '0';
	    else
		switch (code)
		{
		case '*':
		    code = 11;
		    break;

		case '#':
		    code = 12;
		    break;

		default:			/* specials: 'b', 'd', 's', 'q' */
		    return FAIL;		/* and FAIL code */
		}

	    /* handle the key, using VoiceKey as the commands file */

	    switch (result = voicecommand(VoiceKey,code,dtmfdigit,dtmfstring))
	    {
	    case FAIL:
	    case DATAFAX:
	    case DATA:
	    case FAX:
	    case DIALBACK:
		return result;

	    case EXIT:				/* exit from voice mode */
		return SUCCESS;

	    case COMMAND:			/* go to expert mode */
		do
		{
		    MsgBuf[0] = '\0';

		    switch (code = dtmfcode())	/* ask for *nn# code */
		    {
		    case FAIL:
			return FAIL;

		    default:
			switch (result = voicecommand(VoiceCmd,code,dtmfdigit,dtmfstring))
			{
			case FAIL:
			case DATAFAX:
			case DATA:
			case FAX:
			case DIALBACK:
			    return result;

			case UNKNOWN:
			    beep(Error);
			    break;
			}
			break;
		    }
		}
		while (result != EXIT);
		break;

	    case UNKNOWN:			/* unknown key */
	    case ERROR:				/* some problem */
		beep(Error);
		break;
	    }

	    firstdigit = 0;
	}
}

/*
**	voicecommand() - handling of VOICE expert commands
*/

int
voicecommand(filename,code,inputdigit,inputstring)
char *filename;
int code;
int (*inputdigit)(void);
char *(*inputstring)(void);

{
	FILE *fp;
	char *p,*q;
	int result;
	char buf[MAXLINE];

	/* open the commands file, which contains lines like:
	 * # comment
	 * 10: actions for code 10
	 *     more actions for code 10
	 * 20: actions for code 20
	 * ... etc
	 */
	if (filename == NULL || (fp = fopen(filename,"r")) == NULL) {
	    debug3(D_RUN, "Open commandfile (%s) fails\n", filename);
	    return UNKNOWN;
	}

	/* find the proper starting point and parse commands */

	while (fgets(buf,sizeof(buf),fp) != NULL)
	{
	    if (buf[0] == '#' || (p = strchr(buf,':')) == NULL)
		continue;

	    *p++ = '\0';

	    if (isdigit(buf[0]) && atoi(buf) == code)
	    {
		/* found the code, now run commands until next label */

		for (;;)
		{
		    while (isspace(*p))		/* skip leading blanks */
			p++;

		    if ((q = strchr(p,'\n')) != NULL) /* remove \n */
			*q = '\0';

		    if (*p != '\0' &&
			(result = docommand(p,inputdigit,inputstring)) != SUCCESS)
		    {
			fclose(fp);
			return result;
		    }

		    do {
			if (fgets(buf,sizeof(buf),fp) == NULL || isdigit(buf[0])) {
			    fclose(fp);
			    return SUCCESS;
			}
		    } while (buf[0] == '#');

		    p = buf;
		}
	    }
	}

	fclose(fp);
	return UNKNOWN;
}


/*
**	docommand() - run a single command line from VOICECMD file
*/

int
docommand(line,inputdigit,inputstring)
char *line;
int (*inputdigit)(void);
char *(*inputstring)(void);

{
	char *p,*argv[NARG];
	int argc,quote;

	debug3(D_RUN, "docommand(\"%s\")\n", line);

	for(argc = 0;argc < NARG;argc++)
		argv[argc] = NULL;

	for(argc = 0;argc < NARG && *line != '\0';){
	    quote = 0;
	    /* Skip leading white space */
	    while(isspace(*line))
		line++;
	    if(*line == '\0')
		break;
	    /* Check for quoted token */
	    if(*line == '"' || *line == '\'')
		quote = *line++;	/* Suppress quote, remember it */
	    argv[argc++] = line;	/* Beginning of token */
	    /* Find terminating delimiter */
	    if(quote){
		/* Find quote, it must be present */
		if((line = strchr(line,quote)) == NULL)
		    return FAIL;

		*line++ = '\0';
	    } else {
		/* Find space or tab. If not present,
		 * then we've already found the last
		 * token.
		 */
		for (p = line; *p; p++)
		    if (isspace(*p))
			break;
		if (*p != '\0')
		    *p++ = '\0';
		line = p;
	    }
	}

	if (argc == 0)
	    return UNKNOWN;			/* nothing, shouldn't happen */

	/* execute command.  kind of an ad-hoc approach, this is */

	/* auplay <filename>
	 * plays a .au audio file (also known as .snd file)
	 */
	if (argc > 1 && !strcmp(argv[0],"auplay"))
	    return auplay(argv[1]);

	/* beep <VTS-string>
	 * sends a +VTS= string to generate audible beep
	 * default is the ERROR beep defined in config file
	 */
	if (!strcmp(argv[0],"beep"))
	    return beep(argc > 1? argv[1] : Error);

	/* checklevel <level>
	 * verify current user has specified level (as a bit field)
	 */
	if (argc > 1 && !strcmp(argv[0],"checklevel")) {
	    if (!(UserLevel & atoi(argv[1])))
		return ERROR;

	    return SUCCESS;
	}

	/* checktime [not] <from_hhmm> <until_hhmm> [<from_weekday> <until_weekday>]
	 * verify current time is between the specified bounds
	 */
	if (argc > 2 && !strcmp(argv[0],"checktime")) {
	    time_t now;
	    struct tm *tm;
	    char **argp;
	    int hhmm,not;

	    time(&now);
	    tm = localtime(&now);
	    hhmm = 60 * tm->tm_hour + tm->tm_min;

	    argp = &argv[0];

	    if ((not = !strcmp(argv[1],"not"))) { /* optional 'not' inverts test */
		argp++;
		if (--argc < 3)
		    return ERROR;
	    }

	    if (not ^ (hhmm >= atoi(argp[1]) && hhmm <= atoi(argp[2])))
		return ERROR;

	    if (argc > 4 &&
		(not ^ (tm->tm_wday >= atoi(argp[3]) &&
			tm->tm_wday <= atoi(argp[4]))))
		return ERROR;

	    return SUCCESS;
	}

	/* command
	 * returns code COMMAND to upper level, to go to command mode
	 */
	if (!strcmp(argv[0],"command"))
	    return COMMAND;

	/* dialback [<number>]
	 * calls number in Number (or specified number) and enters voice mode
	 */
	if (!strcmp(argv[0],"dialback")) {
	    if (argc > 1)
		strcpy(Number,argv[1]);

	    if (!Number[0])
		return ERROR;

	    if (Synth != NULL)
		say("now dialling back");
	    else
		play("gong","#bd");

	    /* hangup existing call and init the MODEM
	     */
	    if (initmodem(TRUE) != SUCCESS)
		return FAIL;

	    sleep(15);

	    if (dialup("",8,Number) == SUCCESS) /* dial in voice mode */
		return DIALBACK;

	    return FAIL;
	}

	/* exit
	 * returns code EXIT to upper level, to exit from voice mailbox
	 */
	if (!strcmp(argv[0],"exit"))
	    return EXIT;

	/* fax <filename> [<number>]
	 * send a FAX file.  when <filename> is N, use FaxPoll file with
	 * Number entered using "readnumber" (4 digits usually).
	 * when <number> specified, dial it.  when <number> is N, use the
	 * Number entered using "readnumber".
	 * because this function needs the line for outgoing dial and/or
	 * will hangup the existing connection, it will return EXIT so that
	 * the session is ended...
	 */

	if (!strcmp(argv[0],"fax")) {
	    char faxfile[MAXLINE];

	    if (argc == 2 && argv[1][0] == 'N') {
		if (FaxPoll == NULL)
		    return ERROR;

		sprintf(ChatBuf,"%04u",atoi(Number));
		sprintf(faxfile,FaxPoll,ChatBuf);

		if (!access(faxfile,R_OK))
		    argv[1] = faxfile;
		else
		    return ERROR;
	    }

	    if (argc >= 3 && argv[2][0] == 'N')
		argv[2] = Number;

	    if (Synth != NULL)
		say("now calling your FAX");
	    else
		play("gong","#bd");

	    if (argc == 2 || argv[2][0] == '\0') {
		/* select FAX mode on this same connection */
#ifdef ZFAX
		return ERROR;			/* cannot do that with ZFAX */
#else
		sprintf(ChatBuf,"AT%s+FCLASS=2;+FAA=0D\r",(Is_ZyXEL? "S38.4=0":""));
		send(ChatBuf);
		if (waitfor("+FCON",LONGTIME))
		    return FAIL;
#endif
		argv[2] = NULL;
	    } else {
		/* hangup existing call and init the MODEM
		 */
		if (initmodem(TRUE) != SUCCESS)
		    return FAIL;

		sleep(5);
	    }

	    if (faxsend(argv[2],argv[1],-1) == SUCCESS)
		return EXIT;

	    return FAIL;
	}

	/* logmesg "string"
	 * logs the specified message in the logfile
	 */
	if (argc > 1 && !strcmp(argv[0],"logmesg")) {
	    logmesg(argv[1]);
	    return SUCCESS;
	}

	/* mode <mode>
	 * returns specified <mode> to next upper level, which is
	 * supposed to switch the call to that mode.
	 */
	if (argc > 1 && !strcmp(argv[0],"mode")) {
	    if (!strcmp(argv[1],"DATAFAX"))
		return DATAFAX;

	    if (!strcmp(argv[1],"DATA"))
		return DATA;

	    if (!strcmp(argv[1],"FAX"))
		return FAX;

	    if (!strcmp(argv[1],"VOICE"))
		return VOICE;

	    return UNKNOWN;
	}

	/* pager [<info>]
	 * calls pager, sending optional info.
	 * when <info> is N, the Number entered using "readnumber" is used.
	 * because this function needs the line for outgoing dial, it will
	 * return EXIT so that the session is ended...
	 */

	if (!strcmp(argv[0],"pager")) {
	    if (PagerType == 0 || PagerNum == NULL)
		return ERROR;

	    if (argc >= 2 && argv[1][0] == 'N')
		argv[1] = Number;

	    if (Synth != NULL)
		say("now calling pager");
	    else
		play("gong","#bd");

	    if (pager(PagerType,PagerNum,argv[1]) == SUCCESS)
		return EXIT;

	    return FAIL;
	}

	/* play [<filename> [<endstring>]]
	 * plays the specified file, or the recorded file by default
	 */
	if (!strcmp(argv[0],"play")) {
	    if (argc < 2 || argv[1][0] == '\0')
		argv[1] = Recordfile;

	    if (argc < 3)
		argv[2] = "#bd";		/* default end characters */

	    if (play(argv[1],NULL) != SUCCESS)	/* verify the file is valid */
		return ERROR;

	    if (play(argv[1],argv[2]) == SUCCESS &&
		(strchr(MsgBuf,'b') != NULL ||	/* fail when we got 'busy' */
		 strchr(MsgBuf,'d') != NULL))	/* ... or 'dialtone' */
		return FAIL;

	    return SUCCESS;
	}

	/* playback [<direc>]
	 * enters message playback mode, optionally for certain directory
	 * default directory is RecDir, the incoming message directory.
	 */
	if (!strcmp(argv[0],"playback"))
	    return playback(argv[1],inputdigit);

	/* readnumber
	 * gets a (phone) number, reads it aloud, and saves in a global
	 */
	if (!strcmp(argv[0],"readnumber")) {
	    if ((p = (*inputstring)()) != NULL) {
		strcpy(Number,p);		/* save it for later use */
		p = Number;

		debug3(D_RUN, "Number: %s\n", Number);

		/* read it aloud. use voice synth when available, else */
		/* play individual digit recordings */

		if (Synth != NULL)
		{
		    char *q = ChatBuf;
		    int ndig = 0;

		    /* convert numeric string into groups of 2 digits */
		    /* except for digit 0, which always ends a group. */
		    /* e.g.: 030715610 becomes 0 30 71 56 10 */
		    /* (this maybe has to be customizable for countries */
		    /* where numbers are structured like #-###-###-#### */
		    /* and one would read them in groups like that) */

		    while (*p != '\0') {
			*q++ = *p;
			ndig++;

			if (*p == '0' || ndig == 2) {
			    *q++ = ' ';
			    ndig = 0;
			}

			p++;
		    }

		    if (q != ChatBuf && *--q != ' ')
			q++;
		    *q = '\0';

		    if (say(ChatBuf) != SUCCESS)
			return FAIL;
		} else {
		    while (*p != '\0') {
			sprintf(ChatBuf,"%c",*p);
			if (play(ChatBuf,"bd") != SUCCESS ||
			    MsgBuf[0] == 'b' || MsgBuf[0] == 'd')
			    return FAIL;

			p++;
		    }
		}
	    }
	    return SUCCESS;
	}

	/* record <filename> <endstring> <vsd-string> <vsm-value> <maxtime>
	 */
	if (!strcmp(argv[0],"record")) {
	    char recordfile[MAXLINE];
	    int enc,maxtime;

	    /* setup the defaults */
	    if (argc < 2 || argv[1][0] == '\0')
		argv[1] = Recordfile;

	    if (argc < 3 || argv[2][0] == '\0')
		argv[2] = "#bdq";

	    if (argc < 4 || argv[3][0] == '\0')
		argv[3] = Silence;

	    if (argc < 5 || argv[4][0] == '\0')
		enc = RecMode;
	    else
		enc = atoi(argv[4]);

	    if (argc < 6)
		maxtime = 120;
	    else
		maxtime = atoi(argv[5]);

	    /* make recording with these values */
	    strcpy(recordfile,argv[1]);

	    if (record(recordfile,argv[2],argv[3],enc,maxtime) == FAIL)
		return FAIL;

	    return SUCCESS;
	}

	/* recLS <LS-nr>
	 * sets the input device (LS) for recording
	 * mainly useful for local testing and recording of messages
	 */
	if (argc > 1 && !strcmp(argv[0],"recLS")) {
	    RecLS = atoi(argv[1]);
	    return SUCCESS;
	}

	/* remove <file>
	 * removes a file, or the current Recordfile when no file
	 * is specified.
	 */
	if (!strcmp(argv[0],"remove")) {
	    if (unlink(argc > 1? argv[1] : Recordfile) != 0)
		return ERROR;

	    return SUCCESS;
	}

	/* run [<envvar>=<value>...] <string> [<args>...]
	 * run program and send result to voice synthesizer
	 */
	if (argc > 1 && !strcmp(argv[0],"run"))
	    return run(argv + 1);

	/* say <string>
	 * calls voice synthesizer to say <string>
	 */
	if (argc > 1 && !strcmp(argv[0],"say"))
	    return say(argv[1]);

	/* userlevel <level>
	 * sets the user's level (after entry of a password)
	 */
	if (argc > 1 && !strcmp(argv[0],"userlevel")) {
	    UserLevel = atoi(argv[1]);
	    sprintf(MsgBuf, "User level %d", UserLevel);
	    logmesg(MsgBuf);
	    return SUCCESS;
	}

	return UNKNOWN;
}


/*
**	pager () - call a pager and send optional info
**
**	because the access to the paging service could be dependent on
**	local conventions, some alteration of this code may be needed.
**	a switch() is provided to insert local varieties of the algorithm.
**	if you need a different algorithm, please add it under a different
**	case, with a value matching your country access code, and please mail
**	the diffs to me so I can include it in a later version...
**
**	currently implemented algorithms:
**
**	31	Netherlands
**		Call the specified number and wait until it answers (VOICE).
**		When numeric info is to be sent, wait for the end of the
**		voice announcement and the appearance of a DIAL tone, then
**		send the info using DTMF (even when local dialling is PULSE).
**		Info is terminated with a # sign.
**		Then, wait for the confirmation voice message and beep.
**		Actually, we cannot match that and wait for the BUSY signal
**		which follows immediately after the beep.
**		Note that the appearance of the second dialtone is dependent
**		on the kind of pager, and therefore it is your responsability
**		to configure the program (defaults file and keys/commands file)
**		in such a way that extra info is sent when the pager expects
**		it, and is not sent for a pager that does not support it.
**
**	312	Alternative for Netherlands
**		Call a certain number in DATA mode, and wait for the prompt
**		for the pager number to call ("=01=").
**		Send the pager number followed by RETURN.  Wait for prompt
**		for numeric information ("=20=").
**		Send numeric information, followed by RETURN.
**		Wait for confirmation ("=80=") and disconnect.
*/

int
pager(type,number,info)
int type;					/* pager type (algorithm) */
const char *number;				/* the phone number to call */
const char *info;				/* (optional) info to send */
{
	int result = FAIL;
	char buf[20];

	debug5(D_RUN, "pager(%d,\"%s\",\"%s\")\n", type, number, info);

	/* hangup existing call and init the MODEM
	 */
	if (initmodem(TRUE) != SUCCESS)
	    return FAIL;

	sleep(5);

	switch (type)
	{
	case 31:				/* Netherlands */
	    if (dialup("",8,number) != SUCCESS) /* dial in voice mode */
		break;

	    /* when info is to be sent, wait for the second dialtone */

	    if (info != NULL && info[0] != '\0') {
		if (Fw_version >= 611) {	/* 6.11 implements 'd' */
		    if (dtmfrecord("bdq",25) < 5 || strchr(MsgBuf,'d') == NULL)
			break;
		} else
		    sleep(5);

		sprintf(ChatBuf,"ATDT%s#;\r",info); /* send info + # */
		send(ChatBuf);
		if (waitfor("OK",LONGTIME))
		    break;
	    }

	    /* now we have to wait until the call is accepted.	we cannot
	     * match the confirmation voice message and beep, but we should
	     * at least wait until BUSY is returned so that the paging
	     * service has hung up the call.  (if we hang up too soon, the
	     * request will be canceled...)
	     */
	    if (dtmfrecord("bdq",25) < 3 || strchr(MsgBuf,'b') == NULL)
		break;

	    result = SUCCESS;
	    break;

	case 312:				/* Netherlands, alternative */
	    /* set 7 bit, EVEN parity (this is what the service expects/sends)
	     */
	    settermio(SEVEN_E);
	    send("AT\r");
	    if (waitfor("OK",MEDMTIME))
		break;

	    /* the number to be dialled depends on the first digit in the
	     * pager number.  what a mess...
	     */
	    sprintf(buf,"065%c5065%c5",number[0],number[0]);

	    /* dialup in data mode.
	     * it seems they only support 2400/NONE, which is good enough
	     * and provides faster training/negotiation...
	     */
	    if (dialup("&N14&K0",0,buf) != SUCCESS)
		break;

	    if (waitfor("*=01=",LONGTIME))
		break;

	    sprintf(ChatBuf,"%s\r",number);	/* send number + CR */
	    send(ChatBuf);

	    if (info != NULL && info[0] != '\0') {
		if (waitfor("*=20=",LONGTIME))
		    break;

		sprintf(ChatBuf,"%s\r",info);	/* send info + CR */
		send(ChatBuf);
	    }

	    if (waitfor("*=80=",LONGTIME))
		break;

	    /* flush the garbage it sends after the confirmation
	     */
	    waitfor("FLUSH",MEDMTIME);
	    result = SUCCESS;
	    break;
	}

	initmodem(TRUE);			 /* done, force a hangup */

	sprintf(MsgBuf,"PAGER call %s",(result == SUCCESS? "OK":"FAILED"));
	if (info != NULL && info[0] != '\0')
	    sprintf(MsgBuf + strlen(MsgBuf)," (%s)",info);

	logmesg(MsgBuf);
	return result;
}


/*
**	dir_select() - select all recorded files
*/

int
dir_select(dirent)
struct dirent *dirent;
{
	if (dirent->d_name[0] == '.')		/* no hidden files/directories */
	    return 0;

	return 1;
}


/*
**	playback() - play the messages in the recording directory and
**		     act upon them.  option: play from another directory
*/

int
playback(direc,inputdigit)
char *direc;					/* directory, or NULL */
int (*inputdigit)(void);			/* selection input function */

{
	int nfiles,i,d;
	struct dirent **namelist = NULL;
	struct stat st;
	char filename[MAXLINE];
	char archive[MAXLINE];

	if (direc == NULL)			/* take 'direc' as dir */
	    direc = RecDir;			/* default to RecDir */

	/* this line may generate a warning for arg #3, depending on */
	/* the version of <dirent.h> */
	nfiles = scandir(direc,&namelist,dir_select,alphasort);

	debug3(D_RUN, "%d files selected to playback\n", nfiles);

	if (nfiles == 0)
	    return SUCCESS;

	d = 1;					/* forward */

	for (i = 0; i >= 0 && i < nfiles; i += d)
	{
	    if (namelist[i] == NULL)		/* deleted file? */
		continue;

	    sprintf(filename, "%s/%.*s", direc, namelist[i]->d_reclen,
						namelist[i]->d_name);

	    debug3(D_RUN, "playback file (%s)\n", filename);

play_again:
	    /* don't detect BUSY here as the message may be a recording of
	     * a busy tone, and the playback mode would be interrupted when
	     * we played that one (I found that the hard way!)
	     * (probably true for DIALTONE as well, can't test that here)
	     */
	    if (play(filename,"#") == FAIL) {
		for (i = 0; i < nfiles; i++)
		    free(namelist[i]);
		free(namelist);
		return FAIL;
	    }

again:
	    switch ((*inputdigit)())
	    {
	    case '0':				/* quit playback */
		for (i = 0; i < nfiles; i++)
		    free(namelist[i]);
		free(namelist);
		return SUCCESS;

	    case '1':				/* play instructions */
		if (play("playbackhelp",NULL) == SUCCESS)
		    play("playbackhelp","#bd");
		else {
		    say("to leave playback mode, press 0.");
		    say("to get these instructions, press 1.");
		    say("to read the message date and time, press 2.");
		    say("to advance to the next message, press 3.");
		    say("to delete the current message, press 4.");
		    say("to archive the current message, press 5.");
		    say("to go to the first message, press 7.");
		    say("to replay the current message, press 8.");
		    say("to go to the last message, press 9.");
		}
		goto again;

	    case '2':				/* read message's date/time */
		if (stat(filename,&st) == 0) {
		    char *p;
		    struct tm *tm,tmnow;
		    time_t now;
		    char buf[MAXLINE],tmp[10];

		    if (Synth != NULL) {
			time(&now);
			tmnow = *localtime(&now);
			tm = localtime(&st.st_mtime);

			if (tm->tm_year == tmnow.tm_year ||
			    (now - st.st_mtime) < (90 * 24 * 3600L)) {
			    if (tm->tm_yday == tmnow.tm_yday)
				p = "%H:%M today";
			    else
				p = "%A, %B %d, %H:%M";
			} else
			    p = "%A, %B %d %Y, %H:%M";

			strftime(buf,sizeof(buf),p,tm);

			if (say(buf) != SUCCESS)
			    return FAIL;
		    } else {
			strftime(buf,sizeof(buf),"%y%m%d%H%M",localtime(&st.st_mtime));

			for (p = buf; *p != '\0'; p++) {
			    sprintf(tmp,"%c",*p);
			    if (play(tmp,"bd") != SUCCESS ||
				MsgBuf[0] == 'b' || MsgBuf[0] == 'd')
				return FAIL;
			}
		    }
		}
		goto again;

	    case '3':				/* next message */
		break;

	    case '4':				/* delete */
		if (unlink(filename) == 0) {
		    if (Synth != NULL)
			say("message deleted");
		    else
			beep("[880,0,3][0,0,1][880,0,1][0,0,1][880,0,1][0,0,5]");
		    free(namelist[i]);
		    namelist[i] = NULL;
		}
		break;

	    case '5':				/* archive message */
		sprintf(archive, "%s/%.*s", Archive, namelist[i]->d_reclen,
						     namelist[i]->d_name);

		debug4(D_RUN, "rename (%s) to (%s)\n", filename, archive);

		if (strcmp(filename,archive) != 0 &&
		    rename(filename,archive) == 0)
		{
		    if (Synth != NULL)
			say("message archived");
		    else
			beep("[880,0,1][0,0,1][880,0,3][0,0,5]");
		    free(namelist[i]);
		    namelist[i] = NULL;
		}
		break;

	    case '7':				/* first message */
		say("first message");
		i = -1;
		d = 1;
		break;

	    case '8':				/* play again */
		goto play_again;

	    case '9':				/* last message */
		say("last message");
		i = nfiles;
		d = -1;
		break;
	    }
	}

	for (i = 0; i < nfiles; i++)
	    free(namelist[i]);
	free(namelist);
	return SUCCESS;
}


/*
**	dtmfdigit() - wait for a single DTMF tone and return it
*/

int
dtmfdigit()

{
	int result;

	/* send prompt tone */

	beep(Prompt1);

	/* record a single tone, 20 seconds max */

	if ((result = dtmfrecord("0123456789*#bdq",20)) <= 0)
	    return result;

	/* make sure we always see a 'b' or 'd' when present */

	if (strchr(MsgBuf,'b'))
	    return 'b';

	if (strchr(MsgBuf,'d'))
	    return 'd';

	/* return the single digit */

	return MsgBuf[0];
}


/*
**	dtmfcode() - wait for a *<digits># DTMF sequence and return value
*/

int
dtmfcode()

{
	char *p;

	/* read a string, remove * and # */

	if ((p = dtmfstring()) == NULL)
	    return FAIL;

	debug3(D_RUN, "dtmfcode() returns %d\n",atoi(p));

	/* return the value */

	return atoi(p);
}


/*
**	dtmfstring() - wait for DTMF sequence (#-terminated)
*/

char *
dtmfstring()

{
	char *p,*where;

	/* send prompt tone */

	beep(Prompt2);

	do
	{
	    /* record a sequence, 20 seconds max */

	    if (dtmfrecord("#bdq",20) <= 0)
		return NULL;

	    /* make sure we always see a 'b' or 'd' when present */

	    if (strchr(MsgBuf,'b') != NULL || strchr(MsgBuf,'d') != NULL)
		return NULL;
	} while (!strcmp(MsgBuf,"#"));		/* ignore only-# entry */

	/* remove everything up to last * (erase character) */

	where = MsgBuf;

	while ((p = strchr(where,'*')) != NULL)
	    where = p + 1;

	/* delete terminating # */

	if ((p = strchr(where,'#')) != NULL)
	    *p = '\0';

	/* delete any spurious 'silence' indications */

	while (*p == 's')
	    p++;

	debug3(D_RUN, "dtmfstring() returns (%s)\n",where);

	/* return the value, it still resides in MsgBuf! */

	return where;
}

/*************************************************************************
**	Voice mode low level routines					**
*************************************************************************/

/*
**	dtmfrecord() - get DTMF info using most efficient method
**
**	entered DTMF string is placed in MsgBuf, recording time is returned
*/

int
dtmfrecord (end,maxtime)
const char *end;			/* set of codes that ends recording */
int maxtime;				/* max recording time in seconds */

{
	char vsd[20];

	/* set the silence detection parameter suitable for the recording */
	/* interval (up to 25.5 seconds) */

	sprintf(vsd,"%d,%d",(Silence != NULL? atoi(Silence) : 20),10 * maxtime);

	/* newer versions allow recording of DTMF without recording voice */
	/* this is done by setting S39.6=1 and then (re-)issueing the +VLS */
	/* command to select the telephone line.  it does not work when */
	/* the external microphone is selected for recording... */

	if (Fw_version >= 610 && (RecLS == 0 || RecLS == 2)) {
	    char *msg;
	    register int ch;
	    sig_t (*oldint)(),(*oldquit)(),(*oldterm)(),(*oldalarm)();

	    debug4(D_RUN, "dtmfrecord(\"%s\",%d)\n", end, maxtime);

	    /* set silence detection for DTMF recording */

	    if (strcmp(CurVSD,vsd)) {
		sprintf(ChatBuf,"AT+VSD=%s\r",vsd);
		send(ChatBuf);
		if (waitfor("OK",MEDMTIME))
		    return FAIL;
		strcpy(CurVSD,vsd);
	    }

	    /* trap signals occurring during recording, so that we can */
	    /* shut up the MODEM before exiting */

	    Signal = 0;
	    oldint = signal(SIGINT, sighandler);
	    oldquit = signal(SIGQUIT, sighandler);
	    oldterm = signal(SIGTERM, sighandler);
	    oldalarm = signal(SIGALRM, sighandler);

	    send("ATS39.6=1+VLS=2\r");		/* enable DTMF recording from phone */
	    if (waitfor(NULL,MEDMTIME)) {	/* sometimes OK, sometimes VCON */
		maxtime = FAIL;
		goto error;
	    }
	    CurLS = 2;

	    /* set a maximal recording time limit */

	    if (maxtime <= 0 || maxtime > 120)	/* sanity check */
		maxtime = 120;

	    alarm(maxtime);

	    settermio(LONGTIME);
	    *(msg = MsgBuf) = '\0';

	    /* read the DTMF characters */

	    for (;;)
	    {
		switch (ch = getch())
		{
		case EOF:
		    break;			/* nothing, try again */

		case DLE:
		    switch (ch = getch())
		    {
		    case EOF:
			break;

		    case ETX:			/* end of message! */
			goto done;

		    default:			/* something else: DTMF or so */
			if (msg < (MsgBuf + sizeof(MsgBuf) - 1)) {
			    *msg++ = ch;	/* place codes in MsgBuf */
			    *msg = '\0';
			}

			/* terminate recording on certain character codes */

			if (strchr(end,ch) != NULL)
			    goto done;

			break;
		    }
		    break;

		default:			/* unexpected character */
		    alarm(0);
		    maxtime = FAIL;		/* return FAIL */
		    goto error;
		}

		if (Signal) {			/* timeout or other signal? */
		    Signal = 0;
		    alarm(0);
		    maxtime = FAIL;		/* return FAIL */
		    goto error;
		}
	    }

done:
	    maxtime -= alarm(0);		/* stop clock & measure time */

error:
	    send("ATS39.6=0+VLS=2\r");		/* turnoff the DTMF recording */
	    if (waitfor(NULL,MEDMTIME))
		return FAIL;
	    CurLS = 2;

	    signal(SIGINT, oldint);
	    signal(SIGQUIT, oldquit);
	    signal(SIGTERM, oldterm);
	    signal(SIGALRM, oldalarm);
#ifdef SA_SIGINFO
	    catchsig(SIGINT);
	    catchsig(SIGQUIT);
	    catchsig(SIGTERM);
	    catchsig(SIGALRM);
#endif

	    debug4(D_RUN, "dtmfrecord() returns %d, MsgBuf (%s)\n", maxtime, MsgBuf);
	    return maxtime;
	}

	/* do a recording with NULL output */

	return record(NULL,end,vsd,ENC_CELP,maxtime);
}

/*
**	record() - record a message from current speaker, or speaker defined
**		   by RecLS
**
**	return number of seconds recorded, 0 for SILENCE, or FAIL
**	expanded filename will be returned in original 'filename' arg!
*/

int
record(filename,end,vsd,enc,maxtime)
char *filename;				/* NULL to discard data (DTMF only) */
const char *end;			/* set of codes that ends recording */
const char *vsd;			/* silence detection params */
int enc;				/* CELP/ADPCM2/ADPCM3 */
int maxtime;				/* max recording time in seconds */
{
	register FILE *fp = NULL;
	register int ch;
	int done;
	int result = 1;
	int cur_ls = -1;
	char *msg;
	sig_t (*oldint)(),(*oldquit)(),(*oldterm)(),(*oldalarm)();
	ZAD_HEADER header;

#ifdef DEBUG
	if (filename != NULL)
	    debug3(D_RUN, "record(\"%s\"", filename);
	else
	    debug2(D_RUN, "record(NULL");
	debug4(D_RUN, ",\"%s\",\"%s\"", end, vsd);
	debug4(D_RUN, ",%d,%d)\n", enc, maxtime);
#endif

	/* set silence detection for message recording */

	if (vsd != NULL && strcmp(CurVSD,vsd)) {
	    sprintf(ChatBuf,"AT+VSD=%s\r",vsd);
	    send(ChatBuf);
	    if (waitfor("OK",MEDMTIME))
		return FAIL;
	    strcpy(CurVSD,vsd);
	}

	/* set vsm scheme */
	/* adapt to "nearest value" when attached modem does not support */
	/* the desired scheme */

	if ((enc = set_vsm(enc)) == FAIL)
	    return FAIL;

	/* create the output file, when requested */

	if (filename != NULL) {
	    /* create filename for recording */

	    if (RecDir != NULL && filename[0] != '/') {
		char buf[MAXLINE];

		sprintf(buf,"%s/%s",RecDir,filename);
		if (strcmp(filename + strlen(filename) - 4,AudioExt))
		    strcat(buf,AudioExt);
		strcpy(filename,buf);
	    }

	    /* %-macro's will be expanded using strftime, to generate */
	    /* a time-dependent recording file */

	    if (strchr(filename,'%') != NULL) {
		time_t now = time(NULL);
		char buf[MAXLINE];

		strftime(buf,sizeof(buf),filename,localtime(&now));
		strcpy(filename,buf);
	    }

	    if ((fp = fopen(filename,"w")) == NULL) {
		sprintf(MsgBuf, "cannot create audiofile (%s)", filename);
		logmesg(MsgBuf);
		return FAIL;
	    }

	    /* prepare an audio file header */

	    memset(&header,0,sizeof(header));
	    memcpy(header.zyxel,ZAD_MAGIC,sizeof(header.zyxel));
	    header.enc = enc;

	    fwrite(&header,sizeof(header),1,fp);
	}

	/* trap signals occurring during recording, so that we can */
	/* shut up the MODEM before exiting */

	Signal = 0;
	oldint = signal(SIGINT, sighandler);
	oldquit = signal(SIGQUIT, sighandler);
	oldterm = signal(SIGTERM, sighandler);
	oldalarm = signal(SIGALRM, sighandler);

	/* when RecLS has been set, save the current LS and set it */

	if (RecLS != 0) {
	    cur_ls = CurLS;

	    if (set_vls(RecLS)) {		/* select speaker */
		debug3(D_RUN, "record() cannot set LS to %d\n", RecLS);
		result = FAIL;
		goto done;
	    }

	    debug4(D_RUN, "record() VLS was %d, is %d\n", cur_ls, RecLS);
	}

	/* set a maximal recording time limit, so that the MODEM will */
	/* not fill the entire disk when the silence detection fails... */

	if (maxtime <= 0 || maxtime > 900)	/* sanity check */
	    maxtime = 120;

	alarm(maxtime);

	/* sound an attention beep */

	if (filename != NULL && beep(RecAttn) != SUCCESS) {
	    result = FAIL;
	    goto done;
	}

	/* start recording */

	send("AT+VRX\r");
	if (waitfor("CONNECT",MEDMTIME)) {
	    send("AT\r");			/* make sure we abort it */
	    result = FAIL;
	    goto done;
	}

	/* don't switch to XON/XOFF mode here.  we are receiving */
	/* binary data and eating these characters is not a good idea... */
	/* let's simply assume we are fast enough to keep up */

	settermio(SHORT255);
	if (Fw_version < 613)			/* >= 6.13: RTS/CTS flow */
	    write(STDOUT,"\021",1);		/* send XON to be sure */
	done = 0;
	*(msg = MsgBuf) = '\0';

	/* read the voice data, it is terminated by DLE-ETX */
	/* or a timeout of MEDMTIME on receiving voice data */

	for (;;)
	{
	    switch (ch = getch())
	    {
	    case EOF:
		debug2(D_RUN, "record() timeout\n");
		goto done;

	    case DLE:
		switch (ch = getch())
		{
		case EOF:
		    debug2(D_RUN, "record() timeout\n");
		    goto done;

		case ETX:			/* end of message! */
		    goto done;

		case DLE:			/* doubled DLE */
		case DC2:			/* resync symbol */
		case FS:			/* resync symbol */
		    if (fp != NULL) {
			if (enc & ENC_RESYNC)	/* when we are doing resync */
			    putc(DLE,fp);	/* the DLE should be in the file */

			putc(ch,fp);
		    }
		    break;

		default:			/* something else: DTMF or so */
		    if (msg < (MsgBuf + sizeof(MsgBuf) - 1)) {
			*msg++ = ch;		/* place codes in MsgBuf */
			*msg = '\0';
		    }

		    /* terminate recording on certain character codes */

		    if (!done && strchr(end,ch) != NULL)
		    {
			debug3(D_RUN, "record() terminated by %c\n", ch);
			send("AT\r");		/* abort recording */
			settermio(SHORTTIME);
			done = 1;
			break;
		    }

		    /* always terminate recording when 's'ilence */
		    /* (and 's' not in the end codes) */

		    if (ch == 's')
		    {
			result = 0;		/* will delete file */
			debug2(D_RUN, "record() terminated by silence\n");
			if (!done) {
			    send("AT\r");	/* abort recording */
			    settermio(SHORTTIME);
			    done = 1;
			}
		    }
		    break;
		}
		break;

	    default:
		if (fp != NULL)
		    putc(ch,fp);		/* normal voice data */

		break;
	    }

	    if (Signal && !done) {
		debug3(D_RUN, "record() terminated by signal %x\n", Signal);
		Signal = 0;
		send("AT\r");			/* abort recording */
		settermio(SHORTTIME);
		done = 1;
	    }
	}

done:
	maxtime -= alarm(0);			/* stop clock & measure time */

	waitfor("VCON",MEDMTIME);		/* ignore the result */

	/* return recorded time when no failure occurred */

	if (result > 0 && maxtime != 0)
	    result = maxtime;

	/* when current LS was saved, restore it now */

	if (cur_ls >= 0 && set_vls(cur_ls))
	    result = FAIL;

	if (filename != NULL)		/* signal end of recording */
	    beep(RecDone);

	if (fp != NULL && fclose(fp) == EOF)
	    result = FAIL;		/* disk full or so */

	if (filename != NULL && result <= 0)
	    unlink(filename);		/* failure or silence, remove the file */

	signal(SIGINT, oldint);
	signal(SIGQUIT, oldquit);
	signal(SIGTERM, oldterm);
	signal(SIGALRM, oldalarm);
#ifdef SA_SIGINFO
	catchsig(SIGINT);
	catchsig(SIGQUIT);
	catchsig(SIGTERM);
	catchsig(SIGALRM);
#endif

	debug4(D_RUN, "record() returns %d, MsgBuf (%s)\n", result, MsgBuf);
	return result;
}


/*
**	beep() - send a tone string
*/

int
beep(s)
const char *s;

{
	char buf[MAXLINE];		/* local buffer for command */

	if (s != NULL) {
	    sprintf(buf,"AT+VTS=%s\r",s);
	    send(buf);
	    return waitfor("OK",LONGTIME);
	}

	return SUCCESS;			/* NULL string, always ok */
}


/*
**	play() - play the specified file to current speaker
**
**	a new version now uses mmap() when available
*/

#ifdef MAP_PRIVATE
int
play(filename,end)
const char *filename;
const char *end;			/* set of codes that ends playing */
{
	register char *p;
	char *q,*msg;
	register int num,c;
	int fd,fdle,dle,done;
	caddr_t addr;
	size_t len;
	sig_t (*oldint)(),(*oldquit)(),(*oldterm)(),(*oldalarm)();
	ZAD_HEADER *header;
	char buf[MAXLINE];

#ifdef DEBUG
	if (end != NULL)
	    debug4(D_RUN, "play(\"%s\",\"%s\")\n", filename, end);
	else
	    debug3(D_RUN, "playcheck(\"%s\")\n", filename);
#endif

	/* when it isn't a full pathname, make it into one (and add extension) */

	if (PlayDir != NULL && filename[0] != '/') {
	    sprintf(buf,"%s/%s",PlayDir,filename);
	    if (strcmp(filename + strlen(filename) - 4,AudioExt))
		strcat(buf,AudioExt);
	    filename = buf;
	}

	/* open the file */

	if ((fd = open(filename,O_RDONLY)) < 0) {
	    if (end != NULL) {
		sprintf(MsgBuf, "cannot open audiofile (%s)", filename);
		logmesg(MsgBuf);
		MsgBuf[0] = '\0';
	    }
	    return FAIL;
	}

	/* map it into a memory area */
	/* a fixed mapping is used as Linux pre-0.99.12 did not support */
	/* dynamic mmap().  this can be changed in the future. */

	addr = (caddr_t)0x40000000;	/* choose an address to mmap */
	len = lseek(fd,0L,SEEK_END);	/* get file length */

	if ((addr = mmap(addr,len,PROT_READ,MAP_PRIVATE|MAP_FIXED,fd,0)) == NULL) {
	    close(fd);
	    sprintf(MsgBuf, "cannot mmap audiofile (%s)", filename);
	    logmesg(MsgBuf);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* check the file header */

	header = (ZAD_HEADER *) addr;

	if (len < sizeof(ZAD_HEADER) ||
	    memcmp(header->zyxel,ZAD_MAGIC,sizeof(header->zyxel)))
	{
	    munmap(addr,len);
	    close(fd);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* set encoding scheme (found in header) */
	/* when modem doesn't support it, value is modified and we FAIL */

	if (set_vsm(header->enc) != header->enc) {
	    munmap(addr,len);
	    close(fd);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* when everything OK up to here and end=NULL, return */
	/* this is used to check audio file existence/correctness */

	if (end == NULL) {
	    debug2(D_RUN, "playcheck() OK\n");
	    munmap(addr,len);
	    close(fd);
	    MsgBuf[0] = '\0';
	    return SUCCESS;
	}

	/* start playing */

	send("AT+VTX\r");
	if (waitfor("CONNECT",MEDMTIME)) {
	    munmap(addr,len);
	    close(fd);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* trap signals occurring during playing, so that we can */
	/* reset the MODEM before exiting */

	Signal = 0;
	oldint = signal(SIGINT, sighandler);
	oldquit = signal(SIGQUIT, sighandler);
	oldterm = signal(SIGTERM, sighandler);
	oldalarm = signal(SIGALRM, sighandler);

	if (Fw_version >= 613)
	    settermio(IMMEDIATE);		/* >= 6.13: hw flow ctl */
	else
	    settermio(XONXOFF);			/* XON/XOFF flow ctl */

	p = q = addr + sizeof(ZAD_HEADER);
	num = len - sizeof(ZAD_HEADER);

	dle = done = 0;
	*(msg = MsgBuf) = '\0';

	for (;;)
	{
	    /* send more audio data when still required */

	    alarm(5);				/* watchdog in case XON is missed */

	    if (header->enc & ENC_RESYNC)	/* resync info in data? */
	    {
		c = num;			/* number of bytes to xfer */

		if (c > 1024)
		    c = 1024;			/* max size of one chunk */

		if (p == q && c > sizeof(ZAD_HEADER))
		    c -= sizeof(ZAD_HEADER);	/* align next transfer */

		if ((c = write(STDOUT,q,c)) >= 0) /* write a chunk */
		{
		    q += c;			/* next chunk */
		    num -= c;
		}
	    }
	    else				/* no resync info */
	    {
		c = 1024;			/* max size of one chunk */

		while (!done && c > 0)
		{
		    fdle = 0;

		    /* send a block of audio data, doubling DLE as needed */

		    while (--num >= 0 && --c >= 0)
			if (*p++ == DLE) {	/* find DLE as we have to double it */
			    fdle = 1;
			    break;
			}

		    if (write(STDOUT,q,p - q) <= 0) /* write data including any DLE */
			break;

		    if (num < 0) {		/* all sent? */
			send(AudioEnd);		/* DLE-ETX ends the data */
			settermio(LONGTIME);
			done = 1;
		    } else {
			q = p - fdle;		/* more to go, update pointer */
						/* next time around, DLE again! */
		    }
		}
	    }

	    /* read the returned characters, can contain DTMF codes etc */

	    while ((c = getch()) != EOF)
	    {
		if (dle)		/* previous was a DLE? */
		{
		    switch (c)		/* what is next character? */
		    {
		    case ETX:		/* end of message! */
			break;		/* ignore when sending */

		    case DLE:		/* doubled DLE */
			break;

		    default:		/* something else: DTMF or so */
			*msg++ = c;	/* place codes in MsgBuf */
			*msg = '\0';

			/* terminate playing on certain passed chars */

			if (!done &&
			    (strchr(end,c) != NULL ||
			     msg > (MsgBuf + sizeof(MsgBuf) - 3)))
			{
			    debug3(D_RUN, "play() terminated by %c\n", c);
			    send(AudioAbort);	/* ETX-DLE-ETX/DC4 */
			    settermio(LONGTIME);
			    done = 1;
			}
			break;
		    }

		    dle = 0;
		} else {
		    switch (c)		/* what is next character? */
		    {
		    case DLE:		/* is it a DLE? */
			dle = 1;
			break;

		    case XON:		/* a flow control char leaked thru? */
		    case XOFF:
			break;		/* ignore it */

		    default:		/* something else, probably ^M */
			settermio(INITIAL);

			if (done) {
			    /* this will be the VCON at end of play */

			    alarm(0);
			    InPtr--;		/* push it back */
			    waitfor("VCON",LONGTIME); /* let it finish */
			} else {
			    /* it came in while we were sending */
			    /* probably DATA/VOICE pressed or so */

			    debug2(D_RUN, "play() terminated by unexpected char\n");
			    InPtr--;		/* push it back */
			    send(AudioAbort);	/* ETX-DLE-ETX/DC4 */
			    waitfor(NULL,LONGTIME); /* let it riddle */
			    send("AT\r");
			    waitfor("OK",MEDMTIME);
			}

			alarm(0);
			munmap(addr,len);
			close(fd);

			signal(SIGINT, oldint);
			signal(SIGQUIT, oldquit);
			signal(SIGTERM, oldterm);
			signal(SIGALRM, oldalarm);
#ifdef SA_SIGINFO
			catchsig(SIGINT);
			catchsig(SIGQUIT);
			catchsig(SIGTERM);
			catchsig(SIGALRM);
#endif

			debug3(D_RUN, "play() returns MsgBuf (%s)\n", MsgBuf);
			return SUCCESS;
		    }
		}
	    }

	    /* see if we have been requested to stop the playing */

	    if (Signal && !done) {
		debug3(D_RUN, "play() terminated by signal %x\n", Signal);
		Signal = 0;
		send(AudioAbort);		/* ETX-DLE-ETX/DC4 */
		settermio(LONGTIME);
		done = 1;
	    }
	}
}

#else
/* version using normal FILE input */

int
play(filename,end)
char *filename;
char *end;				/* set of codes that ends playing */
{
	FILE *fp;
	char *p,*q,*msg;
	int num;
	int c;
	int dle;
	int done;
	sig_t (*oldint)(),(*oldquit)(),(*oldterm)(),(*oldalarm)();
	ZAD_HEADER header;
	char buf[BUFSIZ];

#ifdef DEBUG
	if (end != NULL)
	    debug4(D_RUN, "play(\"%s\",\"%s\")\n", filename, end);
	else
	    debug3(D_RUN, "playcheck(\"%s\")\n", filename);
#endif

	/* when it isn't a full pathname, make it into one (and add extension) */

	if (PlayDir != NULL && filename[0] != '/') {
	    sprintf(buf,"%s/%s",PlayDir,filename);
	    if (strcmp(filename + strlen(filename) - 4,AudioExt))
		strcat(buf,AudioExt);
	    filename = buf;
	}

	/* open the file */

	if ((fp = fopen(filename,"r")) == NULL) {
	    if (end != NULL) {
		sprintf(MsgBuf, "cannot open audiofile (%s)", filename);
		logmesg(MsgBuf);
		MsgBuf[0] = '\0';
	    }
	    return FAIL;
	}

	/* read and check the file header */

	if (fread(&header,sizeof(header),1,fp) != 1 ||
	    memcmp(header.zyxel,ZAD_MAGIC,sizeof(header.zyxel)))
	{
	    fclose(fp);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* set encoding scheme (found in header) */
	/* when modem doesn't support it, value is modified and we FAIL */

	if (set_vsm(header.enc) != header.enc) {
	    fclose(fp);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* when everything OK up to here and end=NULL, return */
	/* this is used to check audio file existence/correctness */

	if (end == NULL) {
	    debug2(D_RUN, "playcheck() OK\n");
	    fclose(fp);
	    MsgBuf[0] = '\0';
	    return SUCCESS;
	}

	/* start playing */

	send("AT+VTX\r");
	if (waitfor("CONNECT",MEDMTIME)) {
	    fclose(fp);
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* trap signals occurring during playing, so that we can */
	/* reset the MODEM before exiting */

	Signal = 0;
	oldint = signal(SIGINT, sighandler);
	oldquit = signal(SIGQUIT, sighandler);
	oldterm = signal(SIGTERM, sighandler);
	oldalarm = signal(SIGALRM, sighandler);

	if (Fw_version >= 613)
	    settermio(IMMEDIATE);		/* >= 6.13: hw flow ctl */
	else
	    settermio(XONXOFF);			/* XON/XOFF flow ctl */

	dle = done = 0;
	*(msg = MsgBuf) = '\0';

	for (;;)
	{
	    alarm(BUFSIZ/200);		/* watchdog in case XON is missed */

	    /* read a block of audio data from the file */

	    if (!done)
	    {
		if ((num = fread(buf,1,sizeof(buf),fp)) > 0)
		{
		    if (header.enc & ENC_RESYNC) /* resync info in data? */
			write(STDOUT,buf,num);
		    else			/* no resync info */
		    {
			/* send this block of audio data, doubling DLE as needed */

			p = q = buf;

			for (;;)
			{
			    while (--num >= 0)
				if (*p++ == DLE) /* find DLE as we have to double it */
				    break;

			    if (write(STDOUT,q,p - q) <= 0) /* write data including DLE */
				break;

			    q = p - 1;		/* next time around, DLE again! */

			    if (num < 0)	/* all sent? */
				break;
			}
		    }
		}
		else
		{
		    send(AudioEnd);	/* DLE-ETX ends the data */
		    settermio(LONGTIME);
		    done = 1;
		}
	    }

	    /* read the returned characters, can contain DTMF codes etc */

	    while ((c = getch()) != EOF)
	    {
		if (dle)		/* previous was a DLE? */
		{
		    switch (c)		/* what is next character? */
		    {
		    case ETX:		/* end of message! */
			break;		/* ignore when sending */

		    case DLE:		/* doubled DLE */
			break;

		    default:		/* something else: DTMF or so */
			*msg++ = c;	/* place codes in MsgBuf */
			*msg = '\0';

			/* terminate playing on certain passed chars */

			if (!done &&
			    (strchr(end,c) != NULL ||
			     msg > (MsgBuf + sizeof(MsgBuf) - 3)))
			{
			    debug3(D_RUN, "play() terminated by %c\n", c);
			    send(AudioAbort);	/* ETX-DLE-ETX/DC4 */
			    settermio(LONGTIME);
			    done = 1;
			}
			break;
		    }

		    dle = 0;
		} else {
		    switch (c)		/* what is next character? */
		    {
		    case DLE:		/* is it a DLE? */
			dle = 1;
			break;

		    case XON:		/* a flow control char leaked thru? */
		    case XOFF:
			break;		/* ignore it */

		    default:		/* something else, probably ^M */
			settermio(INITIAL);

			if (done) {
			    /* this will be the VCON at end of play */

			    alarm(0);
			    InPtr--;		/* push it back */
			    waitfor("VCON",LONGTIME); /* let it finish */
			} else {
			    /* it came in while we were sending */
			    /* probably DATA/VOICE pressed or so */

			    debug2(D_RUN, "play() terminated by unexpected char\n");
			    InPtr--;		/* push it back */
			    send(AudioAbort);	/* ETX-DLE-ETX/DC4 */
			    waitfor(NULL,LONGTIME); /* let it riddle */
			    send("AT\r");
			    waitfor("OK",MEDMTIME);
			}

			alarm(0);
			fclose(fp);
			signal(SIGINT, oldint);
			signal(SIGQUIT, oldquit);
			signal(SIGTERM, oldterm);
			signal(SIGALRM, oldalarm);
#ifdef SA_SIGINFO
			catchsig(SIGINT);
			catchsig(SIGQUIT);
			catchsig(SIGTERM);
			catchsig(SIGALRM);
#endif

			debug3(D_RUN, "play() returns MsgBuf (%s)\n", MsgBuf);
			return SUCCESS;
		    }
		}
	    }

	    /* see if we have been requested to stop the playing */

	    if (Signal && !done) {
		debug3(D_RUN, "play() terminated by signal %x\n", Signal);
		Signal = 0;
		send(AudioAbort);		/* ETX-DLE-ETX/DC4 */
		settermio(LONGTIME);
		done = 1;
	    }
	}

	debug2(D_RUN, "play() fell through\n");
	return FAIL;				/* 'unreachable' */
}
#endif

/*
**	auplay() - play audio file
**
**	this function is not particularly efficient, but at least it
**	offers the capability to play audio files...
*/

int
auplay(filename)
char *filename;
{
	int fd,ret;
	char buf[4];
	extern int read();

	/* open the file */

	if ((fd = open(filename,O_RDONLY)) < 0)
	    return FAIL;

	/* when it starts with .snd, assume 8k/s */

	if (read(fd,buf,4) == 4 && !memcmp(buf,".snd",4)) {
	    lseek(fd,32L,SEEK_SET);
	    ret = ausend(fd,read8k);
	} else {
	    lseek(fd,0L,SEEK_SET);
	    ret = ausend(fd,read);
	}

	close(fd);
	return ret;
}

/*
**	say() - call voice synthesizer program to read string
**
*/

int
say(string)
const char *string;
{
	int ifd,ofd,ret;
	pid_t pid;

	if (Synth == NULL)
	    return FAIL;

	debug3(D_RUN, "say(\"%s\")\n", string);

	/* start the synthesizer process, return pid and pipe descriptors */

	if (startsynth(&pid,&ifd,&ofd) != SUCCESS)
	    return FAIL;

	/* send the string to the synthesizer process so it can */
	/* start off preparing the output */

	write(ofd,string,strlen(string));
	close(ofd);

	/* transcode the audio data and send it to the modem */

	ret = ausend(ifd,auread);

	/* wait for the synthesizer process to finish */

	close(ifd);
	alarm(10);
	waitpid(pid,&ofd,0);
	alarm(0);

	return ret;
}

/*
**	run() - run an external program and read stdout on synthesizer
**
*/

int
run(argv)
char *argv[];
{
	int ifd,ofd,ret,i;
	pid_t spid,rpid;
	char *prog;

	debug3(D_RUN, "run(\"%s\")\n", argv[0]);

	/* start the synthesizer process, return pid and pipe descriptors */

	if (Synth != NULL && startsynth(&spid,&ifd,&ofd) != SUCCESS)
	    return FAIL;

	/* start the program to run */

	switch (rpid = fork())
	{
	case -1:				/* fork() failure */
	    if (Synth != NULL) {
		close(ofd);
		close(ifd);
		alarm(10);
		waitpid(spid,&ofd,0);
		alarm(0);
	    }

	    logmesg("cannot fork 'run' process");
	    return FAIL;

	case 0:					/* child process */
	    close(STDIN);
	    open("/dev/null",O_RDWR);		/* STDIN to /dev/null */

	    if (Synth != NULL)
		dup2(ofd,STDOUT);		/* STDOUT to synthesizer */
	    else {
		close(STDOUT);
		open("/dev/null",O_RDWR);	/* STDOUT to /dev/null */
	    }

	    close(STDERR);
	    open("/dev/null",O_RDWR);		/* STDERR to /dev/null */

	    for (i = 3; i < 20; i++)
		close(i);			/* close remaining files */
	    while (close(i) == 0)
		i++;

	    while (isalpha(argv[0][0]) && strchr(argv[0],'=') != NULL) {
		putenv(argv[0]);		/* put VAR=value in env */
		argv++;
	    }

	    prog = argv[0];

	    if ((argv[0] = strrchr(Synth,'/')) != NULL) /* get basename */
		argv[0]++;
	    else
		argv[0] = prog;

	    for (i = 1; argv[i] != NULL; i++)	/* substitute magic args */
		if (argv[i][0] == '@' && strlen(argv[i]) == 2)
		    switch (argv[i][1])
		    {
		    case 'C':			/* @C -> last received CONNECT */
			argv[i] = Connect? Connect:"";
			break;

		    case 'L':			/* @L -> line (Device) name */
			argv[i] = Device;
			break;

		    case 'N':			/* @N -> readnumber value */
			argv[i] = Number;
			break;

		    case 'R':			/* @R -> current recording file name */
			argv[i] = Recordfile;
			break;

		    case 'S':			/* @S -> system nodename */
			argv[i] = SysName;
			break;
		    }

#ifdef	DEBUG
	    if (Dfp != NULL) {
		fclose(Dfp);
		Dfp = NULL;
	    }
#endif	/* DEBUG */

	    for (i = 0; i < NSIG; i++)		/* set all signals to SIG_DFL */
		signal(i, SIG_DFL);

	    nice(9);				/* decrease priority */
	    execv(prog,argv);
	    logmesg("cannot execute 'run' process");
	    exit(1);

	default:				/* parent process */
	    if (Synth != NULL)
		close(ofd);
	    break;
	}

	if (Synth != NULL) {
	    /* transcode the audio data and send it to the modem */

	    ret = ausend(ifd,auread);

	    /* wait for the synthesizer process to finish */

	    close(ifd);
	    alarm(10);
	    waitpid(spid,&ofd,0);
	    alarm(0);
	} else
	    ret = SUCCESS;

	/* wait for 'run' process to finish */

	alarm(60);
	waitpid(rpid,&ofd,0);
	alarm(0);

	return ret;
}

/*
**	startsynth() - start the voice synthesizer as a child process
**
**	return process id and filedescriptors of input and output pipes
**	via args
*/

int
startsynth(pid,ifd,ofd)
pid_t *pid;
int *ifd,*ofd;
{
	int ret1,ret2,i,argc;
	int pipe0[2],pipe1[2];
	char *p,*q;
	char *argv[NARG];

	/* setup the pipes we need */

	ret2 = -1;
	if ((ret1 = pipe(pipe0)) || (ret2 = pipe(pipe1))) {
	    if(ret1 == 0) {
		close(pipe0[0]);
		close(pipe0[1]);
		if(ret2 == 0) {
		    close(pipe1[0]);
		    close(pipe1[1]);
		}
	    }
	    logmesg("cannot setup pipes for synth process");
	    return FAIL;
	}

	/* start the voice synthesizer process */

	switch (*pid = fork())
	{
	case -1:				/* fork() failure */
	    close(pipe0[0]);
	    close(pipe0[1]);
	    close(pipe1[0]);
	    close(pipe1[1]);
	    logmesg("cannot fork synth process");
	    return FAIL;

	case 0:					/* child process */
	    dup2(pipe0[0],STDIN);
	    dup2(pipe1[1],STDOUT);
	    close(STDERR);
	    open("/dev/null",O_RDWR);		/* STDERR to /dev/null */

	    for (i = 3; i < 20; i++)
		close(i);			/* close remaining files */
	    while (close(i) == 0)
		i++;

	    if ((p = strrchr(Synth,'/')) != NULL) /* get Synth basename */
		p++;
	    else
		p = Synth;

	    argc = 0;				/* build argvector */
	    memset(argv,0,sizeof(argv));

	    argv[argc++] = p;

	    if (SynthFlags != NULL) {		/* add options */
		strcpy(q = ChatBuf,SynthFlags);

		while ((p = strchr(q,' ')) != NULL) {
		    *p++ = '\0';
		    argv[argc++] = q;
		    q = p;
		}

		argv[argc++] = q;
	    }

#ifdef	DEBUG
	    if (Dfp != NULL) {
		fclose(Dfp);
		Dfp = NULL;
	    }
#endif	/* DEBUG */

	    for (i = 0; i < NSIG; i++)		/* set all signals to SIG_DFL */
		signal(i, SIG_DFL);

	    setuid(UuUid);			/* decrease risk */
	    setgid(UuGid);
	    nice(5);				/* decrease priority */
	    execv(Synth,argv);
	    logmesg("cannot execute synth process");
	    exit(1);

	default:				/* parent process */
	    close(pipe0[0]);
	    close(pipe1[1]);
	    break;
	}

	*ofd = pipe0[1];			/* return descriptors */
	*ifd = pipe1[0];

	return SUCCESS;
}

/*
**	ausend() - send audio data from a pipe to the modem
**
**	the data is transcoded from the values read from the pipe into
**	ADPCM, and sent to the modem at proper pace.
*/

int
ausend(fd,rdf)
int fd;
int (*rdf)(int fd,char *buf,size_t size);
{
	char *msg,*optr;
	register int c;
	int dle,done,enc,inptr,inlen;
	size_t num;
	sig_t (*oldint)(),(*oldquit)(),(*oldterm)(),(*oldalarm)();
	char ibuf[4096],obuf[BUFSIZ+2];
#define getpipe()	(inptr >= inlen && (inlen = (*rdf)(fd,ibuf,sizeof(ibuf))) <= (inptr = 0)? EOF : (int)(unsigned char)ibuf[inptr++])
#define putres(res)	{ if (res == DLE) *optr++ = DLE; *optr++ = res; }

	/* set voice data encoding scheme */
	/* try to get ADPCM4, fallback when not available */

	if ((enc = set_vsm(ENC_ADPCM4)) == FAIL) {
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* start playing mode */

	send("AT+VTX\r");
	if (waitfor("CONNECT",MEDMTIME)) {
	    MsgBuf[0] = '\0';
	    return FAIL;
	}

	/* trap signals occurring during saying, so that we can */
	/* reset the MODEM before exiting */

	Signal = 0;
	oldint = signal(SIGINT, sighandler);
	oldquit = signal(SIGQUIT, sighandler);
	oldterm = signal(SIGTERM, sighandler);
	oldalarm = signal(SIGALRM, sighandler);

	if (Fw_version >= 613)
	    settermio(IMMEDIATE);		/* >= 6.13: hw flow ctl */
	else
	    settermio(XONXOFF);			/* XON/XOFF flow ctl */

	Delta = 5;				/* initialize the coder */
	EstMax = 0;
	Leakage = Fw_version >= 613? 4093 : 3973;

	Samples = 0;

	dle = done = inptr = inlen = 0;		/* initialize buffering */
	*(msg = MsgBuf) = '\0';
	optr = obuf;

	alarm(120);				/* guard the synth process */

	for (;;)
	{
	    /* read audio data from the pipe and send it to modem as ADPCM */
	    /* select packing algorithm (2, 3 or 4 bits per sample) */

	    switch (enc)
	    {
	    case ENC_ADPCM2:
		while (!done && optr < (obuf + sizeof(obuf) - 2))
		{
		    register int res;

		    if ((inlen - inptr) < 4 && (num = optr - obuf) != 0) {
			write(STDOUT,obuf,num);
			debug3(D_CHAT, "write: %d bytes\n",num);
			optr = obuf;
		    }

		    if ((c = getpipe()) == EOF) /* get a byte (buffered) */
			break;			/* quit when no more data */

		    res = adpcm(ENC_ADPCM2,c) << 6;	/* XX------ */

		    if ((c = getpipe()) == EOF)
			break;

		    res |= adpcm(ENC_ADPCM2,c) << 4;	/* --XX---- */

		    if ((c = getpipe()) == EOF)
			break;

		    res |= adpcm(ENC_ADPCM2,c) << 2;	/* ----XX-- */

		    if ((c = getpipe()) == EOF)
			break;

		    res |= adpcm(ENC_ADPCM2,c);		/* ------XX */

		    putres(res);
		}
		break;

	    case ENC_ADPCM3:
		while (!done && optr < (obuf + sizeof(obuf) - 6))
		{
		    register int res;

		    if ((inlen - inptr) < 8 && (num = optr - obuf) != 0) {
			write(STDOUT,obuf,num);
			debug3(D_CHAT, "write: %d bytes\n",num);
			optr = obuf;
		    }

		    if ((c = getpipe()) == EOF) /* get a byte (buffered) */
			break;			/* quit when no more data */

		    res = adpcm(ENC_ADPCM3,c) << 5;	/* XXX----- */

		    if ((c = getpipe()) == EOF)
			break;

		    res |= adpcm(ENC_ADPCM3,c) << 2;	/* ---XXX-- */

		    if ((c = getpipe()) == EOF)
			break;

		    res |= (c = adpcm(ENC_ADPCM3,c)) >> 1; /* ------XX */
		    putres(res);
		    res = (c & 0x01) << 7;		/* X------- */

		    if ((c = getpipe()) == EOF)
			break;

		    res |= adpcm(ENC_ADPCM3,c) << 4;	/* -XXX---- */

		    if ((c = getpipe()) == EOF)
			break;

		    res |= adpcm(ENC_ADPCM3,c) << 1;	/* ----XXX- */

		    if ((c = getpipe()) == EOF)
			break;

		    res |= (c = adpcm(ENC_ADPCM3,c)) >> 2; /* -------X */
		    putres(res);
		    res = (c & 0x03) << 6;		/* XX------ */

		    if ((c = getpipe()) == EOF)
			break;

		    res |= adpcm(ENC_ADPCM3,c) << 3;	/* --XXX--- */

		    if ((c = getpipe()) == EOF)
			break;

		    res |= adpcm(ENC_ADPCM3,c);		/* -----XXX */

		    putres(res);
		}
		break;

	    case ENC_ADPCM4:
		while (!done && optr < (obuf + sizeof(obuf) - 2))
		{
		    register int res;

		    if ((inlen - inptr) < 2 && (num = optr - obuf) != 0) {
			write(STDOUT,obuf,num);
			debug3(D_CHAT, "write: %d bytes\n",num);
			optr = obuf;
		    }

		    if ((c = getpipe()) == EOF) /* get a byte (buffered) */
			break;			/* quit when no more data */

		    res = adpcm(ENC_ADPCM4,c) << 4;	/* XXXX---- */

		    if ((c = getpipe()) == EOF)
			break;

		    res |= adpcm(ENC_ADPCM4,c);		/* ----XXXX */

		    putres(res);
		}
		break;
	    }

	    /* flush-out the buffer, send AudioEnd when nothing buffered */
	    /* (this indicates the routines above could not produce more */
	    /* data, and we're done.  set flag to avoid further reading) */

	    if (!done)
		if ((num = optr - obuf) != 0) {
		    write(STDOUT,obuf,num);
		    debug3(D_CHAT, "write: %d bytes\n",num);
		    optr = obuf;
		} else {
		    send(AudioEnd);	/* DLE-ETX ends the data */
		    settermio(LONGTIME);
		    done = 1;
		}

	    /* read the returned characters, can contain DTMF codes etc */

	    while ((c = getch()) != EOF)
	    {
		if (dle)		/* previous was a DLE? */
		{
		    switch (c)		/* what is next character? */
		    {
		    case ETX:		/* end of message! */
			break;		/* ignore when sending */

		    case DLE:		/* doubled DLE */
			break;

		    default:		/* something else: DTMF or so */
			*msg++ = c;	/* place codes in MsgBuf */
			*msg = '\0';

			/* terminate saying on certain passed chars */

			if (!done &&
			    (strchr("#bd",c) != NULL ||
			     msg > (MsgBuf + sizeof(MsgBuf) - 3)))
			{
			    debug3(D_RUN, "say() terminated by %c\n", c);
			    send(AudioAbort);	/* ETX-DLE-ETX/DC4 */
			    settermio(LONGTIME);
			    done = 1;
			}
			break;
		    }

		    dle = 0;
		} else {
		    switch (c)		/* what is next character? */
		    {
		    case DLE:		/* is it a DLE? */
			dle = 1;
			break;

		    case XON:		/* a flow control char leaked thru? */
		    case XOFF:
			break;		/* ignore it */

		    default:		/* something else, probably ^M */
			settermio(INITIAL);

			if (done) {
			    /* this will be the VCON at end of say */

			    InPtr--;		/* push it back */
			    waitfor("VCON",LONGTIME); /* let it finish */
			} else {
			    /* it came in while we were sending */
			    /* probably DATA/VOICE pressed or so */

			    debug2(D_RUN, "say() terminated by unexpected char\n");
			    InPtr--;		/* push it back */
			    send(AudioAbort);	/* ETX-DLE-ETX/DC4 */
			    waitfor(NULL,LONGTIME); /* let it riddle */
			    send("AT\r");
			    waitfor("OK",MEDMTIME);
			}

			alarm(0);
			signal(SIGINT, oldint);
			signal(SIGQUIT, oldquit);
			signal(SIGTERM, oldterm);
			signal(SIGALRM, oldalarm);
#ifdef SA_SIGINFO
			catchsig(SIGINT);
			catchsig(SIGQUIT);
			catchsig(SIGTERM);
			catchsig(SIGALRM);
#endif

			debug3(D_RUN, "say() returns MsgBuf (%s)\n", MsgBuf);
			return SUCCESS;
		    }
		}
	    }

	    /* see if we have been requested to stop the saying */

	    if (Signal && !done) {
		debug3(D_RUN, "say() terminated by signal %x\n", Signal);
		Signal = 0;
		send(AudioAbort);		/* ETX-DLE-ETX/DC4 */
		settermio(LONGTIME);
		done = 1;
	    }
	}
}

/*
**	adpcm() - ADPCM coder
**
**	this routine takes an 8-bit value and returns the n-bit ADPCM code
**
**	unfortunately there are no real specs.	ZyXEL releases a program
**	to convert soundblaster files to ADPCM, and updates this every few
**	ROM releases to adjust some coefficients.  the routine below is
**	patterned after this program (vcnvt.c)
*/

int
adpcm(enc,edata)
int enc;					/* encoding, ENC_ADPCMx */
register int edata;				/* value to encode */
{
	int tmpcompbit,databit,signbit;

#ifdef ULAW_AUDIO
	edata = (edata & 0x80)? Ulaw[edata & 0x7f] : -Ulaw[edata];
#else
	edata = ((edata - 128) << 6) + 32;	/* 8-bit unsigned data */
#endif

	databit = signbit = 0;

	/* Check for the waveform data and quantize this data */
	if (edata -= EstMax) {
	    tmpcompbit = 1 << enc;
	    /* ----------------------------------------------------- */
	    /* If the data is negative, set flag and change the sign */
	    /* ----------------------------------------------------- */
	    if (edata < 0) {
		edata = -edata;
		signbit = tmpcompbit;
	    }
	    /* --------------------------------------------------- */
	    /* Quantize the waveform data, delta value is adaptive */
	    /* --------------------------------------------------- */
	    while (((edata -= Delta) > 0) && --tmpcompbit)
		databit += 1;
	}

	if (enc == ENC_ADPCM4 && databit == 0)
	    signbit = 8;

	/* the leakage */
	EstMax = (((long)EstMax * Leakage) + 2048) / 4096;

	if ((Delta & 1) && !signbit)
	    EstMax++;

	/* Calculate the Delta and EstMax */
	if (signbit)
	    EstMax -= (Delta >> 1) + (long)Delta * databit;
	else
	    EstMax += (Delta >> 1) + (long)Delta * databit;

	Delta = (((long)Delta * MaxTbl[enc][databit]) + 8192) >> 14;

	return signbit + databit;
}

/*
**	auread() - read audio data
**
**	this is basically just read(), but when no data is available
**	for an extended period of time, some dummy data is returned...
**
**	the audio player, once started, needs 9600 samples per second.
**	the amount of data returned from this routine is computed from
**	that (or actually, the timeout on select is varied accordingly).
*/

int
auread(fd,buf,size)
int fd;
char *buf;
size_t size;
{
	int num;
	long usec;
	fd_set select_set;
	struct timeval timeval;

	FD_ZERO(&select_set);			/* all bits off */
	FD_SET(fd,&select_set);			/* set bit for spec'd fd */

	if (Samples == 0)			/* first time? */
	    usec = 100000;			/* wait 100ms for data */
	else
	{
	    gettimeofday(&timeval,NULL);	/* find elapsed time */
	    usec = 1000000 * (timeval.tv_sec - MatchTime.tv_sec) +
		    timeval.tv_usec - MatchTime.tv_usec;

	    usec = (10000 * Samples) / 96 - usec; /* subtract for 9600 sam/s */
	    usec -= 100000;			/* 100ms safety margin */

	    if (usec < 10000)			/* minimum of 10ms */
		usec = 10000;
	}

	timeval.tv_sec = usec / 1000000;	/* define timeout */
	timeval.tv_usec = usec % 1000000;

	if (select(fd + 1,&select_set,NULL,NULL,&timeval) > 0) {
	    if ((num = read(fd,buf,size)) > 0)
		Samples += num;

	    return num;
	}

	if (size > 2400)
	    size = 2400;			/* max bytes to return */

#ifdef ULAW_AUDIO
	memset(buf,0xff,size);
#else
	memset(buf,128,size);
#endif
	Samples += size;
	return size;
}

/*
**	read8k() - like read() but do dirty conversion from 8k to 9k6
*/

int
read8k(fd,buf,size)
int fd;
char *buf;
size_t size;
{
	char *p,*q;
	size_t num,n;

	/* read 5/6 of requested data at the end of the buffer */

	num = read(fd, q = buf + (size / 6), (5 * size) / 6);
	p = buf;

	/* copy bytes, duplicating each 5th byte */

	for (n = num; n >= 5; n -= 5) {
	    *p++ = *q;
	    *p++ = *q++;
	    *p++ = *q++;
	    *p++ = *q++;
	    *p++ = *q++;
	    *p++ = *q++;
	}

	while (n-- > 0)
	    *p++ = *q++;

	/* return number of converted bytes */

	return (int)(p - buf);
}

/*
**	set_vls() - set audio recording/playback device
**
*/

int
set_vls(ls)
int ls;						/* desired device */
{
	char buf[20];

	sprintf(buf,"AT+VLS=%d\r",ls);
	send(buf);

	if (waitfor(NULL,MEDMTIME) ||		/* sometimes OK, sometimes VCON */
	    !strcmp(Match,"ERROR"))		/* but not ERROR... */
	    return FAIL;

	CurLS = ls;
	return OK;
}

/*
**	set_vsm() - set audio encoding scheme
**
**	Input is the encoding scheme as used in ZFAX audio files.
**	This is not the same as the AT+VSM value sent to the modem!
*/

int
set_vsm(enc)
int enc;					/* desired encoding */
{
	int sync = 0;
	int vsm;
	char buf[20];

	/* check the encoding value.  see if the modem supports it. */

	switch (enc)
	{
	case ENC_CELP:
	    if (!Fw_plus)			/* not a PLUS, cannot record CELP */
		enc = ENC_ADPCM2;		/* do ADPCM2 instead... */

	    vsm = enc + 1;			/* VSM value is one higher */
	    break;

	case ENC_ADPCM2|ENC_RESYNC:
	case ENC_ADPCM3|ENC_RESYNC:
	    if (Fw_version < 610)		/* resync requires 6.10 */
		enc -= ENC_RESYNC;
	    else
		sync = 1;

	    /* fall-through */

	case ENC_ADPCM2:
	case ENC_ADPCM3:
	    vsm = (enc & ~ENC_RESYNC) + 1;
	    break;				/* supported by all modems */

	case ENC_ADPCM4|ENC_RESYNC:
	    if (Fw_version < 610)		/* resync requires 6.10 */
		enc -= ENC_RESYNC;
	    else
		sync = 1;

	    /* fall-through */

	case ENC_ADPCM4:
	    if (Fw_version < 613 ||		/* requires 6.13 firmware */
		Speed < 57600)			/* and 57600 bps */
		enc -= (ENC_ADPCM4-ENC_ADPCM3); /* else do ADPCM3 instead... */

	    vsm = (enc & ~ENC_RESYNC) + 1;
	    break;

	case ENC_ADPCM3N|ENC_RESYNC:		/* new ADPCM3 + resync */
	    if (Fw_version < 610)		/* resync requires 6.10 */
		enc -= ENC_RESYNC;
	    else
		sync = 1;

	    /* fall-through */

	case ENC_ADPCM3N:			/* new ADPCM3 */
	    if (Fw_version < 613) {		/* requires 6.13 firmware */
		enc -= (ENC_ADPCM3N-ENC_ADPCM3); /* do ADPCM3 instead... */
		vsm = (enc & ~ENC_RESYNC) + 1;
		break;
	    }

	    vsm = 30;
	    break;

	default:
	    return FAIL;			/* unsupported value */
	}

	if (sync == CurSync && vsm == CurVSM)	/* encoding already OK? */
	    return enc;

	sprintf(buf,"ATS39.7=%d+VSM=%d\r",sync,vsm);
	send(buf);

	if (waitfor("OK",MEDMTIME))
	    return FAIL;

	CurVSM = vsm;				/* save the set encoding */
	CurSync = sync;

	return enc;				/* return modified encoding */
}

/*************************************************************************
**	Misc utility functions						**
*************************************************************************/

/*
**	push() - start a process and 'push' it onto an fd
**
**	STDIN of the childprocess will be the fd passed as a parameter, and
**	STDOUT of the childprocess will replace that fd. thus, the childprocess
**	is effectively 'pushed' onto the fd, providing a filter function on
**	the file data.
**
**	The PID of the process is returned via parameter 1, as the caller
**	should waitpid() on it.
**
*/

int
push(pid,fd,program,argv)
pid_t *pid;					/* pid returned here */
int *fd;					/* is input file, becomes pipe */
char *program;					/* program name (full pathname) */
char *argv[];					/* arguments (ends in NULL) */

{
	int i;
	int pipe1[2];
	pid_t fpid;
	char *p;

	/* get program basename */

	if ((p = strrchr(program,'/')) != NULL)
	    argv[0] = p + 1;
	else
	    argv[0] = program;

	/* setup the pipe we need */

	if (pipe(pipe1) != 0) {
	    logmesg("cannot setup pipe");
	    return FAIL;
	}

	/* start the GS process */

	switch (fpid = fork())
	{
	case -1:				/* fork() failure */
	    close(pipe1[0]);
	    close(pipe1[1]);
	    logmesg("cannot fork");
	    return FAIL;

	case 0:					/* child process */
	    dup2(*fd,STDIN);			/* input file */
	    dup2(pipe1[1],STDOUT);
	    close(STDERR);
	    open("/dev/null",O_RDWR);		/* STDERR to /dev/null */

	    for (i = 3; i < 20; i++)
		close(i);			/* close remaining files */
	    while (close(i) == 0)
		i++;

#ifdef	DEBUG
	    if (Dfp != NULL) {
		fclose(Dfp);
		Dfp = NULL;
	    }
#endif	/* DEBUG */

	    for (i = 0; i < NSIG; i++)		/* set all signals to SIG_DFL */
		signal(i, SIG_DFL);

	    setuid(UuUid);			/* decrease risk */
	    setgid(UuGid);
	    nice(5);				/* decrease priority */
	    execv(program,argv);
	    sprintf(MsgBuf,"cannot execute %s",program);
	    logmesg(MsgBuf);
	    exit(1);

	default:				/* parent process */
	    close(pipe1[1]);
	    close(*fd);
	    *fd = pipe1[0];
	    *pid = fpid;
	    break;
	}

	return SUCCESS;
}

/*
**	Fputs() - does fputs() with '\' and '@' expansion
**
**	Returns EOF if an error occurs.
*/

int
Fputs(s,fd)
register const char *s;
int fd;
{
	char c, n, buf[32];
	time_t tim;

	while ((c = *s++) != '\0')
	{
	    if ((c == '@') && (n = *s++))
	    {
		switch (n)
		{
		case 'B':	/* speed (baud rate) */
			sprintf(buf, "%d", Speed);
			if (Fputs(buf,fd) == EOF)
				return EOF;
			break;
		case 'D':	/* date */
			tim = time(NULL);
			strftime(buf,sizeof(buf),"%d %B %Y",localtime(&tim));
			if (Fputs(buf,fd) == EOF)
				return EOF;
			break;
		case 'L':	/* line */
			if (Device && Fputs(Device,fd) == EOF)
				return EOF;
			break;
		case 'S':	/* system node name */
			if (SysName && Fputs(SysName,fd) == EOF)
				return EOF;
			break;
		case 'T':	/* time */
			tim = time(NULL);
			strftime(buf,sizeof(buf),"%X",localtime(&tim));
			if (Fputs(buf,fd) == EOF)
				return EOF;
			break;
		case 'U':	/* number of active users */
			sprintf(buf, "%d", Nusers);
			if (Fputs(buf,fd) == EOF)
				return EOF;
			break;
		case 'V':	/* version */
			if (Version && Fputs(Version,fd) == EOF)
				return EOF;
			break;
		case '@':	/* in case '@@' was used */
			if (write(fd, &n, 1) != 1)
				return EOF;
			break;
		}
	    } else {
		    if (c == '\\')
			    s = unquote(s, &c);
		    /* we're in raw mode: send CR before every LF
		     */
		    if (c == '\n' && write(fd, "\r", 1) != 1)
			    return EOF;
		    if (c && write(fd, &c, 1) != 1)
			    return EOF;
	    }
	}
	return SUCCESS;
}


/*
**	getuname() - retrieve the system's node name
**
**	Returns pointer to name or a zero-length string if not found.
*/

char *
getuname()
{
#ifdef	DOUNAME
	struct utsname uts;
#endif	/* DOUNAME */
	static char name[80];

	name[0] = '\0';

#ifdef	DOUNAME				/* dig it out of the kernel */
	if (uname(&uts) != FAIL)
		strcpy(name, uts.nodename);
#endif	/* DOUNAME */

#ifdef	PHOSTNAME			/* get it from the shell */

	if (strlen(name) == 0) {
		FILE *cmd;
		if ((cmd = popen(PHOSTNAME, "r")) != NULL) {
			fgets(name, sizeof(name), cmd);
			pclose(cmd);
			name[strlen(name)-1] = '\0';
		}
	}

#endif	/* PHOSTNAME */

	return name;
}


/*
**	filespresent() - checks if files are present in a given dir
*/
int
filespresent(dirname)
const char *dirname;
{
	DIR *dirp;
	struct dirent *de;
	int found = 0;

	if ((dirp = opendir(dirname)) == NULL)
	    return 0;			/* no dir, so no files */

	while ((de = readdir(dirp)) != 0)
	    if (de->d_name[0] != '.')
	    {
		found++;		/* found a non-hidden file! */
		break;			/* might as well quit */
	    }

	closedir(dirp);
	return found;
}

/*
**	rip() - remove trailing CR and spaces from a string
**
**	returns length of remaining string
*/
int
rip(string)
char *string;
{
	char *p;

	if ((p = strchr(string,'\r')) != NULL)
	    *p = '\0';
	else
	    p = string + strlen(string);

	while (--p >= string && *p == ' ')
	    *p = '\0';

	return (int)(++p - string);
}

/*
**	settermio() - setup tty termio values (depending on state)
*/

void
settermio(state)
int state;
{
#ifdef __linux__
	int divisor;
#endif

	switch (state) {
	case INITIALIZE:			/* initialize everything */
		ioctl(STDIN, TCGETS, &Termio);
#ifdef __linux__
		if (ioctl(STDIN, TIOCGSERIAL, &Serial) < 0)
		    Serial.type = 0;
		else
		    Serial.flags &= ~ASYNC_SPD_MASK;
#endif

		switch (Speed)
		{
		case 9600:
			Cbaud = B9600;
			break;
		case 19200:
			Cbaud = B19200;
			break;
		case 38400:
			Cbaud = B38400;
			break;
#if defined(__linux__) && defined(USE_SPD_HI)
		case 57600:
			if (Serial.type != 0) {
			    Serial.flags |= ASYNC_SPD_HI;
			    Cbaud = B38400;
			    break;
			}
		case 115200:
			if (Serial.type != 0) {
			    Serial.flags |= ASYNC_SPD_VHI;
			    Cbaud = B38400;
			    break;
			}
#else
# ifdef B57600
		case 57600:
			Cbaud = B57600;
			break;
# endif
# ifdef B76800
		case 76800:
			Cbaud = B76800;
			break;
# endif
# ifdef B115200
		case 115200:
			Cbaud = B115200;
			break;
# endif
#endif
		default:
#ifdef __linux__
			if (Serial.type != 0) {
			    divisor = Serial.baud_base / Speed;
			    if ((divisor * Speed) == Serial.baud_base) {
				Serial.flags |= ASYNC_SPD_CUST;
				Serial.custom_divisor = divisor;
				Cbaud = B38400;
				break;
			    }
			}
#endif
			sprintf(MsgBuf, "Invalid speed (%d)",Speed);
			logmesg(MsgBuf);
			Cbaud = B9600;
			break;
		}

#ifdef __linux__
		if (Serial.type != 0)
		    ioctl(STDIN,TIOCSSERIAL,&Serial);
#endif

		/* initial settings - reasonable to talk to the MODEM
		 */
		Termio.c_iflag = BRKINT | INPCK;
		Termio.c_oflag = 0;
		Termio.c_cflag = Cbaud | CS8 | CREAD | HUPCL | CLOCAL | CRTSCTS;
		Termio.c_lflag = 0;
#ifdef N_TTY
		Termio.c_line = N_TTY;
#endif

		/* set c_cc[] chars to reasonable values
		 */
		Termio.c_cc[VINTR] = CINTR;
		Termio.c_cc[VQUIT] = CQUIT;
		Termio.c_cc[VERASE] = CERASE;
		Termio.c_cc[VKILL] = CKILL;
		Termio.c_cc[VEOF] = CEOF;
		Termio.c_cc[VSTART] = CSTART;
		Termio.c_cc[VSTOP] = CSTOP;
		Termio.c_cc[VSUSP] = CSUSP;
		Termio.c_cc[VREPRINT] = CRPRNT;
		Termio.c_cc[VWERASE] = CWERASE;
		Termio.c_cc[VLNEXT] = CLNEXT;

		Termio.c_cc[VTIME] = 0;		/* block read for 1 char */
		Termio.c_cc[VMIN] = 1;

		ioctl(STDIN, TCSETSF, &Termio);
		return;

	case INITIAL:				/* back to initial state */
		Termio.c_iflag = BRKINT | INPCK;
		Termio.c_oflag = 0;
		Termio.c_cflag = Cbaud | CS8 | CREAD | HUPCL | CLOCAL | CRTSCTS;
		Termio.c_lflag = 0;

		Termio.c_cc[VTIME] = 0;
		Termio.c_cc[VMIN] = 1;
		break;

	case DROPDTR:				/* lower DTR line */
		Termio.c_cflag &= ~CBAUD;
#ifdef CBAUDEX
		Termio.c_cflag &= ~CBAUDEX;
#endif
		break;

	case SHORTTIME:				/* read() returns quickly */
		if (Termio.c_cc[VTIME] == 2 && Termio.c_cc[VMIN] == 0)
		    return;			/* already set! */

		Termio.c_cc[VTIME] = 2;		/* 200ms */
		Termio.c_cc[VMIN] = 0;
		break;

	case SHORT255:				/* read() returns quickly */
		Termio.c_cc[VTIME] = 2;		/* 200ms */
		Termio.c_cc[VMIN] = 255;	/* prefer to get some chars */
		break;

	case MEDMTIME:				/* read() returns medium */
		if (Termio.c_cc[VTIME] == (Is_ZyXEL? 10 : 25) &&
		    Termio.c_cc[VMIN] == 0)
		    return;			/* already set! */

		Termio.c_cc[VTIME] = Is_ZyXEL? 10 : 25; /* 1 or 2.5 second */
		Termio.c_cc[VMIN] = 0;
		break;

	case LONGTIME:				/* read() returns slowly */
		if (Termio.c_cc[VTIME] == 150 && Termio.c_cc[VMIN] == 0)
		    return;			/* already set! */

		Termio.c_cc[VTIME] = 150;	/* 15 seconds */
		Termio.c_cc[VMIN] = 0;
		break;

	case XONXOFF:				/* XON/XOFF handshake */
		Termio.c_iflag |= IXON | IXOFF;
		Termio.c_cflag &= ~CRTSCTS;
	case IMMEDIATE:				/* immediate return on read */
		Termio.c_cc[VTIME] = 0;
		Termio.c_cc[VMIN] = 0;
		break;

	case WATCHDCD:				/* SIGHUP when DCD lost */
		Termio.c_cflag &= ~CLOCAL;
		break;

	case SEVEN_E:				/* 7-bit even parity */
		Termio.c_iflag |= ISTRIP;
		Termio.c_cflag &= ~CS8;
		Termio.c_cflag |= PARENB | CS7;
		break;

	case FINAL:				/* after login */
		Termio.c_iflag &= ~(BRKINT | INPCK);
		if (FinalXoff)
		    Termio.c_iflag |= IXON | IXANY;
		Termio.c_oflag |= OPOST | TAB3;
		Termio.c_cflag &= ~CLOCAL;
		Termio.c_lflag |= ISIG | ICANON | ECHO | ECHOE | ECHOK;
		break;
	}

	ioctl(STDIN, TCSETSW, &Termio);		/* wait until sent, then set */
}


#ifdef TIOCMGET
/*
**	modemstat() - return status of MODEM handshake lines
*/

int
modemstat()

{
	int mstat;

	if (ioctl(0, TIOCMGET, &mstat) < 0)	/* read MODEM line status */
	    mstat = -1;				/* failed, assume not supported */

	debug3(D_CHAT, "modemstat(): %04o\n", mstat);
	return mstat;
}
#endif


/*
**	huphandler() - handles SIGHUP when enabled
*/

sig_t
huphandler(sig)
int sig;
{
	debug2(D_RUN, "SIGHUP caught\n");
	logmesg("CARRIER lost");
	send("ATZ\r");
	settermio(DROPDTR);
	closeline();
	exit(EXIT_FAILURE);
}

/*
**	inthandler() - handles SIGINT when not trapped elswhere
*/

sig_t
inthandler(sig)
int sig;
{
	debug2(D_RUN, "SIGINT caught\n");
	logmesg("BREAK received");
	send("ATZ\r");
	settermio(DROPDTR);
	closeline();
	exit(EXIT_FAILURE);
}

/*
**	sighandler() - handles misc signals when required
*/

sig_t
sighandler(sig)
int sig;
{
	if (sig >= NSIG)
	    sig = 0;

	Signal |= 1 << sig;		/* save the signal number as a flag */
}

#ifdef SA_SIGINFO
/*
**	sigcatcher() - catch all signals, log info (when possible)
*/

sig_t
sigcatcher(sig,si)
int sig;
siginfo_t *si;
{
	int fd,save;
	time_t tim;
	char buf[80];

	signal(sig, SIG_DFL);		/* reset to default handling */

	sprintf(buf, LOGFILE, MyName);	/* make logfile name */

	if ((fd = open(buf, O_WRONLY | O_APPEND)) >= 0) /* open logfile */
	{
	    save = dup(STDERR);		/* put it on descriptor STDERR */
	    dup2(fd,STDERR);
	    close(fd);

	    tim = time(NULL);
	    sprintf(buf, "%.24s %s: Unexpected signal", ctime(&tim), Device);

	    if (si != NULL)
		psiginfo(si,buf);	/* signal info to stderr (sigh) */
	    else
		psignal(sig,buf);	/* same without detail (sigh sigh) */

	    close(STDERR);		/* restore original STDERR */
	    dup2(save,STDERR);
	    close(save);
	}

	raise(sig);			/* deliver it again to really handle it */
}

/*
 *	catchsig() - direct a signal to the above catcher
 */

void
catchsig(sig)
int sig;

{
	struct sigaction sa;

	memset(&sa,0,sizeof(sa));
	sa.sa_handler = sigcatcher;
	sa.sa_flags = SA_SIGINFO;
	sigaction(sig,&sa,NULL);
}
#endif


/*************************************************************************
**	Lockfile handling for sharing of single port with UUCP		**
*************************************************************************/

/*
**	makelock() - attempt to create a lockfile
**
**	Returns FAIL if lock could not be made (line in use).
*/

int
makelock(name)
char *name;
{
	int fd, pid;
	char *temp, buf[MAXLINE+1];
#ifdef	ASCIIPID
	char apid[16];
#endif	/* ASCIIPID */

	debug3(D_LOCK, "makelock(%s) called\n", name);

	/* first make a temp file
	 */
	strcpy(buf, Lock);
	if ((temp = strrchr(buf, '/')) != NULL)
		strcpy(temp + 1, "TM.XXXXXX");

	if ((fd = creat((temp = mktemp(buf)), 0444)) == FAIL) {
		sprintf(MsgBuf, "cannot create tempfile (%s)", temp);
		logmesg(MsgBuf);
		return FAIL;
	}
	debug3(D_LOCK, "temp = (%s)\n", temp);

	/* owner uucp, to prevent possible trouble when the program
	 * crashes without removing the lockfile
	 */
	chown(temp, UuUid, UuGid);

	/* put my pid in it
	 */
#ifdef	ASCIIPID
	sprintf(apid, "%10d\n", getpid());
	write(fd, apid, strlen(apid));
#else
	pid = getpid();
	write(fd, (char *)&pid, sizeof(pid));
#endif	/* ASCIIPID */
	close(fd);

	/* link it to the lock file
	 */
#ifdef DEBUG
	errno = 0;
#endif
	while (link(temp, name) == FAIL) {
		debug3(D_LOCK, "link(temp,name) failed, errno=%d\n", errno);
		if (errno == EEXIST) {		/* lock file already there */
			if ((pid = readlock(name)) == FAIL)
				continue;
			if (pid != 0 && kill(pid, 0) == FAIL && errno == ESRCH) {
				/* pid that created lockfile is gone */
				if (unlink(name) == 0)
					continue;
			}
		}
		debug2(D_LOCK, "lock NOT made\n");
		unlink(temp);
		return FAIL;
	}
	debug2(D_LOCK, "lock made\n");
	unlink(temp);
	return SUCCESS;
}

/*
**	checklock() - test for presense of valid lock file
**
**	Returns TRUE if lockfile found, FALSE if not.
*/

boolean
checklock(name)
char *name;
{
	int pid;
	struct stat st;

	debug3(D_LOCK, "checklock(%s) called\n", name);

	if ((stat(name, &st) == FAIL) && errno == ENOENT) {
		debug2(D_LOCK, "stat failed, no file\n");
		return FALSE;
	}

	if ((pid = readlock(name)) == FAIL) {
		debug2(D_LOCK, "couldn't read lockfile\n");
		return FALSE;
	}

	if (pid != 0 && kill(pid, 0) == FAIL && errno == ESRCH) {
		debug2(D_LOCK, "no active process has lock, will remove\n");
		unlink(name);
		return FALSE;
	}

	debug2(D_LOCK, "active process has lock, return TRUE\n");
	return TRUE;
}

/*
**	readlock() - read contents of lockfile
**
**	Returns pid read or FAIL on error.
*/

int
readlock(name)
char *name;
{
	int fd, n;
	int pid = 0;
#ifdef	ASCIIPID
	char apid[16];
#endif	/* ASCIIPID */

	if ((fd = open(name, O_RDONLY)) == FAIL)
		return FAIL;

#ifdef	ASCIIPID
	n = read(fd, apid, sizeof(apid) - 1);
	close(fd);

	if (n > 0) {			/* not empty file */
	    apid[n] = '\0';

# ifdef BOTHPID
	    if (n == sizeof(pid))
		memcpy(&pid,apid,sizeof(pid));	/* binary pid */
	    else
# endif
	    if (sscanf(apid,"%d",&pid) != 1) /* ascii pid */
		pid = 0;
	}
#else
	n = read(fd, (char *)&pid, sizeof(pid));
	close(fd);

	if (n == 0)
		pid = 0;		/* empty file */
#endif	/* ASCIIPID */

	debug3(D_LOCK, "read %d from the lockfile\n", pid);
	return pid;
}

/*
**	rmlocks() - remove lockfile(s), if they are mine
*/

void
rmlocks()
{
	if (Lock[0] != '\0' && readlock(Lock) == getpid())
	    unlink(Lock);
}



/*************************************************************************
**	Chatting functions using UUCP-like chat scripts			**
*************************************************************************/

/*
**	chat() - handle expect/send sequence to Device
**
**	Returns FAIL if an error occurs.
*/

int
chat(s)
char *s;
{
	register int state = EXPECT;
	boolean finished = FALSE, if_fail = FALSE;
	char c, *p;
	char word[MAXLINE+1];		/* buffer for next word */

	debug3(D_CHAT, "chat(%s) called\n", s);

	while (!finished) {
		p = word;
		while (((c = (*s++ & 0177)) != '\0') && c != ' ' && c != '-')
			/*
			 *  SMR - I don't understand this, because if c is \0
			 *  then it is 0, isn't it?  If so we end the loop and
			 *  terminate the word anyway.
			 *
			*p++ = (c) ? c : '\177';
			 */
			*p++ = c;

		finished = (c == '\0');
		if_fail = (c == '-');
		*p = '\0';

		switch (state) {
		case EXPECT:
			if (expect(word) == FAIL) {
				if (if_fail == FALSE)
					return FAIL;	/* no if-fail seq */
			} else {
				/* eat up rest of current sequence
				 */
				if (if_fail == TRUE) {
					while ((c = (*s++ & 0177)) != '\0' &&
						c != ' ')
						;
					if (c == '\0')
						finished = TRUE;
					if_fail = FALSE;
				}
			}
			state = SEND;
			break;
		case SEND:
			if (send(word) == FAIL)
				return FAIL;
			state = EXPECT;
			break;
		}
	}
	debug2(D_CHAT, "chat() successful\n");
	return (SUCCESS);
}


/*
**	unquote() - decode char(s) after a '\' is found.
**
**	Returns the pointer s; decoded char in *c.
*/

const char	valid_oct[] = "01234567";
const char	valid_dec[] = "0123456789";
const char	valid_hex[] = "0123456789aAbBcCdDeEfF";

const char *
unquote(s, c)
const char *s;
char *c;
{
	int value, base;
	char n;
	const char *valid;

	n = *s++;
	switch (n) {
	case 'b':
		*c = '\b';	break;
	case 'c':
		if ((n = *s++) == '\n')
			*c = '\0';
		else
			*c = n;
		break;
	case 'f':
		*c = '\f';	break;
	case 'n':
		*c = '\n';	break;
	case 'r':
		*c = '\r';	break;
	case 's':
		*c = ' ';	break;
	case 't':
		*c = '\t';	break;
	case '\n':
		*c = '\0';	break;	/* ignore NL which follows a '\' */
	case '\\':
		*c = '\\';	break;	/* '\\' will give a single '\' */
	default:
		if (isdigit(n)) {
			value = 0;
			if (n == '0') {
				if (*s == 'x') {
					valid = valid_hex;
					base = 16;
					s++;
				} else {
					valid = valid_oct;
					base = 8;
				}
			} else {
				valid = valid_dec;
				base = 10;
				s--;
			}
			while (strpbrk(s, valid) == s) {
				value = (value * base) + (int) (*s - '0');
				s++;
			}
			*c = (char) (value & 0377);
		} else {
			*c = n;
		}
		break;
	}
	return s;
}


/*
**	send() - send a string to stdout
*/

int
send(s)
register const char *s;
{
	register int retval = SUCCESS;
	char ch;
	int inbuf;
	long diff;
	struct timeval now;
	char buf[MAXBUF];

	debug2(D_CHAT, "   SEND: ");

	/* before we send anything, make sure at least 20ms has passed */
	/* since the last matched input.  this is required because the */
	/* modem is not ready immediately after sending the response, */
	/* but requires slightly more delay....	 bummer! */

	gettimeofday(&now,NULL);

	if ((diff = now.tv_sec - MatchTime.tv_sec) <= 1) {
	    diff *= 1000000;			/* make microseconds */
	    diff += now.tv_usec - MatchTime.tv_usec;

	    if ((diff = 20000 - diff) > 0) {	/* calculate diff to 20ms */
		usleep(diff);			/* sleep the remainder */
		debug3(D_CHAT, "[%ldms] ", (diff+500)/1000);
	    }

	    MatchTime.tv_sec = 0;		/* don't sleep again */
	}

	debug2(D_CHAT, "(");

	if (!strcmp(s, "\"\"")) {	/* ("") used as a place holder */
		debug2(D_CHAT, "[nothing])\n");
		return retval;
	}

	inbuf = 0;

	while ((ch = *s++) != '\0')
	{
	    if (ch == '\\')
	    {
		switch (*s)
		{
		case 'p':		/* '\p' == pause */
			if (inbuf && write(STDOUT, buf, inbuf) != inbuf) {
				retval = FAIL;
				break;
			}
			inbuf = 0;
			debug2(D_CHAT, "[pause]");
			usleep(200000);
			s++;
			continue;
		case 'd':		/* '\d' == delay */
			if (inbuf && write(STDOUT, buf, inbuf) != inbuf) {
				retval = FAIL;
				break;
			}
			inbuf = 0;
			debug2(D_CHAT, "[delay]");
			usleep(1000000);
			s++;
			continue;
		case 'K':		/* '\K' == BREAK */
			if (inbuf && write(STDOUT, buf, inbuf) != inbuf) {
				retval = FAIL;
				break;
			}
			inbuf = 0;
			debug2(D_CHAT, "[break]");
			ioctl(STDOUT, TCSBRK, 0);
			s++;
			continue;
		default:
			s = unquote(s, &ch);
			break;
		}
	    }
	    debug3(D_CHAT, ((ch < ' ') ? "^%c" : "%c"),
			   ((ch < ' ') ? ch | 0100 : ch));
	    buf[inbuf++] = ch;
	    if (inbuf == MAXBUF) {
		if (write(STDOUT, buf, inbuf) != inbuf) {
		    retval = FAIL;
		    break;
		}
		inbuf = 0;
	    }
	}
	if (inbuf && write(STDOUT, buf, inbuf) != inbuf)
		retval = FAIL;
	debug3(D_CHAT, ") -- %s\n", (retval == SUCCESS) ? "OK" : "FAILED");
	return retval;
}


/*
**	expect() - look for a specific string on stdin
*/

static jmp_buf env;	/* here so expalarm() sees it */

int
expect(s)
register const char *s;
{
	register int i,ch;
	int expfail;			/* warning about these 3 variables... */
	int retval;			/* ... being clobbered are given ... */
	sig_t (*oldalarm)();		/* ... by some versions of GCC */
	char c, *p, word[MAXLINE+1];

	if (!strcmp(s, "\"\"")) {	/* ("") used as a place holder */
		debug2(D_CHAT, "   EXPECT: ([nothing])\n");
		gettimeofday(&MatchTime,NULL);	/* save current time */
		return SUCCESS;
	}

	expfail = EXPFAIL;		/* default fail timeout */

	/* look for escape chars in expected word
	 */
	for (p = word; (c = (*s++ & 0177)) != '\0';) {
		if (c == '\\') {
			if (*s == 'T') {	/* change expfail timeout */
				if (isdigit(*++s)) {
					s = unquote(s, &c);
					/* allow 3 - 255 second timeout */
					if ((expfail = ((unsigned char) c)) < 3)
						expfail = 3;
				}
				continue;
			} else
				s = unquote(s, &c);
		}
		*p++ = (c) ? c : '\177';
	}
	*p = '\0';

	settermio(INITIAL);
	oldalarm = signal(SIGALRM, expalarm);
	alarm((unsigned) expfail);

	if (setjmp(env)) {	/* expalarm returns non-zero here */
		debug3(D_CHAT, "[timed out after %d seconds]\n", expfail);
		signal(SIGALRM, oldalarm);
		gettimeofday(&MatchTime,NULL);	/* save current time */
		return FAIL;
	}

	debug3(D_CHAT, "   EXPECT: <%d> (", expfail);
	debug1(D_CHAT, word);
	debug2(D_CHAT, "), GOT: ");

	retval = FAIL;
	p = ChatBuf;

	while ((ch = getch()) != EOF) {
		debug3(D_CHAT, ((ch < ' ') ? "^%c" : "%c"),
			       ((ch < ' ') ? ch | 0100 : ch));
		*p++ = (char) ((int) ch & 0177);
		*p = '\0';
		if ((int)strlen(ChatBuf) >= (int)strlen(word)) {
			for (i=0; ChatBuf[i]; i++)
				if (expmatch(&ChatBuf[i], word)) {
					retval = SUCCESS;
					break;
				}
		}
		if (retval == SUCCESS)
			break;
	}
	alarm((unsigned) 0);
	signal(SIGALRM, oldalarm);
	debug3(D_CHAT, " -- %s\n", (retval == SUCCESS) ? "got it" : "Failed");
	gettimeofday(&MatchTime,NULL);		/* save current time */
	return retval;
}


/*
**	waitfor() - wait for a certain response (CR/LF terminated)
*/

int
waitfor(word, state)
const char *word;
int state;
{
	int retval = FAIL;
	int ch;
	int bufpos;
#ifdef DEBUG
	struct timeval t1;
	time_t elapsed = 0;
#endif

	settermio(state);			/* set requested handling */

	debug3(D_CHAT, "   WAITFOR: <%d> (", Termio.c_cc[VTIME]);
	debug1(D_CHAT, word);
	debug2(D_CHAT, "), GOT: ");

#ifdef DEBUG
	if (Debug & D_CHAT)
	    gettimeofday(&t1,NULL);		/* for timing */
#endif

	Match = ChatBuf;			/* position for match */
	bufpos = 0;

	/* discard any leading CR and LF characters (from previous line) */

	while ((ch = getch()) != EOF && (ch == '\r' || ch == '\n' || ch == '\0'))
	    debug3(D_CHAT, "^%c", ch | 0100);

	if (ch == EOF)
	    goto done;

	ChatBuf[bufpos++] = ch;			/* character from above */
	debug3(D_CHAT, ((ch < ' ') ? "^%c" : "%c"),
		       ((ch < ' ') ? ch | 0100 : ch));

	/* keep filling the chatbuf until the read times out */

	while (bufpos < (sizeof(ChatBuf) - 1) && (ch = getch()) != EOF)
	{
	    debug3(D_CHAT, ((ch < ' ') ? "^%c" : "%c"),
			   ((ch < ' ') ? ch | 0100 : ch));

	    if (ch == '\0' || (ChatBuf[bufpos++] = ch) != '\n')
		continue;			/* wait for linefeed */

	    /* we have received a linefeed, check for matches in buffer */

	    ChatBuf[bufpos] = '\0';		/* 0-terminate */

	    if (word == NULL || expmatch(Match, word)) {
		retval = SUCCESS;
		break;
	    }

	    Match = &ChatBuf[bufpos];		/* start looking here */
	}

done:
	ChatBuf[bufpos] = '\0';			/* 0-terminate */
	gettimeofday(&MatchTime,NULL);		/* save current time */

#ifdef DEBUG
	if (Debug & D_CHAT) {
	    elapsed = (MatchTime.tv_usec - t1.tv_usec) / 1000 +
			(MatchTime.tv_sec - t1.tv_sec) * 1000;
	}
	debug4(D_CHAT, " -- %s (%d ms)\n", (retval == SUCCESS) ? "got it" : "Failed", elapsed);
#endif
	return retval;
}


/*
**	expmatch() - compares expected string with the one gotten
*/

boolean
expmatch(got, exp)
register const char *got;
register const char *exp;
{
	if (*exp == '*') {			/* wildcard? */
	    exp++;

	    while (*got)			/* match at all positions */
		if (expmatch(got++,exp))
		    return TRUE;

	    return FALSE;
	} else {
	    while (*exp)
		if (*got++ != *exp++)
		    return FALSE;		/* no match */

	    return TRUE;
	}
}


/*
**	expalarm() - called when expect()'s SIGALRM goes off
*/

sig_t
expalarm(sig)
int sig;
{
	longjmp(env, sig);
}


/*************************************************************************
**	Defaults file functions						**
*************************************************************************/

/*
**	defbuild() - create in-core list of defaults
**
**	Returns (DEF**)NULL if no defaults file found or an error occurs.
*/

DEF **
defbuild(filename)
char *filename;
{
	register int i;
	register DEF *dp;
	register DEF *next;
	FILE *fp;
	char *fname, defname[MAXLINE+1], buf[MAXLINE+1];
	static DEF *deflist[MAXDEF+1];		/* in-core list */
	struct stat st;

	debug3(D_DEF, "defbuild(%s) called\n",
			((filename == NULL) ? "NULL" : filename));

	/* look to see if there's a DEFAULTS/MyName.Device file
	 */
	sprintf(buf, "%s", DEFAULTS);
	strcat(buf, ".%s");
	sprintf(defname, buf, MyName, Device);
	debug3(D_DEF, "looking for %s\n", defname);
	if ((stat(defname, &st) == FAIL) && errno == ENOENT) {	/* Nope */
		debug2(D_DEF, "stat failed, no file\n");
		sprintf(defname, DEFAULTS, MyName);
	}

	fname = (filename != NULL) ? filename : defname;

	/* if fname doesn't begin with a '/', assume it's a
	 * filename to be made "DEFAULTS/fname"
	 */
	if (*fname != '/') {
		sprintf(defname, DEFAULTS, fname);
		fname = defname;
	}

	debug3(D_DEF, "fname = (%s)\n", fname);

	if ((fp = defopen(fname)) == NULL) {
		debug2(D_DEF, "defopen() failed\n");
		return NULL;		/* couldn't open file */
	}

	for (i=0; i < MAXDEF; i++) {
		if ((dp = defread(fp)) == NULL)
			break;
		if ((next = (DEF *) malloc((unsigned) sizeof(DEF))) == NULL) {
			logmesg("malloc() failed: defaults list truncated");
			break;
		}
		next->name = dp->name;
		next->value = dp->value;
		deflist[i] = next;
		debug5(D_DEF, "deflist[%d]: name=(%s), value=(%s)\n",
				i, deflist[i]->name, deflist[i]->value);
	}
	deflist[i] = NULL;	/* terminate list */
	defclose(fp);
	debug2(D_DEF, "defbuild() successful\n");
	return deflist;
}


/*
**	defvalue() - locate the value in "deflist" that matches "name"
**
**	Returns (char*)NULL if no match is made.
*/

char *
defvalue(deflist, name)
register DEF **deflist;
register const char *name;
{
	debug3(D_DEF, "defvalue(%s) called\n", name);

	if (deflist != NULL)
		for (; *deflist != NULL; deflist++)
			if (!strcmp(name, (*deflist)->name)) {
				debug3(D_DEF, "defvalue returns (%s)\n",
						(*deflist)->value);
				return (*deflist)->value;  /* normal exit */
			}

	debug2(D_DEF, "defvalue returns NULL\n");
	return NULL;
}


/*
**	defopen() - open the defaults file
**
**	Returns (FILE*)NULL if file not found or an error occurs.
*/

FILE *
defopen(filename)
register char *filename;
{
	if (filename != NULL)
		return fopen(filename, "r");

	return NULL;
}


/*
**	defread() - read a line from the defaults file
**
**	Returns (DEF*)NULL if an error occurs.
*/

DEF *
defread(fp)
register FILE *fp;
{
	register char *p;
	char buf[MAXLINE+1];	/* buffer large enough for 1 line */
	static DEF def;

	do {
		if (fgets(buf, sizeof(buf), fp) == NULL)
			return NULL;	/* no more lines */

	} while ((buf[0] == '#') || (buf[0] == '\n'));
	  /* SMR - ignore comment lines */

	buf[strlen(buf)-1] = '\0';		/* rm trailing \n */

	/* lines should be in the form "NAME=value"
	 */
	if ((p = strchr(buf, '=')) == NULL) {
		sprintf(MsgBuf, "bad defaults line: %s", buf);
		logmesg(MsgBuf);
		return NULL;
	}
	*p++ = '\0';		/* split into two fields, name and value */
	def.name = strdup(buf);
	def.value = strdup(p);

	return &def;
}


/*
**	defclose() - closes the defaults file
**
**	Returns EOF if an error occurs.
*/

int
defclose(fp)
register FILE *fp;
{
	return fclose(fp);
}


/*************************************************************************
**	Error and Event logging						**
*************************************************************************/

/*
**	logmesg() - display/log a message
*/

void
logmesg(msg)
register const char *msg;
{
	register FILE *fp;
	time_t tim;
	char logfile[80];

	sprintf(logfile, LOGFILE, MyName);	/* make logfile name */

	if ((fp = fopen(logfile, "a")) != NULL ||
	    (fp = fopen(logfile, "w")) != NULL) /* REJ: Linux can't append to /dev/console */
	{
		tim = time(NULL);
		fprintf(fp, "%.24s %s: %s\n", ctime(&tim), Device, msg);
		fclose(fp);
	}

#ifdef	TRYMAIL
	else {
		char buf[MAXLINE];

		sprintf(buf, "%s %s", MAILER, NOTIFY);
		if ((fp = popen(buf, "w")) != NULL) {
			fprintf(fp, "From: %s %s\n", MyName, Device);
			fprintf(fp, "To: %s\n", NOTIFY);
			fprintf(fp, "Subject: %s problem\n\n", MyName);
			fprintf(fp, "%s: %s\n", Device, msg);
			pclose(fp);
		}
	}
#endif	/* TRYMAIL */
}


/*************************************************************************
**	Debugging functions						**
*************************************************************************/

#ifdef	DEBUG

/*
**	debug() - an fprintf to the debug file
**
**	Only does the output if the requested level is "set."
*/

/*VARARGS2*/
void
debug(int lvl, const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	if ((Dfp != NULL) && (Debug & lvl)) {
		vfprintf(Dfp, fmt, args);
		fflush(Dfp);
	}
	va_end(args);
}

/*
**	dprint() - like debug(), but shows control chars
*/

void
dprint(lvl, word)
int lvl;
const char *word;
{
	const char *p, *fmt;
	char ch;

	if (Debug & lvl) {
		if (word == NULL)
		    fprintf(Dfp,"(null)");
		else {
		    p = word;
		    while ((ch = *p++) != '\0') {
			if (ch < ' ') {
				fmt = "^%c";
				ch = ch | 0100;
			} else {
				fmt = "%c";
			}
			fprintf(Dfp, fmt, ch);
		    }
		}
		fflush(Dfp);
	}
}

#endif	/* DEBUG */
