/* $Header: /home/klaus/mgetty/voice/RCS/vmodem.c,v 1.23 1994/10/28 21:02:21 klaus Exp $
 *
 * voice modem specific functions
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#ifndef _NOSTDLIB_H
#include <stdlib.h>
#endif
#include <fcntl.h>

#include "mgetty.h"
#include "policy.h"
#include "voclib.h"
#include "tio.h"

#define FRACT(x,m) ((int)((x)*(m)+0.5))

int voice_modem;

static int vio_tab[][3]={{2,8,16},{2,8,16},{0,1,2}};
static char* vm_header[] = {"ZyXEL", "Dolph", "RockJ"};

int mg_init_voice _P1( (fd), int fd )
{
    lprintf( L_MESG, "voice code version %s", voice_rel);

    /* find out what kind of voice modem this is */
    voice_modem = VM_ZYXEL;
    
    if ( voice_command( "AT+FCLASS=8", "VCON|OK", fd ) != ERROR ) {
	/* ZyXEL or Dolphin */
	if( voice_command( "ATI", "1496|OK", fd ) == 0 ) {
	    voice_command( "", "OK", fd );
	    voice_modem = VM_ZYXEL;
	} else {
	    voice_modem = VM_DOLPHIN;
	}
    } else if ( voice_command( "AT#CLS=8", "VCON|OK", fd) != ERROR ) {
	/* ROCKWELL */
	voice_modem = VM_ROCKWELL;
    } else {
	lprintf( L_MESG, "voice init failed, using data/fax mode");
	return FAIL;
    }

    switch(voice_modem) {
      case VM_ZYXEL:
	lprintf(L_NOISE, "modem type ZyXEL");
#ifdef DIST_RING
	if (ZYXEL_ROM >= 611) {
	    if ( voice_command( DIST_RING_INIT,
			       "VCON|OK", fd ) == ERROR ) {
		lprintf( L_WARN,
			"coudn't initialize distinctive RING" );
	    }
	}
#endif
	break;
      case VM_DOLPHIN:
	lprintf(L_NOISE, "modem type DOLPHIN");
	break;
      case VM_ROCKWELL:
	lprintf(L_NOISE, "modem type ROCKWELL");
	break;
    }
    return SUCCESS;
}

void
voice_mode_init _P1((fd), int fd)
{
    char cmd[256];
    
    switch(voice_modem) {
      case VM_ZYXEL:
	if (ZYXEL_ROM >= 611) {
	    if ( voice_command( "ATS39.7=0 S39.6=1 +VIT=60",
			       "VCON|OK", fd ) == ERROR ) {
		lprintf( L_WARN,
			"voice init failed, continuing" );
	    }
	}
	if (ZYXEL_ROM >= 612) {
	    sprintf( cmd, "AT+VDH=%d +VDD=%d",
		     FRACT(VOICE_DTMF_THRESHOLD, 31),
		     (int)((VOICE_DTMF_LEN + 3)/6));
	    if ( voice_command( cmd, "VCON|OK", fd )==ERROR ) {
		lprintf( L_WARN, "setting DTMF threshold didn't work" );
	    }
	}
	if (ZYXEL_ROM >= 613) {
	    if ( voice_command( "AT+FLO=2", "VCON|OK", fd ) == ERROR ) {
		lprintf( L_WARN, "can't turn on hardware flow control" );
	    }
	}	    
	break;
      case VM_DOLPHIN:
	/* should anything be done here? */
	break;
      case VM_ROCKWELL:
	/* turn off autobauding, this also turns on the
	 * inactivity timer.
	 * Use #BDR=x to set the bps rate to x * 2400 bps.
	 * #BDR=16 means use 38400 bps, as far as I know other
	 * bit rates don't work anyway...
	 * Also set deadman timer to 60 seconds.
	 */

	if (voice_command( "ATS30=60 #BDR=16", "VCON|OK", fd ) == ERROR ) {
	    lprintf( L_WARN, "can't set baud rate");
	}
	break;
    }
}

void
voice_mode_off _P1((fd), int fd)
{
    switch(voice_modem) {
      case VM_ZYXEL:
      case VM_DOLPHIN:
	voice_command( "AT+VLS=0 +FCLASS=0", "OK", fd );
	break;
      case VM_ROCKWELL:
	voice_command( "AT#CLS=0", "OK", fd );
	break;
    }
}

boolean
voice_format_supported _P1((compr), int compr)
{
    boolean known=FALSE;
    
    switch(voice_modem) {
      case VM_ZYXEL:
	if(compr!=1) known=TRUE;
	break;
      case VM_DOLPHIN:
      case VM_ROCKWELL:
	break;
    }
    return known;
}

int
voice_data_rate _P1((compr), int compr)
{
    int rate=0;
    
    switch(voice_modem) {
      case VM_ZYXEL:
	rate = 1200 * compr;
	break;
      case VM_DOLPHIN:
	rate = 1200 * compr; /* FIXME: is this right? */
	break;
      case VM_ROCKWELL:
	rate = 900 * compr;
	break;
    }

    return rate;
}

static void
voice_set_volume _P2((fd, speaker), int fd, boolean speaker)
{
    double vol;
    char cmd[256];
    
    vol = speaker ? SPEAKER_PLAY_VOLUME : SPEAKER_ANSWER_VOLUME;
    
    switch(voice_modem) {
      case VM_ZYXEL:
	sprintf( cmd, "ATL%d", FRACT(vol, 7) );
	break;
      case VM_DOLPHIN:
      case VM_ROCKWELL:
	sprintf( cmd, "ATL%d", FRACT(vol, 3) );
	break;
    }

    if ( voice_command( cmd, "VCON|OK", fd ) == ERROR )
    {
	lprintf( L_WARN, "ATL -> some error, continuing");
    }
}

void
enter_data_mode _P1((answer_mode), int answer_mode)
{
    answer_mode &= (ANSWER_FAX | ANSWER_DATA);
    if (answer_mode == 0) {
	lprintf(L_AUDIT, "fax and data both disabled, hanging up");
	exit(1);
    }
    
    switch (voice_modem) {
      case VM_ZYXEL:
      case VM_DOLPHIN:
	switch (answer_mode) {
	  case ANSWER_FAX | ANSWER_DATA:
	    /* let the modem decide if it's fax or data */
	    lprintf(L_NOISE, "Trying fax/data connection...");
	    voice_command("AT+FCLASS=2 +FAA=1", "OK", STDIN);
	    break;
	  case ANSWER_FAX:
	    /* answer in FAX mode only */
	    /* FIXME: class 2.0 ? */
	    voice_command("AT+FCLASS=2 +FAA=0", "OK", STDIN);
	    break;
	  case ANSWER_DATA:
	    /* answer in DATA mode only */
	    voice_command("AT+FCLASS=0 +FAA=0", "OK", STDIN);
	    break;
	}
	break;
      case VM_ROCKWELL:
	switch (answer_mode) {
	  case ANSWER_FAX | ANSWER_DATA:
	    /* let the modem decide if it's fax or data */
	    lprintf(L_NOISE, "Trying fax/data connection...");
	    voice_command("AT#CLS=2 +FAA=1", "OK", STDIN);
	    break;
	  case ANSWER_FAX:
	    /* answer in FAX mode only */
	    /* FIXME: class 2.0 ? */
	    voice_command("AT#CLS=2 +FAA=0", "OK", STDIN);
	    break;
	  case ANSWER_DATA:
	    /* answer in DATA mode only */
	    voice_command("AT#CLS=0 +FAA=0", "OK", STDIN);
	    break;
	}
	break;
    }
}

int
voice_beep _P3((fd, voice_io, beep), int fd, int voice_io, char *beep )
{
    char cmd[256];

    voice_set_volume(fd, voice_io==VIO_SPEAKER);

    switch(voice_modem) {
      case VM_ZYXEL:
      case VM_DOLPHIN:
	sprintf( cmd, "AT+VLS=%d", vio_tab[voice_modem][voice_io] );
	break;
      case VM_ROCKWELL:
	sprintf( cmd, "AT#VLS=%d #VBT=12", vio_tab[voice_modem][voice_io] );
	break;
    }
	
    if ( voice_command( cmd, "VCON|OK", fd ) == ERROR )
    {
	lprintf( L_WARN, "init error, abort beep!" );
	return ERROR;
    }
    
    switch(voice_modem) {
      case VM_ZYXEL:
      case VM_DOLPHIN:
	sprintf( cmd, "AT+VTS=%s", beep );
	break;
      case VM_ROCKWELL:
	sprintf( cmd, "AT#VTS=%s", beep );
	break;
    }

    if ( voice_command( cmd , "VCON|OK", fd ) == ERROR )
    {
	lprintf( L_WARN, "beep failed, aborting" );
	return ERROR;
    }
    return 0;
}

int /* compression */
voice_send_init _P3((fd, vocfd, voice_io), int fd, int vocfd, int voice_io)
{
    char cmd[128], buf[16];
    TIO vtio;
    int flow;
    int r;
    int compr=ERROR;

    if (voice_modem == VM_ZYXEL && ZYXEL_ROM >= 613) {
	flow = FLOW_HARD;
    } else {
	flow = FLOW_XON_OUT;
    }

    /* read zfax header */
    r = read(vocfd, buf, 16);
    
    if (r >= 16 && strncmp(buf, vm_header[voice_modem], 5) == 0) {
	compr=buf[10] + 1;
	lprintf(L_MESG,	"reading zfax header, compression type %d", compr);
    } else {
	lprintf(L_WARN, "No ZFAX header found, aborting!");
	return ERROR;
    }
        
    tio_get(fd, &vtio);
    fcntl(fd, F_SETFL, O_RDWR);
    tio_mode_sane(&vtio, TRUE);
    tio_set_flow_control(fd, &vtio, FLOW_XON_OUT);
    tio_mode_raw(&vtio);
    tio_set(fd, &vtio);

    voice_mode_init(fd);
    voice_set_volume(fd, voice_io==VIO_SPEAKER);

    switch(voice_modem) {
      case VM_ZYXEL:
      case VM_DOLPHIN:
	sprintf( cmd, "AT+VSM=%d +VLS=%d",
		compr, vio_tab[voice_modem][voice_io] );
	break;
      case VM_ROCKWELL:
	sprintf( cmd, "AT#VBS=%d #VLS=%d",
		compr, vio_tab[voice_modem][voice_io] );
	break;
    }
	
    if ( voice_command( cmd, "VCON|OK", fd ) == ERROR )
    {
	lprintf( L_WARN, "init failed, abort voice send!");
	return ERROR;
    }

    switch(voice_modem) {
      case VM_ZYXEL:
      case VM_DOLPHIN:
	sprintf( cmd, "AT+VTX" );
	break;
      case VM_ROCKWELL:
	sprintf( cmd, "AT#VTX" );
	break;
    }
	
    if ( voice_command( cmd, "CONNECT", fd ) == ERROR )
    {
	lprintf( L_WARN, "voice send failed, aborting" );
	return ERROR;
    }
    return compr;
}
    
int
voice_record_init _P6((fd, compr, voice_io, silence, threshold, header),
		      int fd, int compr, int voice_io,
		      int silence, double threshold, char *header)
{
    char cmd[128];
    TIO vtio;
    int flow;
    
    /* Setup tty interface
     * Do not set c_cflag, assume that caller has set bit rate,
     * hardware handshake, ... properly
     */

    if (voice_modem == VM_ZYXEL && ZYXEL_ROM >= 613) {
	flow = FLOW_HARD;
    } else {
	flow = FLOW_XON_IN;
    }
    
    tio_get(fd, &vtio);
    fcntl(fd, F_SETFL, O_RDWR);
    tio_mode_sane(&vtio, TRUE);
    tio_set_flow_control(fd, &vtio, flow);
    tio_mode_raw(&vtio);
    tio_map_cr(&vtio, FALSE);
    tio_set(fd, &vtio);
    
    voice_mode_init(fd);
    voice_set_volume(fd, voice_io==VIO_SPEAKER);

    strcpy(header, vm_header[voice_modem]);
    header[5]='\002';
    switch(voice_modem) {
      case VM_ZYXEL:
	sprintf( cmd, "AT+VSM=%d +VLS=%d +VSD=%d,%d",
		 compr, vio_tab[voice_modem][voice_io],
		 FRACT(threshold, 31), silence );
	break;
      case VM_DOLPHIN:
	sprintf( cmd, "AT+VSM=%d +VLS=%d +VSD=%d,%d",
		 compr, vio_tab[voice_modem][voice_io],
		 FRACT(threshold, 31), silence );
	break;
      case VM_ROCKWELL:
	sprintf( cmd, "AT#VBS=%d #VSS=%d #VSD=1 #VSP=%d #VLS=%d",
		 compr, FRACT(threshold, 3), silence,
		 vio_tab[voice_modem][voice_io]);
	break;
    }

    if ( voice_command( cmd, "VCON|OK", fd ) == ERROR )
    {
	lprintf( L_WARN, "init failed, abort voice recording!" );
	return ERROR;
    }

    switch(voice_modem) {
      case VM_ZYXEL:
      case VM_DOLPHIN:
	sprintf( cmd, "AT+VRX" );
	break;
      case VM_ROCKWELL:
	sprintf( cmd, "AT#VRX" );
	break;
    }

    if ( voice_command( cmd, "CONNECT", fd ) == ERROR )
    {
	lprintf( L_WARN, "voice recording failed, aborting" );
	return ERROR;
    }
    return SUCCESS;
}
