/*
 * apcd.c - Daemon for the APC Smart UPS
 *
 * Copyright (c) 1995 Pavel Korensky
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL PAVEL KORENSKY BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF PAVEL
 * KORENSKY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * PAVEL KORENSKY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND PAVEL KORENSKY HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

/*	
 * Version:
 *	
 * $Id: apcd.c,v 1.5 1995/05/23 07:25:08 root Exp root $
 *	
 *	
 * History:
 *	
 * $Log: apcd.c,v $
 * Revision 1.5  1995/05/23  07:25:08  root
 * First public ALPHA version
 *
 * Revision 1.4  1995/05/23  01:07:40  root
 * Parameters are on the command line, instead of config.h file
 *
 * Revision 1.3  1995/05/23  00:25:43  root
 * System shutdown with UPS switch off was added
 *
 * Revision 1.2  1995/05/21  21:10:56  root
 * Some small fixes
 *
 * Revision 1.1  1995/05/21  20:15:13  root
 * Initial revision
 *
 *
 *
 *
 *	
 */	


#include "apcd.h"
#include "version.h"

static char *version="$Id: apcd.c,v 1.5 1995/05/23 07:25:08 root Exp root $";

UPSINFO		myUPS;
int		killme;
int		port;
char		use_port[100];
int		power_timer;
int		alarmup,alarmdown,pending,alarmcount,wasmsg;
struct termios	oldtio, newtio;


void main(int argc, char *argv[])
{
	char msg[100];

	if (argc != 3) {
		printf("usage: apcd port timeout\n");
		exit(1);
	}
	strcpy(use_port,argv[1]);
	power_timer=atoi(argv[2]);
	printf("APC SmartUPS daemon started on port %s with timeout %d mins.\n",use_port,power_timer); 
	
	killme = 0;
	/* Initialize system log */
	openlog("apcd", LOG_PID | LOG_NDELAY | LOG_PERROR, LOG_LOCAL2);
	
		
	/* Become daemon */
	start_daemon();
	signal_setup();	
	syslog(LOG_INFO,"Starting apcd version %s",VERSION);
	setup_tty();
	alarmup=0;
	alarmdown=0;
	pending=0;
	wasmsg=0;
	while(!killme) {
		fillUPS(port,&myUPS); 
		sleep(1); 
		if (pending) {
			alarmcount--;
			if (alarmcount == 0) {
				go_down();
				pending=0;
			}
			if(((alarmcount % 60) == 0) && (alarmcount != 0)) {
				sprintf(msg,"\n\nAPC Daemon: Power failure, system will go down in %d minutes.\n",alarmcount/60);
				mesall(msg);
				wasmsg=1;
			}	
		}
		if (alarmup) {
			if(!pending) {
				alarmcount=power_timer*60;
				pending=1;
			}
			alarmup=0;
		}
		if (alarmdown) {
			if(wasmsg) {
				sprintf(msg,"\n\nAPC Daemon: Power restored, shutdown cancelled\n");
				mesall(msg);
				wasmsg=0;
			}
			alarmcount=0;
			pending=0;
			alarmdown=0;
		}
		
	}
	if(killme == 1) {
		syslog(LOG_INFO,"Ending apcd version %s",VERSION);
		tcsetattr(port,TCSANOW,&oldtio);
		close(port);
		closelog();

	}
	if(killme == 2) {
		mesall("APC Daemon: SYSTEM IS GOING DOWN NOW !!!");
		do_shutdown();
	}
};


/* Setup of the communication port. Hope it will work
 */

void setup_tty()
{
	port=open(use_port,O_RDWR | O_NOCTTY);
	if (port < 0) {
		syslog(LOG_ERR,"Unable to open port %s",use_port);
		exit(-1);
	}
	tcgetattr(port,&oldtio); /* Save old settings */
	newtio.c_cflag = DEFAULT_SPEED | CS8 | CLOCAL | CREAD;
	newtio.c_iflag = IGNPAR; /* Ignore errors, raw input */
	newtio.c_oflag = 0; /* Raw output */
	newtio.c_lflag = 0; /* No local echo */	
	newtio.c_cc[VMIN] = 1;
	newtio.c_cc[VTIME] = 0;
	tcflush(port,TCIFLUSH);
	tcsetattr(port,TCSANOW,&newtio);
}


/* Become a daemon, release stdin, stdout etc.
 */

void start_daemon()
{
	int	pid;
	
	close(0);
	close(1);
	close(2);
	if ((pid=fork()) < 0) {
		syslog(LOG_ERR,"Unable to fork");
		exit(1);
	}
	if (pid != 0) exit(0);
};


/* Setup various signal handlers. Code here is adapted from diald program
 * which is (c) Eric Schenk.
 */

void signal_setup()
{
	sigset_t	sigmask;
	struct	sigaction sa;
	
	sigemptyset(&sigmask);
	sigaddset(&sigmask, SIGINT); /* Termination requested */
	sigaddset(&sigmask, SIGTERM); /* Termination requested */
	sigaddset(&sigmask, SIGUSR1); /* UPS on line */	
	sigaddset(&sigmask, SIGUSR2); /* UPS on the battery */
	sigaddset(&sigmask, SIGHUP); /* Dump the UPS stats to the file */
	
#define SIGNAL(s,handler) { \
		sa.sa_handler = handler; \
		if (sigaction(s, &sa, NULL) < 0) { \
			syslog(LOG_ERR, "sigaction(%d) failed ", s); \
			exit(1); \
		} \
	}

	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	SIGNAL(SIGINT,sig_intr);
	SIGNAL(SIGTERM,sig_term);
	SIGNAL(SIGUSR1,go_on_line);
	SIGNAL(SIGUSR2,go_on_batt);
	SIGNAL(SIGHUP,sig_hup);
}

void sig_intr(int sig)
{
	syslog(LOG_INFO,"SIGINTR Termination requested");
	killme = 1;
}	

void sig_term(int sig)
{
	syslog(LOG_INFO,"SIGTERM Termination requested");
	killme=1;
}	

void sig_hup(int sig)
{
}


void go_on_batt(int sig)
{
	syslog(LOG_INFO,"UPS is going on battery");
	alarmup=1;
}

void go_on_line(int sig)
{
	syslog(LOG_INFO,"UPS is going on-line");
	alarmdown=1;
}

void go_down()
{
	syslog(LOG_INFO,"System is going down - power failure");
	killme=2;
}

int getline(int fd, char *s)
{
	int i,ending;
	char c;
	
	i=0;
	ending=0;
	
	while (!ending) {
		read(fd,&c,1);
		switch(c) {
			case UPS_ON_BATT: raise(SIGUSR2);
				  	  break;
			case UPS_ON_LINE: raise(SIGUSR1);
					  break;
			case 	    '\n': ending=1;
					  break;
				 default: s[i++]=c;
				 	  break;
		}	
	}
	s[i]='\0';
	return(0);
}

int fillUPS (int fd,UPSINFO *ups)
{
	char	answer[MAXLINE];
	char	q;

	q='Y';
	write(fd,&q,1);
	getline(fd,answer);
	
	q=BATT_FULL;
	write(fd,&q,1);
	getline(fd,answer);
	ups->BatLoad=atof(answer);
	
	q=UPS_LINE_MIN;
	write(fd,&q,1);
	getline(fd,answer);
	ups->LineMin=atof(answer);
	
	q=UPS_LINE_MAX;
	write(fd,&q,1);
	getline(fd,answer);
	ups->LineMax=atof(answer);
	
	q=UPS_LOAD;
	write(fd,&q,1);
	getline(fd,answer);
	ups->UPSLoad=atof(answer);
	
	q=LINE_FREQ;
	write(fd,&q,1);
	getline(fd,answer);
	ups->LineFreq=atof(answer);
	
	q=LINE_VOLTAGE;
	write(fd,&q,1);
	getline(fd,answer);
	ups->LineVoltage=atof(answer);
	
	q=OUTPUT_VOLTAGE;
	write(fd,&q,1);
	getline(fd,answer);
	ups->OutputVoltage=atof(answer);
	
	q=UPS_TEMP;
	write(fd,&q,1);
	getline(fd,answer);
	ups->UPSTemp=atof(answer);
	
	q=BATT_VOLTAGE;
	write(fd,&q,1);
	getline(fd,answer);
	ups->BattVoltage=atof(answer);
	
	q=UPS_STATUS;
	write(fd,&q,1);
	getline(fd,answer);
	ups->Status=atoi(answer);
	
	return(0);
}		


/* mesusr() and mesall() function are actually parts of shutdown source
 * I am using them for sending messages before shutdown
 */

void mesusr(char *mess,struct utmp *ut)
{
	int fd;
	char term[40] = {'/','d','e','v','/',0};

	(void)strncat(term,ut->ut_line,sizeof(ut->ut_line));
	if((fd=open(term, O_RDWR | O_NONBLOCK)) < 0)
		return;
	write(fd,mess,strlen(mess));
	close(fd);
}

void mesall(char *mess)
{
	struct utmp *ut;
	utmpname(_PATH_UTMP);
	setutent(); 
	ut=getutent();
	while((ut = getutent())) {
		if(ut->ut_type == USER_PROCESS)
			mesusr(mess,ut);
	}
	endutent();
}


/*
 *
 * From here, there are parts of the source from shutdown.c which
 * is a part of linux-utils-2.1
 *
 */

void write_wtmp(), unmount_disks(), unmount_disks_ourselves();



void
do_shutdown()
{
	struct itimerval new,old;
	char a;
	
/*	setpriority(PRIO_PROCESS, 0, PRIO_MIN); */

	chdir("/");

	signal(SIGPIPE, SIG_IGN);
	signal(SIGINT,  SIG_IGN);

	/* do syslog message... */
	syslog(LOG_INFO, "System cleanup"); 
	closelog();
	sleep(1);
	kill(1, SIGTSTP);	/* tell init not to spawn more getty's */
	write_wtmp();
	sync();
	signal(SIGTERM, SIG_IGN);
	setpgrp();		/* so the shell wont kill us in the fall */
	/* a gentle kill of all other processes except init */
	kill(-1, SIGTERM);
	sleep(2);

	/* now use brute force... */
	kill(-1, SIGKILL);

	/* turn off accounting */
	acct(NULL);

	sync();
	sleep(2);
	/* unmount disks... */
	unmount_disks();
	sync();
	sleep(1);
	signal(SIGALRM,send_second_z); /* After SIGALRM send the second 'Z' */
	new.it_interval.tv_sec = 0;
	new.it_interval.tv_usec = 0;
	new.it_value.tv_sec = 1;
	new.it_value.tv_usec = 500000;
	a = 'Z';
	write(port,&a,1);	
	setitimer(ITIMER_REAL,&new,&old);	/* and start the 1500ms	interval */
	while(1);				/* endless loop till SIGALRM */
		
	/* NOTREACHED */
	exit(0); /* to quiet gcc */
}

void send_second_z(int sig)
{
	int a;
	
	a = 'Z';
	write(port,&a,1);	 
};


void
write_wtmp()
{
	/* write in wtmp that we are dying */
	int fd;
	struct utmp ut;
	
	memset((char *)&ut, 0, sizeof(ut));
	strcpy(ut.ut_line, "~");
	memcpy(ut.ut_name, "shutdown", sizeof(ut.ut_name));

	time(&ut.ut_time);
	ut.ut_type = BOOT_TIME;
	
	if((fd = open(_PATH_WTMP, O_WRONLY|O_APPEND)) > 0) {
		write(fd, (char *)&ut, sizeof(ut));
		close(fd);
	}
}

void
unmount_disks()
{
	/* better to use umount directly because it may be smarter than us */

	int pid;
	int result;
	int status;

	sync();
	if ((pid = fork()) < 0) {
		printf("Cannot fork for umount, trying manually.\n");
		unmount_disks_ourselves();
		return;
	}
	if (!pid) {
		execl(_PATH_UMOUNT, UMOUNT_ARGS, NULL);
		printf("Cannot exec %s, trying umount.\n", _PATH_UMOUNT);
		execlp("umount", UMOUNT_ARGS, NULL);
		printf("Cannot exec umount, trying manually.\n");
		unmount_disks_ourselves();
		exit(0);
	}
	while ((result = wait(&status)) != -1 && result != pid)
		;
	if (result == -1 || status) {
		printf("Running umount failed, trying manually.\n");
		unmount_disks_ourselves();
	}
}

void
unmount_disks_ourselves()
{
	/* unmount all disks */

	FILE *mtab;
	struct mntent *mnt;
	char *mntlist[128];
	int i;
	int n;
	char *filesys;
	
	sync();
	if (!(mtab = setmntent(_PATH_MTAB, "r"))) {
		printf("Cannot open %s.\n", _PATH_MTAB);
		return;
	}
	n = 0;
	while (n < 100 && (mnt = getmntent(mtab))) {
		mntlist[n++] = strdup(mnt->mnt_fsname[0] == '/' ?
			mnt->mnt_fsname : mnt->mnt_dir);
	}
	endmntent(mtab);

	/* we are careful to do this in reverse order of the mtab file */

	for (i = n - 1; i >= 0; i--) {
		filesys = mntlist[i];
#ifdef DEBUGGING
		printf("umount %s\n", filesys);
#else
		if (umount(mntlist[i]) < 0)
			printf("Couldn't umount %s\n", filesys);
#endif
	}
}
 