/* cspawnd.c: spawns gettys on demand.
 *
 * todd j. derr <tjd@haven.zets.org>
 *
 * 	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 * v1.1: 4 April 1995 - Please edit config.h before building.
 * 
 * some code in open_getty() was taken from:
 *
 *       open.c: open a vt to run a new command (or shell).
 *	 Copyright (c) 1994 by Jon Tombs <jon@gtex02.us.es>
 *
 * Thanks to Andries Brouwer <aeb@cwi.nl> for kernel support,
 *	and the idea for this in kbd-0.90's spawn_console.c
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <fcntl.h>
#include <syslog.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/vt.h>
#include <sys/wait.h>
#include <linux/kd.h>
#include "config.h"

#ifdef TTY
#include <pwd.h>
#include <grp.h>

uid_t tty_uid=-1;
gid_t tty_gid=-1;
#endif

#ifndef CLEANUP
#undef  CLEANUP_INTERVAL
#define CLEANUP_INTERVAL	-1	/* sleep forever */
#endif

void sigusr1(),sigchld(),close_all_vcs(),quit();
int opencons();
#ifdef TTY
void remove_utmp(pid_t,uid_t,gid_t);
void open_getty(int, gid_t);
#else
void remove_utmp(pid_t);
void open_getty(int);
#endif

int main(int argc,char *argv[])
{
	char *basename;
	int sflags,cons_fd;
	struct sigaction usr1,chld;
#ifdef TTY
   struct group *ttyg;
   struct passwd *ttyn;
#endif


#ifdef DEBUG
	sflags=LOG_PID|LOG_PERROR;
#else
	sflags=LOG_PID;
#endif

	if((basename=strrchr(argv[0],'/')) != NULL)
		++basename;
	else
		basename=argv[0];
	
	if(getuid())
		exit(9);	/* exit silently if we're not root */

	openlog(basename,LOG_PID|LOG_PERROR,LOG_DAEMON);

#ifndef DEBUG
	switch(fork()) {	/* become a daemon */

		case -1:
			syslog(LOG_ERR,"fork: %m\r\n");
			quit();

		case 0:
			closelog();
			openlog(basename,sflags,LOG_DAEMON);
			close(0);
			close(1);
			close(2);
			if(setsid()<0) {
			    syslog(LOG_ERR,"setsid(): %m\r\n");
			    quit();
			}
			break;

		default:
			closelog();
			exit(0);
    	}
#endif
	
#ifdef TTY
	/* we use this in utmp.c to change tty modes as well. */
	if((ttyn=getpwnam(TTYUSER)))
	{
		tty_uid=ttyn->pw_uid;
	}
	else
	{
		syslog(LOG_WARNING,"can't get uid for '%s': %m\r\n",TTYUSER);
		tty_uid=-1;
	}

	if((ttyg=getgrnam(TTY)))
	{
		tty_gid=ttyg->gr_gid;
	}
	else
	{
		syslog(LOG_WARNING,"can't get gid for '%s': %m\r\n",TTYUSER);
		tty_gid=-1;
	}
#endif
	usr1.sa_handler=sigusr1;
	chld.sa_handler=sigchld;

	sigemptyset(&usr1.sa_mask);
	sigaddset(&usr1.sa_mask,SIGUSR1);
	sigaddset(&usr1.sa_mask,SIGCHLD);

	chld.sa_mask=usr1.sa_mask;
	usr1.sa_flags=chld.sa_flags=0;
	usr1.sa_restorer=chld.sa_restorer=NULL;

	if(sigaction(SIGUSR1,&usr1,NULL)<0)
	{
		syslog(LOG_ERR,"sigaction(SIGUSR1): %m\r\n");
		quit();
	}

	if(sigaction(SIGCHLD,&chld,NULL)<0)
	{
		syslog(LOG_ERR,"sigaction(SIGCHLD): %m\r\n");
		quit();
	}

	if((cons_fd=opencons()) < 0) quit();	/* we're in bad shape */

	if(ioctl(cons_fd, KDSIGACCEPT, (long) SIGUSR1)<0) {
		syslog(LOG_ERR,"ioctl(KDSIGACCEPT): %m\r\n");
		quit();
	}

	close(cons_fd);

	/* we basically wait forever for signals, waking up
	 * every CLEANUP_INTERVAL seconds to close all inactive vc's
	 */

	while(1) {
		sleep(CLEANUP_INTERVAL);
#ifdef CLEANUP
		close_all_vcs();
#endif
	}
	/* NOTREACHED */
	closelog();
	return -argc;	/* makes gcc shut up */
}

void quit()
{
	syslog(LOG_ERR,"Exiting...\r\n");
	closelog();
	exit(2);
}

#ifdef CLEANUP
void close_all_vcs()
{
	int fd;
#ifdef DEBUG
	syslog(LOG_DEBUG,"cleaning up...\r\n");
#endif
	
	if((fd=opencons()) < 0) return;

	if (ioctl(fd, VT_DISALLOCATE, 0) < 0) {
		syslog(LOG_WARNING,"ioctl(VT_DISALLOCATE): %m\r\n");
	}

	close(fd);
}
#endif

void sigusr1(void)
{
	int fd;

	if((fd=opencons()) < 0) return;

#ifdef TTY
	open_getty(fd,tty_gid);
#else
	open_getty(fd);
#endif
	close(fd);

}

void sigchld(void)
{
	int status;
	pid_t pid;

#ifdef DEBUG
	syslog(LOG_DEBUG,"caught a SIGCHLD...\r\n");
#endif

	while((pid=waitpid(-1,&status,WNOHANG))>0)
	{
#ifdef DEBUG
		syslog(LOG_DEBUG,"SIGCHLD: pid %d.\r\n",pid);
#endif

#ifdef TTY
		remove_utmp(pid,tty_uid,tty_gid);
#else
		remove_utmp(pid);
#endif
	}
}

/* we do this rather than keeping it open all the time so our daemon
 * doesn't get associated with a VC, which can cause problems.
 * returns an fd for /dev/console, which we should close ASAP.
 */

int opencons()
{
	int fd;
	if ((fd = open("/dev/console",O_WRONLY,0)) < 0) {
		syslog(LOG_WARNING,"open /dev/console: %m\r\n");
  	}
	return fd;
}
