/* TArg.h --- The argument input window class  
   
   Copyright (C) 1993  A.Matthias
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   */

#ifndef _TArg_h
#define _TArg_h

extern "C" {
#include <ncurses/ncurses.h>
}
#include "globals.h"
#include "config.h"
#include "lineedit.h"

#define DLGHT 5
#define DLGWT 60

class TArg
{
private:

#define OK_TEXT  "[Return=OK]"
#define ESC_TEXT "[ESC=Cancel]"

  WINDOW *wptr, *frameptr;
  char PS1 [PATH_MAX], DEFAULT [PATH_MAX];
  int x1, y1, x2, y2;
  bool shown;
  int hascolor;

public:
  TArg( char* ps1, char* defaults )
  {
    x1 = (SCR_X/2)-(DLGWT/2);
    x2 = (SCR_X/2)+(DLGWT/2);
    y1 = (SCR_Y/2)-(DLGHT/2);
    y2 = (SCR_Y/2)+(DLGHT/2)+3;

    frameptr = newwin( y2-y1, x2-x1, y1, x1 );
    box( frameptr, FRAME_CHAR_V, FRAME_CHAR_H );
    y1++; x1++; y2--; x2--;
    wptr=newwin( y2-y1, x2-x1, y1, x1 );

    //    hascolor = has_colors();
    hascolor=0;  // Because of repaint problems if color is set

    if ( hascolor )
    {
      start_color();
      init_pair( 1, COLOR_WHITE, COLOR_BLUE );
      wattron( wptr, COLOR_PAIR( 1 ) );
      touchwin( wptr );
    }      
    leaveok( wptr, FALSE );
    leaveok( frameptr, FALSE );
    shown=FALSE;
    setPS1( ps1, defaults );
    cbreak();
    werase( wptr );
    // Write the prompt
    wmove( wptr, 0, ((x2-x1)/2)-(strlen(PS1)/2)-1 );
    waddstr( wptr, PS1 );
    // Write the OK and CANCEL buttons
    wmove( wptr, y2-y1-1, ((x2-x1)/2)-((x2-x1)/4)-(strlen(OK_TEXT)/2)-1 );
    waddstr( wptr, OK_TEXT );
    wmove( wptr, y2-y1-1, ((x2-x1)/2)+((x2-x1)/4)-(strlen(ESC_TEXT)/2)-1 );
    waddstr( wptr, ESC_TEXT );
    // Write the inverted input line
    wmove( wptr, 2, 0 );
    wstandout( wptr );
    wclrtoeol( wptr );
    wmove( wptr, 2, 0 );
    show();
  }
  
  ~TArg()
  {
    shown = FALSE;
    delwin( wptr );
  }
  
  void show( void )
  { 
    if ( shown == FALSE )
    {
      shown=TRUE;
    }
    touchwin( frameptr ); 
    wrefresh( frameptr ); 
    touchwin( wptr ); 
    wrefresh( wptr ); 
  }
  
  void prompt( void )
  { mvwaddstr( wptr, 0, 0, PS1 ); wclrtoeol( wptr ); show(); }
  
  // setPS1() adds a trailing separator and default info
  void setPS1( char *ps1, char* defaults )
  {
    strcpy( DEFAULT, defaults );
    sprintf( PS1, "%s [%s]%s\n", ps1, DEFAULT, ARG_SEPARATOR );
  }


  // Returns a pointer to buf or NULL if user escaped

  char* input( char* buf )
  {
    char userline[PATH_MAX];
    int ulpos = 0, ch;
    bool ul_dirty;

    userline[0]='\0';
    ulpos = 0;
    ch=0;
    ul_dirty=TRUE;
    keypad( wptr, TRUE );
    meta( wptr, TRUE );
    
    do
      {
	if ( ul_dirty == TRUE )
	  {
	    wmove( wptr, 2, 0 );
	    wstandout( wptr );
	    wclrtoeol( wptr );
	    wmove( wptr, 2, 0 );
	    wputline( wptr, userline, ulpos, INVERSE );
	    show();
	    ul_dirty=FALSE;
	  }
	noecho();
	
	ch = wgetch( wptr );
	
	if ( ch == KEY_LEFT )
	  {
	    if (ulpos>0) ulpos--;
	    ul_dirty=TRUE;
	    continue;
	  }
	
	if ( ch == KEY_RIGHT )
	  {
	    if (ulpos<(int)strlen(userline)) ulpos++;
	    ul_dirty=TRUE;
	    continue;
	  }
	
	// DELLN_CHAR deletes the input line
	if ( ch == DELLN_CHAR )
	  {
	    ulpos = 0;
	    userline[0]='\0';
	    ul_dirty=TRUE;
	    
	    continue;
	  }
	
	if ( ch == 10 ) 
	  {
	    continue;  /* don't append CR to line */
	  }

	/* delete character */
	if ( ch == KEY_DC )
	  {
	    if ( ulpos < (strlen( userline )-1) )  
	      {
		ul_dirty=TRUE;
		strcpy( &userline[ulpos], &userline[ulpos+1] );
	      }
	    continue;
	  }
	    	  
	if ( ( (ch == erasechar()) || (ch == BS_CHAR) )  && (ulpos>0) )
	  {
	    ul_dirty=TRUE;
	    strcpy( &userline[ulpos-1], &userline[ulpos] );
	    ulpos--;
	    continue;
	  }
	else
	  if ( (ch != erasechar()) && (ch!=BS_CHAR) )
	    {
	      ul_dirty=TRUE;
	      strinsch( userline, ch, ulpos );
	      ulpos++;
	      continue;
	    }
      }
    while ( (ch != ESCAPE_CHAR) && (ch != 10 ) );
    
    if ( ( strlen( userline ) == 0 ) && ( ch != ESCAPE_CHAR ) )
      strcpy( buf, DEFAULT );
    else
      strcpy( buf, userline );

    if ( ch == ESCAPE_CHAR ) 
      return( NULL );
    else
      return( buf );
  }
  
  WINDOW* getwptr( void )
  { return( wptr ); }
  
};


#endif


