/* mkfifo -- make fifo's (named pipes)
   Copyright (C) 1990, 1991 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* Options:
   -m, --mode=mode	Set the mode of created fifo's to MODE, which is
			symbolic as in chmod and uses the umask as a point of
			departure.

   David MacKenzie <djm@ai.mit.edu>  */

#ifdef HAVE_CONFIG_H
#if defined (CONFIG_BROKETS)
/* We use <config.h> instead of "config.h" so that a compilation
   using -I. -I$srcdir will use ./config.h rather than $srcdir/config.h
   (which it would do because it found this file in $srcdir).  */
#include <config.h>
#else
#include "config.h"
#endif
#endif

#include <stdio.h>
#include <getopt.h>
#include <sys/types.h>
#include "system.h"
#include "modechange.h"
#include "version.h"

#ifdef NLS
#include <nl_types.h>
#include <locale.h>
#endif

void error ();

static void usage ();

/* The name this program was run with. */
char *program_name;

/* If non-zero, display usage information and exit.  */
static int show_help;

/* If non-zero, print the version on standard output and exit.  */
static int show_version;

static struct option const longopts[] =
{
  {"mode", required_argument, NULL, 'm'},
  {"help", no_argument, &show_help, 1},
  {"version", no_argument, &show_version, 1},
  {NULL, 0, NULL, 0}
};

#ifdef NLS
/* Message catalog descriptor */
static nl_catd catfd = -1;

/* Is the catalog currently open? */
static int cat_is_open = 0;

/* mkfifo uses the thirteenth set within the fileutils.cat */
static int set_num = 13;
#endif


void
main (argc, argv)
     int argc;
     char **argv;
{
  unsigned short newmode;
  struct mode_change *change;
  char *symbolic_mode;
  int errors = 0;
  int optc;

  program_name = argv[0];
  symbolic_mode = NULL;

#ifdef NLS
  setlocale(LC_MESSAGES,"");
  catinit ();
#endif

#ifndef S_ISFIFO
#ifdef NLS
  error (4, 0, catgets(catfd, set_num, 1, "fifo files not supported"));
#else
  error (4, 0, "fifo files not supported");
#endif
#else
  while ((optc = getopt_long (argc, argv, "m:", longopts, (int *) 0)) != EOF)
    {
      switch (optc)
	{
	case 0:
	  break;
	case 'm':
	  symbolic_mode = optarg;
	  break;
	default:
	  usage (1);
	}
    }

  if (show_version)
    {
      printf ("%s\n", version_string);
#ifdef NLS
      catclose(catfd);
#endif
      exit (0);
    }

  if (show_help)
    usage (0);

  if (optind == argc)
    usage (1);

  newmode = 0666 & ~umask (0);
  if (symbolic_mode)
    {
      change = mode_compile (symbolic_mode, 0);
      if (change == MODE_INVALID)
#ifdef NLS
        error (1, 0, catgets(catfd, set_num, 2, "invalid mode"));
#else
	error (1, 0, "invalid mode");
#endif
      else if (change == MODE_MEMORY_EXHAUSTED)
#ifdef NLS
        error (1, 0, catgets(catfd, set_num, 3, "virtual memory exhausted"));
#else
	error (1, 0, "virtual memory exhausted");
#endif
      newmode = mode_adjust (newmode, change);
    }

  for (; optind < argc; ++optind)
    {
      if (mkfifo (argv[optind], newmode))
	{
#ifdef NLS
          error (0, errno, catgets(catfd, set_num, 4,
               "cannot make fifo `%s'"), argv[optind]);
#else
	  error (0, errno, "cannot make fifo `%s'", argv[optind]);
#endif
	  errors = 1;
	}
    }

#ifdef NLS
  catclose(catfd);
#endif
  exit (errors);
#endif
}

#ifdef S_ISFIFO
static void
usage (status)
     int status;
{

#ifdef NLS
  catinit ();
#endif

  if (status != 0)
#ifdef NLS
    fprintf (stderr, catgets(catfd, set_num, 5,
             "Try `%s --help' for more information.\n"), program_name);
#else
    fprintf (stderr, "Try `%s --help' for more information.\n",
	     program_name);
#endif
  else
    {
#ifdef NLS
      printf (catgets(catfd, set_num, 6,
              "Usage: %s [OPTION] PATH...\n"), program_name);
      printf (catgets(catfd, set_num, 7, "\
\n\
  -m, --mode MODE   set permission mode (as in chmod), not 0666 - umask\n\
      --help        display this help and exit\n\
      --version     output version information and exit\n")); 
#else
      printf ("Usage: %s [OPTION] PATH...\n", program_name);
      printf ("\
\n\
  -m, --mode MODE   set permission mode (as in chmod), not 0666 - umask\n\
      --help        display this help and exit\n\
      --version     output version information and exit\n");
#endif
    }
#ifdef NLS
  catclose(catfd);
#endif
  exit (status);
}
#endif

catinit ()
{
    if (!cat_is_open)
        catfd = catopen("fileutils",MCLoadBySet);
    cat_is_open = 1;
}
