/* readrc.c

   Reads the user's .Safedelrc file and scans
   the section requested and passes back a
   structure with the information found.

   Passed: pointer to name of rc file
           pointer to the name of the section

   Returns: pointer to created structure or NULL

   NOTE: the calling routine must free the storage
         malloc'd by this routine.
*/

#include <stdlib.h>
#include <stdio.h>
#include <sys/stat.h>
#include <string.h>
#include <ctype.h>
#include "safedelete.h"

#define SAFEDAYS 1
#define COMPRESS 2
#define SUFFIXES 3 

/* Global Variables */

rcstruct RCitems;

/* ================================================ */

/* VerifyEntry

   Verifies an entry from the .Safedelrc file.

   Passed: pointer to the entry

   Returns: 0 - entry is invalid
            PREFIX - entry is in form: a*
            SUFFIX - entry is in form: *a
            INFIX  - entry is in form: a*a
            FULLNAME - entry is in form: a
            ALLFILES - entry is: *
*/

int  VerifyEntry(char *Entry)
{
int i, NumStars;
char *wrkptr;

/* First make sure entry has only 1 "*" */

  NumStars=0;
  wrkptr=Entry;
  for(i=0; i < strlen(Entry); i++)
  {
    if(*wrkptr == '*')
      NumStars++;
    wrkptr++;
  }
  if(NumStars > 1)
    return 0;

/* Now see what type of entry we have */

  if(NumStars == 0)
    return FULLNAME;

  if(strlen(Entry) == 1)
    return ALLFILES;

  if(*Entry == '*')
    return SUFFIX;

  if(*(Entry+strlen(Entry)-1) == '*')
    return PREFIX;

  return INFIX;
}

/* ================================================= */

/* ReadRC

   Read the .Safedelrc file and parse its contents.

   Passed: pointer to .Safedelrc filename
           pointer to section name to read

   Returns: pointer to structure rcstruct or NULL
*/

rcstruct *ReadRC(char *RCFileName, char *SectName)
{
int  SectType;
int  i, NumEnts;
static int BeenHere = 0;

FILE *frc;

char DaysSect[] = "safedays";
char CompSect[] = "compression";
char SuffSect[] = "suffix";
char WorkBuf[WORK_BUF_LEN];
char *wrkptr1, *wrkptr2, *wrkptr3, *palloc, *stringptr;

struct stat FileInfo;

safedays_struct *safedays = NULL;
compression_struct *compression = NULL;
suffix_struct *suffix = NULL;

/* Do some initialization work */

  if(! BeenHere)
  {
    RCitems.DaysStruct=NULL;
    RCitems.NumDaysEnts=0;
    RCitems.CompStruct=NULL;
    RCitems.NumCompressEnts=0;
    RCitems.SuffStruct=NULL;
    RCitems.NumSuffixes=0;
    BeenHere=1;
  }

  SectType=0;
  NumEnts=0;

/* Make sure the rc file exists before going further */

  if(stat(RCFileName, &FileInfo) != 0)
    return NULL;

/* Open the rc file and look for the requested section */

  frc=fopen(RCFileName, "r");
  while(fgets(WorkBuf, WORK_BUF_LEN, frc) != NULL)
  {
    if(*WorkBuf == '#' || *WorkBuf == ' ')
      continue;

    if(*WorkBuf == '[')
    {
      for(i=0; i < strlen(WorkBuf); i++)
        WorkBuf[i]=tolower(WorkBuf[i]);

      if(strncmp(WorkBuf+1, SectName, strlen(SectName)) == 0)
        break;
    }
  }

/* Allocate space for the structure.
   We get 4 times the size of the rc
   file just to make sure we have enough.
   The area is split into two parts:
   the first part is for the structures, 
   the second part is for the character strings.
*/

  palloc=malloc(4*FileInfo.st_size);
  stringptr=palloc+(3*FileInfo.st_size);

  if(strncmp(SectName, DaysSect, strlen(DaysSect)) == 0)
  {
    SectType=SAFEDAYS;
    safedays=RCitems.DaysStruct=(safedays_struct *)palloc;
    NumEnts=(3*FileInfo.st_size)/sizeof(safedays_struct);
  }

  if(strncmp(SectName, CompSect, strlen(CompSect)) == 0)
  {
    SectType=COMPRESS;
    compression=RCitems.CompStruct=(compression_struct *)palloc;
    NumEnts=(3*FileInfo.st_size)/sizeof(compression_struct);
  }

  if(strncmp(SectName, SuffSect, strlen(SuffSect)) == 0)
  {
    SectType=SUFFIXES;
    suffix=RCitems.SuffStruct=(suffix_struct *)palloc;
    NumEnts=(3*FileInfo.st_size)/sizeof(suffix_struct);
  }

  if(SectType == 0)
    return NULL;

/* We found the section we're looking for.
   Read the records till we find the 
   end-of-section indicator filling in 
   the structure as we go.

   wrkptr1 points to the start of each filename
   wrkptr2 is used to find the end of each filename
   wrkptr3 points to the value associated for each
*/

  while(fgets(WorkBuf, WORK_BUF_LEN, frc) != NULL)
  {
    if(*WorkBuf == '[')
      break;

    if(*WorkBuf == '#' || *WorkBuf == ' ')
      continue;

    wrkptr1=strchr(WorkBuf, '\n');
    while(isspace(*wrkptr1))
      wrkptr1--;
    *(wrkptr1+1)='\0';   /* indicate end of record */

    wrkptr1=WorkBuf;
    wrkptr3=strchr(wrkptr1, ' ');
    while(*wrkptr3 == ' ')
      wrkptr3++;

    for(wrkptr2=wrkptr1; *wrkptr1 != ' '; wrkptr2++)
    {
      if(*wrkptr2 == ',' || *wrkptr2 == ' ')
      {
        switch(SectType)
        {
          case SAFEDAYS : i=(int)(wrkptr2-wrkptr1);
                          strncpy(stringptr, wrkptr1, i);
                          *(stringptr+i)='\0';
                          if((safedays->Type=VerifyEntry(stringptr)) > 0)
                          {
                            switch(safedays->Type)
                            {
                              case PREFIX : safedays->PreLen=i-1;
                                            safedays->SufLen=0;
                                            break;

                              case SUFFIX : safedays->PreLen=0;
                                            safedays->SufLen=i-1;
                                            break;

                              case INFIX  : safedays->PreLen=(int)(strchr(stringptr, '*')-stringptr);
                                            safedays->SufLen=strlen(stringptr)-safedays->PreLen-1;
                                            break;

                              case FULLNAME : safedays->PreLen=0;
                                              safedays->SufLen=i;
                                              break;

                              case ALLFILES : safedays->PreLen=safedays->SufLen=0;
                                              break;
                            }
                            safedays->FileName=stringptr;
                            stringptr+=i+1;
                            safedays->Days=atoi(wrkptr3);
                            RCitems.NumDaysEnts++;
                            if(RCitems.NumDaysEnts >= NumEnts)
                            {
                              fclose(frc);
                              return &RCitems;
                            }
                            safedays++;
                          }
                          if(*wrkptr2 == ',')
                            wrkptr1=++wrkptr2;
                          else
                            wrkptr1=wrkptr2;
                          break;

          case COMPRESS : i=(int)(wrkptr2-wrkptr1);
                          strncpy(stringptr, wrkptr1, i);
                          *(stringptr+i)='\0';
                          if((compression->Type=VerifyEntry(stringptr)) > 0)
                          {
                            switch(compression->Type)
                            {
                              case PREFIX : compression->PreLen=i-1;
                                            compression->SufLen=0;
                                            break;

                              case SUFFIX : compression->PreLen=0;
                                            compression->SufLen=i-1;
                                            break;

                              case INFIX  : compression->PreLen=(int)(strchr(stringptr, '*')-stringptr);
                                            compression->SufLen=strlen(stringptr)-compression->PreLen-1;
                                            break;

                              case FULLNAME : compression->PreLen=0;
                                              compression->SufLen=i;
                                              break;

                              case ALLFILES : compression->PreLen=compression->SufLen=0;
                                              break;
                            }
                            compression->FileName=stringptr;
                            stringptr+=i+1;
                            if(strcmp(wrkptr3, "none") == 0 || strchr(wrkptr3, '/') == NULL)
                            {
                              strcpy(stringptr, "none");
                              compression->CompCmd=stringptr;
                              compression->UncompCmd=stringptr;
                            }
                            else
                            {
                              i=(int)(strchr(wrkptr3, '/')-wrkptr3);
                              strncpy(stringptr, wrkptr3, i);
                              compression->CompCmd=stringptr;
                              *(stringptr+i)='\0';
                              stringptr+=i+1;
                              strcpy(stringptr, wrkptr3+i+1);
                              compression->UncompCmd=stringptr;
                            }
                            stringptr+=strlen(stringptr)+1;
                            RCitems.NumCompressEnts++;
                            if(RCitems.NumCompressEnts >= NumEnts)
                            {
                              fclose(frc);
                              return &RCitems;
                            }
                            compression++;
                          }
                          if(*wrkptr2 == ',')
                            wrkptr1=++wrkptr2;
                          else
                            wrkptr1=wrkptr2;
                          break;

          case SUFFIXES : i=(int)(wrkptr2-wrkptr1);
                          strncpy(stringptr, wrkptr1, i);
                          *(stringptr+i)='\0';
                          suffix->CmdName=stringptr;
                          stringptr+=i+1;
                          strcpy(stringptr, wrkptr3);
                          suffix->Suffix=stringptr;
                          stringptr+=strlen(stringptr)+1;
                          RCitems.NumSuffixes++;
                          if(RCitems.NumSuffixes >= NumEnts)
                          {
                            fclose(frc);
                            return &RCitems;
                          }
                          suffix++;
                          if(*wrkptr2 == ',')
                            wrkptr1=++wrkptr2;
                          else
                            wrkptr1=wrkptr2;
                          break;
        }
      }
    }
  }
  fclose(frc);
  return &RCitems;
}
