
/* undelete.c
   Undeletes the files saved by the safedelete
   program.

   Uses ncurses to perform terminal I/O to the
   Linux console or xterm window.

   Usage:  undelete [ -i | --info ] [ filename ]

   If filename is specified, the most recent
   copy of that file is restored to its original
   location.  The filename can consist of
   the complete path+filename, partial path
   with filename, or just the filename.  If
   the complete path is not specified, it
   is assumed to be relative to the current
   directory (where the undelete command
   was issued).

   If no filename is specified, the list of 
   files in the .safedelete.log is shown as
   a scrollable list allowing the users to 
   choose which file(s) to undelete.

   If the file to be undeleted already exists
   the user is prompted for permission to
   overlay the existing file.

   If the -i or --info options are specified,
   only information on the deleted file is given.
   If no filename is given with the -i/--info
   options, information on all files is displayed.
*/

#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <stdio.h>
#include <pwd.h>
#include <grp.h>
#include <string.h>
#include <ctype.h>
#include <ncurses.h>
#include <time.h>
#include <utime.h>
#include <getopt.h>
#include "safedelete.h"

/* #define BACK_ROW (int)((LINES-24)/2) */

#define BACK_ROW 0
#define BACK_COL (int)((COLS-80)/2)
#define LIST_ROW BACK_ROW+3
#define LIST_COL BACK_COL+1
#define LIST_BEG 0

/* Global variables */

int ListEnd = 17;
int PartialLastLine = 0;

/* ================================================================ */

/* UndeleteFile
   Scans the users .safedelete.log looking for the fully
   qualified filename.  Once found, the grunged filename
   is obtained and the file is tar'ed back to its original
   location.

   Passed: pointer to fully qualified filename
           pointer to name of safedelete directory
	   pointer to path of user's .safedelete.log
           pointer to path of user's .Safedelrc
           pointer to new directory, new filename, or NULL
	   pointer to existing file information or NULL
                   relative record number to undelete or -1 if last occurence of file
                  undelete flag - TRUE => undelete the file, FALSE => remove log entry only
	   
   Returns: 0 - file successfully undeleted.
            1 - filename not found in .safedelete.log
            2 - saved file not found in safedelete directory
            3 - undelete failed, reason unknown
*/
int  UndeleteFile(char *FileName, char *SafeDir, char *SafeDelLog, char *SafeDelRC,
                  char *NewName, struct stat *CurrInfo,
                  int FileNum, bool UndelFlag)
{
char WorkBuf[WORK_BUF_LEN];
char WorkBuf2[WORK_BUF_LEN];

char *LogList, *palloc, *wrkptr, *wrkptr2;

char *GrungedFN;
char *UncompressCmd;
char *FileSuffix;

int  i, j, k;
long LogSize;
time_t NewestMtime;
FILE *flog, *flink;
struct stat FileInfo;

file_struct *Orig, *Curr;
struct utimbuf origtime;

/* Open the user's .safedelete.log and read the whole
   thing into storage where we can easily manipulate it.
*/

  wrkptr=NULL;
  j=k=0;
  stat(SafeDelLog, &FileInfo);
  LogSize=FileInfo.st_size;
  LogList=palloc=malloc(LogSize);
  flog=fopen(SafeDelLog, "r");
  fread(LogList, LogSize, 1, flog);
  fclose(flog); 

/* Now search the whole list looking for the most recent
   occurence of the file only if the relative file number is -1.
*/

  if(FileNum == -1)
  {
    j=0;
    NewestMtime=-1;
    while(j < LogSize)
    {
      memcpy(&k, LogList, 2);    /* get length of this record */
      if(strcmp(FileName, LogList+2) == 0)
      {
        wrkptr2=LogList+2+strlen(LogList+2)+1;
        wrkptr2+=strlen(wrkptr2)+1;
        if(*wrkptr2 == 127)
          wrkptr2++;
        Curr=(file_struct *) wrkptr2;
        if(Curr->Mtime > NewestMtime)
        {
          NewestMtime=Curr->Mtime;
          wrkptr=LogList;
        }
      }
      j+=k;
      LogList+=k;
    }

    if(wrkptr == NULL)
    {
      free(palloc);
      return 1;
    }
  }
  else
  {

/* Find the record number specified by FileNum */

    for(j=0; j < FileNum; j++)
    {
      memcpy(&k, LogList, 2);
      LogList+=k;
    }
    wrkptr=LogList;

/* Make sure this is really the filename we're looking for */

    if(strcmp(FileName, wrkptr+2) != 0)
    {
      free(palloc);
      return 1;
    }
  }

/* Set up some work pointers */

  LogList=wrkptr;
  wrkptr+=2+strlen(LogList+2)+1;    /* point to start of grunged filename */
  GrungedFN=wrkptr;
  wrkptr2=wrkptr+strlen(wrkptr)+1;  /* start of original inode info */
  if(*wrkptr2 == 127)
    wrkptr2++;
  Orig=(file_struct *) wrkptr2;     /* point to start of original file info */
  if(S_ISREG(Orig->Mode) && ! (Orig->FileFlag & NOCOMPRESS))
  {

/* Normal file and it was compressed */

    wrkptr2+=sizeof(file_struct);
    UncompressCmd=wrkptr2;
    wrkptr2+=strlen(wrkptr2)+1;
    FileSuffix=wrkptr2;
  }
  else
  {

/* File was not normal or it was not compressed */

    UncompressCmd=NULL;
    FileSuffix=NULL;
  }

  if(UndelFlag)
  {

/* We found the record we're looking for.  First make
   sure the grunged file really exists.
*/

    sprintf(WorkBuf, "%s/%s", SafeDir, GrungedFN);
    if(stat(WorkBuf, &FileInfo) != 0)
    {
      free(palloc);
      return 2;
    }

/* If we have an old format entry just tar it back */

    if(Orig->Version == 0)
    {
      sprintf(WorkBuf, "tar -xzf %s/%s > /dev/null", SafeDir, GrungedFN);
      getcwd(WorkBuf2, WORK_BUF_LEN);
      chdir("/");
      system(WorkBuf);
      chdir(WorkBuf2);
    }
    else
    {

/* Rename the file to its original name */

      i=ParseFileName(LogList+2);
      sprintf(WorkBuf2, "%s/%s", SafeDir, LogList+2+i+1);
      rename(WorkBuf, WorkBuf2);

/* If this was a regular file and it was compressed
   uncompress it.
*/

      if(UncompressCmd)
      {
        sprintf(WorkBuf, "%s%s", WorkBuf2, FileSuffix);
        rename(WorkBuf2, WorkBuf);
        sprintf(WorkBuf, "%s %s%s", UncompressCmd, WorkBuf2, FileSuffix);
        system(WorkBuf);
      }

/* If this is a regular file, copy it to its ultimate
   destination, whether that be a new directory
   or a new filename.
*/
      if(S_ISREG(Orig->Mode))
      {
        if(NewName == NULL)
        {
          sprintf(WorkBuf, "cp %s %s", WorkBuf2, LogList+2);
          system(WorkBuf);
          strcpy(FileName, LogList+2);
        }
        else
        {
          sprintf(WorkBuf, "cp %s %s", WorkBuf2, NewName);
          system(WorkBuf);
          strcpy(FileName, NewName);
        }

/* Reset the permissions, uid, gid, etc of
   the new file.
*/

        chmod(FileName, Orig->Mode);
        chown(FileName, Orig->Uid, Orig->Gid);
        origtime.actime=Orig->Atime;
        origtime.modtime=Orig->Mtime;
        utime(FileName, &origtime);
      }

/* This is a link, create a new one */

      else
      {
        flink=fopen(WorkBuf2, "r");
        fgets(WorkBuf, WORK_BUF_LEN, flink);
        fclose(flink);

/* If user wants link to have a new name, give it one */

        if(NewName == NULL)
        {
          symlink(WorkBuf, LogList+2);
          strcpy(FileName, LogList+2);
        }
        else
        {
          symlink(WorkBuf, NewName);
          strcpy(FileName, NewName);
        } 
      }
    }

/* Make sure that the file was really
   undeleted.
*/

    if(lstat(FileName, &FileInfo) != 0 &&
      stat(FileName, &FileInfo) != 0)
      {
        free(palloc);
        return 3;
      }

/* If we just overlaid the file, compare
   the size and time stamps to see if
   we really undeleted the file.  If they
   are equal we have to assume that the
   undelete failed for some reason.
*/

    if(CurrInfo)
    {
      if(FileInfo.st_size == CurrInfo->st_size &&
         FileInfo.st_mtime == CurrInfo->st_mtime)
      {
        free(palloc);
        return 3;
      }
    }
    strcpy(WorkBuf2, SafeDir);
    strcat(WorkBuf2, "/");
    j=ParseFileName(LogList+2);
    strcat(WorkBuf2, LogList+2+j+1);
  }
  else
  {
    strcpy(WorkBuf2, SafeDir);
    strcat(WorkBuf2, "/");
    strcat(WorkBuf2, GrungedFN);
  }

/* Now that the file is undeleted we need
   to do some cleanup.  Delete the saved
   file from the safedelete directory and
   remove the entry in the user's safedelete
   log.
*/

  unlink(WorkBuf2);

  *(LogList+2)='$';   /* Show this file is undeleted */

/* Write the user's log back, minus the
   entry for the file we just undeleted.
*/

  flog=fopen(SafeDelLog, "w");
  j=k=0;
  LogList=palloc;
  while(j < LogSize)
  {
    memcpy(&k, LogList, 2);
    if(*(LogList+2) == '/')
      fwrite(LogList, k, 1, flog);
    j+=k;
    LogList+=k;
  }

  fclose(flog);
  free(palloc);

  return 0;
}

/* ======================================================== */

/* FillListWin
   Fills in the window containing the list of files currently deleted and
   the date each was deleted.

   Passed: pointer to the list window
              pointer to the file pointer array
             number of files in the array
             number of current selected file
             number of line currently containing selection indicator '>'
             number of the file which occupies the first line in the window

   Returns: nothing
*/

void FillListWin(WINDOW *ListWin, char *FilePtr[],
                int NumFiles, int *SelFile, int FileLine[],
                int *FirstFile)
{
char WorkBuf[WORK_BUF_LEN];
char *wrkptr, *wrkptr2;
int  i, j, k;

/* Do some initialization work */

  werase(ListWin);
  PartialLastLine=0;

/* If there are no files to display, just return */

  if(NumFiles <= 0)
    return;

/* FileLine contains the line number in the window where
   a particular filename starts (e.g. FileLine[4]=5 means
   that the 4th filename begins on line 5 in the ListWin
   window).  We use this array for moving the '>' indicator
   up and down.  We initialize the entire array to some
   arbitrary large limit so we can tell when we get to the
   bottom of the window.  This just makes coding the
   scrolling functions easier.
*/

  for(i=0; i < MAX_FILES; i++)
    FileLine[i]=99;

/* If the last file in the list is also in the first line
   of the window and has been undeleted, restart list
   from the beginning.
*/

  if(*FirstFile >= NumFiles)
    *SelFile=*FirstFile=0;

/* Look at each filename and put it into the screen buffer.
   Also split filenames that are longer than what will fit
   on one line.
*/

  j=LIST_BEG;
  for(i=*FirstFile; i < NumFiles; i++)
  {
    FileLine[i]=j;
    wrkptr=FilePtr[i]+2;
    k=strlen(wrkptr);
    wrkptr2=wrkptr+k+1;
    if(k > 60)
    {

/* Filename longer than one line, write as many full lines
   as we can.
*/

      for(k/=60; k > 0; k--)
      {
        mvwaddnstr(ListWin, j, 3, wrkptr, 60);
        j++;
        if(j > ListEnd)
	{
          PartialLastLine=1;
          break;
	}
        wrkptr+=60;
      }

/* Write the last line of the filename, if it will fit in the window */

      if(j > ListEnd)
        break;
      k=strlen(FilePtr[i]+2)%60;
      mvwaddnstr(ListWin, j, 3, wrkptr, k);
    }

/* Full filename fits on one line, place it there */

    else
      mvwaddnstr(ListWin, j, 3, wrkptr, k);

/* Add the date stamp following the filename */

    strncpy(WorkBuf, wrkptr2+1, 2);
    WorkBuf[2]='/';
    strncpy(WorkBuf+3, wrkptr2+3, 2);
    WorkBuf[5]='/';
    strncpy(WorkBuf+6, wrkptr2+5, 2);
    mvwaddnstr(ListWin, j, 67, WorkBuf, 8);
    j++;
    if(j > ListEnd)
      break;
  }

/* Add a file selection indicator that will move vertically
   along the left side of the file names.
*/

  mvwaddch(ListWin, FileLine[*SelFile], 1, '>');
}

/* ================================================================ */

/* GetFileInfo

   This routine scans an entry from the .safedelete.log
   and creates a series of null-delimited strings 
   containing the information for the file.

   Passed: pointer .safedelete.log entry
           pointer to area to receive strings

   returns: nothing
*/

void GetFileInfo(char *LogEntry, char *StringArea)
{
struct passwd *UidInfo;
struct group *GidInfo;
struct tm *tm;

int i, k;
char *wrkptr;
char TimeBuf[9];

char *Permissions[8] = {"---", "--x", "-w-",
                       "-wx", "r--", "r-x",
                       "rw-", "rwx" };

file_struct *Orig;

/* Get file permissions */

  wrkptr=LogEntry+2;
  wrkptr+=strlen(wrkptr)+1;
  wrkptr+=strlen(wrkptr)+1;
  if(*wrkptr == 127)
    wrkptr++;
  Orig=(file_struct *)wrkptr;
  if(Orig->Version == 0)
  {
    strcpy(StringArea, "Permissions: (unknown)");
    i=strlen(StringArea)+1;
    strcpy(StringArea+i, "Owners login id: (unknown)");
    i+=strlen(StringArea+i)+1;
    strcpy(StringArea+i, "Owners group id: (unknown)");
    i+=strlen(StringArea+i)+1;
    strcpy(StringArea+i, "Size: (unknown)");
    i+=strlen(StringArea+i)+1;
    strcpy(StringArea+i, "Last accessed: (unknown)");
    i+=strlen(StringArea+i)+1;
    strcpy(StringArea+i, "Last modified: (unknown)");
    i+=strlen(StringArea+i)+1;
    *(StringArea+i)='\0';
  }
  else
  {
    strcpy(StringArea, "Permissions: -");
    if(S_ISDIR(Orig->Mode))
      *(StringArea+13)='d';
    if(S_ISLNK(Orig->Mode))
      *(StringArea+13)='l';
    if(S_ISCHR(Orig->Mode))
      *(StringArea+13)='c';
    if(S_ISBLK(Orig->Mode))
      *(StringArea+13)='b';
    if(S_ISFIFO(Orig->Mode))
      *(StringArea+13)='f';
    if(S_ISSOCK(Orig->Mode))
      *(StringArea+13)='s';
    k=(Orig->Mode>>6)&0x0007;
    strcat(StringArea, Permissions[k]);
    k=(Orig->Mode>>3)&0x0007;
    strcat(StringArea, Permissions[k]);
    k=Orig->Mode&0x0007;
    strcat(StringArea, Permissions[k]);
    i=strlen(StringArea)+1;

/* Get owner info */

    if((UidInfo=getpwuid(Orig->Uid)) == NULL)
      sprintf(StringArea+i, "Owners login id: %d", Orig->Uid);
    else
      sprintf(StringArea+i, "Owners login name: %s", UidInfo->pw_name);
    i+=strlen(StringArea+i)+1;

    if((GidInfo=getgrgid(Orig->Gid)) == NULL)
      sprintf(StringArea+i, "Owners group id: %d", Orig->Gid);
    else
      sprintf(StringArea+i, "Owners group name: %s", GidInfo->gr_name);
    i+=strlen(StringArea+i)+1;

/* Show size of file */

    sprintf(StringArea+i, "Size: %li", Orig->Size);
    i+=strlen(StringArea+i)+1;

/* Show the date and time stamps */

    tm=localtime(&Orig->Atime);
    strftime(TimeBuf, 9, "%m/%d/%y", tm);
    sprintf(StringArea+i, "Last accessed: %s", TimeBuf);
    i+=strlen(StringArea+i)+1;

    tm=localtime(&Orig->Mtime);
    strftime(TimeBuf, 9, "%m/%d/%y", tm);
    sprintf(StringArea+i, "Last modified: %s", TimeBuf);
    i+=strlen(StringArea+i)+1;

/* Get symbolic link info, if needed */

    if(S_ISLNK(Orig->Mode))
    {
      wrkptr+=sizeof(file_struct);
      strcpy(StringArea+i, "Link to: ");
      strcat(StringArea+i, wrkptr);
      i+=strlen(StringArea+i)+1;
    }

/* Add terminating null */
 
    *(StringArea+i)='\0';
  }
}

/* ================================================================ */

/* ShowFileInfo
   This routine goes through the user's .safedelete.log looking
   for the specified filename.  When found, a variety of file information
   is displayed.

   Passed: pointer to the user's .safedelete.log
           pointer to the filename or NULL

   Returns: nothing
*/

void ShowFileInfo(char *SafeDelLog, char *FileName)
{
struct stat FileInfo;

FILE *flog;

char *LogList, *palloc, *wrkptr1, *wrkptr2;
char WorkBuf[WORK_BUF_LEN];

int  i, k, GotFile;
long j;

/* First we read the entire .safedelete.log into memory so
   it's easy to manipulate.
*/

  stat(SafeDelLog, &FileInfo);
  LogList=palloc=malloc(FileInfo.st_size);
  flog=fopen(SafeDelLog, "r");
  fread(LogList, FileInfo.st_size, 1, flog);
  fclose(flog);

/* Now that we've got the entire log in memory, just
   go through it looking for the file the user wants.
*/

  j=GotFile=0;
  while(j < FileInfo.st_size)
  {
    memcpy(&k, LogList, 2);  /* get length of this record */
    wrkptr1=LogList+2;

    if(FileName == NULL || strcmp(FileName, wrkptr1) == 0)
    {
      GotFile=1;

/* Show file name */

      printf("\n%s\n", wrkptr1);

      GetFileInfo(LogList, WorkBuf);

/* Show file information */

      wrkptr2=WorkBuf;
      for(i=0; i < 7; i++)
      {
        if(strlen(wrkptr2) > 0)
	{
          printf("\t");
          printf(wrkptr2);
          printf("\n");
	}
        wrkptr2+=strlen(wrkptr2)+1;
      }
    }
    j+=k;
    LogList+=k;
  }

/* Issue message if file wasn't found */

  if(! GotFile)
    fprintf(stderr, "undelete: %s: not found in .safedelete.log\n",
            FileName);

  free(palloc);
}

/* ================================================================ */

/* ListFiles

   This routine reads the .safedelete.log and displays
   the names of the files that are currently safely deleted.
   If a particular filename is given, only those entries that
   match it are listed.  If FileFlag is non-zero, the
   safedeleted filename is also listed.

   Passed: pointer to the SafeDelete log filename
            pointer to the filename to look for or NULL
           flag - 1=list safedeleted filename  0=list original filename only
          size of SafeDelete log (in bytes)
*/

void ListFiles(char *SafeDelLog, char *FileName, int FileFlag, off_t FileSize)
{
char *wrkptr, *wrkptr2, *palloc;
int  j, k, GotIt;
FILE *flog;

  GotIt=0;

/* Read the .safedelete.log and list the entries */

  wrkptr=palloc=malloc(FileSize);
  flog=fopen(SafeDelLog, "r");
  fread(wrkptr, FileSize, 1, flog);
  fclose(flog);
  j=k=0;
  while(j < FileSize)
  {
    memcpy(&k, wrkptr, 2);
    if(FileName != NULL)
    {
      if(strcmp(FileName, wrkptr+2) != 0)
      {
        wrkptr+=k;
        j+=k;
        continue;
      }
    }
    GotIt=1;
    printf("%s", wrkptr+2);
    if(FileFlag)
    {
      wrkptr2=wrkptr+2+strlen(wrkptr+2)+1;
      printf("\t%s", wrkptr2);
    }
    printf("\n");
    wrkptr+=k;
    j+=k;
  }
  free(palloc);
  if(! GotIt)
    printf("undelete: %s: not found in .safedelete.log\n",
           FileName);
}

/* ================================================================ */

/* SortLog

   This routine performs a quick sort of the
   FilePtr array.  I had to write my own because
   the qsort function apparently doesn't understand
   the concept of indirection and can't sort strings
   by just swapping pointers in an array.
*/

void SortLog(char *FilePtr[], int left, int right)
{
int i, j;
char *wrkptr, *temp;

  i=left;
  j=right;
  wrkptr=FilePtr[(left+right)/2]+2;

  do
  {
    while(strcmp(FilePtr[i]+2, wrkptr) < 0 && i < right) i++;
    while(strcmp(FilePtr[j]+2, wrkptr) > 0 && j > left) j--;
    if(i <= j)
    {
      temp=FilePtr[i];
      FilePtr[i]=FilePtr[j];
      FilePtr[j]=temp;
      i++;
      j--;
    }
  } while(i <= j);

  if(left < j)
    SortLog(FilePtr, left, j);

  if(i < right)
    SortLog(FilePtr, i, right);
}

/* ================================================================ */

/* GetInput

   Reads input from a specified window and places
   the characters entered into the area provided.

   Passed: pointer to the window
           pointer to an area to receive the input
           length of the area
           starting line number of cursor
           starting column number of cursor
           length of line on the screen
           flag indicating whether or not to clear the window
            before displaying it

   Returns: length of the area
*/

int GetInput(WINDOW *window, char *Area, int AreaLen, int LineNum, int ColNum, int LineLen,
            bool RefreshFlag)
{
static int i = 0;
static int WinLine = 0, WinCol = 0;
int j = 0;
int MaxCol, MaxChars;

/* Set maximum column number in the window */

  MaxChars=(AreaLen <= LineLen ? AreaLen : LineLen);
  MaxCol=MaxChars+ColNum; 

/* Clear input window to blanks, if needed */

  if(RefreshFlag)
  {
    for(j=ColNum; j <= MaxCol; j++)
      mvwaddch(window, LineNum, j, ' ');

    WinLine=LineNum;
    WinCol=ColNum;
    i=0;
  }

/* Display the window */

  touchwin(window);
  wrefresh(window);

/* Show the cursor */

  curs_set(TRUE);

/* Get input from user */

  while(i < MaxChars)
  {
    while((j=mvwgetch(window, WinLine, WinCol)) == ERR);

    if(j < KEY_MIN || j > KEY_MAX)
      j&=255;

    if(j == KEY_F(3))
      break;    /* F3 key */

    if(j == 27)
      break;    /* Esc key */

    if(j == 10)
      break;    /* Enter key */

    if(j == KEY_DC || j == KEY_BACKSPACE)
    {                             /* Backspace or Delete key */
      if(i == 0)
        continue;

      WinCol--;
      mvwaddch(window, WinLine, WinCol, ' ');
      wrefresh(window);
      i--;
      *(Area+i)=' ';
      continue;
    }

    if(j < 32)
      continue;      /* Invalid character */

    if(WinCol < MaxCol)
    {
      mvwaddch(window, WinLine, WinCol, j);
      wrefresh(window);
      *(Area+i)=j;
      WinCol++;
      i++;
    }
    else
      beep();
  }

/* Hide the cursor */

  curs_set(FALSE);

  if(j == 10)
  {
    *(Area+i)='\0';
    return i;
  }
  else
  {
    *Area='\0';
    return 0;
  }
}

/* ================================================================ */

/* Main processing routine

   This routine parses the command
   line looking for any options or
   arguments.  If any are found, they
   are processed accordingly.
*/
   
int  main(int argc, char **argv)
{
char FileName[FILE_NAME_LEN];
char NewFileName[FILE_NAME_LEN];
char SafeDir[SAFE_DIR_LEN];
char SafeDelLog[DEL_LOG_LEN];
char SafeDelRC[SAFE_RC_LEN];
char WorkBuf[WORK_BUF_LEN];
char WorkBuf2[WORK_BUF_LEN];
char c;

char CmdName[] = "undelete";

struct passwd *UserInfo;

file_struct *Orig;

int  i, j, k, ccode;
char *LoginName, *HomeDir, *wrkptr, *wrkptr2;
char *palloc = NULL;
char *FilePtr[MAX_FILES+1];
int  FileLine[MAX_FILES+1];
int  NumFiles, SelFile, FirstFile;

int  InfoFlag, ListFlag, FileFlag, BadOptFlag;

struct stat FileInfo, NewInfo;
FILE *flog;

struct tm *tm;

static struct option LongOpts[] =
{
    {"info", no_argument, NULL, 'i'},
    {"list", no_argument, NULL, 'l'},
    {"file", no_argument, NULL, 'f'},
    {0     , 0          , 0   ,   0}
};

bool NotDone = TRUE;
bool RefreshWin;

/* ListWin is the window containing the scrollable
   list of files.  BackWin is the background window.
*/

WINDOW *ListWin, *BackWin, *VerifyWin;
WINDOW *InfoWin, *BackInfoWin;
WINDOW *WarnWin, *NewnameWin;

/* Do some preliminary initializtion */

  HomeDir=getenv("HOME");
  if(HomeDir == NULL)
  {
    LoginName=getlogin();
    UserInfo=getpwnam(LoginName);
    HomeDir=UserInfo->pw_dir;
  }
  strcpy(SafeDelLog, HomeDir);
  strcat(SafeDelLog, "/.safedelete.log");
  strcpy(SafeDir, HomeDir);
  strcat(SafeDir, SAFE_DIR_NAME);
  strcpy(SafeDelRC, HomeDir);
  strcat(SafeDelRC, "/.Safedelrc");

/* Check permissions of safedelete directory and log */

  if(CheckPerms(SafeDir, SafeDelLog, CmdName) != 0)
    return 1;

/* Get info on .safedelete.log, we'll need it later */

  ccode=stat(SafeDelLog, &FileInfo);
  if(ccode == 0 && FileInfo.st_size > 0)
  {
    palloc=malloc(FileInfo.st_size);
    flog=fopen(SafeDelLog, "r");
    fread(palloc, FileInfo.st_size, 1, flog);
    fclose(flog);
    if(*palloc == '/')
    {
      printf("undelete: your .safedelete.log is still in the old format\n");
      printf("undelete: it will be converted using the safecnvt command\n");
      system("safecnvt");

/* Now that it's converted, let's just double check */

      flog=fopen(SafeDelLog, "r");
      fread(palloc, FileInfo.st_size, 1, flog);
      fclose(flog);
      if(*palloc == '/')
      {
        printf("undelete: conversion failed, cannot undelete any files\n");
        free(palloc);
        return 1;
      }
      else
        printf("undelete: your .safedelete.log was successfully converted\n");
    }
    free(palloc);
  }


/* Process any options found on the command line */

  InfoFlag=0;
  ListFlag=0;
  FileFlag=0;
  BadOptFlag=0;

  while((i=getopt_long(argc, argv, "ilf", LongOpts, &j)) != -1)
  {
    switch(i)
    {
      case 'i': InfoFlag=1;
                break;

      case 'l' : ListFlag=1;
                 break;

      case 'f' : FileFlag=1;
                 break;

      case '?' :
      default  : BadOptFlag=1;
                 break;
    }
  }

/* If a bad option was found, don't go any further */

  if(BadOptFlag)
    return 1;

/* The following code is done only when no
   filename is given on the command line.
*/

  if(argc == optind)
  {

/* See if user is requesting information on all files
   (user issued:  undelete -i)
*/

    if(InfoFlag)
    {
      if(FileInfo.st_size < 30)
      {
        fprintf(stderr, "undelete: no files to undelete\n");
        return 1;
      }
      ShowFileInfo(SafeDelLog, NULL);
      return 0;
    }

/* See if user just wants a list of the files displayed
   (user issued: undelete -l or undelete -lf)
*/

    if(ListFlag)
    {
      if(FileInfo.st_size < 30)
      {
        fprintf(stderr, "undelete: no files to undelete\n");
        return 1;
      }
      ListFiles(SafeDelLog, NULL, FileFlag, FileInfo.st_size);
      return 0;
    }
  }

/* See if a filename was specified on the command line.
   If so, see if it is an absolute path or a relative path.  If
   relative, prefix it with the current directory path.
*/

  if(argc > optind)
  {
    wrkptr=argv[optind];

    if(*wrkptr == '/')
      strcpy(FileName, wrkptr);
    else
    {
      getcwd(FileName, 256);
      if(*wrkptr == '.')
      {
        while(*wrkptr == '.')
	{

/* See if user specified ./ */

           if(strncmp(wrkptr, "./", 2) == 0)
	   {
              wrkptr+=2;
              continue;
	   }

/* See if user specified ../ */

           if(strncmp(wrkptr, "../", 3) == 0)
	   {
             wrkptr+=3;
             j=ParseFileName(FileName);
             if(j == 0)
               strcpy(FileName, "/");
             else
               FileName[j]='\0';
             continue;
	   }

/* None of the above, must be a hidden file */

           break;
	}
        strcat(FileName, "/");
        strcat(FileName, wrkptr);
      }
      else
      {
        strcat(FileName, "/");
        strcat(FileName, wrkptr);
      }
    }

/* At this point we have a fully qualified path and filename
   (whether provided by the user or built above).   See if the
   user is only requesting information on it.
*/

    if(InfoFlag)
    {
      ShowFileInfo(SafeDelLog, FileName);
      return 0;
    }

/* See if the user just wants the entries listed */

    if(ListFlag)
    {
      ListFiles(SafeDelLog, FileName, FileFlag, FileInfo.st_size);
      return 0;
    }

/* Make sure we have write access to the directory. */

    j=ParseFileName(FileName);
    FileName[j]='\0';

    if(CheckPerms(FileName, NULL, CmdName) != 0)
      return 1;
    else
      FileName[j]='/';

/* Now see if the file exists.  If it does, ask the user if they 
   want to overlay it.
*/

    if(stat(FileName, &FileInfo) == 0 ||
      lstat(FileName, &FileInfo) == 0)
    {
      printf("File already exists, overlay it? (y|n) ");
      i=getchar();
      if(i != 'y' && i != 'Y')
        return 1;
      ccode=UndeleteFile(FileName, SafeDir, SafeDelLog, SafeDelRC, NULL, &FileInfo,
                         -1, TRUE);
    }
    else
      ccode=UndeleteFile(FileName, SafeDir, SafeDelLog, SafeDelRC, NULL, NULL,
                         -1, TRUE);

    switch(ccode)
    {
      case 0 : printf("undelete: %s: successfully undeleted\n",
                     FileName);
               break;

      case 1 : fprintf(stderr,
               "undelete: %s: not found in .safedelete.log\n",
               FileName);
               break;

      case 2 : fprintf(stderr,
               "undelete: safedelete file containing %s not found\n",
               FileName);
               break;

      case 3 : fprintf(stderr,
	       "undelete: undelete failed, reason unknown\n");
	       break;
	
      default : break;
    }

    return ccode;  
  }
  else
  {
  
/* No file name was provided on the command line.
   First see if any files can be undeleted.  If there are,
   show them on the screen and let the user choose
   which file(s) to undelete.
*/

    ccode=stat(SafeDelLog, &FileInfo);
    if(ccode != 0 || FileInfo.st_size < 30)
    {
      fprintf(stderr, "undelete: no files to undelete\n");
      return 1;
    }

/* To make the list of files easier to read, we
   first sort them into alphabetic sequence.
   Then we copy the sorted entries back into the
   original log file and delete the sorted copy.
*/

    printf("\nSorting entries, please wait...\n");
    wrkptr=palloc=malloc(FileInfo.st_size);
    flog=fopen(SafeDelLog, "r");
    fread(wrkptr, FileInfo.st_size, 1, flog);
    fclose(flog);

    NumFiles=j=0;
    while(j < FileInfo.st_size)
    {
      memcpy(&k, wrkptr, 2);  /* get length of this record */
      FilePtr[NumFiles]=wrkptr;
      NumFiles++;
      if(NumFiles > MAX_FILES)
      {
        fprintf(stderr, "\nMore than %d entries found, unable to perform sort\n", 
                MAX_FILES);
        fprintf(stderr, "Press any key to continue...\n");
        getchar();
        break;
      }
      j+=k;
      wrkptr+=k;
    }

    if(NumFiles > 1 && NumFiles <= MAX_FILES)
    {
      SortLog(FilePtr, 0, NumFiles-1);

      flog=fopen(SafeDelLog, "w");
      for(i=0; i < NumFiles; i++)
      {
        memcpy(&k, FilePtr[i], 2);
        fwrite(FilePtr[i], k, 1, flog);
      }
      fclose(flog);
    }

/* Initialize the ncurses screen */

    if(!(stdscr=initscr()))
    {
      fprintf(stderr, "undelete: initscr failed\n");
      return 1;
    }

/* We need at least a 24x80 screen to do our thing */

    if((LINES < 24) || (COLS < 80))
    {
      fprintf(stderr, "undelete: screen too small, need 24x80, this screen only %dx%d\n", LINES, COLS);
      return 1;
    }

/* Set the end of the file list window */

    ListEnd=LINES-8;

/* Create the background window */

    if(! (BackWin=newwin(LINES-1, 80, BACK_ROW, BACK_COL)))
    {
      fprintf(stderr, "undelete: can't create background window\n");
      return 1;
    }

/* Create the file list window */

    if(! (ListWin=newwin(LINES-7, 78, LIST_ROW, LIST_COL)))
    {
      fprintf(stderr, "undelete: can't create list window\n");
      return 1;
    }

/* Create the verify window */

    if(! (VerifyWin=newwin(1, 80, BACK_ROW+(LINES-1), BACK_COL)))
    {
      fprintf(stderr, "undelete: can't create verify window\n");
      return 1;
    }

/* Create the information background window */

    if(! (BackInfoWin=newwin(11, 42, BACK_ROW+7, BACK_COL+19)))
    {
      fprintf(stderr, "undelete: can't create info background window\n");
      return 1;
    }

/* Create informataion window */

    if(! (InfoWin=newwin(9, 40, BACK_ROW+8, BACK_COL+20)))
    {
      fprintf(stderr, "undelete: can't create information window\n");
      return 1;
    }

/* Create warning window */

    if(! (WarnWin=newwin(3, 64, BACK_ROW+10, BACK_COL+8)))
    {
      fprintf(stderr, "undelete: can't create warning window\n");
      return 1;
    }

/* Create new name window */

    if(! (NewnameWin=newwin(5, 64, BACK_ROW+10, BACK_COL+8)))
    {
      fprintf(stderr, "undelete: can't create newname window\n");
      return 1;
    }

/* See if we can use colors.  If so, set them up.
   If not, then it could be a vt100 or xterm so we
   use black & white with various attributes.
*/

    start_color();
    if(has_colors())
    {
      init_pair(1, COLOR_WHITE, COLOR_BLUE);
      init_pair(2, COLOR_WHITE, COLOR_BLACK);
      init_pair(3, COLOR_WHITE, COLOR_RED);
      wattrset(BackWin, COLOR_PAIR(1));
      wattrset(ListWin, COLOR_PAIR(2));
      wattrset(VerifyWin, COLOR_PAIR(3));
      wattrset(BackInfoWin, COLOR_PAIR(1));
      wattrset(InfoWin, COLOR_PAIR(1));
      wattrset(WarnWin, COLOR_PAIR(3));
      wattrset(NewnameWin, COLOR_PAIR(1));
    }
    else
    {
      wattrset(BackWin, A_NORMAL);
      wattrset(ListWin, A_BOLD);
      wattrset(VerifyWin, A_BOLD);
      wattrset(BackInfoWin, A_NORMAL);
      wattrset(InfoWin, A_BOLD);
      wattrset(WarnWin, A_BOLD);
      wattrset(NewnameWin, A_NORMAL);
    }

/* Clear the windows to blanks */

    werase(BackWin);
    werase(ListWin);
    werase(VerifyWin);
    werase(BackInfoWin);
    werase(NewnameWin);

/* Draw some borders and lines in the various windows */

    wborder(BackWin, ACS_VLINE, ACS_VLINE, ACS_HLINE, ACS_HLINE,
            ACS_ULCORNER, ACS_URCORNER, ACS_LLCORNER, ACS_LRCORNER);

    wmove(BackWin, 2, 0);
    whline(BackWin, ACS_HLINE, 80);
    mvwaddch(BackWin, 2, 0, ACS_LTEE);
    mvwaddch(BackWin, 2, 79, ACS_RTEE);

    wmove(BackWin, LINES-4, 0);
    whline(BackWin, ACS_HLINE, 80);
    mvwaddch(BackWin, LINES-4, 0, ACS_LTEE);
    mvwaddch(BackWin, LINES-4, 79, ACS_RTEE);

    wborder(BackInfoWin, ACS_VLINE, ACS_VLINE, ACS_HLINE, ACS_HLINE,
            ACS_ULCORNER, ACS_URCORNER, ACS_LLCORNER, ACS_LRCORNER);
    mvwaddstr(BackInfoWin, 10, 10, " Press F3 to return ");

    wborder(WarnWin, ACS_VLINE, ACS_VLINE, ACS_HLINE, ACS_HLINE,
            ACS_ULCORNER, ACS_URCORNER, ACS_LLCORNER, ACS_LRCORNER);

    wborder(NewnameWin, ACS_VLINE, ACS_VLINE, ACS_HLINE, ACS_HLINE,
            ACS_ULCORNER, ACS_URCORNER, ACS_LLCORNER, ACS_LRCORNER);
    mvwaddstr(NewnameWin, 1, 2, "Enter new file/path name:");
    mvwaddstr(NewnameWin, 4, 12, " Press Enter to accept or F3 to return ");


/* Now add the headings */

    mvwaddstr(BackWin, 1, 4, "File Name");
    mvwaddstr(BackWin, 1, 66, "Date Deleted");
    mvwaddstr(BackWin, LINES-3, 9, "Quit     ");
    waddstr(BackWin, "Up     ");
    waddstr(BackWin, "Down     ");
    waddstr(BackWin, "Select     ");
    waddstr(BackWin, "Remove     ");
    waddstr(BackWin, "Info     ");
    waddstr(BackWin, "Newname");

/* Put the .safedelete.log entries in the ListWin window */

    SelFile=FirstFile=0;
    FillListWin(ListWin, FilePtr, NumFiles, &SelFile, FileLine,
                &FirstFile);

/* Terminal options:
   don't echo the user-typed characters, don't wait for
   input, make cursor invisible, don't wait for carriage
   return while waiting for input and enable the keypad.
*/

    noecho();
    nodelay(stdscr, FALSE);
    cbreak();
    curs_set(FALSE);
    keypad(stdscr, TRUE);
    keypad(InfoWin, TRUE);
    keypad(WarnWin, TRUE);
    keypad(NewnameWin, TRUE);
    wtimeout(InfoWin, 0);
    wtimeout(NewnameWin, 0);

/* Display the screen images */

    move(0, 0);
    wnoutrefresh(BackWin);
    wnoutrefresh(ListWin);
    wnoutrefresh(VerifyWin);
    doupdate();

/* Get input from the user and process it */

    while(NotDone)
    {
      i=getch();
      if(i < KEY_MIN || i > KEY_MAX)
        i&=255;
      switch(i)
      {
        case KEY_UP :
        case 'u' :
        case 'U' : werase(VerifyWin);
                  move(0, 0);
                  wrefresh(VerifyWin);
                  if(SelFile <= 0)
                    break;
                  mvwaddch(ListWin, FileLine[SelFile], 1, ' ');
                  SelFile--;
                  if(SelFile < FirstFile)
		  {
                    FirstFile--;
                    FillListWin(ListWin, FilePtr, NumFiles,
                                &SelFile, FileLine, &FirstFile);
		  }
                  else
		    mvwaddch(ListWin, FileLine[SelFile], 1, '>');
                  move(0, 0);
                  wrefresh(ListWin);
                  break;

        case KEY_DOWN :
        case 'd' :
        case 'D' : werase(VerifyWin);
                  move(0, 0);
                  wrefresh(VerifyWin);

/* If the last line in the window contains a partial filename,
   we just scroll down 1 filename to show the complete
   filename of that file.  We leave the '>' on the same filename.
   We do this only for the last filename in the list.
*/

                  if(SelFile == NumFiles-1)
		  {
                    while(PartialLastLine)
		    {
                      mvwaddch(ListWin, FileLine[SelFile], 1, ' ');
                      FirstFile++;
                      FillListWin(ListWin, FilePtr, NumFiles,
                                  &SelFile, FileLine, &FirstFile);
                      mvwaddch(ListWin, FileLine[SelFile], 1, '>');
		    }
		  }
                  else
		  {
                    mvwaddch(ListWin, FileLine[SelFile], 1, ' ');
                    SelFile++;
                    if(FileLine[SelFile] > ListEnd)
		    {
                      while(PartialLastLine)
		      {
                        FirstFile++;
                        FillListWin(ListWin, FilePtr, NumFiles,
                                    &SelFile, FileLine, &FirstFile);
                        mvwaddch(ListWin, FileLine[SelFile], 1, ' ');
		      }
                      FirstFile++;
                      FillListWin(ListWin, FilePtr, NumFiles,
                                  &SelFile, FileLine, &FirstFile);
		    }
                    else
                      mvwaddch(ListWin, FileLine[SelFile], 1, '>');
		  }
                  move(0, 0);
                  wrefresh(ListWin);
                  break;

        case 's' :
        case 'S' : werase(VerifyWin);
                  move(0, 0);
                  wrefresh(VerifyWin);
                  if(SelFile < 0)
                    break;
                  wrkptr=FilePtr[SelFile]+2;

/* wrkptr points to the fully qualified filename.
   Make sure we have write access to the directory
   where the file is going to be placed in.
*/

                  j=ParseFileName(wrkptr);
                  *(wrkptr+j)='\0';
                  if(access(wrkptr, W_OK) != 0)
                  {
                    mvwaddstr(VerifyWin, 0, 5,
                              "Error getting write access to the directory");
                    move(0, 0);
                    wrefresh(VerifyWin);
                    *(wrkptr+j)='/';
                    break;
                  }
                  else
                   *(wrkptr+j)='/';

/* See if the file already exists.  If so, see
   if the user wants to overlay it
*/

                  if(lstat(wrkptr, &FileInfo) == 0 ||
                     stat(wrkptr, &FileInfo) == 0)
		  {
                    touchwin(WarnWin);
                    mvwaddstr(WarnWin, 1, 1,
                    "            File already exists, overlay it? (y|n)            ");
                    wnoutrefresh(WarnWin);
                    move(0, 0);
                    doupdate();

                    c=getch();

                    touchwin(ListWin);
                    wnoutrefresh(ListWin);
                    move(0, 0);
                    doupdate();

                    if(c != 'y' && c != 'Y')
                      break;

		    ccode=UndeleteFile(wrkptr, SafeDir, SafeDelLog, SafeDelRC,
                                       NULL, &FileInfo,
                                       SelFile, TRUE);
		  }
                  else
                    ccode=UndeleteFile(wrkptr, SafeDir, SafeDelLog, SafeDelRC,
                                       NULL, NULL,
                                       SelFile, TRUE);
	  
                  switch(ccode)
		  {
                   case 0 : if(lstat(wrkptr, &FileInfo) != 0)
                              stat(wrkptr, &FileInfo);
                            tm=localtime(&FileInfo.st_mtime);
                            strftime(WorkBuf+80, 9, "%m/%d/%y", tm);
                            sprintf(WorkBuf,
                            "File undeleted   size: %li   last modified: %s",
                            FileInfo.st_size, WorkBuf+80);
                            mvwaddstr(VerifyWin, 0, 5, WorkBuf);

/* The UndeleteFile routine physically rewrites the .safedelete.log
   so mimic what it does by removing the filename from the list.
*/

                            for(i=SelFile; i < NumFiles; i++)
                              FilePtr[i]=FilePtr[i+1];
                            NumFiles--;
                            if(SelFile >= NumFiles)
                              SelFile--;

/* Now refresh the list window with the new list */

                            FillListWin(ListWin, FilePtr, NumFiles,
                                        &SelFile, FileLine, &FirstFile);
                            move(0, 0);
                            wrefresh(ListWin);
                            break;

		   case 1 : mvwaddstr(VerifyWin, 0, 5,
                                      "File not found in .safedelete.log");
                            break;

		   case 2 : mvwaddstr(VerifyWin, 0, 5,
                                      "Safedelete file not found, undelete failed");
                            break;

		   case 3 : mvwaddstr(VerifyWin, 0, 5,
				      "Unknown error, file not undeleted");
		            break;
		     
                  default : break;

		  }
                  move(0, 0);
                  wrefresh(VerifyWin);
                  break;

        case 'n' :
        case 'N' : werase(VerifyWin);
                  move(0, 0);
                  wrefresh(VerifyWin);
                  if(SelFile < 0)
                    break;
                  wrkptr=FilePtr[SelFile]+2;

                  wrkptr2=wrkptr+strlen(wrkptr)+1;
                  wrkptr2+=strlen(wrkptr2)+1;
                  if(*wrkptr2 == 127)
                    wrkptr2++;
                  Orig=(file_struct *) wrkptr2;
                  if(Orig->Version == 0)
		  {
                    touchwin(WarnWin);
                    mvwaddstr(WarnWin, 1, 1,
                    " Entry in old format, cannot rename (press any key to return) ");
                    wnoutrefresh(WarnWin);
                    move(0, 0);
                    doupdate();

                    c=getch();

                    touchwin(ListWin);
                    wnoutrefresh(ListWin);
                    move(0, 0);
                    doupdate();
                    break;
		  }

                  RefreshWin=TRUE;

/* Loop till user enters valid filename or presses F3 */

                  c='n';
                  while(c != 'y')
		  {

/* Get new filename or path(+filename) from user */

                    j=GetInput(NewnameWin, WorkBuf2, FILE_NAME_LEN, 3, 2, 60,
                               RefreshWin);

                    touchwin(ListWin);
                    wnoutrefresh(ListWin);
                    werase(VerifyWin);
                    move(0, 0);
                    doupdate();
                    RefreshWin=FALSE;

                    strcpy(NewFileName, WorkBuf2);

                    if(j > 0)
                    {

/* See if user specified ~/ for directory */

                      if(*NewFileName == '~')
                      {
                        strcpy(NewFileName, HomeDir);
                        strcat(NewFileName, WorkBuf2+1);
                      }

/* See if user specified absolute path or just a file name */

                      if(*NewFileName != '/')
                      {
                        j=ParseFileName(wrkptr);
                        strncpy(NewFileName, wrkptr, j+1);
                        strcpy(NewFileName+j+1, WorkBuf2);
                      }

/* Check status of path or filename */

                      if(stat(NewFileName, &NewInfo) != 0)
               	      {
                        j=ParseFileName(NewFileName);
                        *(NewFileName+j)='\0';
                        ccode=stat(NewFileName, &NewInfo);
                        if(ccode != 0 || ! S_ISDIR(NewInfo.st_mode))
  		        {
                          mvwaddstr(VerifyWin, 0, 5,
                                    "Invalid directory/filename, try again");
                          move(0, 0);
                          wrefresh(VerifyWin);
                          continue;
	  	        }
		      }

/* If user entered just a directory, append the filename to it */

                      else
                      {
                        if(S_ISDIR(NewInfo.st_mode))
                        {
                          j=ParseFileName(wrkptr);
                          strcat(NewFileName, wrkptr+j);
                        }
                        j=ParseFileName(NewFileName);
                        *(NewFileName+j)='\0';
                      }

/* See if we have write access to the directory */

                      if(access(NewFileName, W_OK) != 0)
                      {
                        mvwaddstr(VerifyWin, 0, 5,
                                  "Error getting write access to the directory");
                        move(0, 0);
                        wrefresh(VerifyWin);
                        continue;
                      }

                      *(NewFileName+j)='/';

/* See if new file already exists */

                      if(lstat(NewFileName, &FileInfo) == 0 ||
                         stat(NewFileName, &FileInfo) == 0)
  		      {
                        touchwin(WarnWin);
                        mvwaddstr(WarnWin, 1, 1,
                        "          New file already exists, overlay it? (y|n)          ");
                        wnoutrefresh(WarnWin);
                        move(0, 0);
                        doupdate();

                        c=wgetch(WarnWin);
 
                        touchwin(ListWin);
                        wnoutrefresh(ListWin);
                        move(0, 0);
                        doupdate();

                        if(c != 'y' && c != 'Y')
                          continue;

		        ccode=UndeleteFile(wrkptr, SafeDir, SafeDelLog, SafeDelRC,
                                           NewFileName, &FileInfo,
                                           SelFile, TRUE);
		      }
                      else
		      {
                        c='y';
                        ccode=UndeleteFile(wrkptr, SafeDir, SafeDelLog, SafeDelRC,
                                           NewFileName, NULL,
                                           SelFile, TRUE);
		      }

                      switch(ccode)
                      {
                        case 0 : if(lstat(NewFileName, &FileInfo) != 0)
                                   stat(NewFileName, &FileInfo);
                                 tm=localtime(&FileInfo.st_mtime);
                                 strftime(WorkBuf+80, 9, "%m/%d/%y", tm);
                                 sprintf(WorkBuf,
                                 "File undeleted   size: %li   last modified: %s",
                                 FileInfo.st_size, WorkBuf+80);
                                 mvwaddstr(VerifyWin, 0, 5, WorkBuf);

/* The UndeleteFile routine physically rewrites the .safedelete.log
   so mimic what it does by removing the filename from the list.
*/

                                 for(i=SelFile; i < NumFiles; i++)
                                   FilePtr[i]=FilePtr[i+1];
                                 NumFiles--;
                                 if(SelFile >= NumFiles)
                                   SelFile--;

/* Now refresh the list window with the new list */

                                 FillListWin(ListWin, FilePtr, NumFiles,
                                            &SelFile, FileLine, &FirstFile);
                                 move(0, 0);
                                 wrefresh(ListWin);
                                 break;

  		        case 1 : mvwaddstr(VerifyWin, 0, 5,
                                           "File not found in .safedelete.log");
                                 break;

  		        case 2 : mvwaddstr(VerifyWin, 0, 5,
                                           "Safedelete file not found, undelete failed");
                                 break;

		        case 3 : mvwaddstr(VerifyWin, 0, 5,
			                   "Unknown error, file not undeleted");
		                 break;
		     
                       default : break;

		      }
                      move(0, 0);
                      wrefresh(VerifyWin);
		    }
                    else
                      c='y';
		  }
                  break;

	case 'r' :
	case 'R' : werase(VerifyWin);
                  move(0, 0);
                  wrefresh(VerifyWin);
                  if(SelFile < 0)
                    break;

/* Make sure the user REALLY wants to remove this entry */

                  mvwaddstr(WarnWin, 1, 1,
                  " Cannot undelete file after entry is removed, proceed? (y|n)  ");

                  touchwin(WarnWin);
                  wnoutrefresh(WarnWin);
                  move(0, 0);
                  doupdate();

                  c=getch();

                  touchwin(ListWin);
                  wnoutrefresh(ListWin);
                  move(0, 0);
                  doupdate();

                  if(c != 'y' && c != 'Y')
                    break;

                  wrkptr=FilePtr[SelFile]+2;

/* Call UndeleteFile to just remove the entry without undeleting the file */

                  ccode=UndeleteFile(wrkptr, SafeDir, SafeDelLog, SafeDelRC, NULL,
                                     NULL, SelFile, FALSE);

                  switch(ccode)
		  {
		    case 0 : mvwaddstr(VerifyWin, 0, 5,
                                      "Entry removed from .safedelete.log");

/* The UndeleteFile routine physically rewrites the .safedelete.log
   so mimic what it does by removing the filename from the list.
*/

                            for(i=SelFile; i < NumFiles; i++)
                              FilePtr[i]=FilePtr[i+1];
                            NumFiles--;
                            if(SelFile >= NumFiles)
                              SelFile--;

/* Now refresh the list window with the new list */

                            FillListWin(ListWin, FilePtr, NumFiles,
                                        &SelFile, FileLine, &FirstFile);
                            move(0, 0);
                            wrefresh(ListWin);
                            break;

	            case 1: mvwaddstr(VerifyWin, 0, 5,
                                      "File not found in .safedelete.log");
                           break;

		   case 2 :
		   case 3 :
		  default : werase(VerifyWin);
                           break;
		  }
                  move(0, 0);
                  wrefresh(VerifyWin);
                  break;

	case 'i' :
       	case 'I' : werase(VerifyWin);
                  move(0, 0);
                  wrefresh(VerifyWin);
                  if(SelFile < 0)
                    break;

                  GetFileInfo(FilePtr[SelFile], WorkBuf);

                  werase(InfoWin);

                  wrkptr=WorkBuf;
                  for(i=1; i < 8; i++)
                  {
                    if(strlen(wrkptr) > 38)
		    {
                      *(wrkptr+38)='\0';
                      mvwaddstr(InfoWin, i, 1, wrkptr);
                      *(wrkptr+38)='a';
		    }
                    else
                      mvwaddstr(InfoWin, i, 1, wrkptr);
                    wrkptr+=strlen(wrkptr)+1;
                  }

                  touchwin(BackInfoWin);
                  wnoutrefresh(BackInfoWin);
                  wnoutrefresh(InfoWin);
                  move(0, 0);
                  doupdate();

                  do
		  {
                    j=getch();
                    if(j < KEY_MIN || j > KEY_MAX)
                      j&=255;
		  }
                  while(j != KEY_F(3) && j != 27);

                  touchwin(ListWin);
                  wnoutrefresh(ListWin);
                  move(0, 0);
                  doupdate();
                  break;

        case 'q' :
        case 'Q' : NotDone=FALSE;
                  break;

        default : break;
      }
    }

/* Clear the screen and delete the windows */

    free(palloc);

    wattrset(ListWin, A_NORMAL);
    wattrset(BackWin, A_NORMAL);
    wattrset(VerifyWin, A_NORMAL);

    werase(ListWin);
    werase(BackWin);
    werase(VerifyWin);

    wrefresh(ListWin);
    wrefresh(BackWin);
    wrefresh(VerifyWin);

    delwin(ListWin);
    delwin(BackWin);
    delwin(VerifyWin);
    delwin(NewnameWin);
    delwin(WarnWin);
    delwin(InfoWin);
    delwin(BackInfoWin);

    curs_set(TRUE);
    endwin();
  }
  return 0;
}
