/****************************************************************************
 *
 *	Program:	list.c
 *	Author:		Marc van Kempen
 *	desc:		navigating through directories and 
 *			displaying of files in various fileformats.
 *
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/param.h>
#include <dirent.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>

#include "curses.h"
#include "mlist.h"
#include "scroll.h"
#include "config.h"
#include "tar.h"

/*****************************************************************************
 *
 *	globals
 *
 *****************************************************************************/

int 	_screenx, 		/* width of the screen */
	_screeny; 		/* height of the screen */

void 	*_sort_func = DEFSORTF;	/* default sort function */

extern char version[];		/* current list version */

char *_help[] = {
  "             Helpscreen",
  "             ---------- ",
  "",
  "Directory view",
  "--------------",
  "",
  "use <i, j, k, m> or arrowkeys to move one step through list",
  "use <n, u> or <Page Down, Page Up> to move one page through list",
  "<e>  : edit file",
  "<d>  : delete file",
  "<c>  : copy file",
  "<p>  : enter new path to display",
  "<q>, <x> : to stop",
  "<^A> : for the beginning of the list",
  "<^E> : for the end of the list",
  "<^R> : redraw the screen",
  "",
  "Command line editing",
  "--------------------",
  "<^A> : beginning of the line",
  "<^E> : end of the line",
  "<DEL>: delete character under the cursor",
  "<BS> : delete character left from the cursor",
  "arrowkeys <left> and <right> to move through the line",
  NULL,
};

 
/*****************************************************************************
 *
 *	main
 *
 *****************************************************************************/

void
main(int argc, char **argv)
{
	DirList		*dir;			/* array of pointers to dirents */
	int		ndirs;

	if (argc > 1) {
		if (chdir(argv[1])) {
			perror("directory error :");
			exit(-1);
		}
	}

	InitScreen();		/* initialize the screen */
	Raw(ON);		/* set the keyboard in raw mode */

	signal(SIGWINCH, (void *) catch_resize_signal);
	
	get_dir(".", &dir, &ndirs);	/* read directory */
	choose_list(dir, ndirs);	/* choose from the directory and display files */

	CloseScreen();		/* restore the tty settings */

	return;
}


/*****************************************************************************
 *
 *	functions
 *
 *****************************************************************************/

void
catch_resize_signal(void)
/*
 *	desc:	catch resize signal and reset the screen dimensions
 *	pre:	a SIGWINCH signal has been caught
 *	post:	_screenx is the new screenwidth
 *		_screeny is the new screenheight
 *		a "Z" has been fed to the terminal to initiate a redraw
 */
{
        struct winsize  win;

	get_screensize(&win);
	_screenx = win.ws_col == 0 ? 80 : win.ws_col;
	_screeny = win.ws_row == 0 ? 24 : win.ws_row;
	ioctl(0, TIOCSTI, "Z");		/* feed a Z to the terminal (simulate a keypress) */
	InitScreen();			/* reinitialize the screen */
	Raw(ON);
}/* catch_resize_signal() */

long
numlen(long num)
/*
 *	desc:	returns the length of the number in digits
 */
{
	char 	numstr[512];

	sprintf(numstr, "%ld", num);
	return(strlen(numstr));
} /* numlen() */

void 
display_list(DirList *dir, int ndirs, int bl, int el, 
		int screenx, int screeny)
/*
 *	desc:	display directory-list
 *	pre:	<dir> contains <ndirs> directory-entries
 *		0 < bl <= el < ndirs
 *		dimension of screen = (screenx X screeny)
 *	post:	list has been displayed from dir[bl] to dir[el]
 */
{
	int		i, j, ncols, nnames, x, n, nrows, lf, ll;
	long		d;
	char		cwd[MAXPATHLEN];
	char		*line, tmpstr[50];

	line = (char *) malloc( screenx );	

	nnames = screeny - 2;

	if (el > ndirs) {		/* check boundary */
		el = ndirs;
	}

	if ((el - bl) / nnames * COLWIDTH > screenx) {
		fprintf(stdout, "display_list: Error in coordinates \n");
		fprintf(stdout, "el=%d, bl=%d, nnames=%d, COLWIDTH=%d, screenx=%d\n",
			el, bl, nnames, COLWIDTH, screenx);
		CloseScreen();
		exit(1);
	}

	ClearScreen();
	MoveCursor(0, 0);
	getwd(cwd);
	Write_to_screen("%s | dir=%s", 2, version, cwd);
	memset(line, ' ', screenx-1);	/* reset the line to spaces */
	line[screenx] = '\0';		/* terminate the line */
	ncols = (int) (el - bl - 1) / nnames + 1;
	nrows = min(nnames, ndirs);
	for (i=0; i < nnames; i++) {
		memset(line, ' ', screenx-1);	/* reset the line to spaces */
		line[screenx] = '\0';		/* terminate the line */
		for (j=0; j<ncols; j++) {
			n = i + bl + j * nnames;
			x = 1 + j * COLWIDTH;
			if (n < ndirs) {
				memcpy(	line+x, 
					dir[n].filename, 
					strlen(dir[n].filename));
       	         		if (!(dir[n].link)) {
       	         			if (!(S_ISDIR(dir[n].filestatus.st_mode))) {
						sprintf(tmpstr, "%d", dir[n].filestatus.st_size);
						memcpy(	line+x+COLWIDTH-4-strlen(tmpstr), 
							tmpstr,
							strlen(tmpstr));
					} else {
						memcpy(line+x+COLWIDTH-4-5, "<DIR>", 5);
					}
				} else {
					lf = strlen(dir[n].filename);
					ll = min(strlen(dir[n].linkname), COLWIDTH - 8 - lf);
					memcpy(line+x+COLWIDTH-4-ll, 
					 	dir[n].linkname + strlen(dir[n].linkname) - ll, 
					 	ll);
					if (strlen(dir[n].linkname) != ll) {
						memset(line + x + COLWIDTH - 4 - ll, 
							'.', 3);
					}
					memset(line + x + lf + 1,
						'-',
						COLWIDTH - 7 - lf - ll);
					line[x+COLWIDTH - 6 - ll] = '>';
				}
				if (j < ncols-1) memcpy(line+x+COLWIDTH-2, "|", 1);
			} else {
				/*ReadCh();*/
				line[x] = 0;
			}
		}
                MoveCursor(i+1, 0);
		Write_to_screen("%s", 1, line);
	}
	
	/* Calculate the size of the directory */
	d = 0;
	for (i=0; i<ndirs; i++) {
		if (!S_ISDIR(dir[i].filestatus.st_mode) && !dir[i].link) {
			d += dir[i].filestatus.st_size;
		}
	}
	MoveCursor(screeny-1, 0);
	Write_to_screen("Files=%d Used=%dK | s=sort c=copy d=del e=edit ?=help a=make r=run p=path", 2, ndirs, (int) d/1024);
	
	return;	

#ifdef 0
	d = 0;							/* total space of files */
	for (i=0; i < el-bl; i++) {
		y = (int) i % nnames;
		x = (int) (i / nnames) * COLWIDTH + 1;
		MoveCursor(y+1, x);
		Write_to_screen("%s", 1, dir[i+bl].filename);
		if (!(dir[i+bl].filestatus.st_mode & S_IFDIR)) {
			MoveCursor(y+1, x+COLWIDTH-4-numlen(dir[i+bl].filestatus.st_size));
			Write_to_screen(" %d", 1, dir[i+bl].filestatus.st_size);
			d += dir[i+bl].filestatus.st_size;
		} else {
			MoveCursor(y+1, x+COLWIDTH-3-5);
			Write_to_screen("<DIR>",0);
		}
	}
	MoveCursor(screeny - 1, 0);
	Write_to_screen("Files=%d Used=%dK | s=sort c=copy d=del e=edit ?=help a=make r=run p=path", 2, ndirs, (int) d/1024);

	/* draw column separators */
	ncols = (int) (el - bl - 1) / nnames + 1;
	for (j=0; j < ncols-1; j++) {
		for (i=0; i < nnames; i++) {
			y = (int) i;
			x = (int) (j + 1) * COLWIDTH - 1;
			MoveCursor(y+1, x);
			Write_to_screen("|", 0);
		}
	}
		
	return;
#endif
} /* display_list() */

void
choose_list(DirList *dir, int ndirs)
/* 
 *	desc:	display the dir-list and choose a file to display
 *	pre:	<dirent> contains <ndirs> directory-entries
 *	post:	true
 */
{
	int		ncols, nnames, nn;	
	int		bl, col, row, listptr, finished, ch;
	struct winsize	win;
	int		del_y_n, oldrow, oldcol, redraw_list, oldlistptr;
	char		curdir[MAXPATHLEN],	 
			olddir[MAXPATHLEN],
			newdir[MAXPATHLEN];
	struct stat	cwd_stat1, cwd_stat2;	
	FILE            *f;
	
	get_screensize(&win);
	_screenx = win.ws_col == 0 ? 80 : win.ws_col;
	_screeny = win.ws_row == 0 ? 24 : win.ws_row;

	/* <ncols> is the # of colums to display */
	/* since we have a screendimension of (screenx X screeny), */
 	/* and a minimal columnwidth of, COLWIDTH, calculate the # */
 	/* of columns possible: ncols = screenx div COLWIDTH */
	/* reserve 2 lines for info, so screeny - 2 directory-entries */
	/* per column */
	/* nnames is the # of names per column */
	ncols = (int) _screenx / COLWIDTH;
	nnames = min(_screeny - 2, ndirs);

	/* <bl> is the beginning of the list, the end of the list is */
	/* calculated from <bl> plus the possible number of names on */
	/* the screen */
	bl = 0;

	display_list(dir, ndirs, bl, bl+ncols*nnames, _screenx, _screeny);	
	MoveCursor(1, 1);
	StartInverse();
	Write_to_screen("%s", 1, dir[0].filename);
	EndInverse();
				
	/* choose from the list */

	col = 0; oldcol = 0;
	row = 1; oldrow = 1;
	listptr = 0; oldlistptr = 0;
	finished = FALSE;
	while (!finished) {
		redraw_list = FALSE;
		ch = getkey();
		/* Save the ttysettings, so external programs dont fuck this one up */
		SaveState(1);
		/* get the modification time of the directory */
		/* to optimize for directory rereads */
		stat(".", &cwd_stat1);
		switch(ch) {
			case 18 : /* CTRL-R or Z */
			case 'Z':
				/* Redraw the screen and recalculate the screenparameters */
				get_dir(".", &dir, &ndirs);
				ncols = (int) _screenx / COLWIDTH;
				nnames = min(_screeny - 2, ndirs);
				bl = 0;
				listptr = 0; oldlistptr = 0;
				col = 0; oldcol = 0;
				row = 1; oldrow = 1;
				redraw_list = TRUE;
				break;
			case KEYUP : /* move up */
			case UPARROW :
				if (listptr > 0) {
					if ((row == 1) && (col == 0)) {
						bl--; 
						redraw_list = TRUE;
					} else {
					if ((row == 1) && (col > 0)) {
						row = nnames;
						col--;
					} else {
						row--;
					} }
					listptr--;
				}
				break;
			case KEYLEFT : /* move left */
			case LEFTARROW :
				if (col > 0) {
					col--;
					listptr -= nnames;
				}
				break;
			case KEYRIGHT : /* move right */
			case RIGHTARROW :
				if ((col < ncols-1) && (listptr + nnames < ndirs)) {
					col++;
					listptr += nnames;
					if (listptr > ndirs) listptr = ndirs;
				}
				break;
			case KEYDOWN : /* move down */
			case DNARROW :
				if (listptr < ndirs-1) {
					if (row < nnames) { /* just move down */
						row++;
					} else { /* move to next column */
					if ((row==nnames) && (col < ncols-1)) {
						row = 1;
						col++;
					} else { /* scroll the screen */
					if ((row==nnames) && (col==ncols-1)) {
						bl++;
						redraw_list = TRUE;
					} } }
					listptr++;
				}
				break;
			case 'n' : /* page down */
			case 'N' :
			case PGDN: 
				if (listptr + ncols * nnames < ndirs) {
					/* we can move one full page down */
					listptr += ncols * nnames - 
						   col * nnames - row + 1;
					bl += ncols * nnames;
					redraw_list = TRUE;
					row = 1; oldrow = 1;
					col = 0; oldcol = 0;
				} else { 
					/* We can't move a full page down */
					/* check if there are enough items to show */
					if (bl + nnames * ncols <  ndirs - 1) {
						listptr = ndirs - 1;
						bl += nnames * ncols;
						col = (int) (listptr - bl) / nnames; oldcol = col;
						row = (int) (listptr - bl) % nnames + 1; oldrow = row;
						redraw_list = TRUE;
					} else {
						listptr = ndirs - 1;
						col = (int) (listptr - bl) / nnames;
						row = (int) (listptr - bl) % nnames + 1;
					}
				}
				break;

			case 'u' : /* page up */
			case 'U' : 
			case PGUP:
				if (listptr - ncols * nnames >= 0) {
					/* move one full page up */
					listptr = listptr - ncols * nnames - col * nnames - row + 1;
					bl -= ncols * nnames;		
					if (bl < 0) {
						bl = 0;
						listptr = 0;
					}
					row = 1; oldrow = 1;
					col = 0; oldcol = 0;
					redraw_list = TRUE;
				} else {
					/* move to top of list */
					if (bl != 0) redraw_list = TRUE;
					listptr = 0;
					bl = 0;
					row = 1; 
					col = 0; 
				}
				break;

			case 1 : /* Begin of list (^A) or (HOME) */
			case HOME :
				if (bl != 0) {
					listptr = 0;
					bl = 0;
					col = 0; oldcol = col;
					row = 1; oldrow = row;
					redraw_list = TRUE;
				} else {
					listptr = 0;
					bl = 0; 
					oldcol = col; col = 0; 
					oldrow = row; row = 1;
				}
				break;

			case 5 : /* End of list (^E) or (End) */
			case END:
				if (ndirs - bl - 1 >= ncols * nnames) {
					redraw_list = TRUE;
				}
				oldcol = col;
				oldrow = row;
				if (ndirs < ncols * nnames) {
					/* less than one page of entries */
					listptr = ndirs - 1;
					bl = 0; 
					col = (int) ((ndirs - 1) / nnames);
					row = (int) ((ndirs - 1) % nnames + 1);
				} else {
					/* more than one page of entries */
					listptr = ndirs - 1;
					bl = listptr - ncols * nnames + 1;
					col = ncols - 1;
					row = nnames;
				}
				if (redraw_list) {
					oldcol = col;
					oldrow = row;
				}
				break;
		
			case '.' : /* decrease directory level */
				getcwd(olddir, MAXPATHLEN);	/* save the current dir */
				chdir("..");
				get_dir(".", &dir, &ndirs);
				listptr = search_dir(olddir, dir, ndirs);
				
				/* calculate new offset on screen */
				/* this should be at about the middle */
				/* of the screen */
				nn = (int) ((ndirs < nnames * ncols) ? 
					(ndirs / 2) : 
					(nnames * ncols / 2));
				nnames = min(_screeny - 2, ndirs);
				if (listptr > nn) {
					bl = listptr - nn;
				} else {
					bl = 0;
				}
				row = (listptr - bl) % (_screeny - 2) + 1; 
				oldrow = row;
				col = (int) (listptr - bl) / (_screeny - 2); 
				oldcol = col;
				break;

			case '/' : /* goto the root directory */
			case '~' : /* goto the $(HOME) directory */
				if (ch == '/') {
					newdir[0] = ch;
					newdir[1] = 0;
				}
				if (ch == '~') {
					strcpy(newdir, getenv("HOME"));
				}
				chdir(newdir);
				get_dir(".", &dir, &ndirs);
				row = 1; 
				oldrow = row;
				col = 0; 
				oldcol = col;
				break;

			case '\n': /* CR, display the file */
				getcwd(olddir, MAXPATHLEN);	/* save the current dir */
   				if (process_file(&dir, &ndirs, listptr)) {
					/* the directory was changed 			*/
					/* first determine if the directory level was 	*/
					/* decreased. Take the size of the cwd as a 	*/
					/* measure, and compare this to olddir 		*/
					if (strlen(getcwd(curdir, MAXPATHLEN)) < strlen(olddir)) {
						/* the dir-level was decreased */
						/* set listptr to the directory you just left */
						listptr = search_dir(olddir, dir, ndirs);
						
						/* calculate new offset on screen */
						/* this should be at about the middle */
						/* of the screen */
						nn = (int) ((ndirs < nnames * ncols) ? 
							(ndirs / 2) : 
							(nnames * ncols / 2));
						nnames = min(_screeny - 2, ndirs);
						if (listptr > nn) {
							bl = listptr - nn;
						} else {
							bl = 0;
						}
						row = (listptr - bl) % (_screeny - 2) + 1; 
						oldrow = row;
						col = (int) (listptr - bl) / (_screeny - 2); 
						oldcol = col;
					} else {
						nnames = min(_screeny - 2, ndirs);
						bl = 0;
						row = 1; oldrow = 1;
						col = 0; oldcol = 0;
						listptr = 0;
					}
				}
				redraw_list = TRUE;
				break;
			case 'e' : /* start editor */
			case 'E' :
				edit_file(dir[listptr].filename);
				redraw_list = TRUE;
				break;
			case 'd' : /* delete file */
			case 'D' : 
				MoveCursor(_screeny - 1, 0);
				CleartoEOLN();
				Write_to_screen("Delete <%s> (y/n) : ", 1, dir[listptr].filename);
				del_y_n = ReadCh();
				if ((del_y_n == 'y') || (del_y_n == 'Y')) {
					if (!delete_file(dir[listptr])) {
						MoveCursor(_screeny - 1, 0);
						CleartoEOLN();
						Write_to_screen("Could not delete <%s>, press any key", 1,
							dir[listptr].filename);
						ReadCh();
					} else {
						get_dir(".", &dir, &ndirs);
						nnames = min(_screeny - 2, ndirs);
						if (listptr == ndirs) {	/* at bottom of list */
							/* same code as for moving up */
							if (listptr > 0) {
								if ((row == 1) && (col == 0)) {
									bl--; 
								} else {
								if ((row == 1) && (col > 0)) {
									row = nnames;
									col--;
								} else {
									row--;
								} }
							listptr--;
							}
						}
					}					
				} else {
					/* skip (don't delete) */
				}
				redraw_list = TRUE;
				break;
			case 'c' :
			case 'C' : /* copy file */
				copy_file(dir[listptr].filename);
				redraw_list = TRUE;
				break;
			case '?' : /* display help */
				display_help();
				redraw_list = TRUE;
				break;
			case 'a' : /* make (compiling and other shit) */
			case 'A' :
				make_file();
				redraw_list = TRUE;
				break;
			case 'r' : /* execute a user command */
			case 'R' :
				exec_usercmd();
				redraw_list = TRUE;
				break;
			case 'p' : /* enter a new path */
			case 'P' :
				if (new_path(&dir, &ndirs, _screeny)) {
					/* the directory was changed */
					nnames = min(_screeny - 2, ndirs);
					bl = 0;
					row = 1; oldrow = 1;
					col = 0; oldcol = 0;
					listptr = 0;
				}
				redraw_list = TRUE;
				break;
			case 's' : /* different sort function */
			case 'S' :
				MoveCursor(_screeny - 1, 0);
				CleartoEOLN();
				Write_to_screen("Sort by: F)ilename E)xtension D)ate S)ize", 0);
				ch = getkey();
				switch(ch) {
					case 'f':
					case 'F':
						_sort_func = dir_alphasort;
						/* dir+1 --> skip the '..'-directory */
						qsort(dir+1, ndirs-1, sizeof(DirList), _sort_func);
						redraw_list = TRUE;
						break;
					case 'e':
					case 'E':
						_sort_func = dir_extsort;
						qsort(dir+1, ndirs-1, sizeof(DirList), _sort_func);
						redraw_list = TRUE;
						break;
					case 'd':
					case 'D':
						_sort_func = dir_datesort;
						qsort(dir+1, ndirs-1, sizeof(DirList), _sort_func);
						redraw_list = TRUE;
						break;
					case 's':
					case 'S':
						_sort_func = dir_sizesort;
						qsort(dir+1, ndirs-1, sizeof(DirList), _sort_func);
						redraw_list = TRUE;
						break;
					default:
						redraw_list = TRUE;
						break;
				}
				ch = 's';
				break;
			case 'q' : /* stop */
			case 'Q' :
			case ' ' :
			case 'x' :
			case 'X' : /* stop and get out in the current directory */
				if ((ch == 'x') || (ch == 'X')) {
					getcwd(curdir, MAXPATHLEN);
				} else {
					strcpy(curdir, ".");
				}
				f = fopen("/tmp/listchdir", "w");
				if (f) {
				  fprintf(f, "%s", curdir);
				  fchmod(fileno(f), S_IRUSR|S_IWUSR|
					            S_IRGRP|S_IWGRP|
					            S_IROTH|S_IWOTH);
				  fclose(f);
				}
				finished = TRUE;
				break;

		}
		
		/* check if the directory was modified */
		stat(".", &cwd_stat2);
		if (cwd_stat1.st_ctime != cwd_stat2.st_ctime) {
			/* reread the directory to make sure to get all changes */
			get_dir(".", &dir, &ndirs);
			redraw_list = TRUE;
		}	
		if (redraw_list) {		/* redraw the whole screen */
			display_list(dir, ndirs, bl, bl + ncols * nnames, _screenx, _screeny);
			MoveCursor(row, col * COLWIDTH+1);
			oldrow = row;  oldcol = col;
			StartInverse();
			Write_to_screen("%s", 1, dir[listptr].filename);
			EndInverse();			
		} else {			/* try to be as efficient as possible */
						/* in redrawing the screen */
			MoveCursor(oldrow, oldcol * COLWIDTH+1);		/* erase '>' */
			Write_to_screen("%s", 1, dir[oldlistptr].filename);
			MoveCursor(row, col * COLWIDTH+1);		/* draw '>' at new position */
			oldrow = row;  oldcol = col;
			StartInverse();
			Write_to_screen("%s", 1, dir[listptr].filename);
			EndInverse();
		}
		oldlistptr = listptr;
		RestoreState(1);
	}
	MoveCursor(_screeny - 1, 0);		/* move cursor to bottom */
	CleartoEOLN();				/* clear to end of line */
	return;
}/* choose_list() */

int
search_dir(char *dirname, DirList *dir, int ndirs)
/*
 *	desc:	search <dirname> in <dir>, 
 *	pre:	<dirname> is the name of the directory to be found
 *		<dir> contains the directory, <ndirs> is the # of entries 
 *	post:	dir[searchdir()]->d_name = sdirname
 */
{
	int	i;
	char 	*sdirname;		/* stripped dirname */
	
	/* strip leading path from dirname */
	/* dirname always contains a '/' when the directory tree can be decreased */
	sdirname = dirname + strlen(dirname);
	while ((*sdirname != '/') && (sdirname > dirname)) sdirname--;
	if (*sdirname != '/') {
		printf("search_dir : Panic, could not find a '/' in (%s)\n", dirname);
		exit(-1);
	} else {
		sdirname++;
	}

	/* now find <sdirname> in <dir> */
	i=0;
	while ((i<ndirs) && (strcmp(sdirname, dir[i].filename))) i++;
	/* i >= ndirs || sdirname==dir[i].filename */
	if ((i >= ndirs) && (strcmp(sdirname, dir[i].filename))) {
		printf("searchdir() : Panic, I could not find (%s)\nin the directory listing\n", dirname);
		exit(-1);
	} else {
		return(i);
	}
} /* search_dir() */

int
new_path(DirList **dir, int *ndirs, int screeny)
/*
 *	desc:	enter a path to display
 *	pre:	true
 *	post:	see desc
 */
{
	char	newpath[MAXPATHLEN];
	int	cont = TRUE, dirchanged = FALSE;
	int	wait_4_key, i, j, lh, lp;
	char	home[MAXPATHLEN];

	while (cont) {
		strcpy(newpath, "");			/* initialize the new path */
		MoveCursor(screeny - 1, 0);		/* get a new path */
		CleartoEOLN();
		Write_to_screen("Enter the new path : ", 0);
		MoveCursor(screeny - 1, 21);
		ReadString(newpath, 40, '_');

		/* strip spaces from newpath */
		i = 0;
		while (newpath[i]) {
			if (newpath[i] == ' ') {
				newpath[i] = '\0';
			} else {
				i++;
			}
		}

		/* parse $(HOME) in newpath*/
		if (getenv("HOME")) {
			strcpy(home, getenv("HOME"));
			lh = strlen(home);
			lp = strlen(newpath);	
			i=0;
			while ((i<lp) && (newpath[i] != '~')) i++;
			if (newpath[i] == '~') {
				if (lp+lh < MAXPATHLEN) {
					/* shift newpath[i+1..lp-1] to 
					   newpath[i+lh..lp-1+lh-1]
					*/
					j=lp-1;
					while (j>i) {
						newpath[lh-1+j] = newpath[j];
						j--;
					}
					/* expand ~ to $HOME */
					while (j<lh) {
						newpath[i+j] = home[j];
						j++;
					}
					/* terminate the new path */
					newpath[lp+lh-1] = 0;
				} else {
					/* home does not fit, don't expand */
				        /* this will automatically generate */
				        /* an error */
				}
			}
		}
		
		dirchanged = chdir(newpath);
		if ((dirchanged == -1) && (strcmp(newpath, "") != 0)) {		
			MoveCursor(screeny - 1, 0);	/* check for input errors */
			CleartoEOLN();
			Write_to_screen("The new path is invalid, reenter", 0);
			wait_4_key = ReadCh();
			*newpath = 0;			/* make an empty string */
			cont = TRUE;
		} else {
			cont = FALSE;
		}
	}

	if (dirchanged == 0) {
		get_dir(".", dir, ndirs);
		return(TRUE);
	} else {
		return(FALSE);
	}
}/* new_path() */

void
exec_usercmd(void)
/*
 *	desc:	execute a user-specified command
 *	pre:	true
 *	post:	?
 */
{
	static char		*prog = NULL;	/* to remember the previous command */

	if (!prog) {
		prog = (char *) malloc( 255 );
		*prog = 0;
	}
	ClearScreen();
	Write_to_screen("Enter your command : ", 0);

	/* read one line from the keyboard */
	MoveCursor(0, 21);
	ReadString(prog, 50, '_');	
	ClearScreen();
	system(prog);
	Write_to_screen("\npress any key...", 0);
	ReadCh();

	return;
}/* exec_usercmd() */

void 
make_file(void)
/*
 *	desc:	make, using makefile in current directory
 *	pre:	true
 *	post:	?
 */
{
	char	prog[MAXPATHLEN];
	
	strcpy(prog, MAKE);
	ClearScreen();
	system(prog);
	Write_to_screen("\npress any key ...\n", 0);
	ReadCh();

	return;
}/* make_file() */

void
display_help(void)
/*
 *	desc:	display some help on keys, etc.
 *	pre:	true
 *	post:	true
 */
{
	int	nl, i;
	char	**p;

	ClearScreen();

	/* determine # of lines in _help[][] */
	nl = 0;
	for (p=_help; *p; p++) {
		nl++;
	}
	
	/* display help per screen */
	i=1;
	printf("%s\n", _help[0]);
	while (i<nl) {
		if (i % (_screeny-1) == 0) {
		        StartInverse();
			Write_to_screen("--More--", 0);
			EndInverse();
			ReadCh();
			printf("\n");
		}
		printf("%s\n", _help[i]);
		i++;
	}
	while (i % (_screeny-1) != 0) {
	  printf("\n");
	  i++;
	}
	StartInverse();
	printf("Press any key ... ");
	EndInverse();
	ReadCh();
		  
	return;
}/* display_help */

int 
delete_file(DirList d)
/* 
 *	desc:	delete <d.filename>
 *	pre:	<d.fname> is a valid filename, distinguish between files 
 *		directories.
 *	post:	returns "<d.filename> is deleted" -> TRUE else -> FALSE
 */
{
	if (d.filestatus.st_mode & S_IFDIR) {	/* it's a directory */
		if (rmdir(d.filename)) {
			return(FALSE);
		} else {
			return(TRUE);
		}
	} else {				/* it's a file */
		if (unlink(d.filename)) {
			return(FALSE);
		} else {
			return(TRUE);
		}
	}
}/* delete_file */

int
copy_file(char *fname)
/*
 *	desc:	copy a file
 *	pre:	<fname> is a valid filename
 *	post:	true
 */
{
	char	prog[MAXPATHLEN], dest[MAXPATHLEN];
	int	i;
	
	MoveCursor(_screeny - 1, 0);
	CleartoEOLN();
	MoveCursor(_screeny - 1, 0);
	Write_to_screen("Copy <%s> to : ", 1, fname);
	ReadString(dest, 40, '_');
	if (strlen(dest) == 0) {
		return(TRUE);
	}
	sprintf(prog, "%s %s %s", CP, fname, dest);
	i = system(prog);
	if (i) {
		MoveCursor(_screeny - 1, 0);
		CleartoEOLN();
		MoveCursor(_screeny - 1, 0);
		Write_to_screen("Could not copy <%s> to <%s>", 2, fname, dest);
		return(FALSE);
	} else {
		return(TRUE);
	}
}/* copy_file */
	
void 
edit_file(char *fname)
/*
 *	desc:	start editor for fname
 *	pre:	<fname> is a filename
 *	post:	true
 */
{
	char	*editor;
	char 	prog[255];

	/* start editor, as specified in $EDITOR, or in EDITOR */
	if (!(editor = getenv("EDITOR"))) {
		editor = EDITOR;		/* no environment variable EDITOR */
	}					/* so take constant: EDITOR (see list.h) */
	sprintf(prog, "%s %s", editor, fname);
	system(prog);

	return;	
} /* edit_file */
				
int
process_file(DirList **dir, int *ndirs, int i)
/*
 *	desc:	process the file depending on its type
 *	pre:	<fname> is a file
 *	post:	returns TRUE if directory changed
 */
{
	char		*fname = (*dir+i)->filename;

	/* get the type of the file : file or directory */
	if ((*dir+i)->filestatus.st_mode & S_IFDIR) {	/* is directory */
		chdir(fname);
		get_dir(".", dir, ndirs);	/* read directory */
		return(TRUE);
	} else {				/* assume it's a displayable file */
		view_file(fname);
		return(FALSE);
	}
} /* process_file() */


int
get_filetype(char *fname)
/*
 *	desc:	determine the filetype of fname
 */
{
	char 			b[5];
	FILE			*f;
	union record		h;

	if ((f=fopen(fname, "r")) == NULL) {
		return(FALSE);
	}
	fread(b, 4, 1, f);
	rewind(f);
	fread(&h, sizeof(union record), 1, f);		/* read record for is_tar() */
	fclose(f);
	b[4] = 0;		/* terminate b, so it can be interpreted as a string */

	/*  the long numbers here are in opposite order as found in the signature of
	 *  the file, eg. the zip signature is 0x504b0304, so we check for a long off
	 *  0x04034b50	
	 */

	if (*(long *)b == 0x04034b50) {				/* zipfile */
		return(ZIPFILE);
	}
	if ((*(long *)b & 0x00ffffff) == 0x00088b1f) {		/* gzipfile */
		return(GZIPFILE);
	}
	if ((*(long *)b & 0x0000ffff) == 0x9d1f) {	/* actually a compressed file */
		return(GZIPFILE);			/* but gzip can handle this   */
	}
	if (strcmp(b, "GIF8") == 0) {
		return(GIFFILE);
	}
	if (strncmp(b, "%!", 2) == 0) {
		return(PSFILE);
	}
	if (is_tar(&h)) {
		return(TARFILE);
	}
	return(TEXTFILE);	
}


void
get_screensize(struct winsize *w)
/*
 *	desc:	get the size of the current screen
 *	pre:	w points to allocated space
 *	post:	w->ws_row = "# of rows"
 *		w->ws_col = "# of colums"
 */
{
	ioctl(0, TIOCGWINSZ, w);		/* get the window size */
	return;
}/* get_screensize() */
