
/*******************************************
 *  Fontedit V0.02  (alpha version)        *
 *                                         *
 * A font editor for Linux fontpak.        *
 *                                         *
 *******************************************/

/*  Copyright (C) 1993  A.M. Kuchling

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

	(Note: for this alpha release, the GNU GPL is not included--
      please ignore the following paragraph)
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.    */


#define SIDE (table >> 1)	/* Which side of the screen we're on */
#define TABLE (table & 1)	/* What table we're in */
#define OFFSET (40*(table >> 1))/* Left/right side of the screen */
#define PIXELCHAR '#'		/* Character used in the pixel grid */

#include <curses.h>
#include <stdio.h>
#include <errno.h>

/* Global variables */

char *fontname[2];
int grid[2][4], NumFonts;
unsigned char edit_font[2][8192];
unsigned char undo_buffer[2][16];
int xoff[2] = {8, 4},			/* Screen location of the two tables */
    yoff[2] = {2, 20};
int limit[4] = {16, 8, 4, 32};		/* Table sizes */

void DrawGrid (int), DoCursor (int, int);


	/* Refresh one side of the screen */
	 
void 
DrawSide (int side)
{
  int i, j, offset = side * 40;
  unsigned char c;

  for (i = 1; i < 4; i++)	/* Display character table (buggy) */
    {
      for (j = 0; j < 32; j++)
	{
	  mvaddch (yoff[1] + i, xoff[1] + offset + j, i * 32 + j);
	}

      mvaddch (yoff[1] + i, xoff[1] + offset - 1, '|');
      mvaddch (yoff[1] + i, xoff[1] + offset + 32, '|');
    }

 	/* Print boxes and headings */

  mvprintw (yoff[1] - 1, xoff[1] + offset - 1, "+--------------------------------+");
  mvprintw (yoff[1] + 4, xoff[1] + offset - 1, "+--------------------------------+");

  mvprintw (yoff[0] + 1, xoff[0] + offset + 9, "Character:");
  mvprintw (yoff[0] - 1, xoff[0] + offset - 1, "+--------+");
  mvprintw (yoff[0] + 16, xoff[0] + offset - 1, "+--------+");
  for (i = 0; i < 16; i++)
    mvprintw (yoff[0] + i, xoff[0] + offset - 1, "|        |");
  mvprintw (0, offset + 15, "Font: %s", fontname[side]);
  DrawGrid (side);		/* Display the current character grid */
  refresh ();
}


	/* Draw the current character on the grid */
	 
void 
DrawGrid (int side)
{
  unsigned char byte;
  int i, j, offset = side * 40;
  int c = grid[side][2] * 32 + grid[side][3];	/* Character number */

  DoCursor (side * 2, 0);	/* Erase cursor */
  for (i = 0; i < 16; i++)	/* Display character */
    {
      move (yoff[0] + i, xoff[0] + offset);
      byte = edit_font[side][c * 32 + i];
      for (j = 0; j < 8; j++)	/* Output the current line */
	{
	  if (byte > 127)
	    addch (PIXELCHAR);
	  else
	    addch (' ');
	  byte = (byte & 127) << 1;
	}
    }
  mvprintw (yoff[0] + 2, xoff[0] + offset + 12, "%3i", c);
  DoCursor (side * 2, 1);
}


	/* Display/erase our simulated cursor */
	 
void 
DoCursor (int table, int mode)
{
  int c;

  move (yoff[TABLE] + grid[SIDE][2 * TABLE],	/* Move to cursor location */
	xoff[TABLE] + grid[SIDE][2 * TABLE + 1] + OFFSET);
  c = inch ();			/* Read character under cursor */
  if (mode == 1)
    standout ();		/* Switch to standout mode */
  move (yoff[TABLE] + grid[SIDE][2 * TABLE],	/* Reset cursor position */
	xoff[TABLE] + grid[SIDE][2 * TABLE + 1] + OFFSET); /* (just in case) */
  addch (c);			/* Output character, now in appropriate mode */
  standend ();			/* Turn off standout */
  refresh ();
}

 
 	/* Handle a cursor key (we're not doing this in keypad() mode */

 void 
CursorKey (int table)
{
  char c;
  int byte, i = -1;

  (void) read (0, &c, 1);
  if (c == 91)			/* It's one of our special keys */
    {
      (void) read (0, &c, 1);
      if (c == 65 || c == 66)
	i = 2 * TABLE;		/* Find appropriate coordinate */
      else if (c == 68 || c == 67)
	i = 2 * TABLE + 1;
      if (i != -1)		/* Just in case it's not a cursor key */
	{
	  DoCursor (table, 0);	/* Erase old cursor */
	  if (c == 65 || c == 68)	/* Decrement location counter, or... */
	    {
	      grid[SIDE][i]--;
	      if (grid[SIDE][i] == -1)
		grid[SIDE][i] += limit[i];
	    }
	  else
	    {
	      grid[SIDE][i]++;	/* ...increment location counter */
	      if (grid[SIDE][i] == limit[i])
		grid[SIDE][i] = 0;
	    }
	  DoCursor (table, 1);	/* Redraw new cursor */
	  if (TABLE == 1)
	    {
	      DrawGrid (SIDE);	/* If we've changed characters, redraw the grid */
	      byte = (grid[SIDE][2] * 32 + grid[SIDE][3]) * 32;
	      for (i = 0; i < 16; i++)	/* and save the character */
		undo_buffer[SIDE][i] = edit_font[SIDE][byte + i];
	      refresh ();
	    }
	}
    }
}

 
 
	/* File handling:  Load and save a font file */

void 
LoadFont (int fontnum)
{
  FILE *fontfile;

  fontfile = fopen (fontname[fontnum], "rb");
  if (fontfile == NULL)
    mvprintw (0, 40 * fontnum, "Error on read");
  else
    {
      fread (edit_font + fontnum * 8192, 8192, 1, fontfile);
      fclose (fontfile);
      mvprintw (0, 40 * fontnum, "               ");
    }
  refresh ();
}

void 
SaveFont (int fontnum)
{
  int i=0;
  FILE *fontfile;
  char tempName[FILENAME_MAX];
  
  while (fontname[fontnum][i]!='\0') 
   { 
    tempName[i]=fontname[fontnum][i];
    i++;
   }
 tempName[i++]='~';
 tempName[i]  ='\0';
 
 unlink(tempName);	/* Erase old backup file */
 rename(fontname[fontnum], tempName);	/* Rename old file to backup name */
  
  fontfile = fopen (fontname[fontnum], "wb");
  if (fontfile == NULL)
    mvprintw (0, 40 * fontnum, "Error on write");
  else
    {
      fwrite (edit_font + fontnum * 8192, 8192, 1, fontfile);
      fclose (fontfile);
      mvprintw (0, 40 * fontnum, "               ");
    }
  refresh ();
}


 
/* The 3 following routines are lifted directly from Jerry Kaidor's
   loadfont.c, with slight modifications.			  */

char 
hinyb (char ch)
{
  ch = ((ch >> 4) & 0x0f) + '0';
  if (ch > '9')
    ch += 7;
  return ch;
}

char 
lonyb (char ch)
{
  ch = (ch & 0x0f) + '0';
  if (ch > '9')
    ch += 7;
  return ch;
}

void 
StoreFont (int side)
{
  char c;
  int i;
  (void) read (0, &c, 1);
  if (c == '1' || c == '2')
    {
      putchar (27);
      putchar ('F');
      putchar ('P');
      putchar (c);
      for (i = 0; i < 8192; i++)
	{
	  putchar (hinyb (edit_font[side][i]));
	  putchar (lonyb (edit_font[side][i]));
	}
    }
}


void 
main (int argc, char *argv[])
{
  int i, byte, table = 0;
  char key, buffer[16];

  /* Look for filenames on command line */
  for (NumFonts = 0, i = 1; NumFonts < 2 && i < argc; i++)
    if (argv[i][0] != '-')
      fontname[NumFonts++] = argv[i];

  if (NumFonts == 0)
    {
      printf ("Must enter at least 1 font name\n");
      exit (1);
    }

  initscr ();
  clear ();			/* Initialize curses */
  cbreak ();
  noecho ();

  grid[0][0] = grid[1][0] = grid[0][1] = grid[1][1] = 0;	/* Initial cursor positions */
  grid[0][3] = grid[1][3] = 0;
  grid[0][2] = grid[1][2] = 2;

  for (i = 0; i < 16; i++)
    buffer[i] = 0;		/* Clear character copy buffer */

  mvprintw (yoff[0], xoff[0] - 4, "==>");	/* Draw the table pointer */
  for (i = 0; i < NumFonts; i++)
    {
      LoadFont (i);
      DrawSide (i);		/* Set up screen */
      DoCursor (2 * i + 1, 1);	/* Show character cursor */
    }

  do				/* Main loop- read a key & handle it */
    {
      (void) read (0, &key, 1);
      if ('A' <= key && key <= 'Z')
	key += 'a' - 'A';
      switch (key)
	{
	case ('\t'):			/* TAB: Switch from grid to table */
	  mvprintw (yoff[TABLE], OFFSET + xoff[TABLE] - 4, "   ");
	  table = (table == NumFonts * 2 - 1) ? 0 : table + 1;
	  mvprintw (yoff[TABLE], OFFSET + xoff[TABLE] - 4, "==>");
	  refresh ();
	  break;

	case (27):		/* Cursor keys */
	  CursorKey (table);
	  break;		

	case (32):			  /* Space: Toggle a bit */
	  if (TABLE == 0)	/* Check if we're in the grid */
	    {
	      byte = 32 * (grid[SIDE][2] * 32 + grid[SIDE][3]) + grid[SIDE][0];	/* Compute affected byte */
	      edit_font[SIDE][byte] ^= 1 << (7 - grid[SIDE][1]);
	      DrawGrid (SIDE);	/* Redraw grid */
	    }
	  break;

	case ('0'):		/* 0-2: Switch to appropriate font */
	case ('1'):
	case ('2'):
	  putchar (27);
	  putchar ('F');
	  putchar ('U');
	  putchar (key);
	  refresh ();
	  break;

	case ('c'):		/* C: Copy character into buffer */
	  byte = 32 * (grid[SIDE][2] * 32 + grid[SIDE][3]);

	  for (i = 0; i < 16; i++)
	    buffer[i] = edit_font[SIDE][byte + i];
	  break;
	 
	case ('l'):		/* L: Load a font */
	  LoadFont (SIDE);
	  DrawGrid (SIDE);
	  break;

	 case ('s'):		/* S: Save a font */
	  SaveFont (SIDE);	
	  break;
	 
	case ('u'):		/* U: Undo changes from buffer */
	  byte = 32 * (grid[SIDE][2] * 32 + grid[SIDE][3]);	
	  for (i = 0; i < 16; i++)
	    edit_font[SIDE][byte + i] = undo_buffer[SIDE][i];
	  DrawGrid (SIDE);
	  break;
	 
	case ('w'):		/* W: Store a font into a codepage */
	  StoreFont (SIDE);	
	  break;

	 case ('y'):		/* Y: Yank from buffer */
	  byte = 32 * (grid[SIDE][2] * 32 + grid[SIDE][3]);
	  for (i = 0; i < 16; i++)
	    edit_font[SIDE][byte + i] = buffer[i];
	  DrawGrid (SIDE);
	  break;
	}
    }
  while (key != 'q');		/* Q: Quit */

  clear ();
  refresh ();			/* Leave a blank screen */
  endwin ();

}
