/*
 * This file is part of fb, the frame buffer device, a grafics card driver for
 *                                linux.
 *
 *      Copyright (C) 1995 Michael Weller (eowmob@exp-math.uni-essen.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Michael Weller (eowmob@exp-math.uni-essen.de or
 * mat42b@aixrs1.hrz.uni-essen.de) Heiderhoefen 116b,
 * D 46049 Oberhausen, Germany.
 */

/*
 * modecc, a compiler for modedefining config files.
 */

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <sys/stat.h>

#include "modecc.h"

char *version = "modecc version 0.0.alpha May 5 1995";
char *source = NULL;
char *cppcom;
int srcline = 1;
int lexerr = 0;
int usecpp = 1;

struct option longoptions[] = {
	{ "help", 0, NULL, 'h' },
	{ "version", 0, NULL, 'v' },
	{ "cpp", 0, NULL, 'c' },
	{ "nocpp", 0, NULL, 'C' },
	{ NULL, 0, NULL, 0 }
	};

extern int optind;
extern char *optarg;

void usage(void) {
	fputs(	"usage: modecc [-hv{cC}] [source]\n"
		"\tTry \"modecc -h\" or \"modecc --help\" for more info\n", stderr);
	exit(2);
}

void help(void) {
	fprintf(stderr,
		"modecc [options] [source] - a compiler for videomode definitions\n"
		"  source        - the source file to compile. See man modecc for more\n"
		"                  about it's format. If not given at all, or given as -\n"
		"                  stdin is read.\n"
		"  -c, --cpp     - pipe input through cpp. (this is the default).\n"
		"                  A shell command to be executed for cpp can be\n"
		"                  set with the environment variable CPP.\n"
		"                  Default is: CPP=\"%s\".\n"
		"		   %%s will be replaced with the filename to process.\n"
		"  -C, --nocpp   - do NOT pipe the input through the C preprocessor.\n"
		"  -h, --help    - display this page.\n"
		"  -v, --version - display version information.\n"
		, DEFCPP);
}

int main(int argc, char *argv[]) {
	int index, value;
	int ph = 0, pv = 0;

	cppcom = getenv("CPP");
	if (!cppcom)
		cppcom = DEFCPP;
	for(;;)	{
		value = getopt_long(argc, argv, "vhCc", longoptions, &index);
		if(value >= 0) {
			switch(value) {
				case 'h':
					ph = 1;
					break;
				case 'v':
					pv = 1;
					break;
				case 'c':
					usecpp = 1;
					break;
				case 'C':
					usecpp = 0;
					break;
				default:
					usage();
			}
		} else	break;
	}
	if (optind + 1 < argc) {
		fprintf(stderr, ERRHEAD "too many source files.\n");
		usage();
	} else if (optind < argc)
		source = argv[optind];

	if (pv) puts(version);
	if (ph) help();

	if (ph || pv)
		exit(0);
		
	/*
	 * construct the filename to be used.
	 */
	if (!source)
		source = "-";
	if (strcmp(source, "-")) {
		/* We are accessing a real file */
		struct stat tmpstat;
		char *newsrc;

		if (stat(source, &tmpstat)) {
			/* Try to append a .m */
			newsrc = malloc(strlen(source) + 3);
			if (!newsrc) {
			    exoom:
				fprintf(stderr, ERRHEAD "Out of memory");
				perror("");
				exit(1);
			}
			strcpy(newsrc, source);
			strcpy(strchr(newsrc, 0), ".m");
			/* When the new file exists use that one */
			if (!stat(newsrc, &tmpstat))
				source = newsrc;
			else {	/*
				 * Fall through using the old file.. cpp or fopen will
				 * cause error message referring to the filename given.
				 */
				free(newsrc);
			}
		}
	}

	if(!usecpp) {
		if (!strcmp(source, "-")) {
			/* nicer for error messages */
			source = "<stdin>";
			yyin = stdin;
		} else {
			yyin = fopen(source, "r");
			if (!yyin) {
				fprintf(stderr, ERRHEAD "Cannot open source file \"%s",
					source);
				perror("\"");
				exit(1);
			}
		}
	} else {
		char *commandline;

		commandline = malloc(strlen(source) + strlen(cppcom) + 3);
		if (!commandline)
			goto exoom;
		sprintf(commandline, cppcom, source);
		yyin = popen(commandline, "r");
		if (!yyin) {
			perror(ERRHEAD "executing preprocessor");
			exit(1);
		}
	}
	yyparse();
	if(usecpp) {
		int err;

		err = pclose(yyin);
		if (err) {
			fprintf(stderr, ERRHEAD "cpp failed, aborting!\n");
			return err > 0 ? err : 1;
		}
	}
	if(lexerr) {
		fprintf(stdout, ERRHEAD "No action taken due to parse errors!\n");
		exit(2);
		}
	return 0;
}
