/*
 * This file is part of fb, the frame buffer device, a grafics card driver for
 *                                linux.
 *
 *     Copyright (C) 1995 Pascal Haible (haible@IZFM.Uni-Stuttgart.DE)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Dipl.-Ing. Pascal Haible (haible@IZFM.Uni-Stuttgart.DE)
 *
 */

#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <sys/ioctl.h>

#include <linux/vt.h>
#define VT_SWITCH_ALLOW	1
#define VT_SWITCH_DENY	0
#include <linux/kd.h>

#include <sys/mman.h>
#include <sys/ipc.h>
#include <sys/shm.h>

#include "libfb.h"

/****************************************/
/* debugging and error stuff            */
/****************************************/
int fb_debug = 9;

char fb_error_text[200];
int fb_verbose = 1;

static char *fb_err_where, *fb_err2, *fb_err3;

#define DEBUG(_level, _stat)				\
	do { if (_level <= fb_debug) { _stat ; } } while (0)

#define ERROR1(_a1)					\
	do { sprintf(fb_error_text, _a1);		\
	     if (fb_verbose) {				\
		fprintf(stderr, "%s", fb_error_text);	\
		fflush(stderr);				\
	     }						\
	   } while (0)
#define ERROR2(_a1,_a2)					\
	do { sprintf(fb_error_text, _a1, _a2);		\
	     if (fb_verbose) {				\
		fprintf(stderr, "%s", fb_error_text);	\
		fflush(stderr);				\
	     }						\
	   } while (0)
#define ERROR3(_a1,_a2,_a3)				\
	do { sprintf(fb_error_text, _a1, _a2, _a3);	\
	     if (fb_verbose) {				\
		fprintf(stderr, "%s", fb_error_text);	\
		fflush(stderr);				\
	     }						\
	   } while (0)
#define ERROR4(_a1,_a2,_a3,_a4)				\
	do { sprintf(fb_error_text, _a1, _a2, _a3, _a4);\
	     if (fb_verbose) {				\
		fprintf(stderr, "%s", fb_error_text);	\
		fflush(stderr);				\
	     }						\
	   } while (0)


#define ENTERING					\
	DEBUG(9, ERROR2("libfb: entering %s ", __FUNCTION__);)

#define LEAVING						\
	DEBUG(9, ERROR2("\tlibfb: leaving %s\n", __FUNCTION__);)

#define DEBUG_INFO(_s)					\
	DEBUG(7, ERROR2("libfb: debugging info: %s\n", _s);)

#define INFO(_s)					\
	DEBUG(5, ERROR2("libfb: info: %s\n", _s);)

#define WARNING(_s)					\
	DEBUG(3, ERROR2("libfb: warning: %s\n", _s);)

#define ERROR(_s)					\
	DEBUG(1, ERROR2("libfb: error: %s\n", _s);)

#define SYS_ERROR(_s1)							\
	do {								\
		/* fb_err_where already set */				\
		fb_err2 = _s1;						\
		fb_err3 = strerror(errno);				\
		/* let the user program fetch the error text -- or: */	\
		ERROR4("libfb: error while doing %s: %s: %s.\n",	\
			fb_err_where, fb_err2, fb_err3);			\
	} while (0)

#define FB_ERROR(_s1)							\
	do {								\
		/* fb_err_where already set */				\
		fb_err2 = _s1;						\
		/* let the user program fetch the error text -- or: */	\
		ERROR3("libfb: error while doing %s: %s.\n",		\
			fb_err_where, fb_err2);				\
	} while (0)


/****************************************/
/* initialization level encoding        */
/****************************************/
volatile int _fb_init_level = 0;
#define INITLEVEL_NONE	0
#define INITLEVEL_INIT	1
#define INITLEVEL_VT	2
#define INITLEVEL_DEV	3
#define INITLEVEL_MODE	4
#define INITLEVEL_SHM	5
#define INITLEVEL_MMAP	6
/* #define INITLEVEL_SWITCHED_OUT	7 */

#define IS_INITLEVEL(_il)	(_fb_init_level == _il)

#define CHECK_INITLEVEL(_il) 						\
    if (!IS_INITLEVEL(_il)) {						\
	DEBUG(1, ERROR2("libfb: error: calling %s() when not appropriate.\n",\
			__FUNCTION__));					\
	DEBUG(1, ERROR3("initlevel is %i, wanted %i.\n",		\
			_fb_init_level, _il));				\
	return -1;							\
    }

#define CHECK_INITLEVEL2(_il1, _il2) 					\
    if ( (!IS_INITLEVEL(_il1)) && (!IS_INITLEVEL(_il2)) ) {		\
	DEBUG(1, ERROR2("libfb: error: calling %s() when not appropriate.\n",\
			__FUNCTION__));					\
	DEBUG(1, ERROR4("initlevel is %i, wanted %i or %i.\n",		\
			_fb_init_level, _il1, _il2));			\
	return -1;							\
    }

#define SET_INITLEVEL(_il) \
    do { _fb_init_level = _il; } while (0)

/****************************************/
/* variables                            */
/****************************************/

/* VT status */
int _fb_initial_vt;
int _fb_vt;
/* int _fb_switch_inhibit; */
int _fb_vt_fd;
int _fb_switched_out;
struct sigaction _fb_sig;
struct sigaction _fb_oldsigusr1;
struct sigaction _fb_oldsigusr2;

int _fb_dev_fd;

fb_info_t _fb_info;
fb_mode_t *_fb_modes_all;
unsigned long _fb_mode;

int _fb_shmsize;
int _fb_shmid;
char *_fb_shmbase;

int _fb_mmapsize;
char *_fb_mmapbase;

char *_fb_base;


/* internal functions */

static int  _fb_vt_switch_away(void);
static int  _fb_vt_switch_back(void);
static int  _fb_vt_switch_deny(void);
static void _fb_switch_away_request(int sig);
static void _fb_switch_back_request(int sig);

static void _fb_switch_away_request(int sig);
static void _fb_switch_back_request(int sig);

int fb_init(void)
{
    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_NONE);

    _fb_initial_vt = -1;
    _fb_vt = -1;
/*     _fb_switch_inhibit = 0; */
    _fb_vt_fd = -1;
    _fb_switched_out = 0;

    _fb_dev_fd = -1;

/*    _fbinfo = NULL; */
    _fb_modes_all = NULL;
    _fb_mode = 0;

    _fb_shmsize = -1;
    _fb_shmid = -1;
    _fb_shmbase = NULL;

    _fb_mmapsize = -1;
    _fb_mmapbase = NULL;

    _fb_base = NULL;

    SET_INITLEVEL(INITLEVEL_INIT);
    LEAVING;
    return 0;
}

int fb_leave(void)
{
    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_INIT);

    if (_fb_modes_all != NULL)
	free(_fb_modes_all);

    SET_INITLEVEL(INITLEVEL_NONE);
    LEAVING;
    return 0;
}

/****************************************/
/* VT open                              */
/****************************************/
int fb_vt_open(void)
{
    int fd;
    char vt_name[1+3+1+3+2+1] = "/dev/tty??";
    struct vt_mode vt_m;
    struct vt_stat vt_s;
    /* int ret; */

    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_INIT);

    fb_err_where = "VT initialization";

    if ((fd = open("/dev/console", O_WRONLY,0)) < 0) {
	SYS_ERROR("/dev/console");
	return errno;
    }
    if ((ioctl(fd, VT_OPENQRY, &_fb_vt) < 0) || (_fb_vt == -1)) {
	close(fd);
	SYS_ERROR("ioctl(VT_OPENQRY)");
	return errno;
    }
    if (close(fd)) {
	SYS_ERROR("close");
	return errno;
    }
    if (_fb_vt > 99) {
	FB_ERROR("can't handle vt > 99");
	return -1;
    }
    sprintf(vt_name, "/dev/tty%d", _fb_vt); /* /dev/tty1-NN */
    DEBUG(5, ERROR2("libfb: info: using vt %i.\n", _fb_vt));
    if ((_fb_vt_fd = open(vt_name, O_WRONLY|O_NDELAY, 0)) < 0) {
	SYS_ERROR(vt_name);
	return errno;
    }
    /* remember which vt is active now */
    if (ioctl(_fb_vt_fd, VT_GETSTATE, &vt_s) != 0) {
	close(_fb_vt_fd);
	SYS_ERROR("ioctl(VT_GETSTATE)");
	return errno;
    }
    _fb_initial_vt = vt_s.v_active;
#if 0
    if ((fd = open("/dev/tty", O_RDWR)) < 0) {
	close(_fb_vt_fd);
	SYS_ERROR("/dev/tty");
	return errno;
    }
    if (ioctl(fd, TIOCNOTTY, 0) != 0) {
	close(_fb_vt_fd);
	close(fd);
	SYS_ERROR("TIOCNOTTY");
	return errno;
    }
    if (close(fd)) {
	close(_fb_vt_fd);
	SYS_ERROR("close");
	return errno;
    }
#endif
/*    _fb_switch_inhibit = 1; */
    if (ioctl(_fb_vt_fd, VT_ACTIVATE, _fb_vt) != 0) {
	/* close(_fb_vt_fd); */
	fb_vt_close();
	SYS_ERROR("ioctl(VT_ACTIVATE)");
	return errno;
    }
    if (ioctl(_fb_vt_fd, VT_WAITACTIVE, _fb_vt) != 0) {
	/* close(_fb_vt_fd); */
	fb_vt_close();
	SYS_ERROR("ioctl(VT_WAITACTIVE)");
	return errno;
    }
    if (ioctl(_fb_vt_fd, VT_GETMODE, &vt_m) != 0) {
	/* close(_fb_vt_fd); */
	fb_vt_close();
	SYS_ERROR("ioctl(VT_GETMODE)");
	return errno;
    }
/*
    signal(SIGUSR1, _fb_switch_away_request);
    signal(SIGUSR2, _fb_switch_back_request);
*/
    _fb_sig.sa_handler = _fb_switch_away_request;
    _fb_sig.sa_mask = sigmask(SIGUSR1) | sigmask(SIGUSR2);
    _fb_sig.sa_flags = 0;
    _fb_sig.sa_restorer = NULL;
    sigaction(SIGUSR1, &_fb_sig, &_fb_oldsigusr1);
    _fb_sig.sa_handler = _fb_switch_back_request;
    sigaction(SIGUSR2, &_fb_sig, &_fb_oldsigusr2);
    vt_m.mode = VT_PROCESS;
    vt_m.relsig = SIGUSR1;
    vt_m.acqsig = SIGUSR2;
    if (ioctl(_fb_vt_fd, VT_SETMODE, &vt_m) != 0) {
	/* close(_fb_vt_fd); */
	fb_vt_close();
	SYS_ERROR("ioctl(VT_SETMODE)");
	return errno;
    }
#if 1
    if (ioctl(_fb_vt_fd, KDSETMODE, KD_GRAPHICS) != 0) {
	/* close(_fb_vt_fd); */
	fb_vt_close();
	SYS_ERROR("ioctl(KD_GRAPHICS)");
	return errno;
    }
#endif
    SET_INITLEVEL(INITLEVEL_VT);
    LEAVING;
    return 0;
}

/****************************************/
/* VT close                             */
/****************************************/
int fb_vt_close(void)
{
    struct vt_mode vt_m;

    ENTERING;  
    CHECK_INITLEVEL(INITLEVEL_VT);

    fb_err_where = "VT closing down";

    if (ioctl(_fb_vt_fd, KDSETMODE, KD_TEXT)) {
	close(_fb_vt_fd);
	SYS_ERROR("ioctl(KD_TEXT)");
	return errno;
    }
    if (ioctl(_fb_vt_fd, VT_GETMODE, &vt_m) == -1) {
	close(_fb_vt_fd);
	SYS_ERROR("ioctl(VT_GETMODE)");
	return errno;
    }
    vt_m.mode = VT_AUTO;
    if (ioctl(_fb_vt_fd, VT_SETMODE, &vt_m)) {
	close(_fb_vt_fd);
	SYS_ERROR("ioctl(VT_SETMODE)");
	return errno;
    }
/*
    signal(SIGUSR1, SIG_DFL);
    signal(SIGUSR2, SIG_DFL);
*/
    sigaction(SIGUSR1, &_fb_oldsigusr1, NULL);
    sigaction(SIGUSR2, &_fb_oldsigusr2, NULL);

    /* don't switch back to the initial vt if we are on an other text vt now! */
    if ((_fb_initial_vt >= 0) && (!_fb_switched_out)) {
	if (ioctl(_fb_vt_fd, VT_ACTIVATE, _fb_initial_vt)) {
	    close(_fb_vt_fd);
	    SYS_ERROR("ioctl(VT_ACTIVATE)");
	    return errno;
	}
	_fb_initial_vt = -1;
    }
    if (close(_fb_vt_fd)) {
	SYS_ERROR("close");
	return errno;
    }
/*     _fb_switch_inhibit = 0; */
    SET_INITLEVEL(INITLEVEL_INIT);
    LEAVING;
    return 0;
}

/****************************************/
/* VT switch request signal handler     */
/****************************************/
static void _fb_switch_away_request(int sig)
{
    if (!_fb_switched_out) {
	_fb_vt_switch_away();
    } else {
	_fb_vt_switch_deny();
    }
}

static void _fb_switch_back_request(int sig)
{
    if (_fb_switched_out) {
	_fb_vt_switch_back();
    } else {
	_fb_vt_switch_deny();
    }
}

/****************************************/
/* do VT switch                         */
/****************************************/
static int _fb_vt_switch_away(void)
{
    int ret;

/*    _fb_switch_inhibit = 1; */

    if ((ret = fb_copy_to_shm()))
	return ret;
    if ((ret = fb_video_off()))
	return ret;
    if ((ret = fb_munmap()))
	return ret;
    /* what should I do if these fail? */
    if ((ret = fb_shm_attach()))
	return ret;
    if ((ret = fb_dev_close()))
	return ret;
    if ((ret = ioctl(_fb_vt_fd, VT_RELDISP, VT_SWITCH_ALLOW)))
	return ret;

    _fb_switched_out = 1;
/*    _fb_switch_inhibit = 0; */

    return 0;
}

static int _fb_vt_switch_back(void)
{
    int ret;

/*     _fb_switch_inhibit = 1; */

    if ((ret = ioctl(_fb_vt_fd, VT_RELDISP, VT_ACKACQ)))
	return ret;
    /* what should I do if these fail? */
    if ((ret = fb_dev_open()))
	return ret;
    if ((ret = fb_set_mode(_fb_mode)))
	return ret;
    if ((ret = fb_shm_detach()))
	return ret;
    if ((ret = fb_mmap()))
	return ret;
    if ((ret = fb_copy_from_shm()))
	return ret;

    _fb_switched_out = 0;
    if ((ret = fb_video_on()))
	return ret;
/*    _fb_switch_inhibit = 0; */
    return 0;
}

static int _fb_vt_switch_deny(void)
{
    /* error checking within the signal handler doesn't make much sense */
    /* and since printf etc. might not be reentrant... */
    if (_fb_vt_fd != -1) {
	ioctl(_fb_vt_fd, VT_RELDISP, VT_SWITCH_DENY);
    }
    return 0;
}

/****************************************/
/* open and close /dev/fb               */
/****************************************/
int fb_dev_open(void)
{
    int _fb_get_all_infos(void);
    int ret;

    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_VT);
    fb_err_where = "open fb device";

    _fb_dev_fd = open("/dev/fb", O_RDWR);
    if (_fb_dev_fd == -1) {
	SYS_ERROR("open(/dev/fb)");
	return errno;
    }

    SET_INITLEVEL(INITLEVEL_DEV);

    /* if we are switching back, no need to get all info again */
    if (_fb_switched_out) {
	LEAVING;
	return 0;
    }

    ret = fb_get_info(&_fb_info);
    if (ret) {
	fb_dev_close();
	return ret;
    }

    ret = fb_get_default_mode(&_fb_mode);
    if (ret) {
	fb_dev_close();
	return ret;
    }

    ret = _fb_get_all_infos();
    if (ret) {
	fb_dev_close();
	return ret;
    }
    LEAVING;
    return 0;
}

int fb_dev_close(void)
{
    int ret;

    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_DEV);
    fb_err_where = "close fb device";

    if (_fb_switched_out) {
	/* nothing to do */
	SET_INITLEVEL(INITLEVEL_VT);
	LEAVING;
	return 0;
    }
    ret = close(_fb_dev_fd);
    _fb_dev_fd = -1;
/*
    if (NULL != _fb_modes_all) {
	free(_fb_modes_all);
	_fb_modes_all = NULL;
    }
*/
    SET_INITLEVEL(INITLEVEL_VT);
    if (ret) {
	SYS_ERROR("close(/dev/fb)");
	return errno;
    }
    LEAVING;
    return 0;
}

/****************************************/
/* info & mode info ioctl()s            */
/****************************************/
int fb_get_info(fb_info_t *pinfo)
{
    int ret;
    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_DEV);
    ret = ioctl(_fb_dev_fd, FB_GETINFO, pinfo);
    LEAVING;
    return ret;
}

int fb_get_default_mode(unsigned long *pmode)
{
    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_DEV);
    *pmode = _fb_info.default_mode;
    LEAVING;    
    return 0;
}

int fb_get_num_modes(void)
{
    int ret;
    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_DEV);
    ret = _fb_info.number_modes;
    LEAVING;
    return ret;
}

int fb_get_mode_info(unsigned long mode, fb_mode_t *pmode)
{
    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_DEV);
    memcpy(pmode, _fb_modes_all + mode, sizeof(fb_mode_t));
    LEAVING;
    return 0;
}

int _fb_get_all_infos(void)
{
    int ret;

    CHECK_INITLEVEL(INITLEVEL_DEV);

    _fb_modes_all = (fb_mode_t *)malloc(fb_get_num_modes()*sizeof(fb_mode_t));
    if (NULL == _fb_modes_all)
	return -1;

    ret = ioctl(_fb_dev_fd, FB_MODES_GET_INFO, _fb_modes_all);
    LEAVING;
    return ret;
}

int fb_set_mode(unsigned long mode)
{
    int ret;

    ENTERING;
    CHECK_INITLEVEL2(INITLEVEL_DEV, INITLEVEL_MODE);
    ret = ioctl(_fb_dev_fd, FB_MODE_SET, &mode);
    if (ret)
	return ret;
    _fb_mode = mode;
    SET_INITLEVEL(INITLEVEL_MODE);
    LEAVING;
    return 0;
}

/****************************************/
/* video on and off                     */
/****************************************/
int fb_video_on(void)
{
    int ret;

    ENTERING;
    if (_fb_switched_out) {
	LEAVING;
	return 0;
    }
    CHECK_INITLEVEL(INITLEVEL_MMAP);
    ret = ioctl(_fb_dev_fd, FB_UNBLANK);
    LEAVING;
    return ret;
}

int fb_video_off(void)
{
    int ret;

    ENTERING;
    if (_fb_switched_out) {
	LEAVING;
	return 0;
    }
    CHECK_INITLEVEL(INITLEVEL_MMAP);
    ret = ioctl(_fb_dev_fd, FB_BLANK);
    LEAVING;
    return ret;
}

/****************************************/
/* shared memory stuff                  */
/****************************************/
int fb_shm_init(void)
{
    key_t key;
    int shmflag;
    int ret;

    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_MODE);
    fb_err_where = "init shared memory";

    /* srand(time(NULL)); */
    key = /* rand() | */ IPC_PRIVATE;
    shmflag = IPC_CREAT | IPC_EXCL | 0600;
    _fb_shmsize = _fb_modes_all[_fb_mode].vidmem;
    DEBUG(7, ERROR2("libfb: fb_shm_init: debugging info: _fb_shmsize = %i\n",
			_fb_shmsize));
    _fb_shmid = shmget(key, _fb_shmsize, shmflag);
    DEBUG(7, ERROR2("libfb: fb_shm_init: debugging info: _fb_shmid = %i\n",
			_fb_shmid));
    if (-1 == _fb_shmid) {
	SYS_ERROR("shmget");
	return errno;
    }

    _fb_shmbase = shmat(_fb_shmid, NULL, 0);
    DEBUG(7, ERROR2("libfb: fb_shm_init: debugging info: _fb_shmbase = 0x%p\n",
			_fb_shmbase));
    if (-1 == (int)_fb_shmbase) {
	SYS_ERROR("shmat");
	return errno;
    }

    ret = shmctl(_fb_shmid, IPC_RMID, (struct shmid_ds *)0);
    if (ret) {
	SYS_ERROR("shmctl(IPC_RMID)");
	return errno;
    }
    SET_INITLEVEL(INITLEVEL_SHM);
    LEAVING;
    return 0;
}

int fb_shm_attach(void) /* called from signal handler */
{
    if (!IS_INITLEVEL(INITLEVEL_SHM))
	return -1;
    if (-1 == _fb_shmid) {
	return -1;
    }

    if (_fb_base != shmat(_fb_shmid, _fb_base, 0)) {
	return -1;
    }
    SET_INITLEVEL(INITLEVEL_DEV);
    return 0;
}

int fb_shm_detach(void) /* called from signal handler */
{
    int ret;
    if (!IS_INITLEVEL(INITLEVEL_MODE))
	return -1;
    if (-1 == _fb_shmid) {
	return -1;
    }
    ret = shmdt(_fb_base);
    SET_INITLEVEL(INITLEVEL_SHM);
    return ret;
}

int fb_shm_close(void)
{
    int ret;

    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_SHM);
    if (-1 == _fb_shmid) {
	FB_ERROR("invalid shmid!");
	return -1;
    }
    ret = shmdt(_fb_shmbase);
    _fb_shmid = -1;
    SET_INITLEVEL(INITLEVEL_DEV);
    LEAVING;
    if (ret)
	SYS_ERROR("shmdt");
    return ret;
}

/****************************************/
/* copy from / to shared memory         */
/****************************************/
int fb_copy_from_shm(void) /* called from signal handler */
{
    unsigned long xdim, y, dy;
    unsigned char *shm, *screen;

    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_MMAP);
    if (-1 == _fb_shmid) {
	return -1;
    }

    shm = _fb_shmbase;
    screen = _fb_mmapbase;
    xdim = _fb_modes_all[_fb_mode].xdim*_fb_modes_all[_fb_mode].bpp_used/8;
    dy = _fb_modes_all[_fb_mode].scanlinelength*_fb_modes_all[_fb_mode].bpp_used/8;
    for (y=0; y<_fb_modes_all[_fb_mode].ydim; y++) {
	memcpy(screen, shm, xdim);
	shm += dy;
	screen += dy;
    }
    LEAVING;
    return 0;
}

int fb_copy_to_shm(void) /* called from signal handler */
{
    unsigned long xdim, y, dy;
    unsigned char *shm, *screen;

    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_MMAP);
    if (-1 == _fb_shmid) {
	return -1;
    }

    shm = _fb_shmbase;
    screen = _fb_mmapbase;
    xdim = _fb_modes_all[_fb_mode].xdim*_fb_modes_all[_fb_mode].bpp_used/8;
    dy = _fb_modes_all[_fb_mode].scanlinelength*_fb_modes_all[_fb_mode].bpp_used/8;
/*
fprintf(stderr, "shm = 0x%p, screen = 0x%p,\n", shm, screen);
fprintf(stderr, "xdim = %lu, ydim = %lu, dy = %lu\n", xdim,
				_fb_modes_all[_fb_mode].ydim, dy);
*/
    for (y=0; y<_fb_modes_all[_fb_mode].ydim; y++) {
	memcpy(shm, screen, xdim);
	shm += dy;
	screen += dy;
    }
    LEAVING;
    return 0;
}

/****************************************/
/* mmap() framebuffer                   */
/****************************************/
int fb_mmap(void)
{
    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_SHM);
    fb_err_where = "mmap framebuffer";

    if (NULL != _fb_mmapbase) {
	FB_ERROR("already mapped");
	return -1;
    }
    _fb_mmapsize = _fb_modes_all[_fb_mode].vidmem2mmap;
    _fb_mmapbase = (char *)mmap((caddr_t)_fb_base, _fb_mmapsize, PROT_READ|PROT_WRITE,
				MAP_SHARED, _fb_dev_fd, (off_t)0);
    if (_fb_mmapbase == (void *)-1) {
	_fb_base = NULL;
	SYS_ERROR("mmap");
	return -1;
    }
    _fb_base = _fb_mmapbase;
    SET_INITLEVEL(INITLEVEL_MMAP);
/*     _fb_switch_inhibit = 0; */
    LEAVING;
    return 0;
}

int fb_munmap(void)
{
    int ret;

    ENTERING;

    if (_fb_switched_out) {
	/* already done */
	SET_INITLEVEL(INITLEVEL_SHM);
	LEAVING;
	return 0;
    }

    CHECK_INITLEVEL(INITLEVEL_MMAP);
    fb_err_where = "unmap framebuffer";

    /* if closing down by hand */
/*     _fb_switch_inhibit = 1; */

    if (NULL == _fb_mmapbase) {
	FB_ERROR("already unmapped");
	return -1;
    }
    ret = munmap(_fb_mmapbase, _fb_mmapsize);
    if (ret) {
	SYS_ERROR("munmap");
	return errno;
    }
    _fb_mmapbase = NULL;
    SET_INITLEVEL(INITLEVEL_SHM);
    LEAVING;
    return 0;
}

int fb_get_base(unsigned char **base)
{
    ENTERING;
    CHECK_INITLEVEL(INITLEVEL_MMAP);
    fb_err_where = "Get framebuffer base";
    if (NULL == _fb_base) {
	FB_ERROR("framebuffer not mapped");
	return -1;
    }
    *base = (unsigned char *)_fb_base;
    LEAVING;
    return 0;
}

/****************************************/
/* misc. testing                        */
/****************************************/
int fb_test_test(void)
{
    int ret;

    ENTERING;
    if (_fb_switched_out) {
	LEAVING;
	return 0;
    }
    CHECK_INITLEVEL(INITLEVEL_MMAP);
    ret = ioctl(_fb_dev_fd, FB_TEST_TEST);
    LEAVING;
    return ret;
}

/****************************************/
/* misc.                                */
/****************************************/
int fb_beep(unsigned short freq, unsigned short ms)
{
    int ret;
    ENTERING;
    if (_fb_init_level < INITLEVEL_VT) {
	DEBUG(1, ERROR2("libfb: error: calling %s() when not appropriate.\n",
			__FUNCTION__));
	DEBUG(1, ERROR3("initlevel is %i, wanted %i.\n",
			_fb_init_level, INITLEVEL_VT));
	return -ENXIO;
    }
    ret = ioctl(_fb_vt_fd, KDMKTONE, ((unsigned long)ms<<16) + freq);
    LEAVING;
    return ret;
}
