/*
 * This file is part of fb, the frame buffer device, a grafics card driver for
 *                                linux.
 *
 *     Copyright (C) 1995 Pascal Haible (haible@IZFM.Uni-Stuttgart.DE)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Dipl.-Ing. Pascal Haible (haible@IZFM.Uni-Stuttgart.DE)
 *
 */

/* et4000.c */

#include <linux/autoconf.h>
#include <linux/module.h>
#include <linux/types.h>	/* NULL */
#include <linux/kernel.h>	/* printk() */
#include <linux/mm.h>
#include <asm/io.h>
#include <linux/errno.h>

#include "fb.h"
#include "et4000.h"
#include "chipset.h"

#define VGA_MAPS 5

/* #define DEBUG */

#ifdef DEBUG
#define DEB(__a) __a
#else
#define DEB(__a)
#endif

#include "fb_debug.h"

/* variables */

#include "et4000_m.h"
fb_mode_t *chip_modesinfo = __chip_modesinfo;

/* this kind of initialization is gcc specific and not portable, but       */
/* who would want to compile a linux kernel module with an other compiler? */

fb_info_t chip_info = {
driver_name:	"et4000",
chipset_name:	"et4000",
copyright:	"(c) 1994 Pascal Haible",
info:		"Tseng ET4000",
rev_major:	0,
rev_minor:	1,
rev_tiny:	0,
totalmem:	1024*1024,
memtype:	0,			/* 0: DRAM, 1: VRAM */
number_modes:	(sizeof(__chip_modesinfo) / sizeof(fb_mode_t)),
default_mode:	0,
};

unsigned long chip_mode = 0;

void chip_do_bank0r(int nr);
void chip_do_bank0w(int nr);
void chip_do_bank0rw(int nr);

/* We rely on these being inited with zeros  */
static struct mapping_desc maps_0r[VGA_MAPS], maps_0w[VGA_MAPS];

struct chip_bank_desc chip_banks[2] = {
	/* VGA bank 0, read only, */
	{0xA0000UL, (64 * 1024) / PAGE_SIZE, 16, 0, chip_do_bank0r, chip_do_bank0rw,
		maps_0r, maps_0r + VGA_MAPS, maps_0r},
	/* VGA bank 0, write only, */
	{0xA0000UL, (64 * 1024) / PAGE_SIZE, 16, 0, chip_do_bank0w, NULL,
		maps_0w, maps_0w + VGA_MAPS, maps_0w},
	};

short chip_num_banks = 2;

static void et4000_unlock(void);
static void et4000_lock(void);
static void et4000_getmem(void);
static int chip_saveregs(struct chip_regs_s *regs);
static int chip_restoreregs(struct chip_regs_s *regs);

unsigned char *chip_test_chipset(void)
{
	unsigned char new, old, val;

	ENTERING;
	/* This is the very first thing to do! */
	vga_color_mono();

	et4000_unlock();
	/* test for Tseng ET3000 / ET4000: */
	/* are bits 0 - 5 of the segment select register r/w ? */
	old = INB(ET4_SEGSEL);
	OUTB(ET4_SEGSEL,0x29);	/* arbitrary value != all 1 */
	new = INB(ET4_SEGSEL);
	OUTB(ET4_SEGSEL,old);

	/* return false if not Tseng */
	if (new != 0x29) {
		et4000_lock();
		return NULL;
	}

	/* test for ET4000 */
	/* test if bit4-7 of crt33 can be cleared */
	OUTB(CRT_I, 0x33);
	old = INB(CRT_D);
	new = old ^ 0xf;
	OUTB(CRT_D, new);
	val = INB(CRT_D);
	OUTB(CRT_D, old);
	if (val==new) {
		/* OK, ET4000 */
		/* tests for /W32 etc. left out */
		et4000_lock();
		LEAVING;
		return "ET4000";
	} else {
		et4000_lock();
		return NULL;
	}
}

int chip_init(void)
{
	ENTERING;

	init_mapdesc(maps_0r, VGA_MAPS);
	init_mapdesc(maps_0w, VGA_MAPS);

	et4000_getmem();

	vga_init();

	LEAVING;
	return 0;
}

void chip_cleanup(void)
{
	DOING;
}

void chip_leave(void)
{
	DOING;
	et4000_lock();
}

unsigned long chip_savestate_getsize(void)
{
	return sizeof(struct chip_regs_s);
}

int chip_graphics(void)
{
	ENTERING;
	chip_restoreregs(&(chip_regs_graph[chip_mode]));
	et4000_unlock();
	LEAVING;
	return 0;
}

static int chip_bank0r = 0;
static int chip_bank0w = 0;

void chip_do_bank0rw(int nr)
{
	DEB(printk(KERN_DEBUG "chip_do_bank0rw(%i)\n",nr);)
	nr &= 15;
	chip_bank0r = nr;
	chip_bank0w = nr;
	OUTB(ET4_SEGSEL, (nr << 4) | nr );
}

void chip_do_bank0r(int nr)
{
	DEB(printk(KERN_DEBUG "chip_do_bank0r(%i)\n",nr);)
	nr &= 15;
	chip_bank0r = nr;
	OUTB(ET4_SEGSEL, (nr << 4) | chip_bank0w );
}

void chip_do_bank0w(int nr)
{
	DEB(printk(KERN_DEBUG "chip_do_bank0w(%i)\n",nr);)
	nr &= 15;
	chip_bank0w = nr;
	OUTB(ET4_SEGSEL, (chip_bank0r << 4) | nr );
}

int chip_blank(void)
{
	DOING;
	vga_screenoff();
	return 0;
}

int chip_unblank(void)
{
	DOING;
	vga_screenon();
	return 0;
}

int chip_setorigin(fb_origin_t *origin)
{
	DOING;
	return -ENXIO;
}

int chip_setcolormap(fb_cmap_t *cmap)
{
	DOING;
	return -ENXIO;
}

int chip_getcolormap(fb_cmap_t *cmap)
{
	DOING;
	return -ENXIO;
}

void chip_test_test(void)
{
#if 0
	unsigned char old, new, val;
	DOING;
	OUTB(CRT_I, 0x33);
	old = INB(CRT_D);
	new = old ^ 0xff;
	OUTB(CRT_D, new);
	val = INB(CRT_D);
/*	OUTB(CRT_D, old); */
	printk(KERN_DEBUG "fb: et4000.c: crt33: old=%i, new=%i, val=%i\n",
				old,new,val);
#endif
#if 0
	unsigned char old, new, val;
	DOING;
	/* extended attribute register att16 */
	INB(IS1_R);		/* reset flip flop */
	OUTB(ATT_IW,0x16);
	old=INB(ATT_R);

	new = old ^ 0x0f;
	INB(IS1_R);		/* reset flip flop */
	OUTB(ATT_IW,0x16);
	OUTB(ATT_IW,new);

	INB(IS1_R);		/* reset flip flop */
	OUTB(ATT_IW,0x16);
	val=INB(ATT_R);

	printk(KERN_DEBUG "fb: et4000.c: att16: old=%i, new=%i, val=%i\n",
				old,new,val);
#endif
}


/*****************************************************************************/
/* chipset private functions */
/*****************************************************************************/

/*
 * ET4000 unlock and lock
 * 
 * ET4000 lock MUST NOT be unlocked after switching back to text mode!
 * ET4000 unlock interferes with a secondary monochrome:
 * there bit 1 of 0x3BF (ET4_HC) enables the second page 0xB8000-0xBFFFF,
 * conflicting with VGA text mode! :-((
 *
 * Both Hercules register accesses are apparently really needed...
 */
static void et4000_unlock(void)
{
	DOING;
	/* get access to extended registers */

	/* strange thing with Hercules register :-( */
	OUTB(ET4_HC,0x03);

	if (INB(0x3cc)&0x01)
		OUTB(0x3d8,0xa0);
	else
		OUTB(0x3b8,0xa0);

	/* deprotect CRT register 0x35 by clearing bit 7 of CRT11 */
	OUTB(CRT_I,0x11);
	OUTB(CRT_D,INB(CRT_D)&0x7F);
}

static void et4000_lock(void)
{
	DOING;
	/* protect CRT register 0x35 by setting bit 7 of CRT11 */
	OUTB(CRT_I,0x11);
	OUTB(CRT_D,INB(CRT_D)|0x80);

	if (INB(0x3cc)&0x01)
		OUTB(0x3d8,0x00);
	else
		OUTB(0x3b8,0x00);
	OUTB(0x3d8,0x29);	/* CGA 80x25 text mode (??) */

	OUTB(ET4_HC,0x01);
}

void chip_savestate(void *state, void *font)
{
	chip_saveregs((struct chip_regs_s *)state);
	if (font) {
		vga_savetextfontdata(font);
		chip_restoreregs((struct chip_regs_s *)state);
	}
}

void chip_restorestate(void *state, void *font)
{
	chip_restoreregs((struct chip_regs_s *)state);
	if (font) {
		vga_restoretextfontdata(font);
		chip_restoreregs((struct chip_regs_s *)state);
	}
}

static int chip_saveregs(struct chip_regs_s *regs)
{
	int i;
	int ret;

	ENTERING;
	if ((ret=vga_saveregs(&regs->vga_regs))<0)
		return ret;
	et4000_unlock();

	/* save extended CRT registers: crt30 - crt37, crt3f */
	for (i=0; i<8; i++) {
		OUTB(CRT_I, 0x30+i);
		regs->crt3x[i]=INB(CRT_D);
	}
	OUTB(CRT_I,0x3f);
	regs->crt3f=INB(CRT_D);

	/* save extended sequencer register seq07 */
	OUTB(SEQ_I,7);
	regs->seq07=INB(SEQ_D);

	/* save some other ET4000 specific registers */
	regs->mc=INB(ET4_MCE);
	regs->segsel=INB(ET4_SEGSEL);

	/* save extended attribute register att16 */
	INB(IS1_R);		/* reset flip flop */
	OUTB(ATT_IW,0x16);
	regs->att16=INB(ATT_R);

	et4000_lock();
	LEAVING;
	return ret+8+5*1;
}

static int chip_restoreregs(struct chip_regs_s *regs)
{
	int i;

	ENTERING;
	et4000_unlock();
	/* write some ET4000 specific registers */
	OUTB(ET4_MCE,regs->mc);
	OUTB(ET4_SEGSEL,regs->segsel);

	/* write extended sequencer register seq07 */
	OUTB(SEQ_I,0x07);
	OUTB(SEQ_D,regs->seq07);

	/* write extended CRT registers crt30 - crt37, crt3f */
	for (i=0; i<6; i++) {
		OUTB(CRT_I,0x30+i);
		OUTB(CRT_D,regs->crt3x[i]);
	}
	OUTB(CRT_I,0x36);
	OUTB(CRT_D, (INB(CRT_D)&0x40) | (regs->crt3x[6]&0xbf) );
	OUTB(CRT_I,0x37);
	OUTB(CRT_D, (INB(CRT_D)&0xbb) | (regs->crt3x[7]&0x44) );
	OUTB(CRT_I,0x3f);
	OUTB(CRT_D,regs->crt3f);

	/* write extended attribute register att16 */
	INB(IS1_R);		/* reset flip flop */
	OUTB(ATT_IW,0x16);
	OUTB(ATT_IW,regs->att16);

	et4000_lock();
	i = vga_restoreregs(&regs->vga_regs);
	LEAVING;
	return i;
}

#if 0
void chip_dumpregs(struct chip_regs_s *regs)
{
  printk(KERN_DEBUG "et4000 ext regs:\n");
  printk(KERN_DEBUG "crt3x = %02x %02x %02x %02x %02x %02x %02x %02x\n",
    regs->crt3x[0],regs->crt3x[1],regs->crt3x[2],regs->crt3x[3],
    regs->crt3x[4],regs->crt3x[5],regs->crt3x[6],regs->crt3x[7]);
  printk(KERN_DEBUG "crt3f = %02x, seq07 = %02x, mc = %02x, segsel = %02x, att16 = %02x\n",
    regs->crt3f,regs->seq07,regs->mc,regs->segsel,regs->att16);
  vga_dumpregs(&regs->vga_regs);
}
#endif

static void et4000_getmem(void)
{
	unsigned long chip_mem;
	unsigned char value;

	ENTERING;
	et4000_unlock();
	OUTB(CRT_I,0x37);
	value = INB(CRT_D);
	chip_mem = (value&0x08) ? 256 : 64;
	switch (value & 3) {
	  case 2: chip_mem *= 2;
		  break;
	  case 3: chip_mem *= 4;
		  break;
	}
	if (value&0x80)
	  chip_mem *= 2;
	OUTB(CRT_I, 0x32);
	if (INB(CRT_D) & 0x80) {
		/* Interleaved memory on ET4000/W32i/p: */
		/* Multiply by 2. */
		chip_mem *= 2;
	}

	/* kB -> bytes */
	chip_mem *= 1024;
	chip_info.totalmem = chip_mem;

	et4000_lock();
	LEAVING;
}

int chip_choose_bank(int mode, int address) {
	if(mode & VM_WRITE) {
		if(mode & VM_READ)
			return 0 | NEEDS_NEXT_BANK;
		else    return 1;
	} else {
		return 0;
	}
}

#if 0
unsigned char *chip_getname(void)
{
	return "ET4000";
}

static struct chip_regs_s chip_regs_saved = {
	{ { 0, },
	  { 0, },
	  { 0, },
	  { 0, },
	  0
	},
	{ 0, },
	0, 0, 0, 0, 0 };

/* temporary hack */
int chip_1024x768x256x70(void)
{

	static struct chip_regs_s chip_regs_graph = {
		{ /* VGA */
		  /* 24xCRT */	{ 0x9B,0x7F,0x7F,0x9F,0x84,0x15,0x24,0xF5,0x00,0x60,0x00,0x00,
		  		  0x00,0x00,0x00,0x00,0x02,0x88,0xFF,0x80,0x60,0xFF,0x25,0xAB },
		  /* 21xATT */	{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
		  		  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00 },
		  /*  9xGRA */	{ 0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF },
		  /*  5xSEQ */	{ 0x03,0x01,0x0F,0x00,0x0E },
		  /*    MIS */	0xEF,
		},
		/* 8xCRT3x */	{ 0x00,0x00,0x20,0x00,0x08,0x00,0x43,0x1F },
		/* CRT3f */	0x00,
		/* SEQ07 */	0xBC,
		/* micro */	0x00,
		/* segsel */	0x00,
		/* ATT16 */	0x00 };

	/* chip_saveregs(&chip_regs_saved); */
	chip_restoreregs(&chip_regs_graph);
	return 0;
}

int chip_textmode(void)
{
	chip_restoreregs(&chip_regs_saved);
	return 0;
}
#endif
