/*
 * This file is part of fb, the frame buffer device, a grafics card driver for
 *                                linux.
 *
 *      Copyright (C) 1995 Michael Weller (eowmob@exp-math.uni-essen.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Michael Weller (eowmob@exp-math.uni-essen.de or
 * mat42b@aixrs1.hrz.uni-essen.de) Heiderhoefen 116b,
 * D 46049 Oberhausen, Germany.
 */

/*
 *
 * MICHAEL WELLER DISCLAIMS ALL WARRANTIES WITH REGARD
 * TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL MICHAEL WELLER BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifndef SVGAFB_MACH32_H
#define SVGAFB_MACH32_H

#include <asm/page.h>		/* PAGE_SIZE */
#include "vga.h"
#include "vga_regs.h"

/* Mach32 memory layout. Warning, we will make this dynamical (later), thus
   no defines here, to make badly designed code miscompile. */

#if 0
#define CHIP_BANK_SHIFT		16	/* 64k */
#define CHIP_BANK_SIZE		(1 << CHIP_BANK_SHIFT)
#define CHIP_BANK_MASK		(~(CHIP_BANK_SIZE-1))
#define CHIP_PHYS_BASE		(0xA0000UL)
#define CHIP_PHYS_SIZE		(CHIP_BANK_SIZE)
#define CHIP_BANK0_BASE		(CHIP_PHYS_BASE)
#endif

#define CHIP_MAX_MEM		(4096*1024) /* Mach32 design restriction!!! */
#define CHIP_MIN_BANK_SIZE	(64*1024)

/* Oh great... We have two opposite extremes for Mach32: One 4M page or many 64K pages,
   static definition results in a 64K pointer array in kernel space.. YUK!.
   Until bank_page_pointers is not dynamical allocated, we restrict to use the VGA
   aperture (this is better for worst case debugging anyway) */

#if 0
#define CHIP_MAX_BANK_SIZE	(CHIP_MAX_MEM)
#else
#define CHIP_MAX_BANK_SIZE	(CHIP_MIN_BANK_SIZE)
#endif

#define CHIP_MAX_PAGES_PER_BANK	(CHIP_MAX_BANK_SIZE/PAGE_SIZE)
#define CHIP_MAX_NUMBER_BANKS	(CHIP_MAX_MEM / CHIP_MIN_BANK_SIZE)
#define CHIP_MAX_NUMBER_PAGES	(CHIP_MAX_NUMBER_BANKS * CHIP_MAX_PAGES_PER_BANK)


#define CHIP_PAGES_PER_BANK	(CHIP_BANK_SIZE/PAGE_SIZE)

/* Actually the Mach32 has two. But it is unclear if the second could be mapped,
   and if it could it is at least 1Meg and in almost all cases 4MB, thus making
   the first bank superflous in the second case, handling the paranoid case of
   a 1MB bank for more than 1Meg vga mem is omitted. */

#define ONE_BANK
#undef TWO_BANKS

/* state of the driver: */
typedef struct  {
	unsigned long mach32_baseptr;
	unsigned long mach32_origin;
	fb_mode_t *mach32_mode;

	unsigned int is_vga:1;
	unsigned int unused:31;

	unsigned char mach32_disp_shadow;
	unsigned char mach32_ati2e; /* shadow of ati2e..
				       update on savestate and
				       whenever atie2e could change.
				       maskout bits 0-3!!! */
	unsigned char page_r;
	unsigned char page_w;
	unsigned char page_rh;
	unsigned char page_wh;
	} status_t;

struct chip_regs_s {
	struct vga_regs_s vga_regs;
	status_t driverstat;
	unsigned char ati_ext_regs[33];
	unsigned short mach32_regs[27];
	};


/* port functions */
/* kernel has outb(value,port) */
#define OUTB(_port,_value) outb(_value,_port)
#define INB(_port) inb(_port)
#define OUTW(_port,_value) outw(_value,_port)
#define INW(_port) inw(_port)

/*Actually these are configurable and should be queried from the card.
  But noone (including ATI) does actually do this */
#define ATIPORT         0x1ce
#define ATIOFF          0x80
#define ATISEL(reg)     (ATIOFF+(reg))
#define ATIOUT(reg, value)	OUTW(ATIPORT, ATISEL(reg)|(((unsigned)value)<<8))
#define CRTOUT(reg, value)	{OUTB(CRT_I,reg); OUTB(CRT_D,value);}

static inline unsigned char ATIIN(unsigned port)
	{
	OUTB(ATIPORT,port);
	return INB(ATIPORT+1);
	}
static inline unsigned char CRTIN(unsigned port)
	{
	OUTB(CRT_I,port);
	return INB(CRT_D);
	}
/* This is kernel code, maximal speed required: */
static inline unsigned char rolb5(unsigned char val)
	{
	__asm__( "rolb $5, %0\n": "=q" (val) : "0" (val) );
	return val;
	}

/* I/O ports, You do not expect me to put a MACH32_ in front of these, do you? */
#define CMD             0x9AE8
#define ALU_FG_FN       0xBAEE
#define ALU_BG_FN       0xB6EE
#define EXT_SCISSOR_B   0xE6EE
#define EXT_SCISSOR_L   0xDAEE
#define EXT_SCISSOR_R   0xE6EE
#define EXT_SCISSOR_T   0xDEEE
#define DP_CONFIG       0xCEEE
#define FRGD_MIX        0xBAE8
#define BKGD_MIX        0xBAE8
#define FRGD_COLOR      0xA6E8
#define CUR_X           0x86E8
#define CUR_Y           0x82E8
#define MAJ_AXIS_PCNT   0x96E8
#define DEST_CMP_FN     0xEEEE
#define MULTI_FUNC_CNTL 0xBEE8
#define EXT_FIFO_STATUS 0x9AEE
#define SUBSYS_CNTL     0x42E8
#define SUBSYS_STAT     0x42E8
#define VGA_DAC_MASK    0x3C6
#define SCRATCH_PAD_0   0x52EE
#define DESTX_DIASTP    0x8EE8
#define DESTY_AXSTP     0x8AE8
#define R_SRC_X         0xDAEE
#define SRC_X           0x8EE8
#define SRC_Y           0x8AE8
#define SRC_X_START     0xB2EE
#define SRC_X_END       0xBEEE
#define SRC_Y_DIR       0xC2EE
#define SCAN_TO_X       0xCAEE
#define DEST_X_START    0xA6EE
#define DEST_X_END      0xAAEE
#define DEST_Y_END      0xAEEE
#define GE_STAT         0x9AE8
#define CONF_STAT1      0x12EE
#define CONF_STAT2      0x16EE
#define MISC_OPTIONS    0x36EE
#define MEM_CFG         0x5EEE
#define MEM_BNDRY       0x42EE
#define LOCAL_CNTL      0x32EE
#define CHIP_ID         0xFAEE
#define EXT_GE_CONF     0x7AEE
#define R_EXT_GE_CONF   0x8EEE
#define DISP_CNTL       0x22E8
#define CLOCK_SEL       0x4AEE
#define GE_PITCH        0x76EE
#define GE_OFFSET_HI    0x72EE
#define GE_OFFSET_LO    0x6EEE
#define CRT_PITCH       0x26EE
#define CRT_OFFSET_HI   0x2EEE
#define CRT_OFFSET_LO   0x2AEE
#define H_DISP          0x06E8
#define H_TOTAL         0x02E8
#define H_SYNC_WID      0x0EE8
#define H_SYNC_STRT     0x0AE8
#define V_DISP          0x16E8
#define V_SYNC_STRT     0x1AE8
#define V_SYNC_WID      0x1EE8
#define V_TOTAL         0x12E8
#define R_H_TOTAL       0xB2EE
#define R_H_SYNC_STRT   0xB6EE
#define R_H_SYNC_WID    0xBAEE
#define R_V_TOTAL       0xC2EE
#define R_V_DISP        0xC6EE
#define R_V_SYNC_STRT   0xCAEE
#define R_V_SYNC_WID    0xD2EE
#define SHADOW_SET      0x5AEE
#define SHADOW_CTL      0x46EE
#define MISC_CTL        0x7EEE
#define R_MISC_CTL      0x92EE
#define DISP_STATUS     0x2E8

#define DAC_W_INDEX     0x02EC
#define DAC_DATA        0x02ED
#define DAC_MASK        0x02EA
#define DAC_R_INDEX     0x02EB

#define DAC0            0x02EC
#define DAC1            0x02ED
#define DAC2            0x02EA
#define DAC3            0x02EB

/* Useful bit masks: */
#define GE_BUSY         0x0200
#define ONLY_8514       1
#define LOCAL_BUS_CONF2 (1<<6)
#define BUS_TYPE        0x000E
#define PCI             0x000E
#define ISA             0x0000
#define Z4GBYTE         (1<<13)

/* Type for internal and EEPROM Mach32 mode tables: */
typedef struct  {
                unsigned char vfifo16,vfifo24;
                unsigned char h_disp, h_total, h_sync_wid, h_sync_strt;
                unsigned short v_total, v_disp, v_sync_strt;
                unsigned char disp_cntl, v_sync_wid;
                unsigned short clock_sel, flags, mask, offset;
                } mode_entry;

#endif
