/*
 * This file is part of fb, the frame buffer device, a grafics card driver for
 *                                linux.
 *
 *     Copyright (C) 1995 Pascal Haible (haible@IZFM.Uni-Stuttgart.DE)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Dipl.-Ing. Pascal Haible (haible@IZFM.Uni-Stuttgart.DE)
 *
 */

/* vga.c */
/* copied a lot from svgalib */

#include <linux/autoconf.h>
#include <linux/module.h>
#include <linux/kernel.h>	/* printk() */
#include <asm/io.h>
#include "fb.h"
#include "vga.h"
#include "config.h"

/* #define DEBUG */

#ifdef DEBUG
#define DEB(__a) __a
#else
#define DEB(__a)
#endif

#include "fb_debug.h"

/* port functions */
/* kernel has outb(value,port) */
#define OUTB(_port,_value) outb(_value,_port)
#define OUTW(_port,_value) outw(_value,_port)
#define INB(_port) inb(_port)

/* #define DISABLE_VIDEO_OUTPUT */

/* variables */
/* color / mono ? */
int CRT_I = CRT_IC;		/* current CRT index register address */
int CRT_D = CRT_DC;		/* current CRT data register address */
int IS1_R = IS1_RC;		/* current input status register addres READ */
int FCR_W = FCR_WC;		/* current Feature Control Register WRITE */

struct vga_regs_s vga_regs_640x350x16 = {
/* 24xCRT */  { 0x5F,0x4F,0x50,0x82,0x54,0x80,0xBF,0x1F,0x00,0x40,0x00,0x00,
		0x00,0x00,0x00,0x00,0x83,0x85,0x5D,0x28,0x0F,0x63,0xBA,0xE3 },
/* 21xATT */  { 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
		0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00, },
/*  9xGRA */  { 0x00,0x0F,0x00,0x20,0x00,0x00,0x05,0x0F,0xFF },
/*  5xSEQ */  { 0x03,0x01,0x0F,0x00,0x06 },
/*    MIS */  0xA3 };

void __vga_delay(void) {
	int i;
	/* busy loop delay -- Erk */
	for (i = 0; i < 10; i++);
}

void vga_screenoff(void)
{
	DOING;
	/* turn off screen */
	OUTB(SEQ_I, 0x01);
	OUTW(SEQ_I, ((((unsigned) INB(SEQ_D)) | 0x20) << 8) | 0x01);
#ifdef DISABLE_VIDEO_OUTPUT
	/* disable video output */
	INB(IS1_R);
	__vga_delay();
	OUTB(ATT_IW, 0x00);
#endif
}

void vga_screenon(void)
{
	DOING;
	/* turn screen back on */
	OUTB(SEQ_I, 0x01);
	OUTW(SEQ_I, ((((unsigned) INB(SEQ_D)) & 0xDF) << 8) | 0x01);
	/* enable video output */
	INB(IS1_R);
	__vga_delay();
	OUTB(ATT_IW, 0x20);
}

/* The XFree86 server uses a slow copy, may help us too ... */
static void slowcpy(unsigned char *dest, unsigned char *src, unsigned bytes)
{
    while (bytes-- > 0)
	*(dest++) = *(src++);
}

void vga_savetextfontdata(void *buf)
{
	ENTERING;
	/* save font data - first select a 16 color graphics mode */
	vga_restoreregs(&vga_regs_640x350x16);

	/* save font data in plane 2 */
	OUTB(GRA_I,0x04);
	OUTB(GRA_D,0x02);
	slowcpy(buf,(unsigned char *)VGA_PHYS_BASE, FB_FONT_BUFFER_SIZE);

#if 0
	/* save font data in plane 3 */
	OUTB(GRA_I,0x04);
	OUTB(GRA_D,0x03);
	slowcpy(buf+FONT_BUF,(unsigned char *)VGA_PHYS_BASE,FONT_BUF);
#endif
	LEAVING;
}

void vga_restoretextfontdata(void *buf)
{
	ENTERING;

	/* restore font data - first select a 16 color graphics mode */
	vga_restoreregs(&vga_regs_640x350x16);

	/* disable Set/Reset Register */
	OUTB(GRA_I,0x01);
	OUTB(GRA_D,0x00);

	/* save font data in plane 2 */
	OUTB(SEQ_I,0x02);
	OUTB(SEQ_D,0x04);
	slowcpy((unsigned char *)VGA_PHYS_BASE, buf, FB_FONT_BUFFER_SIZE);

#if 0
	/* save font data in plane 3 */
	OUTB(SEQ_I,0x02);
	OUTB(SEQ_D,0x08);
	slowcpy((unsigned char *)VGA_PHYS_BASE,buf+FONT_BUF,FONT_BUF);
#endif

	LEAVING;
}

int vga_savepalette_getsize(void)
{
	DOING;
	return 3*256;
}

int vga_savepalette(unsigned char *r, unsigned char *g, unsigned char *b)
{
	int i;

	DOING;
	/* save graphics mode palette - first select palette index 0 */
	OUTB(PEL_IR, 0);

	for (i=0; i<256; i++) {
		__vga_delay();
		*(r++) = INB(PEL_D);
		__vga_delay();
		*(g++) = INB(PEL_D);
		__vga_delay();
		*(b++) = INB(PEL_D);
	}
	return 0;
}

void vga_restorepalette(unsigned char *r, unsigned char *g, unsigned char *b)
{
	int i;

	DOING;
	/* restore saved palette */
	OUTB(PEL_IW, 0);

	/* read RGB components - index is autoincremented */
	for (i=0; i<256; i++) {
		__vga_delay();
		OUTB(PEL_D,*(r++));
		__vga_delay();
		OUTB(PEL_D,*(g++));
		__vga_delay();
		OUTB(PEL_D,*(b++));
	}
}

int vga_saveregs_getsize(void)
{
	DOING;
	return sizeof(struct vga_regs_s);
}

int vga_saveregs(struct vga_regs_s *regs)
{
	int i;
	DOING;
	for (i=0; i<CRT_C; i++) {
		OUTB(CRT_I,i);
		regs->crt[i] = INB(CRT_D);
	}
	for (i=0; i<ATT_C; i++) {
		INB(IS1_R);
		__vga_delay();
		OUTB(ATT_IW,i);
		__vga_delay();
		regs->att[i] = INB(ATT_R);
		__vga_delay();
	}
	for (i=0; i<GRA_C; i++) {
		OUTB(GRA_I,i);
		regs->gra[i] = INB(GRA_D);
	}
	for (i=0; i<SEQ_C; i++) {
		OUTB(SEQ_I,i);
		regs->seq[i] = INB(SEQ_D);
	}
	regs->misc = INB(MIS_R);

	return CRT_C + ATT_C + GRA_C + SEQ_C + 1;
}

int vga_restoreregs(struct vga_regs_s *regs)
{
	int i;

	DOING;
	/* update misc output register */
	OUTB(MIS_W,regs->misc);

	/* synchronous reset on */
	OUTB(SEQ_I,0x00);
	OUTB(SEQ_D,0x01);

	/* write sequencer registers */
	OUTB(SEQ_I,1);
	OUTB(SEQ_D,regs->seq[1]|0x20);
	for (i=2; i<SEQ_C; i++) {
		OUTB(SEQ_I,i);
		OUTB(SEQ_D,regs->seq[i]);
	}

	/* synchronous reset off */
	OUTB(SEQ_I,0x00);
	OUTB(SEQ_D,0x03);

	/* deprotect CRT registers 0-7 */
	OUTB(CRT_I,0x11);
	OUTB(CRT_D,INB(CRT_D)&0x7F);

	/* write CRT registers */
	for (i=0; i<CRT_C; i++) {
		OUTB(CRT_I,i);
		OUTB(CRT_D,regs->crt[i]);
	}

	/* write graphics controller registers */
	for (i=0; i<GRA_C; i++) {
		OUTB(GRA_I,i);
		OUTB(GRA_D,regs->gra[i]);
	}

	/* write attribute controller registers */
	for (i=0; i<ATT_C; i++) {
		INB(IS1_R);		/* reset flip-flop */
		__vga_delay();
		OUTB(ATT_IW,i);
		__vga_delay();
		OUTB(ATT_IW,regs->att[i]);
		__vga_delay();
	}

	return 0;
}

void vga_dumpregs(struct vga_regs_s *regs)
{
  printk(KERN_DEBUG "vga regs:\n");
  printk(KERN_DEBUG "crt 0-23:\n");
  printk(KERN_DEBUG "%02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
    regs->crt[0],regs->crt[1],regs->crt[2],regs->crt[3],regs->crt[4],regs->crt[5],
    regs->crt[6],regs->crt[7],regs->crt[8],regs->crt[9],regs->crt[10],regs->crt[11]);
  printk(KERN_DEBUG "%02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
    regs->crt[12],regs->crt[13],regs->crt[14],regs->crt[15],regs->crt[16],regs->crt[17],
    regs->crt[18],regs->crt[19],regs->crt[20],regs->crt[21],regs->crt[22],regs->crt[23]);
  printk(KERN_DEBUG "att 0-20:\n");
  printk(KERN_DEBUG "%02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
    regs->att[0],regs->att[1],regs->att[2],regs->att[3],regs->att[4],regs->att[5],
    regs->att[6],regs->att[7],regs->att[8],regs->att[9],regs->att[10],regs->att[11]);
  printk(KERN_DEBUG "%02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
    regs->att[12],regs->att[13],regs->att[14],regs->att[15],regs->att[16],regs->att[17],
    regs->att[18],regs->att[19],regs->att[20]);
  printk(KERN_DEBUG "gra 0-8:\n");
  printk(KERN_DEBUG "%02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
    regs->gra[0],regs->gra[1],regs->gra[2],regs->gra[3],regs->gra[4],regs->gra[5],
    regs->gra[6],regs->gra[7],regs->gra[8]);
  printk(KERN_DEBUG "seq 0-4: %02x %02x %02x %02x %02x\n",
    regs->seq[0],regs->seq[1],regs->seq[2],regs->seq[3],regs->seq[4]);
  printk(KERN_DEBUG "misc: %02x\n",regs->misc);
}

void vga_color_mono(void)
{
	DOING;
	/* if IOA (bit 0 of Misc. Output Register) is not set, the */
	/* VGA is in mono emulation, some regs go from 0x3Dx to 0x3Bx */
	if (INB(MIS_R)&0x01) {
		CRT_I = CRT_IC;
		CRT_D = CRT_DC;
		IS1_R = IS1_RC;
		FCR_W = FCR_WC;
	} else {
		CRT_I = CRT_IM;
		CRT_D = CRT_DM;
		IS1_R = IS1_RM;
		FCR_W = FCR_WM;
	}
}

void vga_init(void)
{
	DOING;
	/* this is already called from chip_test_chipset() */
	/* vga_color_mono(); */
}
