/*\
|*| --------------------------------------------------------------------------
|*|	S3 Vision964 chipset driver
|*| --------------------------------------------------------------------------
|*|
|*|	Copyright (C) 1995 by Steffen Seeger, [seeger@physik.tu-chemnitz.de]
|*|
|*|	This program is free software; you can redistribute it and/or modify
|*|	it under the terms of the GNU General Public License as published by
|*|	the Free Software Foundation; either version 2, or (at your option)
|*|	any later version.
|*|
|*|	This program is distributed in the hope that it will be useful,
|*|	but WITHOUT ANY WARRANTY; without even the implied warranty of
|*|	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|*|	GNU General Public License for more details.
|*|
|*|	You should have received a copy of the GNU General Public License
|*|	along with this program; see the file COPYING.  If not, write to
|*|	the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
|*|
|*| ---------------------------------------------------------------------------
\*/


/*\
|*|	Kernel includes
\*/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <asm/io.h>

extern void request_region(unsigned int from, unsigned int num,
                           const char *name);
extern int  check_region  (unsigned int from, unsigned int num);
extern void release_region(unsigned int from, unsigned int num);

/*\
|*|     GGI includes
\*/
#undef DEBUG_IO

#include "graphdev.h"
#include "kgi-module.h"
#include "S3/Vision964.h"
#include "S3/s3_io.h"

#define DRIVER_VERSION	"v0.00 ALPHA"

/*\
|*|	exported variables
\*/
#define IO_BASE   0x03B0
#define IO_REGION 0x03E0-IO_BASE

unsigned short INS1r;
unsigned short CRTI,CRTD;
unsigned short ATCR;
unsigned short FCTRLw;


static void
s3_unlock(void)
{
  outbCRT(CR38_MAGIC, CRT_LOCK1);	/* unlock CR30-CR3F */
  outbCRT(CR39_MAGIC, CRT_LOCK2);	/* unlock CR40-CRFF */
}

static void 
s3_lock(void)
{
  outbCRT(0, CRT_LOCK1);		/* lock CR30-CR3F */
  outbCRT(0, CRT_LOCK2);		/* lock CR40-CRFF */
}

static void
outbMISC(unsigned char value)
{
  outb(value, MISCw);			/* put value      */
  if (value & MISC_COLOR_IO)		/* adjust IO base */
    { CRTI=CRTIc; CRTD=CRTDc; ATCR=ATCRc; INS1r=INS1rc; FCTRLw=FCTRLwc; }
   else
    { CRTI=CRTIm; CRTD=CRTDm; ATCR=ATCRm; INS1r=INS1rm; FCTRLw=FCTRLwm; };
}

#define NoSEQRegs	0x05
#define	NoCRTRegs	0x80
#define NoGRCRegs	0x09
#define NoATCRegs	0x15

struct StateTable
{
   unsigned char MISC;
   unsigned char FCTRL;
   unsigned char SEQ[NoSEQRegs];
   unsigned char CRT[NoCRTRegs];
   unsigned char GRC[NoGRCRegs];
   unsigned char ATC[NoATCRegs];
};

static struct StateTable InitialState;

static struct StateTable
DefaultState =
 {
     0x23, 0x00,					/* MISC, FCTRL */
   {
     0x03, 0x00, 0x0F, 0x00, 0x0E			/* SEQ */
   },
   {
     0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x10,	/* CRT */
     0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xE3,		/* 0x1x */
     0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,		/* 0x2x */
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

     0x00, 0x09, 0x00, 0x20, 0x10, 0x00, 0x00, 0x00,		/* 0x3x */
     0x4A, 0xAD, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

     0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,		/* 0x4x */
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

     0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00,		/* 0x5x */
     0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00,

     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,		/* 0x6x */
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,		/* 0x7x */
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
   },
   {
     0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x05, 0x0F, 0xff /* GRC */
   },
   {
     0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,	/* ATC */
     0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
     0xC0, 0x00, 0x0F, 0x00, 0x00
   }
 };


static void
SaveSEQRegs(unsigned char *buf)
{ int i;
  for(i=0; i < NoSEQRegs; i++)
   {
     outb(i, SEQI); *(buf++)=inb(SEQD);
     #ifdef DEBUG_IO
       if (!(i & 0xF))
         printk("\n%.2x: ",i);
       printk("%.2X, ",*(buf-1));
     #endif
   }
  #ifdef DEBUG_IO
    printk("\n");
  #endif
}

static void
RestoreSEQRegs(unsigned char *buf)
{ register unsigned char i;
  for(i=0; i < NoSEQRegs; i++)
   { outb(i,SEQI); outb(*(buf++), SEQD);}
}

static void
SaveCRTRegs(unsigned char *buf)
{ int i;
  for(i=0; i < NoCRTRegs; i++)
   { 
     outb(i, CRTI); *(buf++)=inb(CRTD);
     #ifdef DEBUG_IO
       if (!(i & 0xF))
         printk("\n%.2x: ",i);
       printk("%.2X, ",*(buf-1));
     #endif
   }
  #ifdef DEBUG_IO
    printk("\n");
  #endif
}

static void
RestoreCRTRegs(unsigned char buf[])
{ register unsigned char i;
  outbCRT(inbCRT(CRT_VSYNCEND) & ~CR11_LOCKTIMING, CRT_VSYNCEND);
  for(i=0x00; i < 0x19; i++) outbCRT(buf[i], i); 
  for(i=0x30; i < 0x3D; i++) outbCRT(buf[i], i);
  for(i=0x40; i < 0x6E; i++)
    switch(i) 
     {
       case 0x42:
	 outbCRT((inbCRT(i) & CR42_DCLK_MASK) | (buf[i] & ~CR42_DCLK_MASK), i); 
	 break;
       case 0x55:
	 outbCRT((inbCRT(i) & CR55_DAC_MASK) | (buf[i] & ~CR55_DAC_MASK), i);
       default: 
         outbCRT(buf[i], i);
     }
}

static void
SaveGRCRegs(unsigned char *buf)
{ int i;
  for(i=0; i < NoGRCRegs; i++) 
   {
     outb(i, GRCI); *(buf++)=inb(GRCD);
     #ifdef DEBUG_IO
       if (!(i & 0xF))
         printk("\n%.2x: ",i);
       printk("%.2X, ",*(buf-1));
     #endif
   }
  #ifdef DEBUG_IO
    printk("\n");
  #endif
}

static void
RestoreGRCRegs(unsigned char *buf)
{ register unsigned char i;
  for(i = 0; i < NoGRCRegs; i++)
   { outb(i, GRCI); outb(*(buf++), GRCD); }
}

static void
SaveATCRegs(unsigned char *buf)
{ int i;
  for(i=0; i < NoATCRegs; i++)
   { 
     inb(FCTRLw); outb(i, ATCI);
     *(buf++)=inb(ATCD);  /* ^^ reset AFF,  ^^ setup index */
     #ifdef DEBUG_IO
       if (!(i & 0xF))
         printk("\n%.2x: ",i);
       printk("%.2X, ",*(buf-1));
     #endif
   }
  inb(FCTRLw); outb(ATCI_ENABLEDISPLAY, ATCI);
}

static void
RestoreATCRegs(unsigned char *buf)
{ register unsigned char i;
  inb(FCTRLr);	/* initialize AFF */
  for(i=0; i < NoATCRegs; i++)
   { outb(i,ATCI); outb(*(buf++),ATCI); }
  outb(ATCI_ENABLEDISPLAY, ATCI);
}

static void SaveState(struct StateTable *state)
 {
   s3_unlock();
   state->MISC  = inb(MISCr);	/* save miscellaneous output reg */
   #ifdef DEBUG_IO
     printk("MISC: %.2X\n",state->MISC);
   #endif
   state->FCTRL = inb(FCTRLr);	/* save feature control reg */
   SaveSEQRegs(state->SEQ);	/* save sequencer regs */
   SaveCRTRegs(state->CRT);	/* save cathode ray tube controller regs */
   SaveGRCRegs(state->GRC);	/* save graphics controller regs */
   SaveATCRegs(state->ATC);	/* save attribute controller regs */
 }

static void RestoreState(struct StateTable *state)
 {
   outbMISC((inb(MISCr) & MISC_CLOCK_MASK) | state->MISC);
   outb(state->FCTRL, FCTRLw);
   RestoreSEQRegs(state->SEQ);
   RestoreCRTRegs(state->CRT);
   RestoreGRCRegs(state->GRC);
   RestoreATCRegs(state->ATC);
 }

static void DumpState(struct StateTable *state)
{ int i;
  printk("%.2x\n",state->MISC);
  printk("%.2x\n",state->FCTRL);
  for(i = 0; i < NoSEQRegs; i++)
   { if(!(i & 0xF)) printk("\n%.2x: ",i); printk("%.2x, ",state->SEQ[i]); }
  for(i = 0; i < NoCRTRegs; i++)
   { if(!(i & 0xf)) printk("\n%.2x: ",i); printk("%.2x, ",state->CRT[i]); }
  for(i = 0; i < NoGRCRegs; i++)
   { if(!(i & 0xF)) printk("\n%.2x: ",i); printk("%.2x, ",state->GRC[i]); }
  for(i = 0; i < NoATCRegs; i++)
   { if(!(i & 0xF)) printk("\n%.2x: ",i); printk("%.2x, ",state->ATC[i]); }
}


static struct GGI_ChipParams
chipinfo =
 {
   0,			/* installed RAM, entered during initialisation */
   2048, 2048,		/* register limits for maximum resolution */
   {16000000, 75000000},/**/
   "S3 Incorporated",	/* manufacturer */
   "Vision 984",	/* model */

      GT_TEXT      	/* supported graphmodes */
    | GT_4BIT
    | GT_8BIT
    | GT_15BIT | GT_16BIT
    | GT_32BIT
 };

static int
Detect_Vision964(void)
{
  #define COPYDEFAULT(X) DefaultState.CRT[(X)]=InitialState.CRT[(X)]
  register unsigned char foo;

  foo = COPYDEFAULT(CRT_CHIPID);
  if((foo & CR30_CHIPID) != CR30_Vision964ID)
    {
      printk("  Vision964 not detected (ID = %.2x).\n", foo & CR30_CHIPID);
      return FALSE;
    }
  TRACE(printk("  Vision964 rev %i detected.\n", foo & CR30_REVISION));


  foo = COPYDEFAULT(CRT_CONFIG1);
   switch(foo & CR36_MEMSIZE_MASK)
    {
      case CR36_1MB: chipinfo.ramsize = 1;	break;
      case CR36_2MB: chipinfo.ramsize = 2;	break;
      case CR36_3MB: chipinfo.ramsize = 3;	break;
      case CR36_4MB: chipinfo.ramsize = 4;	break;
      case CR36_6MB: chipinfo.ramsize = 6;	break;
      case CR36_8MB: chipinfo.ramsize = 8;	break;
      default:  chipinfo.ramsize = 1;
                printk("%s: can't determine VRAM size, assume 1MB.\n",__FILE__);
    }

   #ifdef DEBUG
    switch(foo & CR36_RAMMASK)
     {
       case CR36_FASTPAGERAM: printk("  %i MB FP RAM, ",  chipinfo.ramsize); break;
       case CR36_EDO_RAM:     printk("  %i MB EDO RAM, ", chipinfo.ramsize); break;
       default:		      printk("  %i MB unknown RAM, ", chipinfo.ramsize);
     }
    switch(foo & CR36_BUSMASK)
     {
       case CR36_PCI      :   printk("PCI bus\n");        break;
       case CR36_VESALOCAL:   printk("Vesa Local bus\n"); break;
       default:               printk("unknown bus\n");
     }
   #endif
   chipinfo.ramsize *= 0x100000; /* convert MB to Byte */

  COPYDEFAULT(CRT_CONFIG2);
  DefaultState.CRT[CRT_MISC1]      |=  (InitialState.CRT[CRT_MISC1]
				     & (CR3A_PCIMASK | CR3A_REFRESHMASK));
  DefaultState.CRT[CRT_SYSCONFIG]  |=  (InitialState.CRT[CRT_SYSCONFIG]
				     & (CR40_BUSMASK));
  DefaultState.CRT[CRT_MEMCONTROL1]|=  (InitialState.CRT[CRT_MEMCONTROL1]
				     & (CR53_64BIT_VRAM));
  DefaultState.CRT[CRT_LAWCONTROL] |=  (InitialState.CRT[CRT_LAWCONTROL]
				     & (CR58_SAM_256 | CR58_VL_T1_LATCH));
/*  DefaultState.CRT[CRT_GOUT_PORT]  |=  (InitialState.CRT[CRT_LAWCONTROL]
				     & (CR5C_GOUT_MASK));
*/
  COPYDEFAULT(CRT_MEMCONTROL3);
  COPYDEFAULT(CRT_MEMCONTROL4);
  COPYDEFAULT(CRT_MEMCONTROL5);
  DefaultState.CRT[CRT_EXTMISC0]   |=  (InitialState.CRT[CRT_EXTMISC0]
				     & (CR65_SAM_MASK));
  DefaultState.CRT[CRT_EXTMISC1]   |=  (InitialState.CRT[CRT_EXTMISC1]
				     & (CR66_SID_MASK | CR66_PCI_DISCONNECT));
  COPYDEFAULT(CRT_CONFIG3);

  /* temporary, using LAW at F3000000 */
  DefaultState.CRT[0x59] = 0xF3;
  DefaultState.CRT[0x5A] = 0x00;
  
  return TRUE;
  #undef COPYDEFAULT
}

/*\
|*|	GGI interface
\*/

int
kgi_ChipsetInit(void)
{
  int detected; 

  TRACE(printk("S3 964 Chipset Driver "DRIVER_VERSION"\n"));

  outb(S3_DISABLE, S3_SUBSYSTEM);	/* chip wakeup */
  outb(S3_WAKEUP,  S3_SETUP);
  outb(S3_ENABLE,  S3_SUBSYSTEM);

  outbMISC(inb(MISCr));	/* adjust IO base */

  SaveState(&InitialState);/* unlock S3 extended regs and save state */
  if((detected = Detect_Vision964()))
    {
      if(! check_region(IO_BASE, IO_REGION))
        { 
          request_region(IO_BASE, IO_REGION, "Vision964");
 	  return TRUE;
        } 
       else   
        {
	  printk("%s: warning: can't get IO region.\n",__FILE__);
          return TRUE;
        }
    }
   else
    return FALSE;
}

void
kgi_ChipsetDone(void)
{
  release_region(IO_BASE, IO_REGION);
  TRACE(printk("S3 Chipset Driver removed\n"));
/*  s3_lock(); */
}

void
kgi_ChipsetGetParameters (struct GGI_ChipParams *params)
{
  *params = chipinfo;
}

int
kgi_ChipsetCheckTiming (struct GGI_Timing *TM)
{
  if(TM->flags & FL_INTERLACE)
   {
     TRACE(printk("%s: sorry, interlaced modes not supported (yet).\n",__FILE__));
     return FALSE; 
   }
  if(TM->clock == 0x7FFFFFFF)	/* first call ??? */
   {
     int Bpp = 1;

     if(TM->tr.graphtype == GT_24BIT)
      TM->tr.graphtype = GT_32BIT;

     if(!(TM->tr.graphtype & chipinfo.graphtypes))
      {
	TRACE(printk("%s: graphtype %i not supported.\n",__FILE__,\
		     TM->tr.graphtype));
	return FALSE;
      }

     switch(TM->tr.graphtype)
      {
	case GT_TEXT:
	  break;

	case GT_32BIT:	Bpp +=4;	/* (4+2+1+1)/2 = 4   Bpp */
	case GT_15BIT:
	case GT_16BIT:	Bpp +=2;	/*   (2+1+1)/2 = 2   Bpp */
	case GT_8BIT:	Bpp +=1;	/*     (1+1)/2 = 1   Bpp */
	case GT_4BIT:			/*         1/2 = 0.5 Bpp */

	  if (TM->tr.xvirtual <=  640) TM->tr.xvirtual =  640; else
	  if (TM->tr.xvirtual <=  800) TM->tr.xvirtual =  800; else
	  if (TM->tr.xvirtual <= 1024) TM->tr.xvirtual = 1024; else
	  if (TM->tr.xvirtual <= 1152) TM->tr.xvirtual = 1152; else
	  if (TM->tr.xvirtual <= 1280) TM->tr.xvirtual = 1280; else
	  if (TM->tr.xvirtual <= 1600) TM->tr.xvirtual = 1600; else
	  if (TM->tr.xvirtual <= 2048) TM->tr.xvirtual = 2048; else
	   {
	     TRACE(printk("%s: %i horizontal pixels are too many.\n",__FILE__,\
			   TM->tr.xvirtual));
	     return FALSE;
	   }

	  if( ((TM->tr.xvirtual * TM->tr.yvirtual * Bpp)/2) > chipinfo.ramsize)
	   {
	     TRACE(printk("%s: not enough VRAM to do virtual %ix%i at %i bpp",\
		   __FILE__, TM->tr.xvirtual, TM->tr.yvirtual, Bpp*4));
	     return FALSE;
	   }

	  TM->tr.xvisible &= 0x7FFFFFF8; TM->tr.yvisible &= 0x7FFFFFF8;

          return TRUE;

	default:	/* this should not happen ... */

	  INTERNALERROR
	  return FALSE;
      }
     return TRUE;
   }
  else /* TM->clock == 0x7FFFFFFF */
   {
     switch(TM->tr.graphtype)
      {
	case GT_TEXT:
	  break;

	case GT_4BIT:
	case GT_8BIT:
	case GT_15BIT:
	case GT_16BIT:
	case GT_32BIT:
	  if (TM->clock < chipinfo.clock.min)
	   {
	     TRACE(printk("%s: DCLK too low.\n",__FILE__));
	     return FALSE;
	   }
	  if (TM->clockdiv > 5)
	   {
	     TRACE(printk("%s: %i:1 multiplex mode not supported",__FILE__,\
		   1 << TM->clockdiv));
	     return FALSE;
	   }
	  if (TM->clock > chipinfo.clock.max)
	   {
	     TRACE(printk("%s: DCLK too high, set to %i kHz",__FILE__,
	     TM->clock/1000));
	     TM->clock = chipinfo.clock.max;
	   }
	  break;

	default:	/* this should not happen ... */
	  INTERNALERROR
	  return FALSE;
      }
     return TRUE;
   }
}

int
kgi_ChipsetSetTiming (struct GGI_Timing *TM)
{
  struct StateTable setup = DefaultState;
  register int foo;
  int EC00 = EC00_IRQ_MASK;
  int EC01 = EC01_ENH_ENB | EC01_LAW_ENB;

  /*\
  |*|	setup horizontal timing
  \*/
 
   foo = (TM->xwidth >> 3) - 1;			/* horizontal display end */
     setup.CRT[0x01] |= foo & 0xFF;
     setup.CRT[0x5D] |= (foo & 0x100) ? CR5D_HDISPLAYEND_B8 : 0;

   foo = (TM->xsyncstart >> 3) - 1;		/* horizontal blank start */
     setup.CRT[0x02] |= foo & 0xFF;
     setup.CRT[0x5D] |= (foo & 0x100) ? CR5D_HBLANKSTART_B8 : 0;

   foo = TM->xsyncstart >> 3;			/* horizontal sync start  */
     setup.CRT[0x04] |= foo & 0xFF;
     setup.CRT[0x5D] |= (foo & 0x100) ? CR5D_HSYNCSTART_B8 : 0;

   foo = TM->xsyncend >> 3;			/* horizontal sync end    */
     setup.CRT[0x05] |= foo & CR05_HSE_MASK;
     setup.CRT[0x5D] |= ((foo-TM->xsyncend) > 0x20) ? CR5D_HSYNCEND_B5 : 0;

   foo = (TM->xsyncend >> 3); 			/* horizontal blank end   */
     setup.CRT[0x03] |= foo & CR03_HBE_MASK;
     setup.CRT[0x05] |= (foo & 0x20) ? CR05_HBLANKEND_B5 : 0;
     setup.CRT[0x5D] |= (((TM->xsyncstart-TM->xsyncend)>>3) & 0x40) ?
                                   CR5D_HBLANKEND_B6 : 0;

   foo = (TM->xend >> 3) - 5;			/* horizontal total       */
     setup.CRT[0x00] |= foo & 0xFF;
     setup.CRT[0x5D] |= (foo & 0x100) ? CR5D_HTOTAL_B8 : 0;

  /*\
  |*|	setup vertical timing
  \*/
 
   foo = TM->ywidth - 1;			/* vertical display end   */
     setup.CRT[0x12] |= foo & 0xFF;
     setup.CRT[0x07] |= (foo & 0x100) ? CR07_VDISPLAYEND_B8  : 0;
     setup.CRT[0x07] |= (foo & 0x200) ? CR07_VDISPLAYEND_B9  : 0;
     setup.CRT[0x5E] |= (foo & 0x400) ? CR5E_VDISPLAYEND_B10 : 0;

   foo = TM->ysyncstart;			/* vertical blank start   */
     setup.CRT[0x15] |= foo & 0xFF;
     setup.CRT[0x07] |= (foo & 0x100) ? CR07_VBLANKSTART_B8  : 0;
     setup.CRT[0x09] |= (foo & 0x200) ? CR09_VBLANKSTART_B9  : 0;
     setup.CRT[0x5E] |= (foo & 0x400) ? CR5E_VBLANKSTART_B10 : 0;
    
   foo = TM->ysyncstart;			/* vertical sync start    */
     setup.CRT[0x10] |= foo & 0xFF;
     setup.CRT[0x07] |= (foo & 0x100) ? CR07_VSYNCSTART_B8  : 0;
     setup.CRT[0x07] |= (foo & 0x200) ? CR07_VSYNCSTART_B9  : 0;
     setup.CRT[0x5E] |= (foo & 0x400) ? CR5E_VSYNCSTART_B10 : 0;

   foo = TM->ysyncend;				/* vertical sync end      */
     setup.CRT[0x11] |= foo & CR11_VSYNCEND_MASK;
 
   foo = TM->ysyncend + 1;			/* end vertical blank     */
     setup.CRT[0x16] |= foo & 0xFF;
  
   foo = TM->yend - 2;				/* vertical total         */
     setup.CRT[0x06] |= foo & 0xFF;
     setup.CRT[0x07] |= (foo & 0x100) ? CR07_VTOTAL_B8  : 0;
     setup.CRT[0x07] |= (foo & 0x200) ? CR07_VTOTAL_B9  : 0;
     setup.CRT[0x5E] |= (foo & 0x400) ? CR5E_VTOTAL_B10 : 0;
 
 
   setup.CRT[0x09]   |= TM->ymagnify & CR09_CHARHEIGHT_MASK;
   setup.CRT[0x09]   |= (TM->flags & FL_DOUBLESCAN) ?
                                   CR09_DOUBLESCAN : 0;


  foo = (TM->xend + TM->xsyncstart) >> 4; /* (... >> 1) >> 3 */
  setup.CRT[0x3B]     |= foo & 0xFF;
  setup.CRT[0x5D]   |= (foo & 0x100) ? CR5D_FIFOSTART_B8 : 0;

  setup.MISC |= TM->xsyncpol ? MISC_NEG_HSYNC : 0;
  setup.MISC |= TM->ysyncpol ? MISC_NEG_VSYNC : 0;

  setup.CRT[CRT_INTERLACESTART]|= TM->yend >> 1;
  if(TM->flags & FL_INTERLACE)
   {
      TRACE(printk("%s:%i: no interlaced modes (yet).\n",__FILE__,__LINE__));
      setup.CRT[CRT_EXTMODE1]  |= CR42_INTERLACED;
      return FALSE;

   }

  setup.CRT[0x66]      |= TM->clockdiv & CR66_VCLK_DIV_MASK;

  switch(TM->tr.graphtype)
   {
     case GT_4BIT:
       setup.CRT[0x50] |= CR50_4BPP;
       EC01 |= EC01_4BPP;
       break;
     case GT_8BIT:
       setup.CRT[0x3A] |= CR3A_NO4BPPMODE;
       setup.CRT[0x50] |= CR50_8BPP;
       break;
     case GT_15BIT:
     case GT_16BIT:
       setup.CRT[0x3A] |= CR3A_NO4BPPMODE;
       setup.CRT[0x50] |= CR50_16BPP;
       setup.CRT[0x67] |= CR67_NO_INVERT_VCLK;
       break;
     case GT_32BIT:
       setup.CRT[0x3A] |= CR3A_NO4BPPMODE;
       setup.CRT[0x50] |= CR50_32BPP;
       break;

     default:
       INTERNALERROR
       return FALSE;       
   }
  foo = TM->tr.xvirtual >> 3;
   setup.CRT[CRT_LOGICALWIDTH]  |= foo & 0xFF;
   setup.CRT[0x51] |= (foo & 0x100) ? CR51_LOGICALWIDTH_B8 : 0;
   setup.CRT[0x51] |= (foo & 0x200) ? CR51_LOGICALWIDTH_B9 : 0;
  switch(TM->tr.xvirtual)
   { 
     case  640: setup.CRT[0x50] |= CR50_640; break;
     case  800: setup.CRT[0x50] |= CR50_800; break;
     case 1024: setup.CRT[0x50] |= CR50_1024; break;
     case 1152: setup.CRT[0x50] |= CR50_1152; break;
     case 1280: setup.CRT[0x50] |= CR50_1280; break;
     case 1600: setup.CRT[0x50] |= CR50_1600; break;
     case 2048: setup.CRT[0x50] |= CR50_2048; break;
     default: INTERNALERROR return FALSE;
   }
  setup.SEQ[0x01] |= SR1_8DOTCLK;
  setup.ATC[0x10] |= AR10_GRAPHICS;
  setup.GRC[0x06] |= GR6_GRAPHMODE;

  setup.CRT[0x66] |= CR66_PIXELBUS_OFF;
 
  outbCRT(inb(CRT_EXTSYNC1) | CR56_SYNC_OFF, CRT_EXTSYNC1); 
  kgi_MonitorSetTiming(TM);
  kgi_RamdacSetTiming(TM);
  kgi_ClockSetTiming(TM);
  RestoreState(&setup);

  outb(EC00 | EC00_RESET, ECR_MODECONTROL);
  outb(EC00, ECR_MODECONTROL);
  outb(EC01, ECR_FCTRL);
  
  #ifdef DEBUG_IO
    DumpState(&setup);
  #endif
  return TRUE;
}

int
kgi_ChipsetIoctl (struct inode *inode, struct file *file, unsigned int cmd,
		  unsigned long arg)
{
  return -1;
}
