/* 
   S3 964 Chipset-Driver - for SPEA Mercury PCI 2MB VRAM on 16 Apr 1995

   Copyright (C) 1995 Andreas Beck - becka@hp.rz.uni-duesseldorf.de

   If you do any modifications, I would like you to send diffs to me
   to allow for collecting a more and more complete set of drivers and
   to improve existing ones.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#undef S3_805
#undef ACCEL_SUPPORT

#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>

/* THIS IS FOR TESTING PURPOSES ! COMMENT OUT FOR RELEASES ! */
#undef suser
#define suser() 1

#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <linux/malloc.h>
#include <linux/fcntl.h>
#include <linux/vt.h>
#include <sys/ioctl.h>
#include <asm/io.h>
#include <asm/pgtable.h>
#include "graphdev.h"
#include "kgi-module.h"
#include "vga_prg.h"
#include "spea_mercury.h"
#include "vga_generic.h"
#include "generic.h"
#include "S3/regs3.h"
#include "S3/s3.h"

char kernel_version[]= UTS_RELEASE;

/*****************************************************************************
*** Chipset characteristics                                                ***
*****************************************************************************/

struct ggi_ChipParams ChipPar = 
	{	2*1024*1024,		/* 2MB RAM */
		2048,
		1024,
#ifdef S3_805
		"Generic",
		"S3-805/801",
#else		
		"SPEA / Video 7",
		"Mercury PCI S3-964",
#endif
		GT_TEXT|GT_4BIT_PLANES|GT_8BIT,	
		/* more is not implemented now ... */
	};

/*****************************************************************************
*** Enable S3 extended register access                                     ***
*****************************************************************************/
void unlock_s3(void)
{ 
	Outb_CRTC(CRT_LOCK1, CR38_MAGIC);	/* unlock CR30-CR3F */
	Outb_CRTC(CRT_LOCK2, CR39_MAGIC);	/* unlock CR40-CRFF */
	Outb_CRTC(CRT_LOCK,Inb_CRTC(CRT_LOCK)&0xcf);	/* unlock H/VTiming */
	Outb_SEQ (SEQ_UNLOCK, SR8_MAGIC);	/* unlock VESAPower */
}

void lock_s3(void)
{ 
	Outb_CRTC(CRT_LOCK1 , 0);		/* lock CR30-CR3F */
	Outb_CRTC(CRT_LOCK2 , 0);		/* lock CR40-CRFF */
	Outb_SEQ (SEQ_UNLOCK, 0);		/* lock VESAPower */
}

/*****************************************
*** Check if this is a vision 964 chip ***
*****************************************/
static int Detect_Vision964(void)
{
	register unsigned char cr36;

	cr36 = Inb_CRTC(CRT_CHIPID);
	if ( (cr36 & CR30_CHIPID) != 0xD0)
	{
		printk("  Vision964 not detected (ID = %.2x).\n", cr36 & CR30_CHIPID);
		return 0;
	}
	printk("  Vision964 rev %i detected.\n", cr36 & CR30_REVISION);

	cr36 = Inb_CRTC(CRT_CONFIG1);
	switch(cr36 & CR36_MEMSIZE_MASK)
	{
		case CR36_1MB: 
			ChipPar.ramsize = 0x100000;break;
		case CR36_2MB:
			ChipPar.ramsize = 0x200000;break;
		case CR36_3MB: 
			ChipPar.ramsize = 0x300000;break;
		case CR36_4MB:
			ChipPar.ramsize = 0x400000;break;
		case CR36_6MB: 
			ChipPar.ramsize = 0x600000;break;
		case CR36_8MB:
			ChipPar.ramsize = 0x800000;break;
		default:
			printk("%s: can't determine VRAM size, assume 1MB.\n",__FILE__);
	}

	switch(cr36 & CR36_RAMMASK)
	{
		case CR36_FASTPAGERAM:
			printk("  %i FP RAM, ",  ChipPar.ramsize); break;
		case CR36_EDO_RAM:
			printk("  %i EDO RAM, ", ChipPar.ramsize); break;
		default:
			printk("  %i unknown RAM, ", ChipPar.ramsize);
	}

	switch(cr36 & CR36_BUSMASK)
	{
		case CR36_PCI      :
			printk("PCI bus\n");        break;
		case CR36_VESALOCAL:
			printk("Vesa Local bus\n"); break;
		default:
			printk("unknown bus\n");
     }

  return 1;
}


/*****************************************************************************
*** Check/Setup a Text-Mode of the given size                              ***
*****************************************************************************/
int CheckText(struct ggi_Timing *TM,int command)
{ 
	int x;

	if ((x=vga_CheckText(TM))) return x;

	TM->clock=75000000;/* Max for text */

	TM->xwidth=TM->tr.xvisible;
	TM->ywidth=TM->tr.yvisible;
  
	TM->ymagnify=1;
  
	/* We can do nothing against RAMDAC objections */
	if ((x=kgi_CheckRamdacTiming(TM,CMD_NONE))) return(x); 

	/* We can do nothing against Monitor objections except asking for
	   a less perfect mode */
	if ((x=kgi_CheckMonitorTiming(TM,CMD_MON_PERFECT)))
		x=kgi_CheckMonitorTiming(TM,CMD_MON_SOME);

	return x;
}  

/* Set board to a defined state and unlock it */
void ResetCard(void)
{
	static char SEQ [5]=
		{ 0x03, 0x01, 0x03, 0x00, 0x02};
	static char GR  [9]=
		{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x0e, 0x00, 0xFF };
	static char ATR[21]=
		{ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x14, 0x07,
  		  0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		  0x0c, 0x00, 0x0f, 0x00, 0x00 };
	static char CRTC[128]=
		{ 0x82, 0x63, 0x69, 0x9d, 0x6a, 0x9d, 0x88, 0xf0,
		  0x00, 0x6f, 0x0e, 0x0f, 0x00, 0x00, 0x00, 0x64,
		  0x51, 0x22, 0x4f, 0x32, 0x1f, 0x51, 0x52, 0xa3,
 		  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
 		  0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 		  0xd0, 0x05, 0x00, 0x00, 0x00, 0x00, 0x9e, 0x0f,
 		  0x48, 0xa5, 0x05, 0x5a, 0x10, 0x10, 0x10, 0x10,
	 	  0x11, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0xff,
 		  0xfc, 0xff, 0xff, 0x00, 0x1f, 0xff, 0xff, 0xdf,
	 	  0x00, 0x00, 0x04, 0x00, 0x38, 0x00, 0x00, 0x00,
	 	  0x03, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
	 	  0x07, 0x80, 0xa1, 0x00, 0x00, 0x00, 0x00, 0x00,
	 	  0xad, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 	  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

	unlock_CRTC();
	unlock_s3();
	RestoreSEQ (SEQ ,   5);		/* Reset all registers to a defined state */
	RestoreCRTC(CRTC, 128);
	RestoreGR  (GR  ,   9);
	RestoreATR (ATR ,  21);
	outw(0x0000,ADVFUNC_CNTL);	/* Disable 8514A part */
}

void SetExtendedTiming(struct ggi_Timing *TM,int xvirtshift)
{
	int	startFIFOfetch;
	int	h;

	startFIFOfetch=((TM->xend+TM->xsyncstart)>>4)-2;
	
	Outb_CRTC(CRT_FIFOSTART,startFIFOfetch&0xff);

	Outb_CRTC(CRT_SYSCONTROL2, 
		  (Inb_CRTC(CRT_SYSCONTROL2)&0xc0)|
		  ((TM->tr.xvirtual>>(4+xvirtshift))&0x30));

	Outb_CRTC(19,TM->tr.xvirtual>>xvirtshift);

	Outb_CRTC(CRT_EV_OVERFLOW,
	(((TM->yend    -2)&0x400)>>10)|
	(((TM->ywidth  -1)&0x400)>> 9)|
	(((TM->ysyncstart)&0x400)>> 8)|
	(((TM->ysyncstart)&0x400)>> 6)|0x40 );

	h = Inb_CRTC(CRT_EH_OVERFLOW)&0x80;	/* Leave BGT alone */
	h|= 	((((TM->xend       >> 3) - 5) & 0x100) >> 8) |
		((((TM->xwidth     >> 3) - 1) & 0x100) >> 7) |
		((((TM->xsyncstart >> 3) - 1) & 0x100) >> 6) |
		((((TM->xsyncstart >> 3)    ) & 0x100) >> 4) |
		((((startFIFOfetch     )    ) & 0x100) >> 2) ;

	if ((TM->xsyncend >> 3) - (TM->xsyncstart >> 3) > 32)
		h |= CR5D_HSYNCEND_B5;   /* add another 32 DCLKs to hsync pulse width */

	if ((TM->xsyncend >> 3) - (TM->xsyncstart >> 3) > 64)
		h |= CR5D_HBLANKEND_B6;   /* add another 64 DCLKs to blank pulse width */

	Outb_CRTC(CRT_EH_OVERFLOW, h);

	Outb_CRTC(CRT_INTERLACESTART,	/* Interlace mode frame offset */
		  TM->tr.xvirtual>>xvirtshift); 
}


int SetText(struct ggi_Timing *TM)
{ 	
	ResetCard();
	kgi_SetClockTiming(TM);
	vga_SetText(TM);
	SetExtendedTiming(TM,2);
#if 0	
	/* DO NOT ENABLE ! This simply doesn't work :-( */
	Outb_CRTC(CRT_MEMCONFIG0,Inb_CRTC(CRT_MEMCONFIG0)|CR31_HISPEEDTXT);
	Outb_CRTC(CRT_MISC1     ,Inb_CRTC(CRT_MISC1)     |CR3A_HISPEEDTXT);
#endif
	return(0);
}  

/*****************************************************************************
*** Check/Setup a 4bit graphics-mode of the given size                     ***
*****************************************************************************/
int Check4bit(struct ggi_Timing *TM,int olderr)
{ 
	int	x, clksave;

#if 0
	if (TM->tr.xvirtual*TM->tr.yvirtual/8>64*1024) 
	{
		TM->tr.xvirtual=800;TM->tr.yvirtual=600;
		return(-ENOMODESUP_CHIP);
	}
#endif

	if (TM->tr.xvisible&0x07) 
		TM->tr.xvisible&=0xfffffff8;	/* This must be a multiple of 8 */

	TM->clock=86000000;	/* FIXME ... */
	TM->ymagnify=1;

	/* We can do nothing against RAMDAC objections */
	if ((x=kgi_CheckRamdacTiming(TM,CMD_NONE))) return(x); 

	clksave=TM->clock;	/* RAMDAC might have limited it */

	while(1)
	{
		TM->xwidth=TM->tr.xvisible;
		TM->ywidth=TM->tr.yvisible*TM->ymagnify;

		switch((x=kgi_CheckMonitorTiming(TM,CMD_MON_PERFECT)))
		{ 
			case MR_Y_TOO_SMALL:
				TM->clock=clksave;
				if (TM->ymagnify++<32) break;	/* Retry with more magn. */
    			  	TM->ymagnify=1;
	    			x=kgi_CheckMonitorTiming(TM,CMD_MON_SOME);
    			  	return x;
			default:
				return x;	/* We can do nothing about it */
		} 
	}  
	
	return(-ENOMODESUP_CHIP);
}  

int Set4bit(struct ggi_Timing *TM)
{ 
	ResetCard();
	kgi_SetClockTiming(TM);
	vga_Set4bit(TM);
	SetExtendedTiming(TM,4);

	return(0);
}  

/*****************************************************************************
*** Check/Setup a 8bit graphics-mode of the given size                     ***
*****************************************************************************/
int Check8bit(struct ggi_Timing *TM,int olderr)
{ 
	int x, clksave;

       if (TM->tr.xvirtual<= 640) TM->tr.xvirtual= 640;
  else if (TM->tr.xvirtual<= 800) TM->tr.xvirtual= 800;
  else if (TM->tr.xvirtual<=1024) TM->tr.xvirtual=1024;
  else if (TM->tr.xvirtual<=1152) TM->tr.xvirtual=1152;
  else if (TM->tr.xvirtual<=1280) TM->tr.xvirtual=1280;
  else if (TM->tr.xvirtual<=1600) TM->tr.xvirtual=1600;
  else if (TM->tr.xvirtual<=2048) TM->tr.xvirtual=2048;
  else { TM->tr.xvirtual=2048;return(-ENOMODESUP_CHIP); }

	if (TM->tr.xvisible&0x07)
		TM->tr.xvisible&=0xfffffff8;	/* This must be a multiple of 8 */

	TM->clock=86000000;
	TM->ymagnify=1;

	/* We can do nothing against RAMDAC objections */
	if ((x=kgi_CheckRamdacTiming(TM,CMD_NONE))) 
		return(x); 

	clksave=TM->clock;	/* RAMDAC might have limited it */

	while(1)
	{
		TM->xwidth=TM->tr.xvisible;
		TM->ywidth=TM->tr.yvisible*TM->ymagnify;

		switch((x=kgi_CheckMonitorTiming(TM,CMD_MON_PERFECT)))
		{ 
			case MR_Y_TOO_SMALL:
				TM->clock=clksave;
    				if (TM->ymagnify++<32) break;	/* Retry with more magn. */
   				TM->ymagnify=1;
    				x=kgi_CheckMonitorTiming(TM,CMD_MON_SOME);
    				return x;
			default:return x;	/* We can do nothing about it */
		} 
	}  

	return(-ENOMODESUP_CHIP);
}  

int Set8bit(struct ggi_Timing *TM)
{ 
	int h,x,m;
	char SEQ [5]=
		{ 0x03, 0x01, 0x0f, 0x00, 0x0e };
	char GR  [9]=
		{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x05, 0x0F, 0xFF };
	char ATR[21]=
		{ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
  		  0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
  		  0x01, 0x00, 0x0f, 0x00, 0x00}; /* FAST 256 Mode ? */

	ResetCard();
	RestoreSEQ(SEQ,5);
	RestoreGR (GR , 9);
	RestoreATR(ATR,21);    

	kgi_SetClockTiming(TM);

	setup_VGA_timing(TM);

	SetExtendedTiming(TM,3);

	Outb_CRTC(20,0x60);
	Outb_CRTC(23,0xAB); /* MONO or 16 : 0xE3 */

	unlock_s3();
	unlock_CRTC();
	kgi_SetRamdacTiming(TM);

	if (TM->tr.xvirtual==2048)
		Outb_CRTC(CRT_MEMCONFIG0,0x8f);
	else 
		Outb_CRTC(CRT_MEMCONFIG0,0x8d);

	Outb_CRTC(CRT_BACKCOMPAT1,0x00);
	Outb_CRTC(CRT_BACKCOMPAT2,CR33_NOBORDER); /* ??? */
	Outb_CRTC(CRT_BACKCOMPAT3,CR34_ENB_FIFO);

	Outb_CRTC(CRT_LOCK,0x00);

	Outb_CRTC(CRT_MISC1,CR3A_NO4BPPMODE|
			    CR3A_ALT_REFRESH|
			    CR3A_REFRESH1);

	Outb_CRTC(CRT_SYSCONFIG,Inb_CRTC(CRT_SYSCONFIG)|CR40_UNLOCK_ENH);

	switch(8)	/* bits per pixel ??? Documentation disapproves ! */
	{
		case  8:Outb_CRTC(CRT_EXTMODE2,0x0);break;
		case 16:
		case 32:Outb_CRTC(CRT_EXTMODE2,0x08);break;
	}

#if 0	/* Register undocumented ... does it exist ? */
	Outb_CRTC(0x44,0);					/* OK */
#endif

	switch(8)
	{	
		case 32:Outb_CRTC(CRT_HWC_MODE,(Inb_CRTC(CRT_HWC_MODE)&0xf2)|0x04);break;
		default:Outb_CRTC(CRT_HWC_MODE,Inb_CRTC(CRT_HWC_MODE)&0xf2);break;
	}							/* SVS3 says 0 ... */

	h=Inb_CRTC(CRT_SYSCONTROL1)&0x0e;		/* Width select */

	switch(8)	/* bits per pixel*/
	{
		case  8:break;
		case 16:h|=CR50_16BPP;break;
		case 32:h|=CR50_32BPP;break;			/* OK */
	}

	switch(TM->tr.xvirtual)
	{
		case  640:h|=CR50_640;break;
		case  800:h|=CR50_800;break;
		case 1152:h|=CR50_1152;break;
		case 1280:h|=CR50_1280;break;
		case 1600:h|=CR50_1600;break;
		case 1024:h|=CR50_1024;break;
		case 2048:h|=CR50_2048;break;
		default  :printk("Illegal xvirtual !\n");break;		/* OK */
	}

	Outb_CRTC(CRT_SYSCONTROL1,h);

#ifdef OLD_MN_PAR
	h = 255;
	{	int clock,mclk;
		clock = TM->clock * 1;
		mclk = 60000;  /* 130 MHz, limit for 964 */
		m = (int)((mclk/1000.0*.72+16.867)*89.736/(clock/1000.0+39)-21.1543);
		m -= 0; /* Madjust */
		if (m > 31) m = 31;
		else if (m < 0) {
		 m = 0;
		h = 16;
		}
	}
	Outb_CRTC(CRT_MEMCONTROL2,m<<3);	/* always 0x38 ??? */
	h -= /*Nadjust*/ 0;		
	if (h < 0) h = 0;
	else if (h > 255) h = 255;
	Outb_CRTC(CRT_MEMCONTROL3,h);	/* always 0x07 ??? */

	printk("M: %d,N: %d\n",m,h);
#else
	Outb_CRTC(CRT_MEMCONTROL2,15<<3);	
	Outb_CRTC(CRT_MEMCONTROL3,15);
	/* Give memory 1:1 to FiFo and others - never seen this fail and the XF
	   method always gives 0,16 for m,n ... */
#endif
	Outb_CRTC(CRT_DACCONTROL,(Inb_CRTC(CRT_DACCONTROL)&0x08)|0x40);	/* ? */
	Outb_CRTC(CRT_LAWCONTROL,CR58_SAM_256);	/* We DO NOT use the linear window right now. */
	/*  Outb_CRTC(CRT_LAW_POS_H,0x0a); */	/* for this would eliminate the paging which  */
	/*  Outb_CRTC(CRT_LAW_POS_L,0x0a); */	/* is desireable for a demo-driver.           */


	h=(TM->xwidth*1)/4+1; /* FIXME? - other bbps, s3drv says 8 ! */
	Outb_CRTC(CRT_MEMCONTROL4,CR61_L_ENABLE|(h>>8));
	Outb_CRTC(CRT_MEMCONTROL5,(h&0xff));

	for(x=0;x<2048/64;x++)
	{ 
		Outb_CRTC(CRT_EXTCONTROL4,x);		/* Bank select */
		memset((void *)0xa0000,0,0x10000);  /* erase graphics RAM */
	}
	Outb_CRTC(CRT_EXTCONTROL4,0);

#if 0
	/*if (TM->tr.xvirtual == 1024) outw(0x0007,ADVFUNC_CNTL);
	else			    outw(0x0003,ADVFUNC_CNTL);*/
#elif 0
	Outb_CRTC(CRT_EXTMISC1,Inb_CRTC(CRT_EXTMISC1)|CR66_PIXELBUS_OFF);
/*	Outb_CRTC(CRT_EXTSYNC1,Inb_CRTC(CRT_EXTSYNC1)|CR56_SYNC_OFF);*/
	outw(0x0001,ADVFUNC_CNTL);
	outw(0x9000,SUBSYS_CNTL);
	outw(0x5000,SUBSYS_CNTL); /* ? 0x4 ?*/
	inw(SUBSYS_STAT); /* ??? */
	outw(MEM_CNTL | VRTCFG_4 | HORCFG_8,MULTIFUNC_CNTL); /* ? 0x4 ?*/
	
#endif

	return(0);
}  


/***************************************************************************/
/* *** ***        I n t e r f a c e   d e f i n i t i o n s        *** *** */
/***************************************************************************/

void kgi_GetChipsetParameters(struct ggi_ChipParams *CP)
{
	*CP=ChipPar;
}

int kgi_CheckChipsetTiming(struct ggi_Timing *TM,int olderr)
{ 
	switch(TM->tr.graphtype)
	{	case GT_TEXT		: 
			return(CheckText(TM,olderr));

		case GT_4BIT_PLANES	: 
			return(Check4bit(TM,olderr));
			
		case GT_8BIT		: 
			return(Check8bit(TM,olderr));
			
		case GT_15BIT		:
		case GT_16BIT		: 
			return(-1);	/* FIXME */

		case GT_24BIT		: 
			TM->tr.graphtype=GT_32BIT;	/* Use 32 bpp instead */

		case GT_32BIT		:
			return(-1);	/* FIXME */

		default     		: 
			return(-1);
	}
}  

int kgi_SetChipsetTiming(struct ggi_Timing *TM)
{ 
	switch(TM->tr.graphtype)
	{	case GT_TEXT		: 
			return( SetText(TM) );

		case GT_4BIT_PLANES	: 
			return( Set4bit(TM) );
			
		case GT_8BIT		: 
			return( Set8bit(TM) );
			
		case GT_15BIT		:
		case GT_16BIT		: 
			return(-1);	/* FIXME */

		case GT_24BIT		: 
			TM->tr.graphtype = GT_32BIT;	/* Use 32 bpp instead */

		case GT_32BIT		:
			return(-1);	/* FIXME */

		default     		: 
			return(-1);
	}
}  


/****************************/
/*** Set start of display ***/
/****************************/

void hw_setdispstart(int dx,int dy)
{
	int x;

	if (CurrState->bgmode) return;
    
	x=dx+dy*CurrState->ModeNow.xvirtual;
  
	Outb_CRTC(0x0d,(x >>  2) & 0xff);    /*  sa2-sa9 */
	Outb_CRTC(0x0c,(x >> 10) & 0xff);    /* sa10-sa17 */
	Outb_CRTC(CRT_EXTCONTROL3,(x >> 18) & 0x0f);

	/* FIXME - why does this not work in graphics-mode ? 
	   Pixel-panning works fine in Text-Mode ... */
	Outb_ATR_CTL(0x13,(Inb_ATR_CTL(0x13)&0xf0)|((x & 3)<<1));
}

/*********************************************
**** Accelerated drawing functions follow ****
*********************************************/

#ifdef ACCEL_SUPPORT
int s3_wait_fifo(int spc_need)
{ int x=1000000000;
  while((inb(GP_STAT)&(0x100>>spc_need))&&--x);
  return(!x);
}
#endif

int setinforec(void) 
{ CurrState->drawinfo=*(void **)parameters;return(0); }

#ifdef ACCEL_SUPPORT

was in setcolor - not correct anymore ... 

  s3_wait_fifo(6);
  outb(FSS_FRGDCOL | MIX_SRC,FRGD_MIX);
  outb(BSS_BKGDCOL | MIX_SRC,BKGD_MIX);
  outw(SCISSORS_T |   10,MULTIFUNC_CNTL);
  outw(SCISSORS_L |   10,MULTIFUNC_CNTL);
  outw(SCISSORS_R |  300,MULTIFUNC_CNTL);
  outw(SCISSORS_B |  300,MULTIFUNC_CNTL);

  s3_wait_fifo(2);
  outw(~0,WRT_MASK);
  outw(PIX_CNTL | 0x0000,MULTIFUNC_CNTL);
      
  outb(CurrState->drawinfo.color,FRGD_COLOR);
#endif


int s3_drawbox(void)
{ 
#ifdef ACCEL_SUPPORT
  if (CurrState->bgmode)   
#endif
    return(-ENODRVSUP_ALWAYS_MMAP);

#ifdef ACCEL_SUPPORT
  if (s3_wait_fifo(6)) return(-ENODRVSUP_NOW_MMAP);
  outw(0xa000,			MULTIFUNC_CNTL);
  outw(*((int *)parameters  ),	CUR_X);
  outw(*((int *)parameters+1),	CUR_Y);
  outw(*((int *)parameters+2),	MAJ_AXIS_PCNT);
  outw(*((int *)parameters+3) | MIN_AXIS_PCNT,	MULTIFUNC_CNTL);
  outw(CMD_RECT | INC_Y | INC_X | DRAW | PLANAR | WRTDATA,CMD);
  return(0);
#endif
}

/*****************************************************************************
****  Graphics  Device  Interface  Funtions                               ****
*****************************************************************************/

int kgi_ChipsetIoctl(struct inode *inode, struct file *file, \
        unsigned int cmd, unsigned long arg)
{ DEV_NUM_SETUP;

  switch(cmd) { 
	case GRAPH_GETVT       :COPYPAR(sizeof(void *));
				return(getvt());
        case GRAPH_SETMODE     :COPYPAR(sizeof(void *));
    		  		return(setmode());
        case GRAPH_GETMODE     :COPYPAR(sizeof(void *));
    				return(getmode());
        case GRAPH_KEEPMODE    :if (suser()) return(keepmode());
    				else return(-EPERM);
        case GRAPH_SETDRAWINFO :COPYPAR(sizeof(void *));
        			return(setinforec());

        case GRAPH_SETVISWIN   :COPYPAR(2*sizeof(int));
    				return(setdispstart());
        case GRAPH_GETVISWIN   :COPYPAR(2*sizeof(int));
    				return(getdispstart());

        case GRAPH_GETSTATESIZE:COPYPAR(sizeof(int *));
    			        return(getstatesize());
        case GRAPH_SAVESTATE   :COPYPAR(sizeof(void *));
    			        return(savestate());
        case GRAPH_RESTORESTATE:COPYPAR(sizeof(void *));
    			        return(restorestate());

        case GRAPH_BGMODE      :return(bgmode(1));
        case GRAPH_FGMODE      :return(bgmode(0));

        case GRAPH_SETDEFFONT  :if (!suser()) return(-EPERM);
        			COPYPAR(sizeof(void *));
        			return(setdeffont());
        case GRAPH_GETDEFFONT  :COPYPAR(sizeof(void *));
				return(getdeffont());

	case GRAPH_TEST	       :return(0);	/* For timing tests */

        case GRAPH_DRAWBOX     :COPYPAR(4*sizeof(int)+sizeof(void *));
    				return(s3_drawbox());

	case GRAPH_DRAWPIXEL   : /* Faster than doing it here */
	case GRAPH_GETPIXEL    :
	case GRAPH_PUTPIXEL    :
	case GRAPH_DRAWHLINE   :
	case GRAPH_GETHLINE    :
	case GRAPH_PUTHLINE    :
	case GRAPH_GETBOX      :
	case GRAPH_PUTBOX      :return(-ENODRVSUP_ALWAYS_MMAP);

        case GRAPH_FILLSCREEN  :/* will make sense when drawbox works */

        default		       :return -ENODRVSUP_ALWAYS_LOWER;
   }
   /* We should never get here ... */
   return -ENODRVSUP_ALWAYS_CANT;
}                                                      

static int nopage_deadlock_check=0;

#define MAXMP 100

static struct Map { 
		struct task_struct *tsk;
		unsigned long lastmap; } map[MAXMP],*mp;

static int mapcnt=0;		

void kgi_GraphmapClose(struct vm_area_struct * area)
{	
	nopage_deadlock_check=0;
	
	for(;mapcnt;mapcnt--)	/* FIXME ... delete only those mappings that belong */
	{			/* to the closing task - not too critical ... */
		mp=&map[mapcnt-1];
		fast_unmap_page_range(mp->tsk,mp->lastmap, 0x10000);
	}
}

unsigned long kgi_GraphmapNopage(struct vm_area_struct * area, unsigned long address,
	unsigned long page, int no_share)
{
	static pte_t *last_pte=NULL;
	static struct task_struct *tsk;
	unsigned long mappedpage=-1;
	
	VM_DEV_NUM_SETUP;
        	
	if (CurrState->bgmode) {	/* A backgrounded process accesses graphics mem ! */
		area->vm_task->blocked &= ~(1<<(SIGSEGV-1));
		send_sig(SIGSEGV, area->vm_task, 1);
        }

	if (nopage_deadlock_check && !pte_young(*last_pte) && tsk==area->vm_task) {
		area->vm_task->blocked &= ~(1<<(SIGBUS-1));
		send_sig(SIGBUS, area->vm_task, 1);
        }

        address -= area->vm_start;
	nopage_deadlock_check=!(address&0xffff);	/* fault at first byte of VGA page ? */
	address &= 0xffff0000;				/* which page to map */

	if (mappedpage!=address)			/* delete all mapping for old page */
 		for(;mapcnt;mapcnt--) 
		{
			mp=&map[mapcnt-1];
			fast_unmap_page_range(mp->tsk,mp->lastmap, 0x10000);
		}

	/* VGA Granularity is 64 K  - address &= PAGE_MASK; is in there,too ... */

#ifdef S3_805
	Outb_CRTC(CRT_LOCK, (Inb_CRTC(CRT_LOCK) & 0xf0) | ((address>>16) & 0x000000f));
#else
	switch(CurrState->ModeNow.graphtype)
	{ case GT_8BIT		: outw((address>>8)|CRT_EXTCONTROL4,0x3d4);/* Bank select */
				  break;
	  case GT_4BIT_PLANES	: Outb_GR_CTL(4,(address>>16)&3); /* Read Map select */
	  			  Outb_GR_CTL(0,address>>16);     /* Color select */
	  			  break;
	  case GT_TEXT          : 
	  default		:break; }
#endif

	address+= area->vm_start;

        map[mapcnt].tsk=area->vm_task;
        map[mapcnt].lastmap=address;
        if (mapcnt++>=MAXMP) {mapcnt--;printk("Whee - to many mappings - going mad !\n");}
        
	if (CurrState->ModeNow.graphtype==GT_TEXT)
		fast_remap_page_range(area->vm_task,address, 0xb8000, 0x10000, area->vm_page_prot);
	else
        	fast_remap_page_range(area->vm_task,address, 0xa0000, 0x10000, area->vm_page_prot);

        if (nopage_deadlock_check)
        	last_pte = pte_offset( pmd_offset( pgd_offset(tsk=area->vm_task, address), address), address);

	invalidate();

        ++area->vm_task->mm->min_flt;	/* do some statistics */
        
	return((unsigned long)NULL);
}

int kgi_ChipsetInit(void)
{
	outb(S3_DISABLE, S3_SUBSYSTEM);	/* chip wakeup */
	outb(S3_WAKEUP,  S3_SETUP);
	outb(S3_ENABLE,  S3_SUBSYSTEM);

	Detect_Vision964();		/* FIXME - should react to rc */
  
	unlock_s3();			/* Unlock s3-regs ... other Inits might need access */
  
	printk("Manufacturer : %s\n",ChipPar.manufact);
	printk("Model        : %s\n",ChipPar.model);
	printk("RAM / Maxsize: %d Bytes (%d x %d) \n",
        	ChipPar.ramsize,
        	ChipPar.xmaxdots,
        	ChipPar.ymaxdots);

	_getsetfont_hw(OpenFont,0);
	return(0);			/* All went OK - install */
}

void kgi_ChipsetCleanup(void) 
{ 
	lock_s3();
}


void ramdac_select_RS23(int reg)
{  Outb_CRTC(CRT_DACCONTROL,(Inb_CRTC(CRT_DACCONTROL) &
				0xFC) | ((reg & 0x0C) >> 2));
}
                             