/*\
|*| ---------------------------------------------------------------------------
|*|	ATT20c505 / ATT20c504 RAMDAC driver
|*| ---------------------------------------------------------------------------
|*|
|*|	Copyright (C) 1995 by Steffen Seeger, [seeger@physik.tu-chemnitz.de]
|*|
|*|	This program is free software; you can redistribute it and/or modify
|*|	it under the terms of the GNU General Public License as published by
|*|	the Free Software Foundation; either version 2, or (at your option)
|*|	any later version.
|*|
|*|	This program is distributed in the hope that it will be useful,
|*|	but WITHOUT ANY WARRANTY; without even the implied warranty of
|*|	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|*|	GNU General Public License for more details.
|*|
|*|	You should have received a copy of the GNU General Public License
|*|	along with this program; see the file COPYING.  If not, write to
|*|	the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
|*|
|*| ---------------------------------------------------------------------------
\*/


#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <asm/io.h>

#undef DEBUG_IO

#include "scrdrv.h"
#include "kgi-module.h"
#include "ATT/20c505.h"
#include "S3/Vision964.h"
#include "S3/s3_io.h"
/* #include "clut.h"	 standard CLUT's */


#define DAC_VERSION "0.00 ALPHA"
#define PCLK DAC_C2_PCLK0     /* or DAC_C2_PCLK1, (doesn't matter on 20SV) */


enum RAMDAC_types 
 {
	ATT20C504_85	= 0,
	ATT20C504_11	= 1,

	ATT20C505_11	= 2,
	ATT20C505_13	= 3,
	ATT20C505_17	= 4,

	DAC_UNKNOWN	= -1
 };

#define ATT20C504 ATT20C504_85
#define ATT20C505 ATT20C505_11

static struct GGI_DacParams att_params[5] =
      {
       { "AT&T Microelectronics", "ATT 20C504-85",
	 DAC_8BIT | DAC_GAMMA,
         {{       0,   75000000},
	  {       0,   85000000},
          {       0,   85000000},
          {       0,   85000000},
	  {	  0,          0},
	  {       0,   85000000},
	  {       0,          0},
	  {       0,          0},

	  {       0,          0},
          {       0,   85000000}}
       },
       { "AT&T Microelectronics", "ATT 20C504-11",
	 DAC_8BIT | DAC_GAMMA, 
         {{        0,  75000000},
	  {        0, 110000000},
          {        0, 110000000},
	  {        0, 110000000},
	  {        0,         0},
	  {	   0, 110000000},
	  {        0,         0},
	  {        0,         0},

	  {        0,         0},
	  {        0, 110000000}}
       },
       { "AT&T Microelectronics", "ATT 20C505-11",
	 DAC_8BIT | DAC_GAMMA | DAC_2XCLOCK,
         {{        0,  75000000},
          {        0, 110000000},
          {        0, 110000000},
          {        0, 110000000},
	  {        0,         0},
	  {	   0, 110000000},
	  {        0,         0},
	  {        0,         0},

          { 45000000,  55000000},
	  {        0, 110000000}}
       },
       { "AT&T Microelectronics", "ATT 20C505-13",
	 DAC_8BIT | DAC_GAMMA | DAC_2XCLOCK,
         {{        0,  85000000},
          {        0, 135000000},
          {        0, 135000000},
          {        0, 135000000},
	  {        0,         0},
          {        0, 135000000},
	  {        0,         0},
 	  {        0,         0},

          { 45000000,  67500000},
	  {        0, 110000000}}
       },
       { "AT&T Microelectronics", "ATT 20C505-17",
	 DAC_8BIT | DAC_GAMMA | DAC_2XCLOCK,
	 {{        0,  85000000},
	  {        0, 135000000},
	  {        0, 170000000},
	  {        0, 170000000},
	  {        0,         0},
	  {        0, 170000000},
	  {        0,         0},
	  {        0,         0},

	  { 45000000,  85000000},
	  {        0, 110000000}}
       }
      };

static struct GGI_DacParams *dac_params = NULL;
static unsigned char CR0,CR1,CR2,CR3;


/*\
|*|	GGI interface
\*/


int
kgi_RamdacInit(void)
{
   unsigned char status = inbDAC(DAC_STATUSREG);

   printk("ATT RAMDAC driver v"DAC_VERSION"\n");

   /* state save has to be done (later) */

   if(status & DAC_ST_ATT_DEVICE)
     switch(status & DAC_ST_ID_MASK)
      {
	case DAC_ATT20C504:
		dac_params = att_params + ATT20C504;
		break;

	case DAC_ATT20C505:
		dac_params = att_params + ATT20C505;
		break;

	case DAC_ATT20C506:
		printk("%s: don't know how to handle ATT20C506 device.\n",\
			__FILE__);
		return FALSE;
	default:
		printk("  unknown chip ID (%.2x).\n", status & DAC_ST_ID_MASK);
		return FALSE;
      }
    else
     {
       printk("  no ATT RAMDAC detected.\n");
       return FALSE;
     }

   TRACE(printk("  %s %s detected.\n", dac_params->manufact, dac_params->model));
   return TRUE;
}

void
kgi_RamdacDone(void)
{
  TRACE(printk("%s %s driver removed.\n", dac_params->manufact, dac_params->model));
}

void
kgi_RamdacGetParameters(struct GGI_DacParams *params)
{
  *params = *dac_params;
}

int
kgi_RamdacCheckTiming(struct GGI_Timing *TM)
{
  enum GGI_DacRangeIndex div = -1;
  int newclock = TM->clock;

  switch(TM->tr.graphtype)
   {
     case GT_TEXT:	div = DAC_MUX_1;  break;
     case GT_1BIT:	div = DAC_MUX_32; break;
     case GT_4BIT:	div = DAC_MUX_8;  break;
     case GT_8BIT:	div = DAC_MUX_4;  break;

     case GT_15BIT:
     case GT_16BIT:	div = DAC_MUX_2;  break;

     case GT_24BIT:    /* same as 32 bpp ??? */
     case GT_32BIT:	div = DAC_MUX_1;  break;

     default:
	TRACE(printk("%s: could not handle graphtype %i\n",\
		     __FILE__, TM->tr.graphtype));
	return FALSE;
   }

 if (newclock < dac_params->clock[div].min)
   {
     TRACE(printk("%s: invalid clock (%i).\n", __FILE__, newclock));
     return FALSE;
   }

  if (dac_params->clock[div].max < newclock)	/* meet limits */
    newclock = dac_params->clock[div].max;

  if (dac_params->clock[DAC_MUX_NORMAL].max < newclock)
    {
      TRACE(printk("%s: clock doubling mode not implemented yet.\n",__FILE__));
      newclock = dac_params->clock[DAC_MUX_NORMAL].max;
    }

  TM->clock    = newclock;
  TM->clockdiv = div;
  TRACE(printk("%s: %i:1 multiplex mode, DCLK = %i kHz\n", __FILE__,\
			    1 << div, TM->clock/1000));
  return TRUE;
}

int
kgi_RamdacSetTiming(struct GGI_Timing *TM)
{
  #define VGAPalette NULL	/* temporary !!! */
  #define GammaMap NULL

  void *clut;
  struct GGI_MonParams monitor;

  /* setup CR0 */
     CR0 = DAC_C0_8BIT;		/* we'll always use 8 bit DAC */
     kgi_MonitorGetParameters(&monitor);
     if (monitor.synctype & SYNC_ON_RED)     CR0 |= DAC_C0_SYNC_ON_RED;
     if (monitor.synctype & SYNC_ON_GREEN)   CR0 |= DAC_C0_SYNC_ON_GREEN;
     if (monitor.synctype & SYNC_ON_BLUE)    CR0 |= DAC_C0_SYNC_ON_BLUE;
     if (monitor.synctype & BLANK_PEDESTRAL) CR0 |= DAC_C0_BLANK_LEVEL;


  /* setup CR1 */
     switch(TM->tr.graphtype)
      {
        case GT_TEXT:	CR1 = 0;		clut = VGAPalette; break;
        case GT_1BIT:	CR1 = DAC_C1_1BPP;	clut = VGAPalette; break;
	case GT_4BIT:	CR1 = DAC_C1_4BPP;	clut = VGAPalette; break;
	case GT_8BIT:	CR1 = DAC_C1_8BPP;	clut = VGAPalette; break;
	case GT_15BIT:	CR1 = DAC_C1_15BPP;	clut = GammaMap;   break;
	case GT_16BIT:	CR1 = DAC_C1_16BPP;	clut = GammaMap;   break;

	case GT_24BIT:	/* same as 32 bpp ??? */
	case GT_32BIT:	CR1 = DAC_C1_24BPP;	clut = GammaMap;   break;

	default:	return FALSE;
      };

  /* CR1 |= DAC_C1_CLUT_BYPASS; */

  /* setup CR2 */
      CR2 =  ((TM->tr.graphtype == GT_TEXT) ? 0:  DAC_C2_ENHANCED_MODE)
           | ((TM->flags & FL_INTERLACE)    ? DAC_C2_INTERLACED : 0)
           | PCLK;

  /* setup CR3 */
     CR3 = 0;		/* we have no clock doubling yet ... */

  /* program CR3,CR2,CR1,CR0 (CR3 looks a bit strange, but is correct ;-) */
     outbDAC(CR0 | DAC_C0_EXT_UNLOCK | DAC_C0_SLEEP_MODE, DAC_CONTROL0);
     outbDAC(DAC_PW_CR3_INDEX, DAC_PW_INDEX);
     outbDAC(CR3, DAC_STATUSREG);

     outbDAC(CR2, DAC_CONTROL2);
     outbDAC(CR1, DAC_CONTROL1);
     outbDAC(CR0, DAC_CONTROL0);

/*     set_clut(0, 256, clut);
*/
    return TRUE;
}


int
kgi_RamdacIoctl (struct inode *inode, struct file *file, unsigned int cmd,
		 unsigned long arg)
{
  /* this may set
       -  dac_params (take care if 504 or 505 was detected).
       -  hardware cursor
       -  CLUT (or gamma correction table).
   */
  return -1;
}

