/* 
   Graphics library for GGI.

   Copyright (C) 1995 Andreas Beck - becka@hp.rz.uni-duesseldorf.de

   If you do any modifications, I would like you to send diffs to me
   to allow for collecting a more and more complete set of drivers and
   to improve existing ones.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define _LIB_COMPILE_

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <signal.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/time.h>
#include <linux/vt.h>
#include <linux/kd.h>
#include <string.h>
#include "../include/graphlib.h"

#define MMAP_SIZE 0x200000

static int ggi_tty_fd=0;
static int gd_hand=-1;			/* Graphics device file handle    */
static int gd_shmid=-1;			/* id-number of shared memory     */
static char *gd_shmem=NULL;		/* address of 1st shmem-attach (not
					   the shadowed one under ggi_membuffer */

char *ggi_membuffer=NULL;		/* mmap-address of graphics space
					   shmem will be here,when switched
					   away,VGA-RAM else */

/* Virtual console switching and memory locking */
static int forbidvtacquire = 0;
static int forbidvtrelease = 0;

/* This should NOT BE USED ! */
int lock_count = 0;
volatile int release_flag = 0;
volatile int acquire_flag = 0;

static int ggi_vt_num;
static int sleep_when_switched_away=0;

enum availlist ioctl_avail[0x400];	/* map of driver capabilities */
int (*emu_func[0x400])(void *par);	/* emulation functions */

struct ggi_Timing_req CurrentMode;
void *StartUpState,*BackgroundState;

#define MAX_EVENT_HANDLERS 30

static void (*EvHandler[MAX_EVENT_HANDLERS])(struct ggi_event);
enum Events EvHandTyp[MAX_EVENT_HANDLERS];
static int handler_count=0;

static char fontbuf[8192];

/*****************************************************************************/

void discard_avail(void)
{ int x;
  for(x=0;x<sizeof(ioctl_avail)/sizeof(ioctl_avail[0]);x++) 
    ioctl_avail[x]=AVM_IOCTL|AVE_TEST;
}

/*************************************/
/* request a service from the driver */
/*************************************/
int ggi_ioctl(int cmd,void *parm) 
{ if (ioctl(gd_hand,cmd,parm)) return(-errno);else return(0); }

/********************************/
/* set any mode (text/graphics) */
/********************************/
int ggi_setmode(struct ggi_Timing_req *tm)
{ int x;
  /* Delete ioctls marked as "unsupported this mode" */
  for(x=0;x<sizeof(ioctl_avail)/sizeof(ioctl_avail[0]);x++) 
    if ((ioctl_avail[x]&AVE_MASK)==AVE_MODE) ioctl_avail[x]=AVM_IOCTL|AVE_TEST;

  ioctl(ggi_tty_fd, KDSETMODE, KD_GRAPHICS);	/* We probably go to graphics now ... */

  if ((x=ggi_ioctl(GRAPH_SETMODE,&tm))>=0) 
  { CurrentMode=*tm;
    if (tm->graphtype==GT_TEXT)
    { struct vt_sizes vts;
      vts.v_rows = tm->yvisible/tm->ytextgrid;
      vts.v_cols = tm->xvisible/8;
      vts.v_scrollsize = 0;
      if (ioctl(ggi_tty_fd, VT_RESIZE, &vts)) perror("VT_RESIZE");
      ioctl(ggi_tty_fd, KDSETMODE, KD_TEXT);
      system("setfont");
    }
  }

  return(x);
}

/**************************/
/* get the controlling vt */
/**************************/
int ggi_getvt(int *vtnum)
{ return(ggi_ioctl(GRAPH_GETVT,&vtnum)); }

/***********************************/
/* return to mode at program start */
/***********************************/
int ggi_startupmode(void)
{ return(ggi_restorestate(StartUpState)); }

/************************/
/* get the current mode */
/************************/
int ggi_getmode(struct ggi_Timing_req *tm)
{ return(ggi_ioctl(GRAPH_GETMODE,&tm));
}

/**************************************************/
/* keep the current mode after closing the device */
/**************************************************/
int ggi_keepmode(void)
{ return(ggi_ioctl(GRAPH_KEEPMODE,NULL)); }

/*************************/
/* save a graphics state */
/*************************/
void *ggi_savestate(void)
{ int sz=0;
  void *ptr,*argp;
  argp=&sz;ggi_ioctl(GRAPH_GETSTATESIZE,&argp);
  if (sz==0||(ptr=malloc(sz))==NULL) return NULL;
  argp=&ptr;ggi_ioctl(GRAPH_SAVESTATE,argp);
  return(ptr);
}

/****************************/
/* restore a graphics state */
/****************************/
int ggi_restorestate(void *ptr)
{ return(ggi_ioctl(GRAPH_RESTORESTATE,&ptr)); }

/*****************/
/* save a screen */
/*****************/
void *ggi_savescreen(void)
{ void *ptr;
  if ((ptr=malloc(CurrentMode.xvirtual*CurrentMode.yvirtual))==NULL) return NULL;
  ggi_getbox(0,0,CurrentMode.xvirtual,CurrentMode.yvirtual,ptr);
  return(ptr);
}
/********************/
/* restore a screen */
/********************/
int ggi_restorescreen(void *ptr)
{ return(ggi_putbox(0,0,CurrentMode.xvirtual,CurrentMode.yvirtual,ptr)); }

/******************/
/* set a textmode */
/******************/
int ggi_textmode(int cols,int rows,int fontsizex,int fontsizey)
{ struct ggi_Timing_req tim={ 	800,600,	/* Default size */
        	                1024,1024,
                	        GT_TEXT,
                        	8,16           };
                        
  if (fontsizex<8||fontsizex> 9) return -EINVAL;
  if (fontsizey<4||fontsizey>32) return -EINVAL;

  tim.xvisible=cols*8;
  tim.yvisible=rows*fontsizey;
  tim.xtextgrid=fontsizex;
  tim.ytextgrid=fontsizey;

  return(ggi_setmode(&tim));
}

/***********************/
/* set a graphics mode */
/***********************/
int ggi_graphmode(int xsize,int ysize,int xvirtual,int yvirtual,int type)
{ struct ggi_Timing_req tim={    800,600,
	                        1024,1024,
	                        GT_8BIT,
	                        8,16      };
  tim.xvisible=xsize;
  tim.yvisible=ysize;
  tim.xvirtual=xvirtual;
  tim.yvirtual=yvirtual;                       
  tim.graphtype=type;

  return(ggi_setmode(&tim));
}

/*****************************************/
/* set the upper-left edge of the screen */
/*****************************************/
int ggi_setviswin(int x,int y)
{ return(ggi_ioctl(GRAPH_SETVISWIN,&x)); }

/***********************************/
/* set a part of the palette-table */
/***********************************/
int ggi_setpalvec(int start,int anz,struct ggi_Palentry *pp)
{ return(ggi_ioctl(RAMDAC_SETPAL,&start)); }

/***********************************/
/* get a part of the palette-table */
/***********************************/
int ggi_getpalvec(int start,int anz,struct ggi_Palentry *pp)
{ return(ggi_ioctl(RAMDAC_GETPAL,&start)); }

/*************************************************************************/

struct ggi_Drawcontext ggi_drwcontext;

/***********************/
/* set drawing color   */
/***********************/
int ggi_setcolor(int x)
{ ggi_drwcontext.fg_color=x;return(0); }

int ggi_setbgcolor(int x)
{ ggi_drwcontext.bg_color=x;return(0); }

/*******************************/
/* draw/get/put a single pixel */
/*******************************/
static int drawpixel256(int *xy)
{ *(ggi_membuffer+*(xy+1)*CurrentMode.xvirtual+*xy)=ggi_drwcontext.fg_color;
  return(0); }

volatile char xyz;

static int drawpixel16(int *xy)
{ char msk;
  int adr;
  msk=0x80>>(*xy&7);
  adr=((*(xy+1)*CurrentMode.xvirtual)>>3)+
      (*xy>>3)+0x10000*ggi_drwcontext.fg_color;
  xyz=*(ggi_membuffer+adr);
  *(ggi_membuffer+adr)=msk;
  return(0); }

static int drawpixel_text(int *xy)
{ *(int *)(ggi_membuffer+*(xy+1)*CurrentMode.xvirtual+*xy*2)=
    0xdb+(ggi_drwcontext.fg_color<<8);
  return(0); }

int ggi_drawpixel(int x,int y)
{ switchavail(GRAPH_DRAWPIXEL,&x);
  switch(CurrentMode.graphtype)
  { case GT_8BIT       :setdoemu(GRAPH_DRAWPIXEL,drawpixel256  ,&x);break;
    case GT_4BIT_PLANES:setdoemu(GRAPH_DRAWPIXEL,drawpixel16   ,&x);break;
    case GT_TEXT       :setdoemu(GRAPH_DRAWPIXEL,drawpixel_text,&x);break; }
  return 0;}

int ggi_getpixel(int x,int y,void *buffer)
{ switchavail(GRAPH_GETPIXEL,&x);
  /* FIXME ! */
  *(char *)buffer=*(ggi_membuffer+y*CurrentMode.xvirtual+x);
  return 0;}



static int putpixel256(int *xy)
{ *(ggi_membuffer+*(xy+1)*CurrentMode.xvirtual+*xy)=*((int *)*(xy+2));
  return(0); }

static int putpixel16(int *xy)
{ char msk;
  int adr;
  msk=0x80>>(*xy&7);
  adr=((*(xy+1)*CurrentMode.xvirtual)>>3)+
      (*xy>>3)+0x10000* *((int *)*(xy+2));
  xyz=*(ggi_membuffer+adr);
  *(ggi_membuffer+adr)=msk;
  return(0); }

static int putpixel_text(int *xy)
{ *(int *)(ggi_membuffer+*(xy+1)*CurrentMode.xvirtual+*xy*2)=
    0xdb+(*((int *)*(xy+2))<<8);
  return(0); }

int ggi_putpixel(int x,int y,void *buffer)
{ switchavail(GRAPH_PUTPIXEL,&x);
  switch(CurrentMode.graphtype)
  { case GT_8BIT       :setdoemu(GRAPH_PUTPIXEL,putpixel256  ,&x);break;
    case GT_4BIT_PLANES:setdoemu(GRAPH_PUTPIXEL,putpixel16   ,&x);break;
    case GT_TEXT       :setdoemu(GRAPH_PUTPIXEL,putpixel_text,&x);break; }
  return 0;}

/**********************************/
/* draw/get/put a horizontal line */
/**********************************/
int ggi_drawhline(int x,int y,int width)
{ switchavail(GRAPH_DRAWHLINE,&x);
  switcha(GRAPH_DRAWHLINE)
  {case AVM_MMAP :if (CurrentMode.graphtype==GT_8BIT)
  		  { memset(ggi_membuffer+y*CurrentMode.xvirtual+x,ggi_drwcontext.fg_color,width);
  		    break;}
   case AVM_LOWER:for(;width>0;width--) ggi_drawpixel(x++,y);break;
   default       :ggi_panic("hline - no method.");break;
  }
  return(0);
}

int ggi_puthline(int x,int y,int width,void *buffer)
{ switchavail(GRAPH_PUTHLINE,&x);
  switcha(GRAPH_PUTHLINE)
  {case AVM_MMAP :if (CurrentMode.graphtype==GT_8BIT)
  		  { memcpy(ggi_membuffer+y*CurrentMode.xvirtual+x,buffer,width);
  		    break;}
   case AVM_LOWER:for(;width>0;width--) {ggi_putpixel(x++,y,(char *)buffer++);}break;
   default       :ggi_panic("hline - no method.");break;
  }
  return(0);
}

int ggi_gethline(int x,int y,int width,void *buffer)
{ switchavail(GRAPH_GETHLINE,&x);
  switcha(GRAPH_GETHLINE)
  {case AVM_MMAP :if (CurrentMode.graphtype==GT_8BIT)
  		  { memcpy(buffer,ggi_membuffer+y*CurrentMode.xvirtual+x,width);
  		    break;}
   case AVM_LOWER:for(;width>0;width--) {ggi_getpixel(x++,y,(char *)buffer++);}break;
   default       :ggi_panic("hline - no method.");break;
  }
  return(0);
}

/********************************/
/* draw/get/put a vertical line */
/********************************/
int ggi_drawvline(int x,int y,int height)
{ switchavail(GRAPH_DRAWHLINE,&x);
  switcha(GRAPH_DRAWHLINE)
  {case AVM_MMAP :{char *ptr;
                   ptr=ggi_membuffer+y*CurrentMode.xvirtual+x;
                   for(;height>0;height--,ptr+=CurrentMode.xvirtual) 
                     *ptr=ggi_drwcontext.fg_color; }
  		  break;
   case AVM_LOWER:for(;height>0;height--) ggi_drawpixel(x,y++);break;
   default       :ggi_panic("hline - no method.");break;
  }
  return(0);
}

int ggi_putvline(int x,int y,int height,void *buffer)
{ switchavail(GRAPH_PUTHLINE,&x);
  switcha(GRAPH_PUTHLINE)
  {case AVM_MMAP :{char *ptr;
                   ptr=ggi_membuffer+y*CurrentMode.xvirtual+x;
                   for(;height>0;height--,ptr+=CurrentMode.xvirtual) 
                     *ptr=*(char *)buffer++; }
  		  break;
   case AVM_LOWER:for(;height>0;height--) {ggi_putpixel(x,y++,(char *)buffer++);}break;
   default       :ggi_panic("hline - no method.");break;
  }
  return(0);
}

int ggi_getvline(int x,int y,int height,void *buffer)
{ switchavail(GRAPH_GETHLINE,&x);
  switcha(GRAPH_GETHLINE)
  {case AVM_MMAP :{char *ptr;
                   ptr=ggi_membuffer+y*CurrentMode.xvirtual+x;
                   for(;height>0;height--,ptr+=CurrentMode.xvirtual) 
                     *(char *)buffer++=*ptr; }
  		  break;  		  break;
   case AVM_LOWER:for(;height>0;height--) {ggi_getpixel(x,y++,(char *)buffer++);}break;
   default       :ggi_panic("hline - no method.");break;
  }
  return(0);
}

/**********************/
/* draw/get/put a box */
/**********************/
int ggi_drawbox(int x,int y,int width,int length)
{ switchavail(GRAPH_DRAWBOX,&x);
  for(;length>0;length--)
   ggi_drawhline(x,y++,width);
  return(0);
}

int ggi_getbox(int x,int y,int width,int length,void *buffer)
{ switchavail(GRAPH_GETBOX,&x);
  for(;length>0;length--)
  { ggi_gethline(x,y++,width,buffer);
    buffer+=width; }
  return(0);
}

int ggi_putbox(int x,int y,int width,int length,void *buffer)
{ switchavail(GRAPH_PUTBOX,&x);
  for(;length>0;length--)
  { ggi_puthline(x,y++,width,buffer);
    buffer+=width; }
  return(0);
}


/*************************************/
/* fill (erase) the (virtual) screen */
/*************************************/
int ggi_fillscreen(void)
{ switchavail(GRAPH_FILLSCREEN,NULL);
  switcha(GRAPH_FILLSCREEN)
  {case AVM_MMAP :if (CurrentMode.graphtype==GT_8BIT) 
  		  { memset(ggi_membuffer,ggi_drwcontext.fg_color,CurrentMode.xvirtual*CurrentMode.yvirtual);
  		    break;}
   case AVM_LOWER:ggi_drawbox(0,  0,CurrentMode.xvirtual,CurrentMode.yvirtual);break;
   default       :ggi_panic("fillscreen - no method.");break;
  }
  return(0);
}

/****************************************/
/* draw a circle - highly optimized ;-) */
/****************************************/
int ggi_drawcircle(int xcenter,int ycenter,int radius)
{ int x,y,od,md,sd;
  switchavail(GRAPH_DRAWCIRCLE,&xcenter);
  x=radius;y=md=0;
  while(x>=y)
  { ggi_drawpixel(xcenter+x,ycenter+y);ggi_drawpixel(xcenter-x,ycenter+y);
    ggi_drawpixel(xcenter+y,ycenter+x);ggi_drawpixel(xcenter-y,ycenter+x);
    ggi_drawpixel(xcenter+x,ycenter-y);ggi_drawpixel(xcenter-x,ycenter-y);
    ggi_drawpixel(xcenter+y,ycenter-x);ggi_drawpixel(xcenter-y,ycenter-x);
    od=md+y+y+1;sd=od-x-x-1;y++;md=od;
    if (abs(sd)<abs(od)) {x--;md=sd;}
  }
  return(0);
}

/**********TEMPORARY************/
/* Blit out a 2-color bitfield */
/*******************************/
int ggi_blit2c(int x,int y,int xwidth,int ywidth,void *field)
{ int xp,bp;
  int color;

  for(;ywidth--;y++)
  { for(xp=0,bp=0x80;xp<xwidth;xp++)
    { color=(*(char *)field & bp) ? ggi_drwcontext.fg_color 
    				  : ggi_drwcontext.bg_color;
      ggi_putpixel(x+xp,y,&color);
      if (!(bp>>=1)) {bp=0x80;field++;}
    }
  }
  return(0);
}

/**********TEMPORARY*******/
/* Put a character at x,y */
/**************************/
int ggi_putch(int x,int y,int c)
{ return(ggi_blit2c(x,y,8,16,fontbuf+(c<<5))); }


/****TEMPORARY*******/
/* graphical printf */
/********************/
int ggi_printf(int x,int y,const char *format,...)
{ va_list arg;
  char buffer[1024],*bf;
  int rc;

  va_start(arg,format);bf=buffer;
  rc=vsprintf(buffer,format,arg);
  
  while(*bf)
  { ggi_putch(x,y,*bf++);
    if ((x+=8)>=CurrentMode.xvirtual) {x=0;y+=16;}
  }
  return(rc);
}

/*******************************/
/* monitor-powersaving-control */
/*******************************/
int ggi_mon_power(enum PowerSave state)
{ return(ggi_ioctl(MONITOR_POWERSAVE,&state)); }

/**********************************/
/* activate a VT - 0 means our VT */
/**********************************/
int ggi_vt_activate(int num)
{ if (num==0) num=ggi_vt_num;
  if (ioctl(ggi_tty_fd, VT_ACTIVATE  , num)||
      ioctl(ggi_tty_fd, VT_WAITACTIVE, num)  ) return -EIO;
  return(0); }

/***************************/
/* get currently active VT */
/***************************/
int ggi_get_active_vt(void)
{ struct vt_stat vtstat;
  if (ioctl(ggi_tty_fd, VT_GETSTATE , &vtstat)) return -EIO;
  return(vtstat.v_active); }

/*****************************************/
/* get a character from the graphics-tty */
/*****************************************/
int ggi_getch(void)
{ char buffer;
  if (read(ggi_tty_fd,&buffer,1)!=1) return(-1);
  else return(buffer); }

/****************************/
/* is a character available */
/****************************/
int ggi_kbhit(void)
{ fd_set readfds;
  struct timeval timeout;
  FD_ZERO( &readfds );
  FD_SET( ggi_tty_fd, &readfds );
  timeout.tv_sec = timeout.tv_usec = 0;
  return(select( FD_SETSIZE , &readfds, NULL, NULL, &timeout )>0);
}

/***********************************/
/* set the switched-away behaviour */
/***********************************/
int ggi_run_in_bg(int yesno)
{ sleep_when_switched_away=!yesno;return(0); }


/*****************************/
/* register an event-handler */
/*****************************/
int ggi_register_evh(int evtype,void (*EvHand)(struct ggi_event))
{ if (handler_count>=MAX_EVENT_HANDLERS) return(-1);
  EvHandler[handler_count]=EvHand;
  EvHandTyp[handler_count]=evtype;
  handler_count++;	/* better we do this really AFTER assigning the values */
  return(0);
}			      

int ggi_send_event(struct ggi_event ev)
{ int x;
  for(x=0;x<handler_count;x++)
  { if (EvHandTyp[x]==EV_ALL||
        EvHandTyp[x]==ev.eventtype) EvHandler[x](ev);
  }
  return(0);
}

/****************************************************************************/
/* SINGAL HANDLING - restore old mode before executing the original handler */
/****************************************************************************/

int ggi_bgmode(void)
{ return(ggi_ioctl(GRAPH_BGMODE,NULL)); }

int ggi_fgmode(void)
{ return(ggi_ioctl(GRAPH_FGMODE,NULL)); }

#define SETSIG2(sa, sig, fun, oldsa)	\
	sa.sa_handler = fun;		\
	sa.sa_flags = 0;		\
	sa.sa_mask = 0;			\
        sigaction(sig, &sa, &oldsa);
                                
static struct sigaction old_handlers[NSIG];

static void int_handler( int v )
{ char buffer[100];
  sprintf(buffer,"Got signal number %d - exiting.\n",v);
  ggi_panic(buffer);
  sigaction(v, &old_handlers[v], NULL);
  raise(v);
}

/*****************************/
/* Virtual console switching */
/*****************************/

void releasevt_signal( int n ) {
	struct sigaction siga;
	struct ggi_event ev;

	if (lock_count) {		/* Not yet used,but might come in handy */
		release_flag = 1;
		return;
	}
	forbidvtacquire = 1;
	SETSIG2(siga, SIGUSR1, releasevt_signal,old_handlers[SIGUSR1]);
	if (forbidvtrelease) {
		forbidvtacquire = 0;
		ioctl(ggi_tty_fd, VT_RELDISP, 0);
		return;
	}

	/* Move screen contents to shared memory buffer. */
	ggi_getbox(0,0,CurrentMode.xvirtual,CurrentMode.yvirtual,gd_shmem);
  
	/* Now map the shared mem to the place where the VGA RAM was */
	munmap((caddr_t)ggi_membuffer,MMAP_SIZE);
	shmat(gd_shmid,ggi_membuffer,0);

	ggi_bgmode();

	discard_avail();		/* Background - acceleration off */
	ioctl(ggi_tty_fd, VT_RELDISP, 1);
	forbidvtacquire = 0;

	ev.eventtype=EV_VT;
	ev.evdata.vt.subtype=VT_SWITCH_AWAY;
	ev.evdata.vt.console=0;		/* Don't know this here - is there a way short of
					   using ioctl GETSTATE ? */
	ggi_send_event(ev);

	if (sleep_when_switched_away)
	   ioctl(ggi_tty_fd, VT_WAITACTIVE, ggi_vt_num);
} 

void acquirevt_signal( int n ) {
	struct sigaction siga;
	struct ggi_event ev;

	if (lock_count) {		/* Not yet used,but might come in handy */
		acquire_flag = 1;
		return;
	}
	forbidvtrelease = 1;
	SETSIG2(siga, SIGUSR2, acquirevt_signal, old_handlers[SIGUSR2]);
	if (forbidvtacquire) {
		forbidvtrelease = 0;
		return;
	}

	ggi_fgmode();			/* start doing real work again */
	/* Remap the VGA RAM to the original address */
        shmdt(ggi_membuffer);
        mmap((caddr_t)ggi_membuffer,
		 MMAP_SIZE,              /* 2048K - FIXME ! */
		 PROT_READ|PROT_WRITE,   /* for reading and writing  */
		 MAP_SHARED,             /* shared mapping */
                 gd_hand,                /* from the graphics-device */
                 0x0                     /* from the very beginning  */
                 );

	discard_avail();		/* Foreground - acceleration on */
	/* Get screen contents from shared memory buffer. */
	ggi_putbox(0,0,CurrentMode.xvirtual,CurrentMode.yvirtual,gd_shmem);

	ioctl(ggi_tty_fd, VT_RELDISP, VT_ACKACQ);
	forbidvtrelease = 0;

	ev.eventtype=EV_VT;
	ev.evdata.vt.subtype=VT_SWITCH_TO;
	ev.evdata.vt.console=0;		/* Don't know this here - is there a way ? */
	ggi_send_event(ev);
}

void takevtcontrol(void) {
    	struct sigaction siga;
	struct vt_mode newvtmode;
	struct vt_mode oldvtmode;

	ggi_vt_activate(0);		/* Must be active on startup anything else
        				   will confuse the vt-switching-code      */
	ioctl(ggi_tty_fd, VT_GETMODE, &oldvtmode);
    	newvtmode = oldvtmode;
    	newvtmode.mode = VT_PROCESS;	/* handle VT changes */
    	newvtmode.relsig = SIGUSR1;	/* I didn't find SIGUSR1/2 anywhere */
    	newvtmode.acqsig = SIGUSR2;	/* in the kernel sources, so I guess */
    					/* they are free */
	SETSIG2(siga, SIGUSR1, releasevt_signal,old_handlers[SIGUSR1]);
    	SETSIG2(siga, SIGUSR2, acquirevt_signal,old_handlers[SIGUSR2]);
    	ioctl(ggi_tty_fd, VT_SETMODE, &newvtmode);
}

static struct termios oldtermios;

void setup_kbd(void)
{  struct termios kbdtermios;

   tcgetattr(ggi_tty_fd, &oldtermios);
   kbdtermios=oldtermios;

   kbdtermios.c_lflag = kbdtermios.c_lflag & ~ (ICANON | ECHO);
   kbdtermios.c_cc[VMIN] = 1;
   kbdtermios.c_cc[VTIME] = 0;
   tcsetattr(ggi_tty_fd, TCSAFLUSH, &kbdtermios);
}

void reset_kbd(void)
{ tcsetattr(ggi_tty_fd, TCSAFLUSH, &oldtermios); }


/*******************************/
/* initialize graphics package */
/*******************************/
void ggi_init(void)
{ struct sigaction siga;
  struct ggi_Drawcontext *drwc;
  void *ptr;
  char str[20];

  /* Open the graphics device */
  if ((gd_hand=open("/dev/graphics",O_RDWR))==-1) {perror("open");exit(1);}

  discard_avail(); /* Don't know what the driver supports now */

  StartUpState=ggi_savestate();			/* Get old state */

  ggi_getvt(&ggi_vt_num);			/* and the vt we are on */
  
  sprintf(str,"/dev/tty%d",ggi_vt_num);
  if ((ggi_tty_fd=open(str,O_RDWR))==-1) {perror("VT open");exit(1);}
  
  ggi_membuffer = mmap(
	(caddr_t)0,		/* Map whereever you like   */
	MMAP_SIZE,		/* 2048K - FIXME ! */
	PROT_READ|PROT_WRITE,	/* for reading and writing  */
	MAP_SHARED,		/* shared mapping */
	gd_hand,		/* from the graphics-device */
	0x0			/* from the very beginning  */
    );
    
  drwc=&ggi_drwcontext;
  ggi_ioctl(GRAPH_SETDRAWINFO,&drwc);

  ptr=&fontbuf;
  ggi_ioctl(GRAPH_GETDEFFONT,&ptr);

  if (ggi_membuffer<=0) ggi_membuffer=NULL;	
  /* This is may fail ... we have no membuffer in this case ! */

  /* So now the buffer for the screen contents when switched away.
     We use shared mem, as we can set the mapping address for it. */
     
  gd_shmid = shmget(IPC_PRIVATE, MMAP_SIZE ,IPC_CREAT | IPC_EXCL | 0600);
  if (-1 == gd_shmid) {fprintf(stderr,"Cannot get shared memory.\n");exit(1);}
  
  gd_shmem = shmat(gd_shmid, NULL, 0);
  if (gd_shmem<=0) {fprintf(stderr,"Cannot attach shared memory.\n");exit(1);}

  if (shmctl(gd_shmid, IPC_RMID, (struct shmid_ds *)0))
    {fprintf(stderr,"Cannot RMID shared memory.\n");exit(1);}

  /* Catch signals so we won't end up in graphics mode ... 
     not really needed anymore due to the restore on close behaviour,
     but the kernel will not recognize that the process that issued 
     KD_GRAPHICS has died until the next vt-switch so ... */
  SETSIG2(siga, SIGINT , int_handler , old_handlers[SIGINT ]);
  SETSIG2(siga, SIGSEGV, int_handler , old_handlers[SIGSEGV]);
  SETSIG2(siga, SIGFPE , int_handler , old_handlers[SIGFPE ]);
  SETSIG2(siga, SIGILL , int_handler , old_handlers[SIGILL ]);
  SETSIG2(siga, SIGTERM, int_handler , old_handlers[SIGTERM]);
  SETSIG2(siga, SIGQUIT, int_handler , old_handlers[SIGQUIT]);
  SETSIG2(siga, SIGBUS , int_handler , old_handlers[SIGBUS ]);

  takevtcontrol();
  setup_kbd();

}

/*********************************/
/* deinitialize graphics package */
/*********************************/
void ggi_exit(void)
{ close(gd_hand);				/* Close the device */
  reset_kbd();					/* Restore old tty state */
  ioctl(ggi_tty_fd, KDSETMODE, KD_TEXT);	/* Make sure this is Text-Mode */
  if (shmdt(gd_shmem)) {fprintf(stderr,"Cannot detach shared memory.\n");exit(1);}
}

/**************************************/
/* graceful shutdown for fatal errors */
/**************************************/
void ggi_panic(char *why) { ggi_exit();fputs(why,stderr);exit(1); }
