#define AUTOSENSE
#undef REAL_DMA_POLL
#define REAL_DMA
#define READ_OVERRUNS
/* #define NDEBUG (NDEBUG_DMA|NDEBUG_MAIN) */

/*
 * Generic Generic NCR5380 driver
 *	
 * Copyright 1993, Drew Eckhardt
 *	Visionary Computing
 *	(Unix and Linux consulting and custom programming)
 *	drew@colorado.edu
 *      +1 (303) 666-5836
 *
 * ALPHA RELEASE 1. 
 *
 * For more information, please consult 
 *
 * NCR 5380 Family
 * SCSI Protocol Controller
 * Databook
 *
 * NCR Microelectronics
 * 1635 Aeroplaza Drive
 * Colorado Springs, CO 80916
 * 1+ (800) 525-2252
 */

/* 
 * TODO : flesh out DMA support 
 */

/*
 * Options :
 *
 * PARITY - enable parity checking.  Not supported.
 *
 * SCSI2 - enable support for SCSI-II tagged queueing.  Untested.
 *
 * USLEEP - enable support for devices that don't disconnect.  Untested.
 *
 * The card is detected and initialized in one of several ways : 
 * 1.  With command line overrides - gsi8=1
 *
 * 2.  With the gsi8_OVERRIDE compile time define.  This macro will force
 *     detection.
 *
 * gsi8 jumper settings :
 * Port Address
 * W8 off	0x148
 *    on	0x248
 *
 * Memory Addresses
 * W6  W4	BIOS	RAM
 * off nc	0xC8000	0xCC000
 * on  off	0xD0000 0xD4000
 * on  on	0xD8000 0xDC000	
 *
 * Interrupts
 * W1  W2  W5	IRQ
 * on  off off	3
 * off on  off  4
 * off off on   5
 * 
 * DMA Level
 * W3 off	1
 *    on	3
 *
 * SCSI ID 
 * W11 W12 W13	
 * msb	   lsb
 *
 * Parity Checking (not supported, allways off)
 * W10	
 *
 * Unknown 
 * W9
 *
 * NC
 * W7
 */
 
/*
 * $Log: gsi8.c,v $
 * Revision 1.1  1993/10/16  22:43:42  drew
 * Initial revision
 *
 */

#include <asm/system.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include "../block/blk.h"
#include "scsi.h"
#include "hosts.h"
#include "gsi8.h"
#include "NCR5380.h"

#include "constants.h"

#include <linux/delay.h>

typedef struct {
	char *signature ;
	unsigned offset;
	unsigned length;
} Signature;
	
static const Signature signatures[] = {
  {"MODEL GSI-8,", 36, 12},
};

#define NUM_SIGNATURES (sizeof(signatures) / sizeof(Signature))

static int dodetect = 1;

/*
 * Function : gsi8_setup(char *str, int *ints)
 *
 * Purpose : LILO command line initialization of the overrides array,
 * 
 * Inputs : str - unused, ints - array of integer paramters with ints[0]
 *	equal to the number of ints.
 *
 */

void gsi8_setup(char *str, int *ints) {
    if (ints[0] != 1) 
	printk("gsi8_setup : usage gsi8={0,1}\n");
    else 
      dodetect = ints[1];
}

static void doreset(struct Scsi_Host *instance) {
  NCR5380_local_declare();
  NCR5380_setup(instance);
  udelay(200);	
  NCR5380_write(MODE_REG, 0);
  NCR5380_write(TARGET_COMMAND_REG, 0);
  NCR5380_write(INITIATOR_COMMAND_REG, ICR_ASSERT_RST);
  udelay(2000);	
  NCR5380_write(INITIATOR_COMMAND_REG, 0);
  udelay(2000);	
  return;
}

static void sleepfor(unsigned long timeout) {
  timeout += jiffies;
  while (timeout > jiffies) ;
}

int gsi8_reset(void) {
    struct Scsi_Host *instance;
    Scsi_Host_Template *hostt;
    
    printk("GSI-8 bus reset!\n");
    cli();
    for (instance = first_instance, hostt = instance->hostt;
	instance && (instance->hostt == hostt); instance = instance->next) 
	doreset(instance);
    /* report success */
    return 0;
}

static int tryarb(struct Scsi_Host *instance)
{
  NCR5380_local_declare();
  int i, arb;
  NCR5380_setup(instance);

  i = inb(GSI_MAGIC_PORT_1);
  NCR5380_write(MODE_REG, 0);
  NCR5380_write(OUTPUT_DATA_REG, 0x80);
  NCR5380_write(MODE_REG, MR_ARBITRATE);
  arb = 0;
  for (i = 0 ; !arb && i < 32765; i++) {
    if (NCR5380_read(INITIATOR_COMMAND_REG) & ICR_ARBITRATION_PROGRESS) arb=1;
  }
  if (arb) {
    __asm__("nop");
    udelay(3);
  }
  NCR5380_write(MODE_REG, 0);
  NCR5380_write(OUTPUT_DATA_REG, 0x00);
  return arb;
}

static void gsi8_intr (int irq);

static struct sigaction sa =  { gsi8_intr, 0, 
    SA_INTERRUPT , NULL };

/* 
 * Function : int gsi8_detect(int hostno)
 *
 * Purpose : initializes GSI-8 driver based on the 
 *	command line / compile time port and irq definitions.
 *
 * Inputs : hostno - id of this SCSI adapter.
 * 
 * Returns : 1 if a host adapter was found, 0 if not.
 *
 */

int gsi8_detect(int hostno) {
    NCR5380_local_declare();
    struct Scsi_Host *instance;

    unsigned char *base;
    unsigned short int io_port;
    unsigned char irq, nirqs, magic1, magic2;
    unsigned char dma_channel;
    int match, j;

    if (!dodetect) {
      printk("Ricoh GSI-8 detection bypassed\n");
      return 0;
    }

    magic1 = inb(GSI_MAGIC_PORT_1);
    magic2 = inb(GSI_MAGIC_PORT_2);

#ifdef GDEBUG
    printk("scsi%d : Ricoh magic-port flags are 0x%02x 0x%02x\n",
	   hostno, magic1, magic2);
#endif

    if (!(magic1 & GSI_JUMPER_W6)) 
      base = (unsigned char *) GSI_BIOS_W6_OFF;
    else if (magic1 & GSI_JUMPER_W4) 
      base = (unsigned char *) GSI_BIOS_W6_ON_W4_ON;
    else 
      base = (unsigned char *) GSI_BIOS_W6_ON_W4_OFF;

    if (magic1 & GSI_JUMPER_W8) 
      io_port = GSI_HIGH_PORT_BASE;
    else
      io_port = GSI_LOW_PORT_BASE;

    irq = IRQ_NONE;
    nirqs = 0;
 
    if (!(magic1 & GSI_JUMPER_W1)) {
      irq = 3;
      ++nirqs;
    }
    if (!(magic1 & GSI_JUMPER_W2)) {
      irq = 4;
      ++nirqs;
    }
    if (!(magic1 & GSI_JUMPER_W5)) {
      irq = 5;
      ++nirqs;
    }

#if defined(REAL_DMA) || defined(REAL_DMA_POLL)
    if (!(magic1 & GSI_JUMPER_W3))
      dma_channel = 1;
    else
      dma_channel = 3;
#else
    dma_channel = DMA_NONE;
#endif

#ifdef GDEBUG
    printk("scsi%d : Ricoh GSI-8, BIOS at 0x%x, RAM at 0x%x, port at 0x%x\n",
	   hostno, (unsigned) base, (unsigned) base + GSI_OFFSET_BIOS_TO_RAM , 
	   io_port);
#endif

    if (nirqs > 1) {
      printk("scsi%d : WARNING - multiple IRQs enabled on Ricoh GSI-8\n! Check W1, W2, W5\n", hostno);
    }

    for (j = 0, match = 0; !match && j < NUM_SIGNATURES; ++j) {
      if (!memcmp ((void *) (base + signatures[j].offset),
		   (void *) signatures[j].signature,
		   signatures[j].length)) 
	match = 1;
    }

    if (!match) {
      return 0;
    }

    instance = scsi_register (hostno, sizeof(struct NCR5380_hostdata));
    instance->base = base;
    instance->io_port = io_port;
    instance->this_id = (magic2 & 0x7);
    instance->irq = irq;
    instance->dma_channel = dma_channel;
    NCR5380_setup(instance);

    if (NCR5380_read(STATUS_REG) != 0) {
      printk("scsi%d : bus jammed, resetting\n", hostno);
      doreset(instance);
      sleepfor(1000);
    }

    if (!tryarb(instance)) {
      printk("scsi%d : Arbitration attempt failed, bailing out\n", hostno);
      scsi_unregister (instance, sizeof(struct NCR5380_hostdata));
      return 0;
    }

    NCR5380_init(instance);

    if ((instance->irq != IRQ_NONE) && (irqaction (instance->irq, &sa))) {
	    printk("scsi%d : IRQ%d not free, interrupts disabled\n", 
		hostno, instance->irq);
	    instance->irq = IRQ_NONE;
	} 

    if (instance->irq == IRQ_NONE) {
	printk("scsi%d : interrupts not enabled. for better interactive performance,\n", hostno);
	printk("scsi%d : please jumper the board for a free IRQ.\n", hostno);
    }

    if ((instance->dma_channel != DMA_NONE) && 
	(request_dma (instance->dma_channel))) {
	    printk("scsi%d : dma channel %d not free, DMA disabled\n", 
		hostno, instance->dma_channel);
	    instance->dma_channel = DMA_NONE;
	} 

    printk("scsi%d : Ricoh GSI-8 at port 0x%x",
	   instance->host_no, instance->io_port);
    if (instance->irq == IRQ_NONE)
	printk (" interrupts disabled");
    else 
	printk (" irq %d", instance->irq);

    if (instance->dma_channel == DMA_NONE) 
	printk (" dma disabled");
    else
	printk (" dma %d", instance->dma_channel);

    printk(" host id %d\n", instance->this_id);
    
    printk(" options CAN_QUEUE=%d  CMD_PER_LUN=%d release=%d",
	CAN_QUEUE, CMD_PER_LUN, GSI8_PUBLIC_RELEASE);
    NCR5380_print_options(instance);
    printk("\n");
    
    return 1;
  }

const char *gsi8_info (void) {
    static const char string[]="";
    return string;
}

/*
  Forbid DMA transfer if no channel is available, if the transfer would hit
  the low page of any 1-megabyte range (broken address decode in the
  card will cause it to reset the NCR5380 in mid-transfer), if the
  transfer would cross a 64k boundary, or if the transfersize
  suggested by the command is zero or inappropriate for this block
  size.

  Otherwise, transfer the entire buffer in one big DMA.
*/

unsigned int gsi8_dma_xfer_len(struct Scsi_Host *instance,
			       Scsi_Cmnd *cmd) {
  NCR5380_local_declare();
  unsigned int p;
  NCR5380_setup(instance);
  if (instance->dma_channel == DMA_NONE) {
    return 0;
  }
  if (cmd->transfersize == 0 ||
      (cmd->SCp.this_residual % cmd->transfersize) != 0) {
    return 0;
  }
  p = (unsigned int) cmd->SCp.ptr;
  if ((p & 0x000FFFFF) < 0x400) {
    return 0;
  }
  if ((p & 0xFFFF0000) != 
      ((p + cmd->SCp.this_residual - 1) & 0xFFFF0000)) {
    return 65536 - (p & 0x0000FFFF);
  }
  return cmd->SCp.this_residual;
}

/*
  Interrupt wrapper for GSI-8.  Try reading the magic ports in a way
  which we believe will re-enable interrupts properly
*/

#include "NCR5380.c"

static void gsi8_intr (int irq) {
  (void) __inb(GSI_MAGIC_PORT_1);
  NCR5380_intr (irq);
}
