#include <stdio.h>
#include <linux/fs.h>
#include <sys/sysmacros.h>
#include <fcntl.h>
#include <linux/major.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>

#include <getopt.h>

int use_symlink = 0;
int filemode = 0600;
int verbose = 0;

/*
 * This program builds entries in /dev/scsi that have names that are tied
 * to the actual device number and lun.   This is pretty solid as long as
 * you have only one scsi controller, but with multiple scsi controllers
 * this all falls down.  The problem is that you can have multiple controllers
 * from one manufacturer, so it is not sufficient to assign an id based upon
 * an index from the driver.  Currently the controller numbers are just the
 * internal index that is assigned by the kernel.
 *
 * A better solution would be for the kernel to assign id numbers for each
 * controller type, and the we could assign numbers based upon the number
 * of the controller located.  This would guarantee that we could generate
 * useful number.
 */


void flush_sdev()
{
  struct dirent * de;
  char filename[60];
  DIR * sdir;
  /*
   * Next, delete all of the existing entries in the /dev/scsi directory.
   * The idea is that if we have added/removed devices, the numbers might have
   * changed.
   */
  sdir = opendir("/dev/scsi");
  while(1==1)
    {
      de = readdir(sdir);
      if(de == NULL) break;
      if(de->d_name[0] != 's' && de->d_name[0] != 'r') continue;
      strcpy(filename, "/dev/scsi/");
      strcat(filename, de->d_name);
      unlink(filename);
    }
  closedir(sdir);
  printf("Flushed old /dev/scsi entries...\n", de->d_name);

}

void build_sdev(int major, int mode, int devtype, char devchar)
{
  int chan;
  int fd;
  char hostname[60];
  int newmode;
  int recreate;
  int h_id;
  int id;
  int lun;
  int minor = 0;
  int uid, gid;
  struct stat statbuf;
  int scsi_id;
  char scsidev[25];
  char linkto[25];
  int status;

  status = stat("/tmp/testdev", &statbuf);
  if(status == 0)
    {
      unlink("/tmp/testdev");
    }

  status = stat("/dev/scsi", &statbuf);
  if(status == -1)
      return;

  while(1==1)
    {
      status = mknod("/tmp/testdev",0600 | devtype ,
		     makedev(major,minor));
      fd = open("/tmp/testdev", mode);
      if(fd == -1) {
	if( verbose == 2 )
	  {
	    fprintf(stderr,"open(%x/%x) returned %d (%d)\n",
		    major, minor, fd, errno);
	  }
	if(major == SCSI_DISK_MAJOR && (minor & 0xf))
	  {
	    minor = (minor & ~0xf) + 16;
	  } else
	    break;
      }
      status = ioctl(fd, SCSI_IOCTL_GET_IDLUN, &id);

      if(status == -1)	{
	if( verbose == 2 )
	  {
	    fprintf(stderr,"probe(%x/%x) returned %d (%d)\n",
		    major, minor, status, errno);
	  }
	close(fd);
	break;
      }

      status = ioctl(fd, SCSI_IOCTL_PROBE_HOST, hostname);
      close(fd);

      if(status == -1)	break;

#if 0
      printf("Found scsi device %x with idlun %x\n", makedev(major, minor), id);
#endif
      h_id = id >> 24;
      chan = (id >> 16) & 0xff;
      lun = (id >> 8 ) & 0xff;
      scsi_id = id & 0xff;
      if(major == SCSI_DISK_MAJOR && (minor & 0xf))
	{
	  sprintf(scsidev,"/dev/scsi/s%ch%dc%di%dl%dp%d",
		  devchar,
		  h_id,
		  chan,
		  scsi_id,
		  lun, (minor & 0xf));
      } else {
	sprintf(scsidev,"/dev/scsi/s%ch%dc%di%dl%d",
		devchar,
		h_id,
		chan,
		scsi_id,
		lun);
      }
      printf("Found scsi device %s %s\n", scsidev, hostname);

      if( use_symlink )
	{
	  /*
	   * Figure out what we should be symlinked to here.
	   */
	  switch(major)
	    {
	    case SCSI_DISK_MAJOR:
	      if( minor & 0xf)
		sprintf(linkto, "/dev/sd%c%d", 
			(minor >> 4) + 'a', (minor & 0xf)); 
	      else
		sprintf(linkto, "/dev/sd%c", 
			(minor >> 4) + 'a'); 

	      break;
	    case SCSI_CDROM_MAJOR:
	      sprintf(linkto, "/dev/sr%d", minor);
	      break;
	    case SCSI_TAPE_MAJOR:
	      sprintf(linkto, "/dev/st%d", minor);
	      break;
	    case SCSI_GENERIC_MAJOR:
	      sprintf(linkto, "/dev/sg%d", minor);
	      break;
	    }

	  symlink(linkto, scsidev);

	  /*
	   * Now make sure that the device the symlink points to actually
	   * exists.  If not, then create that device.
	   */
	  status = stat(scsidev, &statbuf);
	  if( status == 0 )
	    {
	      if( statbuf.st_rdev != makedev(major, minor) )
		{
		  fprintf(stderr,
			  "Warning - device %s does not point to expected device\n", linkto);
		}
	    }
	  else
	    {
	      /*
	       * Unable to stat the file.  Assume this is because it did not
	       * exist, so we create it.
	       */
	      newmode = devtype | 
		(major != SCSI_CDROM_MAJOR ? filemode : (filemode & ~0222));
	      status = mknod(linkto, newmode,
			     makedev(major,minor));
	      fprintf(stderr,"Creating %s\n", linkto);
	    }
	  
	  if( major == SCSI_TAPE_MAJOR )
	    {
	      sprintf(scsidev,"/dev/scsi/rs%ch%dc%di%dl%d",
		      devchar,
		      h_id,
		      chan,
		      scsi_id,
		      lun);
	      sprintf(linkto, "/dev/rst%d", minor);
	      symlink(linkto, scsidev);
	      /*
	       * Now make sure that the device the symlink points to actually
	       * exists.  If not, then create that device.
	       */
	      status = stat(scsidev, &statbuf);
	      if( status == 0 )
		{
		  if( statbuf.st_rdev != makedev(major, minor) )
		    {
		      fprintf(stderr,
			      "Warning - device %s does not point to expected device\n", linkto);
		    }
		}
	      else
		{
		  /*
		   * Unable to stat the file.  Assume this is because it did not
		   * exist, so we create it.
		   */
		  newmode = devtype | 
		    (major != SCSI_CDROM_MAJOR ? filemode : (filemode & ~0222));
		  status = mknod(linkto, newmode,
				 makedev(major,minor));
		  fprintf(stderr,"Creating %s\n", linkto);
		}
	    }
	}
      else
	{
	  /*
	   * First test to see if a device already exists.  If so,
	   * compare the device number to make sure that it is correct,
	   */
	  newmode = devtype | 
	    (major != SCSI_CDROM_MAJOR ? filemode : (filemode & ~0222));

	  recreate = 1;
	  uid = gid = -1;
	  status = stat(scsidev, &statbuf);
	  if( status == 0 )
	    {
	      recreate = 0;
	      /*
	       * Make sure we have the correct device type too.
	       */
	      if( (statbuf.st_mode & S_IFMT) != devtype )
		{
		  unlink(scsidev);
		  recreate = 1;
		}
	      /*
	       * Compare the device number.  If something changed, then 
	       * unlink it so that we can recreate it.  Save the mode of
	       * the thing so that we can keep the same permissions.
	       */
	      else if( statbuf.st_rdev != makedev(major, minor) )
		{
		  newmode = statbuf.st_mode;
		  uid = statbuf.st_uid;
		  gid = statbuf.st_gid;
		  unlink(scsidev);
		  recreate = 1;
		}
	    }

	  /*
	   * If we need to recreate the device, then do it.
	   */
	  if( recreate )
	    {
	      status = mknod(scsidev, newmode,
			     makedev(major,minor));
	      if( uid != -1 )
		{
		  chown(scsidev, uid, gid);
		}
	      if( major == SCSI_TAPE_MAJOR )
		{
		  sprintf(scsidev,"/dev/scsi/rs%ch%dc%di%dl%d",
			  devchar,
			  h_id,
			  chan,
			  scsi_id,
			  lun);
		  status = mknod(scsidev, newmode,
				 makedev(major,minor + 128));
		  if( uid != -1 )
		    {
		      chown(scsidev, uid, gid);
		    }
		}
	    }
	}

      unlink("/tmp/testdev");
      minor += 1;
    }
  unlink("/tmp/testdev");
}


/*
 * Build a /dev/scsi tree that contains the correct device entries.
 */

usage()
{
  fprintf(stderr,"Usage: scsidev [-f] [-l]\n");
}

main(int argc, char * argv[])
{
  char c;
  int force = 0;

  while ((c = getopt(argc, argv, "flvm:")) != EOF)
    {
      switch (c)
	{
	case 'f':
	  force = 1;
	  break;
	case 'm':
	  filemode = strtoul(optarg, 0, 0);
	  break;
	case 'l':
	  use_symlink = 1;
	  break;
	case 'v':
	  verbose++;
	  break;
	default:
	  usage();
	  exit(1);
	  break;
	}
    }

  if( force ) flush_sdev();

  build_sdev(SCSI_DISK_MAJOR,    O_RDONLY, S_IFBLK, 'd');
  build_sdev(SCSI_CDROM_MAJOR,   O_RDONLY, S_IFBLK, 'r');
  build_sdev(SCSI_TAPE_MAJOR,    O_RDONLY, S_IFCHR, 't');
  build_sdev(SCSI_GENERIC_MAJOR, O_RDWR,   S_IFCHR, 'g');
}
