// -*- C++ -*-

// Config file handling

// (C) Copyright 1994 Jeremy Fitzhardinge <jeremy@sw.oz.au>
// This code is distributed under the terms of the
// GNU General Public Licence.  See COPYING for more details.

#pragma implementation

#include "ConfFile.h"
#include "pushd.h"

#include <String.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <sys/param.h>
#include <fstream.h>
#include <strstream.h>

#include <string.h>
#include <errno.h>

ConfFile::ConfFile(const Path &path)
	 :name(path)
{
	list = NULL;

	parse();
	changed = 0;
}

ConfFile::~ConfFile()
{
	unparse();
	for(list_t *next; list; list = next)
	{
		next = list->next;
		delete list;
	}
}

void
ConfFile::unparse()
{
	if (!changed)
		return;

	String path = name.path();
	ofstream out(path, ios::out|ios::trunc, 0644);

	if (out.bad())
	{
		cout << "Failed to open config file " << path << "\n";
		return;
	}

	for(list_t *p = list; p != NULL; p = p->next)
		out << p->name <<": "  << p->val << "\n";

	changed = 0;
}

void
ConfFile::parse()
{
	String path = name.path();
	ifstream in(path);

	if (in.bad())
		return;

	String line;
	int ret;
	
	while((ret = readline(in, line)) > 0)
	{
		static const Regex sep_rx("^\\([a-zA-Z_]+\\):[ \t]*\\(.*\\)$");

		if (line[0] == '#')
			continue;

		if (!line.matches(sep_rx))
		{
			cerr << "Bad syntax: \"" << line << "\"\n";
			continue;
		}
		int start, len;
		
		sep_rx.match_info(start, len, 1);
		String n(line.at(start, len));
		sep_rx.match_info(start, len, 2);
		String v(line.at(start, len));

		set(n, v);
	}
}

ConfFile::list_t *
ConfFile::find(const String &name) const
{
	list_t *p;
	
	for(p = list; p != NULL; p = p->next)
		if (name == p->name)
			return p;
	return NULL;
}
	
int
ConfFile::get(const String &name, String &val) const
{
	list_t *p;

	if ((p = find(name)) == NULL)
		return 0;
	val = p->val;
	return 1;
}

void
ConfFile::set(const String &name, const String &val)
{
	list_t *p;

	p = find(name);

	if (p == NULL)
	{
		p = new list_t;
		p->name = name;
		p->next = list;
		list = p;
	}
	p->val = val;

	changed = 1;
}

void
ConfFile::setnum(const String &name, long val)
{
	char buf[32];
	ostrstream str(buf, sizeof(buf));

	str << val << '\0';

	set(name, String(buf));
}

int
ConfFile::getnum(const String &name, long &val) const
{
	String v;
	
	if (!get(name, v))
		return 0;

	val = atol(v);

	return 1;
}
