// -*- C++ -*-

// Base directory class

// (C) Copyright 1994 Jeremy Fitzhardinge <jeremy@sw.oz.au>
// This code is distributed under the terms of the
// GNU General Public Licence.  See COPYING for more details.

#pragma implementation

#include "dir.h"
#include "ftpfs.h"
#include "ftpdir.h"
#include "ftplink.h"
#include "ftpconn.h"
#include "pushd.h"

#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <stdio.h>
#include <string.h>

dir::dir(ftpfs &fs, const String &name, dir *up)
    :DirInode(fs, fs.genhand(), up), sem(1),
     name(name, up ? &up->name : (Path *)NULL), filesys(fs), parent(up)
{
	mode = S_IFDIR | 0555;
	atime = ctime = mtime = time(0);
	uid = fs.uid;
	gid = fs.gid;
}

void
dir::init()
{
	int dfd = open();
	
	if (dfd == -1)
		return;
	
	pushd here(dfd);
	
	DIR *dir;
	if ((dir = opendir(".")) == NULL)
	{
		perror("topdir::topdir opendir of cache failed");
		return;
	}

	struct dirent *de;

	while((de = readdir(dir)) != NULL)
	{
		struct stat st;

		if (lstat(de->d_name, &st) == -1)
		{
			fprintf(stderr, "stat of %s failed: %s\n",
				de->d_name, strerror(errno));
			continue;
		}

		Inode *ino;
		String name(de->d_name);

		if ((ino = newinode(this, &st, name)) == NULL)
			continue;
		
		link(name, ino);
	}
	closedir(dir);
}

int
dir::open()
{
	String p = path();

	int ret = ::open(p, O_RDONLY);

	if (ret == -1)
	{
		int err = errno;
		fprintf(stderr, "open of %s failed: %s\n",
			(const char *)p, strerror(errno));
		errno = err;
	}
	return ret;
}

int
dir::do_iput(const up_preamble &, upp_repl &, const upp_iput_s &)
{
	if (nlink == 0)
		delete this;
	return 0;
}

void
dir::beforeop(up_ops)
{
	sem.wait();
}

void
dir::afterop(up_ops)
{
	sem.signal();
}
